#!/usr/bin/env python3
"""
This script provides a CLI interface with subcommands to run CodeQL static analyser
"""

from pathlib import Path
import sys
import argparse

sys.path.append(str(Path(__file__).parent))

try:
    from tools import create_db, analyze_db, sb_check, parse_csv, filter_csv, delete_db
except ImportError as e:
    print(f"Error in importing tool modules: {e}")
    print(f"Ensure create_db.py and analyze_db.py are in the tools directory")
    sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description="RTEMS CodeQL CLI wrapper",
        formatter_class=argparse.RawDescriptionHelpFormatter)

    # Subparsers for subcommands
    subparsers = parser.add_subparsers(dest="command",
                                       help="Available commands",
                                       required=True,
                                       metavar="<command>")

    # Subparsers for checking if codeql is present in path
    check_codeql_path = subparsers.add_parser(
        "path_check",
        help="Checks if CodeQL is added to path",
        description=
        "Checks if codeql is added to path important for creating and analyzing the database"
    )

    # Subparsers for creating database
    create_db_parser = subparsers.add_parser(
        "create_db",
        help="Create a CodeQL database for the RTEMS source.",
        description=
        "Builds a CodeQL database from the RTEMS source code. Requires a source directory and a build command."
    )
    create_db_parser.add_argument(
        "--rtems-source",
        type=str,
        help="Absolute path to RTEMS source code directory.",
        metavar="<path>")
    create_db_parser.add_argument(
        "--rtems-db-name",
        type=str,
        default="rtems-db",
        help=
        "Name of the CodeQL database to create default 'rtems-db' name is used.",
        metavar="<name>")
    create_db_parser.add_argument(
        "--rtems-build",
        required=True,
        help=
        "Path containing build.sh file to build the RTEMS source (e.g., './build.sh').",
        metavar="<path>")

    # Subparser for analysing database
    analyze_db_parser = subparsers.add_parser(
        "analyze_db",
        help="Analyze an existing CodeQL database.",
        description=
        "Runs queries against a specified CodeQL database and outputs the results to a CSV file."
    )
    analyze_db_parser.add_argument(
        "--rtems-db-path",
        required=True,
        type=str,
        help=
        "Path to CodeQL database directory (eg: rtems-db/cpp for cpp and rtems-db/python)",
        metavar="<path>")
    analyze_db_parser.add_argument(
        "--rtems-report-output",
        required=True,
        type=str,
        help="Output path for the resulting CSV report file.",
        metavar="<path>")
    analyze_db_parser.add_argument(
        "--rtems-query",
        type=str,
        required=True,
        help=
        "Path to a .ql query file or a .qls query suite, or a directory of queries.",
        metavar="<path>")
    analyze_db_parser.add_argument(
        "--rtems-db-analyze-ram",
        type=int,
        default=8000,
        help=
        "RAM in MB to allocate to the CodeQL analyzer (default: %(default)sMB).",
        metavar="<MB>")
    analyze_db_parser.add_argument(
        "--rtems-db-analyze-threads",
        type=int,
        default=4,
        help=
        "Number of threads for the CodeQL analyzer to use (default: %(default)s).",
        metavar="<count>")
    analyze_db_parser.add_argument(
        "--csv-format",
        default="line-column",
        choices=["line-column", "line"],
        help="Format for location in the CSV output (default: %(default)s).")

    # Subparser for deleting database
    deleting_db_parser = subparsers.add_parser(
        "delete_db",
        help="Deletes the database generated by codeql",
    )
    deleting_db_parser.add_argument(
        "--rtems-db-path",
        required=True,
        type=str,
        help="Path to CodeQL database directory (eg: rtems-db/",
        metavar="<path>")

    # Subparser for excluding paths
    excluding_path = subparsers.add_parser(
        "exclude_path",
        help="Filter CSV rows based on glob patterns in specified path column.",
        description=
        "Excludes paths present under 'exclude_patterns.txt' using glob patterns"
    )
    excluding_path.add_argument("--input-csv",
                                required=True,
                                type=str,
                                help="Path to the input CSV file.",
                                metavar="<path>")
    excluding_path.add_argument("--output-csv",
                                required=True,
                                type=str,
                                help="Path to the output CSV file.",
                                metavar="<path>")
    excluding_path.add_argument(
        "--excluding-file",
        required=True,
        help=
        "Path to the text file containing exclusion glob patterns (one per line).",
        metavar="<path>")
    excluding_path.add_argument(
        "--column",
        default="Path",
        metavar="<column>",
        help=
        "The name of the column in the CSV that contains the paths. (default: %(default)s)"
    )

    args = parser.parse_args()

    try:
        # For Creating Database
        if args.command == "create_db":
            print(f"Starting 'create_db' command...")
            db_name = args.rtems_db_name
            source_root = Path(args.rtems_source).expanduser().resolve()
            build_command = Path(args.rtems_build).expanduser().resolve()

            if not source_root.exists():
                print(
                    f"Source root not found for RTEMS codebase {source_root}")
                exit(1)
            elif not build_command.exists():
                print(f"build.sh not found at {build_command}")
                exit(1)
            create_db.create_codeql_database(db_name, source_root,
                                             build_command)
        # For analysing the database
        elif args.command == "analyze_db":
            print(f"Starting 'analyze_db' command...")
            db_path = Path(args.rtems_db_path)
            output_file = Path(args.rtems_report_output)
            query = Path(args.rtems_query)
            ram = args.rtems_db_analyze_ram
            threads = args.rtems_db_analyze_threads
            csv_format = args.csv_format

            success = analyze_db.run_analysis(db_path, output_file, query, ram,
                                              threads, csv_format)
            if success:
                try:
                    original_file_path = output_file
                    print("Adding headers to CSV report...")
                    parse_csv.format_csv(original_file_path)
                    print("Successfully added headers.")
                    # Remove the original file
                    print(f"Removing original file:{original_file_path}")
                    print("Original file removed.")
                except Exception as e:
                    print(f"Failed to format CSV:{e}")
                    sys.exit(1)
            else:
                print("Skipping CSV formatting due to analysis failure.")
                sys.exit(1)

        # For CodeQL path-check
        elif args.command == "path_check":
            print(
                f"Starting 'sb_check' command checking if CodeQL is in path..."
            )
            status = sb_check.is_codeql_in_path()
            if status == True:
                print("CodeQL is set to path")
            else:
                print("CodeQL is not in your path. Please add it manually.")

        # For deleting database
        elif args.command == "delete_db":
            print(f"Starting 'delete_db command...'")
            db_path = Path(args.rtems_db_path)
            delete_db.delete_codeql_db(db_path)

        elif args.command == "exclude_path":
            print(
                f"Starting 'exclude_path' command excluding the paths mentioned in 'exclude_pattern.txt'"
            )
            input_file_path = Path(args.input_csv)
            output_file_path = Path(args.output_csv)
            exclusion_file_path = Path(args.exclude_file)
            path_column_name = args.column

            exclusion_patterns_list = filter_csv.load_exclusion_patterns(
                exclusion_file_path)

            if exclusion_patterns_list:
                filter_csv.exclude_path(input_file_path, output_file_path,
                                        path_column_name,
                                        exclusion_patterns_list)
            else:
                print(
                    "No exclusion patterns loaded or exclusion file not found. Exiting."
                )
        print("\nCommand execution completed successfully")

    except Exception as e:
        print(f"\nAn error occurred during command execution: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
