/* SPDX-License-Identifier: BSD-2-Clause */

/*
 *  COPYRIGHT (c) 2011.
 *  On-Line Applications Research Corporation (OAR).
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <tmacros.h>

#include <rtems/score/threadimpl.h>

const char rtems_test_name[] = "SPSIMPLESCHED 2";

/* forward declarations to avoid warnings */
rtems_task Init(rtems_task_argument argument);
rtems_task Test_task(rtems_task_argument argument);
void ObtainRelease(bool suspendIdle);

/*
 *  Keep the names and IDs in global variables so another task can use them.
 */
rtems_id   Idle_id;
rtems_id   Task_id[ 3 ];         /* array of task ids */
rtems_name Task_name[ 3 ];       /* array of task names */
rtems_name Semaphore_name[ 2 ];
rtems_id   Semaphore_id[ 2 ];

rtems_task Test_task(
  rtems_task_argument unused
)
{
  (void) unused;

  rtems_id          tid;
  rtems_status_code status;

  status = rtems_task_ident( RTEMS_WHO_AM_I, RTEMS_SEARCH_ALL_NODES, &tid );
  directive_failed( status, "wake after" );

  for ( ; ; ) {
    status = rtems_task_wake_after( RTEMS_YIELD_PROCESSOR );
    directive_failed( status, "yield" );
  }
}

void ObtainRelease( bool suspendIdle )
{
  rtems_status_code   status;

  if (suspendIdle) {
    puts( "INIT - Suspend Idle Task");
    status = rtems_task_suspend( Idle_id );
    directive_failed( status, "rtems_task_suspend idle" );
  }

  puts( "INIT - Obtain priority ceiling semaphore - priority increases" );
  status= rtems_semaphore_obtain( Semaphore_id[1], RTEMS_DEFAULT_OPTIONS, 0 );
  directive_failed( status, "rtems_semaphore_obtain" );

  puts( "INIT - Obtain priority ceiling semaphore - priority decreases" );
  status = rtems_semaphore_release( Semaphore_id[1] );
  directive_failed( status, "rtems_semaphore_release" );

  if (suspendIdle) {
    puts( "INIT - Resume Idle Task");
    status = rtems_task_resume( Idle_id );
    directive_failed( status, "rtems_task_resume idle" );
  }
}

rtems_task Init(
  rtems_task_argument argument
)
{
  (void) argument;

  rtems_status_code   status;
  Status_Control      score_status;

  TEST_BEGIN();

  score_status = _Objects_Name_to_id_u32(
    rtems_build_name( 'I', 'D', 'L', 'E' ),
    RTEMS_SEARCH_LOCAL_NODE,
    &Idle_id,
    &_Thread_Information.Objects
  );
  rtems_test_assert( score_status == STATUS_SUCCESSFUL );

  /*
   * Create the semaphore. Then obtain and release the
   * semaphore with no other tasks running.
   */
  puts( "INIT - Create priority ceiling semaphore" );
  Semaphore_name[ 1 ] = rtems_build_name( 'S', 'M', '1', ' ' );
  status = rtems_semaphore_create(
    Semaphore_name[ 1 ],
    1,
    RTEMS_BINARY_SEMAPHORE | RTEMS_PRIORITY_CEILING | RTEMS_PRIORITY,
    2,
    &Semaphore_id[ 1 ]
  );
  directive_failed( status, "rtems_semaphore_create of SM1" );
  ObtainRelease( false );

  /*
   * Create test task and obtain release the semaphore with
   * one other task running.
   */
  puts( "INIT - create task 1" );
  Task_name[ 1 ] = rtems_build_name( 'T', 'A', '1', ' ' );
  status = rtems_task_create(
    Task_name[ 1 ], 1, RTEMS_MINIMUM_STACK_SIZE * 2, RTEMS_DEFAULT_MODES,
    RTEMS_DEFAULT_ATTRIBUTES, &Task_id[ 1 ]
  );
  status = rtems_task_start( Task_id[ 1 ], Test_task, 1 );
  ObtainRelease( false );

  /*
   * Create a a second test task and obtain release the semaphore
   * with both tasks running.
   */
  puts( "INIT - create task 2" );
  Task_name[ 1 ] = rtems_build_name( 'T', 'A', '2', ' ' );
  status = rtems_task_create(
    Task_name[ 2 ], 1, RTEMS_MINIMUM_STACK_SIZE * 2, RTEMS_DEFAULT_MODES,
    RTEMS_DEFAULT_ATTRIBUTES, &Task_id[ 2 ]
  );
  status = rtems_task_start( Task_id[ 2 ], Test_task, 1 );
  ObtainRelease( false );

  /*
   * Obtain and release the semaphore with the idle task suspended.
   */
  ObtainRelease( true );

  /*  End the Test */
  TEST_END();
  rtems_test_exit(0);
}

/* configuration information */

#define CONFIGURE_SCHEDULER_SIMPLE
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_SIMPLE_CONSOLE_DRIVER

#define CONFIGURE_MAXIMUM_TASKS             3
#define CONFIGURE_MAXIMUM_SEMAPHORES        2

#define CONFIGURE_INITIAL_EXTENSIONS RTEMS_TEST_INITIAL_EXTENSION

#define CONFIGURE_RTEMS_INIT_TASKS_TABLE

#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)

#define CONFIGURE_INIT_TASK_INITIAL_MODES   RTEMS_NO_PREEMPT
#define CONFIGURE_INIT_TASK_PRIORITY        4

#define CONFIGURE_INIT
#include <rtems/confdefs.h>
/* end of include file */
