/* SPDX-License-Identifier: BSD-2-Clause */

/**
 * @file
 *
 * @brief PCI IRQ Library
 */

/*
 * COPYRIGHT (c) 2010 Cobham Gaisler AB.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* IRQ handling does not have so much with PCI to do, this library depends
 * on the BSP to implement shared interrupts.
 */

#ifndef __PCI_IRQ_H__
#define __PCI_IRQ_H__

#include <rtems/rtems/intr.h>
#include <rtems/score/basedefs.h>

/* PCI Handler (ISR) called when IRQ is generated by any of the PCI devices
 * connected to the same PCI IRQ Pin. This has been defined the same way as
 * rtems_interrupt_handler in order for BSPs to "direct-map" the register
 * and unregister functions rtems_interrupt_handler_install/remove
 */
typedef void (*pci_isr)(void *arg);

/* Get assigned system IRQ to a PCI Device. If no IRQ 0 is returned */
extern int pci_dev_irq(pci_dev_t dev);

/* Register shared PCI IRQ handler, but does not enable it. The system interrupt
 * number is read from the PCI board's PCI configuration space header iline
 * field. The iline field is initialized by the PCI subsystem during start up,
 * the ipin field is translated into a system IRQ and written to iline. The
 * board's driver should use the iline field as the irq argument to this
 * function.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
static inline int pci_interrupt_register(int irq, const char *info,
						pci_isr isr, void *arg)
{
	return rtems_interrupt_handler_install(irq, info,
					       RTEMS_INTERRUPT_SHARED, isr,
					       arg);
}

/* Unregister previously registered shared PCI IRQ handler
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
static inline int pci_interrupt_unregister(int irq, pci_isr isr,
						  void *arg)
{
	return rtems_interrupt_handler_remove(irq, isr, arg);
}

/* Enable shared PCI IRQ handler. This function will unmask the interrupt
 * controller and mark this interrupt handler ready to handle interrupts. Note
 * that since it is a shared interrupt handler service the interrupt may
 * already be enabled, however no calls to this specific handler is made
 * until it is enabled.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
static inline void pci_interrupt_unmask(int irq)
{
	(void)rtems_interrupt_vector_enable((rtems_vector_number)irq);
}

/* Disable shared PCI IRQ handler. This function will mask the interrupt
 * controller and mark this interrupt handler not ready to receive interrupts.
 * Note that since it is a shared interrupt handler service the interrupt may
 * still be enabled, however no calls to this specific handler is made
 * while it is disabled.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
static inline void pci_interrupt_mask(int irq)
{
	(void)rtems_interrupt_vector_disable((rtems_vector_number)irq);
}

/* Acknowledge the interrupt controller by writing to the interrupt controller.
 * Note that since it is a shared interrupt handler service, clearing the
 * interrupt source may affect other ISRs registered to this IRQ.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
static inline void pci_interrupt_clear(int irq)
{
	(void)rtems_interrupt_clear((rtems_vector_number)irq);
}

#endif /* !__PCI_IRQ_H__ */
