/* SPDX-License-Identifier: GPL-2.0+-with-RTEMS-exception */

/*
 *  This file contains the RBTX4938 console IO package.
 */

/*
 *  Author:     Craig Lebakken <craigl@transition.com>
 *
 *  COPYRIGHT (c) 1996 by Transition Networks Inc.
 *
 *  To anyone who acknowledges that this file is provided "AS IS"
 *  without any express or implied warranty:
 *      permission to use, copy, modify, and distribute this file
 *      for any purpose is hereby granted without fee, provided that
 *      the above copyright notice and this notice appears in all
 *      copies, and that the name of Transition Networks not be used in
 *      advertising or publicity pertaining to distribution of the
 *      software without specific, written prior permission.
 *      Transition Networks makes no representations about the suitability
 *      of this software for any purpose.
 *
 *  Derived from bsps/no_cpu/no_bsp/console/console.c:
 *
 *  COPYRIGHT (c) 1989-1999.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.org/license/LICENSE.
 */

#include <ctype.h>

#include <rtems/console.h>
#include <rtems/libio.h>
#include <bsp.h>

#include "yamon_api.h"

/* PMON entry points */
int mon_read(int fd, char *buf, int cnt);    /* stdin is fd=0 */
int mon_write(int fd, char *buf, int cnt);    /* stdout is fd=1 */


/*  console_initialize
 *
 *  This routine initializes the console IO driver.
 */
rtems_device_driver console_initialize(
  rtems_device_major_number  major,
  rtems_device_minor_number  minor,
  void                      *arg
)
{
  (void) minor;
  (void) arg;

  rtems_status_code status;

  status = rtems_io_register_name(
    "/dev/console",
    major,
    (rtems_device_minor_number) 0
  );

  if (status != RTEMS_SUCCESSFUL)
    rtems_fatal_error_occurred(status);

  return RTEMS_SUCCESSFUL;
}

/*  inbyte
 *
 *  This routine reads a character from the SOURCE.
 */
static char inbyte( void )
{
  char buf[10];

  /*
   *  If polling, wait until a character is available.
   */
  while (YAMON_FUNC_GETCHAR(buf) == YAMON_FALSE);

  return (buf[0]);
}

/*  outbyte
 *
 *  This routine transmits a character out the SOURCE.  It may support
 *  XON/XOFF flow control.
 */
static void outbyte(
  char ch
)
{
  char buf[10];

  /*
   *  If polling, wait for the transmitter to be ready.
   *  Check for flow control requests and process.
   *  Then output the character.
   */
  buf[0] = ch;

  YAMON_FUNC_PRINT_COUNT(buf,1);
}

/*
 *  Open entry point
 */

rtems_device_driver console_open(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void                    * arg
)
{
  (void) major;
  (void) minor;
  (void) arg;

  return RTEMS_SUCCESSFUL;
}

/*
 *  Close entry point
 */
rtems_device_driver console_close(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void                    * arg
)
{
  (void) major;
  (void) minor;
  (void) arg;

  return RTEMS_SUCCESSFUL;
}

/*
 * read bytes from the serial port. We only have stdin.
 */
rtems_device_driver console_read(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void                    * arg
)
{
  (void) major;
  (void) minor;

  rtems_libio_rw_args_t *rw_args;
  char *buffer;
  int maximum;
  int count = 0;

  rw_args = (rtems_libio_rw_args_t *) arg;

  buffer = rw_args->buffer;
  maximum = rw_args->count;

  for (count = 0; count < maximum; count++) {
    buffer[ count ] = inbyte();
    if (buffer[ count ] == '\n' || buffer[ count ] == '\r') {
      buffer[ count++ ]  = '\n';
      break;
    }
  }

  rw_args->bytes_moved = count;
  return (count >= 0) ? RTEMS_SUCCESSFUL : RTEMS_UNSATISFIED;
}

/*
 * write bytes to the serial port. Stdout and stderr are the same.
 */
rtems_device_driver console_write(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void                    * arg
)
{
  (void) major;
  (void) minor;

  int count;
  int maximum;
  rtems_libio_rw_args_t *rw_args;
  char *buffer;

  rw_args = (rtems_libio_rw_args_t *) arg;

  buffer = rw_args->buffer;
  maximum = rw_args->count;

  for (count = 0; count < maximum; count++) {
    if ( buffer[ count ] == '\n') {
      outbyte('\r');
    }
    outbyte( buffer[ count ] );
  }

  rw_args->bytes_moved = maximum;
  return 0;
}

/*
 *  IO Control entry point
 */
rtems_device_driver console_control(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void                    * arg
)
{
  (void) major;
  (void) minor;
  (void) arg;

  return RTEMS_SUCCESSFUL;
}

#include <rtems/bspIo.h>

static void RBTX4938_output_char(char c) { outbyte( c ); }

BSP_output_char_function_type           BSP_output_char = RBTX4938_output_char;
BSP_polling_getchar_function_type       BSP_poll_char = NULL;

