/*
 * QEMU model of the AXIQSPI Memory Map for AXI QSPI
 *
 * Copyright (c) 2020 Xilinx Inc.
 *
 * Partially autogenerated by xregqemu.py 2019-11-21.
 * Written by: Joe Komlodi<komlodi@xilinx.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "qemu/osdep.h"
#include "hw/sysbus.h"
#include "hw/register.h"
#include "qemu/bitops.h"
#include "qemu/log.h"
#include "qemu/fifo.h"
#include "qapi/error.h"
#include "qemu/error-report.h"
#include "hw/irq.h"
#include "migration/vmstate.h"
#include "hw/qdev-properties.h"
#include "hw/dma-ctrl.h"

#include "sysemu/dma.h"
#include "hw/ssi/ssi.h"
#include "hw/fdt_generic_util.h"

#define TYPE_XLNX_AXIQSPI "xlnx,axi-quad-spi-3.2"

#define XLNX_AXIQSPI(obj) \
     OBJECT_CHECK(XlnxAXIQSPI, (obj), TYPE_XLNX_AXIQSPI)

#ifndef XLNX_AXIQSPI_ERR_DEBUG
#define XLNX_AXIQSPI_ERR_DEBUG 0
#endif

#define DB_PRINT(...) do { \
    if (XLNX_AXIQSPI_ERR_DEBUG) { \
        qemu_log(__VA_ARGS__); \
    } \
} while (0)

REG32(DGIER, 0x1c)
    FIELD(DGIER, GIE, 31, 1)
REG32(IPISR, 0x20)
    FIELD(IPISR, COMMAND_ERROR, 13, 1)
    FIELD(IPISR, LOOPBACK_ERROR, 12, 1)
    FIELD(IPISR, MSB_ERROR, 11, 1)
    FIELD(IPISR, SLAVE_MODE_ERROR, 10, 1)
    FIELD(IPISR, CPOL_CPHA_ERROR, 9, 1)
    FIELD(IPISR, DRR_NOT_EMPTY, 8, 1)
    FIELD(IPISR, SLAVE_SELECT_MODE, 7, 1)
    FIELD(IPISR, TXFIFO_HALF_EMPTY, 6, 1)
    FIELD(IPISR, DRR_OVERRUN, 5, 1)
    FIELD(IPISR, DRR_FULL, 4, 1)
    FIELD(IPISR, DTR_UNDERRUN, 3, 1)
    FIELD(IPISR, DTR_EMPTY, 2, 1)
    FIELD(IPISR, SLAVE_MODF, 1, 1)
    FIELD(IPISR, MODF, 0, 1)
REG32(IPIER, 0x28)
    FIELD(IPIER, COMMAND_ERROR, 13, 1)
    FIELD(IPIER, LOOPBACK_ERROR, 12, 1)
    FIELD(IPIER, MSB_ERROR, 11, 1)
    FIELD(IPIER, SLAVE_MODE_ERROR, 10, 1)
    FIELD(IPIER, CPOL_CPHA_ERROR, 9, 1)
    FIELD(IPIER, DRR_NOT_EMPTY, 8, 1)
    FIELD(IPIER, SLAVE_SELECT_MODE, 7, 1)
    FIELD(IPIER, TX_FIFO_HALF_EMPTY, 6, 1)
    FIELD(IPIER, DRR_OVERRUN, 5, 1)
    FIELD(IPIER, DRR_FULL, 4, 1)
    FIELD(IPIER, DTR_UNDERRUN, 3, 1)
    FIELD(IPIER, DTR_EMPTY, 2, 1)
    FIELD(IPIER, SLAVE_MODF, 1, 1)
    FIELD(IPIER, MODF, 0, 1)
REG32(SRR, 0x40)
REG32(SPICR, 0x60)
    FIELD(SPICR, LSB_FIRST, 9, 1)
    FIELD(SPICR, MASTER_TRANSACTION_INHIBIT, 8, 1)
    FIELD(SPICR, MANUAL_SLAVE_SELECT_ASSERTION_ENABLE, 7, 1)
    FIELD(SPICR, RX_FIFO_RESET, 6, 1)
    FIELD(SPICR, TX_FIFO_RESET, 5, 1)
    FIELD(SPICR, CPHA, 4, 1)
    FIELD(SPICR, CPOL, 3, 1)
    FIELD(SPICR, MASTER, 2, 1)
    FIELD(SPICR, SPE, 1, 1)
    FIELD(SPICR, LOOP, 0, 1)
REG32(SPISR, 0x64)
    FIELD(SPISR, COMMAND_ERROR, 10, 1)
    FIELD(SPISR, LOOPBACK_ERROR, 9, 1)
    FIELD(SPISR, MSB_ERROR, 8, 1)
    FIELD(SPISR, SLAVE_MODE_ERROR, 7, 1)
    FIELD(SPISR, CPOL_CPHA_ERROR, 6, 1)
    FIELD(SPISR, SLAVE_MODE_SELECT, 5, 1)
    FIELD(SPISR, MODF, 4, 1)
    FIELD(SPISR, TX_FULL, 3, 1)
    FIELD(SPISR, TX_EMPTY, 2, 1)
    FIELD(SPISR, RX_FULL, 1, 1)
    FIELD(SPISR, RX_EMPTY, 0, 1)
REG32(SPI_DTR, 0x68)
REG32(SPI_DRR, 0x6c)
REG32(SPI_SSR, 0x70)
REG32(SPI_TXFIFO_OR, 0x74)
REG32(SPI_RXFIFO_OR, 0x78)

#define XLNX_AXIQSPI_R_MAX (R_SPI_RXFIFO_OR + 1)

/* These registers are only used if AXIQSPI is in XIP mode */
REG32(XIP_CONFIG_REG, 0x60)
    FIELD(XIP_CONFIG_REG, CPOL, 1, 1)
    FIELD(XIP_CONFIG_REG, CPHA, 0, 1)
REG32(XIP_STATUS_REG, 0x64)
    FIELD(XIP_STATUS_REG, AXI_TRANSACTION_ERROR, 4, 1)
    FIELD(XIP_STATUS_REG, CPOL_CPHA_ERROR, 3, 1)
    FIELD(XIP_STATUS_REG, MASTER_MODF, 2, 1)
    FIELD(XIP_STATUS_REG, RX_FULL, 1, 1)
    FIELD(XIP_STATUS_REG, RX_EMPTY, 0, 1)

#define XLNX_AXIQSPI_XIP_R_MAX (R_XIP_STATUS_REG + 1)

#define AXIQSPI_NUM_CS_MAX 32
#define AXIQSPI_NUM_CS_MIN 1
#define AXIQSPI_SRR_RESET_VAL 0x0000000A

/*
 * Extended addressing bits for relevant flash chips.
 * These are used when we need to check if we're changing between
 * 3- and 4-byte addressing.
 */
#define SP_EXTADDR_BIT 0x80
#define MI_EXTADDR_BIT 0x01
#define MA_EXTADDR_BIT 0x20

#define MAKE_32BIT_MASK(len) (uint32_t)MAKE_64BIT_MASK(0, len)

/* SPI mode values match those used in the DTS */
typedef enum {
    AXIQSPI_MODE_STD = 0,
    AXIQSPI_MODE_DUAL = 1,
    AXIQSPI_MODE_QUAD = 2,
    AXIQSPI_MODE_INVALID = 3
} AXIQSPIMode;

/* SPI memory values match those used in the DTS */
typedef enum {
    SPI_MEM_MIXED = 0,
    SPI_MEM_WINBOND = 1,
    SPI_MEM_MICRON = 2,
    SPI_MEM_SPANSION = 3,
    SPI_MEM_MACRONIX = 4,
    SPI_MEM_INVALID = 5
} SPIMemType;

/* Commands are taken from the AXIQSPI support command list */
typedef enum {
    NOP = 0x00,
    WRITE_STATUS_REG = 0x01,
    PAGE_PROGRAM = 0x02,
    READ_DATA = 0x03,
    WRITE_DISABLE = 0x04,
    READ_STATUS_REG = 0x05,
    WRITE_ENABLE = 0x06,
    READ_STATUS_REG_2 = 0x07,
    FAST_READ = 0x0B,
    FAST_READ_4B = 0x0C,
    PAGE_PROGRAM_4B = 0x12,
    READ_4B = 0x13,
    AUTOBOOT_REG = 0x15,
    BANK_REGISTER_READ = 0x16,
    BANK_REGISTER_WRITE = 0x17,
    ERASE_FAST_BOOT_REGISTER = 0x18,
    SUBSECTOR_ERASE = 0x20,
    ERASE_4B = 0x21,
    READ_PASSWORD_4B = 0x27,
    WRITE_PASSWORD = 0x28,
    UNLOCK_PASSWORD = 0x29,
    READ_SECURITY_REGISTER = 0x2B,
    WRITE_LOCK_REG = 0x2C,
    READ_LOCK_REG = 0x2D,
    WRITE_SECURITY_REGISTER = 0x2F,
    CLEAR_STATUS_REG_1 = 0x30,
    WRITE_STATUS_REG_2 = 0x31,
    QUAD_PAGE_PROGRAM = 0x32,
    QUAD_PAGE_PROGRAM_4B = 0x34,
    READ_CONF_REG = 0x35,
    MI_ENTER_QUAD_IO_MODE = 0x35,
    QUAD_INPUT_FAST_IO = 0x38,
    FAST_READ_DUAL = 0x3B,
    FAST_READ_DUAL_4B = 0x3C,
    QUAD_PAGE_PROGRAM_IO_4B = 0x3E,
    PROGRAM_OTP = 0x42,
    READ_UNIQUE_ID = 0x4B,
    SP_OTP_READ = 0x4B,
    MI_OTP_READ = 0x4B,
    CLEAR_STATUS_REG = 0x50,
    ERASE_32K = 0x52,
    READ_FLASH_DISCOVERABLE_PARAMS = 0x5A,
    SUBSECTOR_ERASE_4B = 0x5C,
    BULK_ERASE = 0x60,
    WRITE_VOLATILE_ENH_CONFIG_REG = 0x61,
    READ_VOLATILE_ENH_CONFIG_REG = 0x65,
    RESET_ENABLE = 0x66,
    WRITE_PROTECT_SEL = 0x68,
    FAST_READ_QUAD = 0x6B,
    READ_QUAD_4B = 0x6C,
    READ_FLAG_STATUS_REG = 0x70,
    ERASE_SUSPEND = 0x75,
    ERASE_RESUME = 0x7A,
    GANG_BLOCK_LOCK = 0x7E,
    WRITE_VOLATILE_CONFIG_REG = 0x81,
    READ_VOLATILE_CONFIG_REG = 0x85,
    PROGRAM_SUSPEND = 0x85,
    PROGRAM_RESUME = 0x8A,
    READ_DEVICE_ID = 0x90,
    GANG_BLOCK_UNLOCK = 0x98,
    RESET_MEMORY = 0x99,
    READ_JEDEC_ID = 0x9F,
    DUAL_FAST_PROGRAM = 0xA2,
    HIGH_PERFORMANCE_MODE = 0xA3,
    WRITE_GLOBAL_FREEZE_BIT = 0xA6,
    PPB_LOCK_BIT_READ = 0xA7,
    READ_GLOBAL_FREEZE_BIT = 0xA7,
    PPB_LOCK_BIT_WRITE = 0xA8,
    EXIT_HIGH_PERFORMANCE_MODE = 0xAB,
    READ_ELECTRONIC_SIGNATURE = 0xAB,
    EXIT_POWER_DOWN = 0xAB,
    QUAD_IO_READ_ID = 0xAF,
    PROGRAM_ERASE_SUSPEND = 0xB0,
    WRITE_NONVOLATILE_CONFIG_REG = 0xB1,
    READ_NONVOLATILE_CONFIG_REG = 0xB5,
    ENTER_4B_ADDR_MODE = 0xB7,
    POWER_DOWN = 0xB9,
    BANK_REGISTER_ACCESS = 0xB9,
    FAST_READ_DUAL_IO = 0xBB,
    FAST_READ_DUAL_IO_4B = 0xBC,
    SET_BURST_READ_LENGTH = 0xC0,
    EXIT_SECURE_OTP = 0xC1,
    DIE_ERASE = 0xC4,
    WRITE_EXTENDED_ADDR_REG = 0xC5,
    READ_EXTENDED_ADDR_REG = 0xC8,
    CHIP_ERASE = 0xC7,
    DUAL_EXTENDED_FAST_PROGRAM = 0xD2,
    BLOCK_ERASE = 0xD8,
    BLOCK_ERASE_4B = 0xDC,
    DYB_READ = 0xE0,
    WRITE_DATA_PROTECTION_REG_BITS = 0xE0,
    READ_VOLATILE_LOCK_4B = 0xE0,
    DYB_WRITE = 0xE1,
    WRITE_VOLATILE_LOCK_4B = 0xE1,
    PPB_READ = 0xE2,
    READ_NONVOLATILE_LOCK = 0xE2,
    WRITE_DATA_PROTECTION_REG = 0xE2,
    PPB_PROGRAM = 0xE3,
    WRITE_NONVOLATILE_LOCK = 0xE3,
    OCTAL_WORD_READ_QUAD_IO = 0xE3,
    PPB_ERASE = 0xE4,
    ERASE_NONVOLATILE_LOCK = 0xE4,
    WRITE_VOLATILE_LOCK_REG = 0xE5,
    PASSWORD_READ = 0xE7,
    PASSWORD_PROGRAM = 0xE8,
    PASSWORD_UNLOCK = 0xE9,
    EXIT_4B_ADDR_MODE = 0xE9,
    FAST_READ_QUAD_IO = 0xEB,
    FAST_READ_QUAD_IO_4B = 0xEC,
    RESET = 0xF0,
    EXIT_QUAD_IO_MODE = 0xF5,
    MODE_BIT_RESET = 0xFF
} FlashCmd;

typedef enum {
    AXIQSPI_STATE_STOPPED = 0,
    AXIQSPI_STATE_TX_CMD = 1,
    AXIQSPI_STATE_TX_ADDR = 2,
    AXIQSPI_STATE_TX_DUMMIES = 3,
    AXIQSPI_STATE_TX = 4
} AXIQSPIState;

typedef struct XlnxAXIQSPIConf {
    bool xip_mode;
    uint16_t fifo_depth;
    uint8_t transaction_width;
    uint32_t tx_width_mask;
    uint8_t spi_mem;
    uint8_t mode;
    uint8_t num_cs;
    uint32_t num_cs_mask;
    uint8_t xip_addr_bits;
    uint64_t xip_base;
} XlnxAXIQSPIConf;

typedef enum {
    LINK_STATE_SINGLE = 1,
    LINK_STATE_DUAL = 2,
    LINK_STATE_QUAD = 4,
} AXIQSPILinkState;

typedef struct XlnxAXIQSPI {
    SysBusDevice parent_obj;
    MemoryRegion iomem;
    XlnxAXIQSPIConf conf;
    Fifo tx_fifo;
    Fifo rx_fifo;
    SSIBus *spi_bus;
    qemu_irq *cs_lines;
    qemu_irq irq;

    AXIQSPIState state;
    AXIQSPILinkState link_state;
    uint8_t addr_bytes_txed;
    uint8_t dummy_bytes_txed;
    size_t bytes_txed;
    uint8_t cmd;
    uint8_t addr_bytes;
    uint8_t num_dummies;
    bool is_addr_change_cmd;
    bool is_4b_addressing;
    bool is_xip_wb_init;
    uint32_t prev_ss;

    MemoryRegion xip_mr;

    uint32_t regs[XLNX_AXIQSPI_R_MAX];
    RegisterInfo reg_info[XLNX_AXIQSPI_R_MAX];
} XlnxAXIQSPI;

static inline uint8_t axiqspi_get_addressing(const XlnxAXIQSPI *s)
{
    if (s->conf.xip_mode) {
        return s->conf.xip_addr_bits == 32 ? 4 : 3;
    } else {
        return s->is_4b_addressing ? 4 : 3;
    }
}

static bool mixed_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case WRITE_STATUS_REG:
    case READ_STATUS_REG:
    case WRITE_DISABLE:
    case WRITE_ENABLE:
    case ERASE_SUSPEND:
    case ERASE_RESUME:
    case READ_UNIQUE_ID:
    case READ_JEDEC_ID:
    case CHIP_ERASE:
        s->addr_bytes = 0;
        s->num_dummies = 0;

        return true;
    case PAGE_PROGRAM:
    case READ_DATA:
    case QUAD_PAGE_PROGRAM:
    case BLOCK_ERASE:
        s->addr_bytes = 3;
        s->num_dummies = 0;

        return true;
    default:
        return false;
    }
}

/*
 * Not the same as mixed mode command parsing.
 * This condenses commands shared between flash chips to avoid
 * repeating ourselves.
 *
 * Returns false if the command wasn't a shared command.
 */
static bool shared_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case READ_JEDEC_ID:
    case WRITE_ENABLE:
    case WRITE_DISABLE:
    case CHIP_ERASE:
    case BULK_ERASE:
    case READ_STATUS_REG:
    case READ_CONF_REG:
    case WRITE_STATUS_REG:
        s->addr_bytes = 0;
        s->num_dummies = 0;

        return true;
    case QUAD_PAGE_PROGRAM:
        s->addr_bytes = axiqspi_get_addressing(s);
        s->num_dummies = 0;

        return true;
    case FAST_READ:
        s->addr_bytes = axiqspi_get_addressing(s);
        s->num_dummies = 1;

        return true;
    case FAST_READ_DUAL:
    case FAST_READ_DUAL_IO:
        s->addr_bytes = axiqspi_get_addressing(s);
        s->num_dummies = 2;

        return true;
    case FAST_READ_QUAD:
    case FAST_READ_QUAD_IO:
        s->addr_bytes = axiqspi_get_addressing(s);
        s->num_dummies = 4;

        return true;
    case PAGE_PROGRAM:
    case READ_DATA:
    case BLOCK_ERASE:
        s->addr_bytes = axiqspi_get_addressing(s);
        s->num_dummies = 0;

        return true;
    case READ_UNIQUE_ID:
    case ERASE_RESUME:
    case ERASE_SUSPEND:
        qemu_log_mask(LOG_UNIMP, "Command %x not implemented\n", s->cmd);

        return false;
    default:
        return false;
    }
}

static bool winbond_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case WRITE_STATUS_REG:
    case EXIT_HIGH_PERFORMANCE_MODE:
        s->addr_bytes = 0;
        s->num_dummies = 0;

        return true;
    case ERASE_32K:
    case SUBSECTOR_ERASE:
    case READ_DEVICE_ID:
    case HIGH_PERFORMANCE_MODE:
        s->addr_bytes = 3;
        s->num_dummies = 0;

        return true;
    case READ_STATUS_REG_2:
    case WRITE_STATUS_REG_2:
    case POWER_DOWN:
        qemu_log_mask(LOG_UNIMP, "Command %x not implemented\n", s->cmd);

        return false;
    default:
        return false;
    }
}

static bool spansion_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case WRITE_STATUS_REG:
    case READ_CONF_REG:
    case BANK_REGISTER_ACCESS:
    case RESET:
    case MODE_BIT_RESET:
        s->addr_bytes = 0;
        s->num_dummies = 0;

        return true;
    case BANK_REGISTER_WRITE:
        s->is_addr_change_cmd = true;
        s->addr_bytes = 0;
        s->num_dummies = 0;

        return true;
    case READ_DEVICE_ID:
        s->addr_bytes = 3;
        s->num_dummies = 0;

        return true;
    case FAST_READ_4B:
        s->addr_bytes = 4;
        s->num_dummies = 1;

        return true;
    case FAST_READ_DUAL_4B:
    case FAST_READ_DUAL_IO_4B:
        s->addr_bytes = 4;
        s->num_dummies = 2;

        return true;
    case READ_QUAD_4B:
        s->addr_bytes = 4;
        s->num_dummies = 4;

        return true;
    case PAGE_PROGRAM_4B:
    case READ_4B:
    case BLOCK_ERASE_4B:
    case QUAD_PAGE_PROGRAM_4B:
        s->addr_bytes = 4;
        s->num_dummies = 0;

        return true;
    case READ_ELECTRONIC_SIGNATURE:
        s->addr_bytes = 0;
        s->num_dummies = 3;

        return true;
    case CLEAR_STATUS_REG:
    case BANK_REGISTER_READ:
    case READ_SECURITY_REGISTER:
    case WRITE_SECURITY_REGISTER:
    case PPB_LOCK_BIT_READ:
    case PPB_LOCK_BIT_WRITE:
    case PPB_ERASE:
    case PASSWORD_READ:
    case PASSWORD_PROGRAM:
    case PASSWORD_UNLOCK:
    case PROGRAM_SUSPEND:
    case PROGRAM_RESUME:
    case DYB_READ:
    case DYB_WRITE:
    case SP_OTP_READ:
    case PROGRAM_OTP:
    case PPB_READ:
    case PPB_PROGRAM:
    case READ_FLASH_DISCOVERABLE_PARAMS:
        qemu_log_mask(LOG_UNIMP, "Command %x not implemented\n", s->cmd);

        return false;
    default:
        return false;
    }
}

static bool micron_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case WRITE_STATUS_REG:
    case READ_NONVOLATILE_CONFIG_REG:
    case READ_VOLATILE_CONFIG_REG:
    case WRITE_VOLATILE_CONFIG_REG:
    case READ_VOLATILE_ENH_CONFIG_REG:
    case WRITE_VOLATILE_ENH_CONFIG_REG:
    case READ_FLAG_STATUS_REG:
    case MI_ENTER_QUAD_IO_MODE:
    case EXIT_QUAD_IO_MODE:
    case ERASE_32K:
    case RESET_MEMORY:
        s->num_dummies = 0;
        s->addr_bytes = 0;

        return true;
    case WRITE_NONVOLATILE_CONFIG_REG:
        s->is_addr_change_cmd = true;
        s->num_dummies = 0;
        s->addr_bytes = 0;

        return true;
    case DUAL_FAST_PROGRAM:
        s->num_dummies = 0;
        s->addr_bytes = axiqspi_get_addressing(s);

        return true;
    case DIE_ERASE:
    case SUBSECTOR_ERASE:
        s->num_dummies = 0;
        s->addr_bytes = axiqspi_get_addressing(s);

        return true;
    case FAST_READ_4B:
        s->num_dummies = 1;
        s->addr_bytes = 4;

        return true;
    case FAST_READ_DUAL_4B:
    case FAST_READ_DUAL_IO_4B:
        s->num_dummies = 2;
        s->addr_bytes = 4;

        return true;
    case READ_QUAD_4B:
        s->num_dummies = 4;
        s->addr_bytes = 4;

        return true;
    case PAGE_PROGRAM_4B:
    case QUAD_PAGE_PROGRAM_4B:
    case QUAD_PAGE_PROGRAM_IO_4B:
    case BLOCK_ERASE_4B:
    case ERASE_4B:
    case SUBSECTOR_ERASE_4B:
    case READ_4B:
        s->num_dummies = 0;
        s->addr_bytes = 4;

        return true;
    case QUAD_IO_READ_ID:
    case ERASE_NONVOLATILE_LOCK:
    case PASSWORD_PROGRAM:
    case WRITE_VOLATILE_LOCK_REG:
    case CLEAR_STATUS_REG:
    case WRITE_EXTENDED_ADDR_REG:
    case READ_EXTENDED_ADDR_REG:
    case POWER_DOWN:
    case EXIT_POWER_DOWN:
    case READ_GLOBAL_FREEZE_BIT:
    case WRITE_GLOBAL_FREEZE_BIT:
    case WRITE_PASSWORD:
    case UNLOCK_PASSWORD:
    case READ_LOCK_REG:
    case WRITE_LOCK_REG:
    case RESET_ENABLE:
    case ENTER_4B_ADDR_MODE:
    case EXIT_4B_ADDR_MODE:
    case READ_FLASH_DISCOVERABLE_PARAMS:
    case MI_OTP_READ:
    case DUAL_EXTENDED_FAST_PROGRAM:
    case QUAD_INPUT_FAST_IO:
    case PROGRAM_OTP:
    case READ_NONVOLATILE_LOCK:
    case WRITE_NONVOLATILE_LOCK:
    case READ_VOLATILE_LOCK_4B:
    case WRITE_VOLATILE_LOCK_4B:
    case READ_PASSWORD_4B:
        qemu_log_mask(LOG_UNIMP, "Command %x not implemented\n", s->cmd);

        return false;
    default:
        return false;
    }
}

static bool macronix_parse_cmd(XlnxAXIQSPI *s)
{
    switch (s->cmd) {
    case RESET_MEMORY:
    case READ_VOLATILE_CONFIG_REG:
    case EXIT_QUAD_IO_MODE:
    case BANK_REGISTER_READ:
    case BANK_REGISTER_WRITE:
        s->num_dummies = 0;
        s->addr_bytes = 0;

        return true;
    case WRITE_STATUS_REG:
        s->is_addr_change_cmd = true;
        s->num_dummies = 0;
        s->addr_bytes = 0;

        return true;
    case READ_4B:
    case FAST_READ_DUAL_IO_4B:
    case READ_QUAD_4B:
    case PAGE_PROGRAM_4B:
    case QUAD_PAGE_PROGRAM_IO_4B:
    case ERASE_4B:
        s->num_dummies = 0;
        s->addr_bytes = 4;

        return true;
    case PAGE_PROGRAM:
    case READ_DEVICE_ID:
    case READ_DATA:
    case QUAD_PAGE_PROGRAM:
    case ERASE_32K:
    case SUBSECTOR_ERASE_4B:
    case BLOCK_ERASE:
    case SUBSECTOR_ERASE:
    case BLOCK_ERASE_4B:
        s->num_dummies = 0;
        s->addr_bytes = axiqspi_get_addressing(s);

        return true;
    case FAST_READ_4B:
        s->num_dummies = 1;
        s->addr_bytes = 4;

        return true;
    case FAST_READ_DUAL_4B:
        s->num_dummies = 2;
        s->addr_bytes = 4;

        return true;
    case QUAD_IO_READ_ID:
    case WRITE_EXTENDED_ADDR_REG:
    case READ_EXTENDED_ADDR_REG:
    case RESET_ENABLE:
    case PROGRAM_OTP:
    case PASSWORD_PROGRAM:
    case READ_LOCK_REG:
    case WRITE_LOCK_REG:
    case READ_STATUS_REG_2:
    case POWER_DOWN:
    case CLEAR_STATUS_REG_1:
    case AUTOBOOT_REG:
    case WRITE_PASSWORD:
    case UNLOCK_PASSWORD:
    case SET_BURST_READ_LENGTH:
    case ERASE_FAST_BOOT_REGISTER:
    case PROGRAM_ERASE_SUSPEND:
    case WRITE_PROTECT_SEL:
    case EXIT_SECURE_OTP:
    case GANG_BLOCK_LOCK:
    case GANG_BLOCK_UNLOCK:
    case WRITE_DATA_PROTECTION_REG_BITS:
    case WRITE_DATA_PROTECTION_REG:
    case PPB_ERASE:
    case READ_SECURITY_REGISTER:
    case WRITE_SECURITY_REGISTER:
    case ENTER_4B_ADDR_MODE:
    case EXIT_4B_ADDR_MODE:
    case READ_FLASH_DISCOVERABLE_PARAMS:
    case READ_PASSWORD_4B:
    case OCTAL_WORD_READ_QUAD_IO:
    case QUAD_INPUT_FAST_IO:
        qemu_log_mask(LOG_UNIMP, "Command %x not implemented\n", s->cmd);

        return false;
    default:
        return false;
    }
}

/* Retrives the dummy byte link state from a command for all flash chips */
static uint8_t shared_get_link_state(uint8_t cmd)
{
    switch (cmd) {
    case QUAD_PAGE_PROGRAM:
    case FAST_READ_QUAD:
    case FAST_READ_QUAD_IO:
    case READ_QUAD_4B:
    case QUAD_PAGE_PROGRAM_4B:
    case QUAD_IO_READ_ID:
    case QUAD_INPUT_FAST_IO:
    case QUAD_PAGE_PROGRAM_IO_4B:
    case OCTAL_WORD_READ_QUAD_IO:
        return LINK_STATE_QUAD;
    case FAST_READ_DUAL:
    case FAST_READ_DUAL_IO:
    case FAST_READ_DUAL_IO_4B:
    case FAST_READ_DUAL_4B:
    case DUAL_FAST_PROGRAM:
    case DUAL_EXTENDED_FAST_PROGRAM:
        return LINK_STATE_DUAL;
    default:
        return LINK_STATE_SINGLE;
    }
}

static bool axiqspi_parse_cmd(XlnxAXIQSPI *s, uint8_t cmd)
{
    bool found;

    /* Standard, non-XIP mode doesn't care about commands */
    if (s->conf.mode == AXIQSPI_MODE_STD && !s->conf.xip_mode) {
        s->num_dummies = 0;
        s->link_state = LINK_STATE_SINGLE;
        s->addr_bytes = 0;
        return true;
    }

    s->is_addr_change_cmd = false;
    s->cmd = cmd;

    /* Shared command handling */
    if (shared_parse_cmd(s)) {
        s->link_state = shared_get_link_state(cmd);
        return true;
    }

    /* Individual flash handling */
    switch (s->conf.spi_mem) {
    case SPI_MEM_WINBOND:
        found = winbond_parse_cmd(s);
        break;
    case SPI_MEM_SPANSION:
        found = spansion_parse_cmd(s);
        break;
    case SPI_MEM_MICRON:
        found = micron_parse_cmd(s);
        break;
    case SPI_MEM_MACRONIX:
        found = macronix_parse_cmd(s);
        break;
    case SPI_MEM_MIXED:
        found = mixed_parse_cmd(s);
        break;
    default:
        g_assert_not_reached();
    }

    if (found) {
        s->link_state = shared_get_link_state(cmd);
    }

    return found;
}

static inline bool axiqspi_tx_fifo_half_empty(XlnxAXIQSPI *s)
{
    if (!s->conf.fifo_depth) {
        return false;
    }

    return (fifo_num_used(&s->tx_fifo) == s->conf.fifo_depth / 2);
}

static bool axiqspi_transaction_width_ok(const XlnxAXIQSPI *s, uint32_t val)
{
    if (s->conf.transaction_width == 32) {
        return true;
    }

    /*
     * If any of the bits fall within this mask, those bits
     * fall within the reserved bits of the register
     */
    return !(val & ~s->conf.tx_width_mask);
}

static uint32_t axiqspi_rx_pop(XlnxAXIQSPI *s)
{
    uint32_t data;

    if (ARRAY_FIELD_EX32(s->regs, SPISR, RX_EMPTY)) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Attempted to read SPI_DRR "
                      "when it was empty\n");
        return 0;
    }

    if (s->conf.fifo_depth) {
        if (ARRAY_FIELD_EX32(s->regs, SPISR, RX_FULL)) {
            ARRAY_FIELD_DP32(s->regs, SPISR, RX_FULL, 0);
        }

        data = fifo_pop32(&s->rx_fifo);
        if (fifo_is_empty(&s->rx_fifo)) {
            ARRAY_FIELD_DP32(s->regs, SPISR, RX_EMPTY, 1);
        }
    } else {
        data = s->regs[R_SPI_DRR];

        ARRAY_FIELD_DP32(s->regs, SPISR, RX_EMPTY, 1);
        ARRAY_FIELD_DP32(s->regs, SPISR, RX_FULL, 0);
    }

    return data;
}

static void axiqspi_update_irq(XlnxAXIQSPI *s)
{
    bool gie = ARRAY_FIELD_EX32(s->regs, DGIER, GIE);
    bool irq = (s->regs[R_IPISR] & s->regs[R_IPIER]);

    qemu_set_irq(s->irq, irq && gie);
}

static uint64_t axiqspi_spi_drr_post_read(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = axiqspi_rx_pop(s);

    return val;
}

static uint64_t axiqspi_ipisr_pre_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = (uint64_t) val64;

    /* This field should not be written to if we do not have FIFOs */
    if (!s->conf.fifo_depth && FIELD_EX64(val64, IPISR, TXFIFO_HALF_EMPTY)) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Attempted to write to "
                      "TXFIFO_HALF_EMPTY with no FIFOs enabled\n");
        val = FIELD_DP32(val, IPISR, TXFIFO_HALF_EMPTY, 0);
    }

    /* IPISR bits are toggled on write */
    return s->regs[R_IPISR] ^= val;
}

static uint64_t axiqspi_ipisr_post_read(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = (uint64_t) val64;

    if (axiqspi_tx_fifo_half_empty(s)) {
        val = FIELD_DP32(val, IPISR, TXFIFO_HALF_EMPTY, 1);
    }

    return val;
}

static void axiqspi_update_cs_lines(XlnxAXIQSPI *s)
{
    uint32_t cs = s->regs[R_SPI_SSR];

    for (uint8_t i = 0; i < s->conf.num_cs; ++i) {
        uint8_t curr_cs = (cs >> i) & 1;
        DB_PRINT("axiqspi: cs[%.2d]=%d\n", i, curr_cs);
        qemu_set_irq(s->cs_lines[i], curr_cs);
    }
}

static void axiqspi_rx_push_fifo_update(XlnxAXIQSPI *s, uint32_t data)
{
    bool fifo_was_empty = fifo_is_empty(&s->rx_fifo);

    fifo_push32(&s->rx_fifo, data);

    /* After pushing the data, update the registers */
    if (fifo_was_empty) {
        ARRAY_FIELD_DP32(s->regs, SPISR, RX_EMPTY, 0);
        if (s->conf.mode == AXIQSPI_MODE_STD &&
            !ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
            ARRAY_FIELD_DP32(s->regs, IPISR, DRR_NOT_EMPTY, 1);
        }
    }
    if (fifo_is_full(&s->rx_fifo)) {
        ARRAY_FIELD_DP32(s->regs, IPISR, DRR_FULL, 1);
        ARRAY_FIELD_DP32(s->regs, SPISR, RX_FULL, 1);
    }
}

static void axiqspi_rx_push(XlnxAXIQSPI *s, uint32_t data)
{
    if (!ARRAY_FIELD_EX32(s->regs, SPISR, RX_FULL)) {
        if (s->conf.fifo_depth) {
            axiqspi_rx_push_fifo_update(s, data);
        } else {
            s->regs[R_SPI_DRR] = data;

            /* After pushing the data, update the registers */
            ARRAY_FIELD_DP32(s->regs, IPISR, DRR_FULL, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, RX_FULL, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, RX_EMPTY, 0);
        }
    } else {
        ARRAY_FIELD_DP32(s->regs, IPISR, DRR_OVERRUN, 1);
    }

    axiqspi_update_irq(s);
}

static void axiqspi_tx_push_fifo_update(XlnxAXIQSPI *s, uint32_t val)
{
    if (ARRAY_FIELD_EX32(s->regs, SPISR, TX_EMPTY)) {
        ARRAY_FIELD_DP32(s->regs, SPISR, TX_EMPTY, 0);
    }

    fifo_push32(&s->tx_fifo, val);
    /* After pushing the data, update our regs accordingly */
    if (fifo_is_full(&s->tx_fifo)) {
        ARRAY_FIELD_DP32(s->regs, SPISR, TX_FULL, 1);
    }
}

static void axiqspi_tx_push(XlnxAXIQSPI *s, uint32_t val)
{
    if (!ARRAY_FIELD_EX32(s->regs, SPISR, TX_FULL)) {
        /*
         * If the user is trying to write data wider than the
         * transaction width, truncate the bits and warn them.
         */
        if (!axiqspi_transaction_width_ok(s, val)) {
            qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Value %d is wider than "
                          "the transaction width, truncating it.\n", val);
            val &= s->conf.tx_width_mask;
        }

        if (s->conf.fifo_depth) {
            axiqspi_tx_push_fifo_update(s, val);
        } else {
            s->regs[R_SPI_DTR] = val;

            /* After pushing the data, update our regs */
            ARRAY_FIELD_DP32(s->regs, SPISR, TX_EMPTY, 0);
            ARRAY_FIELD_DP32(s->regs, SPISR, TX_FULL, 1);
        }
    } else {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Attempted to write to SPI_DTR "
                      "when it was full\n");
    }
}

static bool axiqspi_tx_pop(XlnxAXIQSPI *s, uint32_t *data)
{
    bool had_data = false;

    if (s->conf.fifo_depth) {
        if (!fifo_is_empty(&s->tx_fifo)) {
            *data = fifo_pop32(&s->tx_fifo);
            had_data = true;

            /* After we get get data, update our registers accordingly */
            if (ARRAY_FIELD_EX32(s->regs, SPISR, TX_FULL)) {
                ARRAY_FIELD_DP32(s->regs, SPISR, TX_FULL, 0);
            }
            if (fifo_is_empty(&s->tx_fifo)) {
                ARRAY_FIELD_DP32(s->regs, SPISR, TX_EMPTY, 1);
                ARRAY_FIELD_DP32(s->regs, IPISR, DTR_EMPTY, 1);
            }
        } else {
            /*
             * In an underrun condition, the core should transmit zeroes.
             * An underrun can only occur when the core is not a master.
             */
            if (!ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
                *data = 0;

                ARRAY_FIELD_DP32(s->regs, IPISR, DTR_UNDERRUN, 1);
            }
        }
    } else {
        if (!ARRAY_FIELD_EX32(s->regs, SPISR, TX_EMPTY)) {
            *data = s->regs[R_SPI_DTR];
            had_data = true;

            /* Register and interrupt updating */
            ARRAY_FIELD_DP32(s->regs, SPISR, TX_EMPTY, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, TX_FULL, 0);
            ARRAY_FIELD_DP32(s->regs, IPISR, DTR_EMPTY, 1);
        } else {
            /*
             * In an underrun condition, the core should transmit zeroes.
             * An underrun can only occur when the core is not a master.
             */
            if (!ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
                *data = 0;

                ARRAY_FIELD_DP32(s->regs, IPISR, DTR_UNDERRUN, 1);
            }
        }
    }

    axiqspi_update_irq(s);

    return had_data;
}

/*
 * We need to keep track of whether the device is in 3 or 4-byte addressing.
 * These functions look at what byte is being transmitted for the
 * current command and they check the extended addressing bit.
 */
static void axiqspi_sp_addressing_update(XlnxAXIQSPI *s, uint32_t data)
{
    if (s->cmd == BANK_REGISTER_WRITE) {
        if (data & SP_EXTADDR_BIT) {
            s->is_4b_addressing = true;
        } else {
            s->is_4b_addressing = false;
        }
    }
}

static void axiqspi_mi_addressing_update(XlnxAXIQSPI *s, uint32_t data)
{
    if (s->cmd == WRITE_NONVOLATILE_CONFIG_REG) {
        if ((s->bytes_txed == 0) && (data & MI_EXTADDR_BIT)) {
            s->is_4b_addressing = true;
        } else {
            s->is_4b_addressing = false;
        }
    } else if (s->cmd == ENTER_4B_ADDR_MODE) {
        s->is_4b_addressing = true;
    } else if (s->cmd == EXIT_4B_ADDR_MODE) {
        s->is_4b_addressing = false;
    }
}

static void axiqspi_mx_addressing_update(XlnxAXIQSPI *s, uint32_t data)
{
    if (s->cmd == WRITE_STATUS_REG) {
        if ((s->bytes_txed == 1) && (data & MA_EXTADDR_BIT)) {
            s->is_4b_addressing = true;
        } else {
            s->is_4b_addressing = false;
        }
    } else if (s->cmd == ENTER_4B_ADDR_MODE) {
        s->is_4b_addressing = true;
    } else if (s->cmd == EXIT_4B_ADDR_MODE) {
        s->is_4b_addressing = false;
    }
}

static void axiqspi_addressing_update(XlnxAXIQSPI *s, uint32_t data)
{
    switch (s->conf.spi_mem) {
    case SPI_MEM_SPANSION:
        axiqspi_sp_addressing_update(s, data);

        break;
    case SPI_MEM_MICRON:
        axiqspi_mi_addressing_update(s, data);

        break;
    case SPI_MEM_MACRONIX:
        axiqspi_mx_addressing_update(s, data);

        break;
    default:
        break;
    }
}

static uint32_t bit_rev(uint32_t data, uint8_t bit_width)
{
    for (uint8_t i = 0; i < bit_width / 2; ++i) {
        uint32_t tmp = !!(data & (1 << i));
        data |= (data & (1 << (bit_width - 1 - i))) >> (bit_width - 1 - i);
        data |= tmp << (bit_width - 1 - i);
    }

    return data;
}

static void axiqspi_state_reset(XlnxAXIQSPI *s)
{
    s->bytes_txed = 0;
    s->addr_bytes_txed = 0;
    s->dummy_bytes_txed = 0;
    s->state = AXIQSPI_STATE_STOPPED;
    s->link_state = LINK_STATE_SINGLE;
}

static void axiqspi_bus_txrx(XlnxAXIQSPI *s, uint32_t data)
{
    if (ARRAY_FIELD_EX32(s->regs, SPICR, LSB_FIRST)) {
        data = bit_rev(data, s->conf.transaction_width);
    }

    if (ARRAY_FIELD_EX32(s->regs, SPICR, LOOP) &&
        s->conf.mode == AXIQSPI_MODE_STD) {
        DB_PRINT("axiqspi: TXRX loopback 0x%x\n", data);
    } else {
        DB_PRINT("axiqspi: TX->0x%x\n", data);
        data = ssi_transfer(s->spi_bus, data);
        DB_PRINT("axiqspi: RX->0x%x\n", data);
    }
    axiqspi_rx_push(s, data);
}

static bool axiqspi_bus_tx_cmd(XlnxAXIQSPI *s)
{
    bool done;
    uint32_t data;

    if (axiqspi_tx_pop(s, &data)) {
        /*
         * Internal logic, such as the number of dummy bytes and
         * addressing bytes, are determined when parsing the command.
         */
        if (axiqspi_parse_cmd(s, data & 0xFF)) {
            DB_PRINT("axiqspi: New command 0x%x\n", data);
            axiqspi_bus_txrx(s, data);

            if (s->addr_bytes) {
                s->state = AXIQSPI_STATE_TX_ADDR;
            } else if (s->num_dummies) {
                s->state = AXIQSPI_STATE_TX_DUMMIES;
            } else {
                s->state = AXIQSPI_STATE_TX;
            }
        } else {
            ARRAY_FIELD_DP32(s->regs, IPISR, COMMAND_ERROR, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, COMMAND_ERROR, 1);

            qemu_log_mask(LOG_GUEST_ERROR,
                          "axiqspi: Unknown command 0x%x\n", data);
        }

        done = false;
    } else {
        done = true;
    }

    return done;
}

static bool axiqspi_bus_tx_addr(XlnxAXIQSPI *s)
{
    bool done;
    uint32_t data;

    if (axiqspi_tx_pop(s, &data)) {
        axiqspi_bus_txrx(s, data);
        ++s->addr_bytes_txed;

        if (s->addr_bytes_txed == s->addr_bytes) {
            if (s->num_dummies) {
                s->state = AXIQSPI_STATE_TX_DUMMIES;
            } else {
                s->state = AXIQSPI_STATE_TX;
            }
        }

        done = false;
    } else {
        done = true;
    }

    return done;
}

static bool axiqspi_bus_tx_dummies(XlnxAXIQSPI *s)
{
    uint32_t dummy;
    bool had_data = false;
    int i;

    DB_PRINT("axiqspi: dummy byte TX start\n");
    for (i = s->dummy_bytes_txed; i < s->num_dummies; ++i) {
        had_data = axiqspi_tx_pop(s, &dummy);
        if (!had_data) {
            break;
        }
        axiqspi_bus_txrx(s, dummy);
        ++s->dummy_bytes_txed;

        /*
         * Transmit extra dummy bytes depending on the link state.
         * These dummy bytes are not put into the RXFIFO
         */
        for (uint8_t j = 1; j < 8 / s->link_state; ++j) {
            DB_PRINT("axiqspi: TX->0x0\n");
            ssi_transfer(s->spi_bus, dummy);
        }
    }
    DB_PRINT("axiqspi: dummy byte TX end\n");

    if (s->dummy_bytes_txed == s->num_dummies) {
        s->state = AXIQSPI_STATE_TX;
    }

    return !had_data;
}

static bool axiqspi_bus_tx_data(XlnxAXIQSPI *s)
{
    bool done;
    uint32_t data;

    if (axiqspi_tx_pop(s, &data)) {
        if (s->is_addr_change_cmd) {
            axiqspi_addressing_update(s, data);
        }
        axiqspi_bus_txrx(s, data);
        ++s->bytes_txed;

        done = false;
    } else {
        done = true;
    }

    return done;
}

static void axiqspi_bus_tx(XlnxAXIQSPI *s)
{
    bool done = false;
    bool auto_ss = !ARRAY_FIELD_EX32(s->regs, SPICR,
                                     MANUAL_SLAVE_SELECT_ASSERTION_ENABLE);
    bool ss_asserted = false;

    if (s->state == AXIQSPI_STATE_STOPPED) {
        return;
    }

    if (!(~s->regs[R_SPI_SSR])) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Transmitting with no SS "
                      "asserted.\n");
    }

    while (!done) {
        if (auto_ss && !ss_asserted) {
            axiqspi_update_cs_lines(s);
            ss_asserted = true;
        }

        switch (s->state) {
        /*
         * The core will stay in the TX_CMD state until it
         * receives a valid command.
         * Note that in standard SPI mode commands aren't checked,
         * meaning that all commands are valid.
         */
        case AXIQSPI_STATE_TX_CMD:
            done = axiqspi_bus_tx_cmd(s);
            break;
        /*
         * The core stays in the TX_ADDR state until it's transmitted all of
         * the address bytes for the particular command
         */
        case AXIQSPI_STATE_TX_ADDR:
            done = axiqspi_bus_tx_addr(s);
            break;
        /* Transmit any needed dummy bytes */
        case AXIQSPI_STATE_TX_DUMMIES:
            done = axiqspi_bus_tx_dummies(s);
            break;
        /*
         * The core will stay in the transferring state until
         * either SPISSR is all 1s or SPI is disabled
         */
        case AXIQSPI_STATE_TX:
            done = axiqspi_bus_tx_data(s);
            break;
        default:
            g_assert_not_reached();
        }
    }

    if (auto_ss) {
        uint32_t old_ss = s->regs[R_SPI_SSR];
        s->regs[R_SPI_SSR] = 0xFFFFFFFF;
        axiqspi_update_cs_lines(s);
        s->regs[R_SPI_SSR] = old_ss;
    }

    axiqspi_update_irq(s);
}

static void axiqspi_do_reset(XlnxAXIQSPI *s)
{
    fifo_reset(&s->tx_fifo);
    fifo_reset(&s->rx_fifo);

    for (size_t i = 0; i < ARRAY_SIZE(s->reg_info); ++i) {
        register_reset(&s->reg_info[i]);
    }

    qemu_set_irq(s->irq, 0);
    axiqspi_update_cs_lines(s);
}

static void axiqspi_xip_do_reset(XlnxAXIQSPI *s)
{
    fifo_reset(&s->rx_fifo);

    for (size_t i = 0; i < ARRAY_SIZE(s->reg_info); ++i) {
        register_reset(&s->reg_info[i]);
    }
}

static void axiqspi_reset(DeviceState *dev)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(dev);

    if (s->conf.xip_mode) {
        axiqspi_xip_do_reset(s);
    } else {
        axiqspi_do_reset(s);
    }
}

static uint64_t axiqspi_srr_pre_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = (uint32_t) val64;

    if (val == AXIQSPI_SRR_RESET_VAL) {
        DB_PRINT("axiqspi: SPISRR Reset\n");
        axiqspi_do_reset(s);
    } else {
        qemu_log_mask(LOG_GUEST_ERROR,
                      "axiqspi: %d written to SRR, not resetting.\n", val);
    }

    return 0;
}

static uint64_t axiqspi_spi_ssr_pre_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = (uint32_t) val64;

    /* Bounds check the write */
    if (~val & ~s->conf.num_cs_mask) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Attempted to write %x to "
                      "SPI_SSR, when the valid bits are 0x%x. "
                      "Truncating extra bits.\n",
                      val, s->conf.num_cs_mask);

        val |= ~s->conf.num_cs_mask;
    }

    s->prev_ss = s->regs[R_SPI_SSR];

    return val;
}

static void axiqspi_spi_ssr_post_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t ss_noninv;
    uint32_t ssr = (uint32_t) val64;
    bool manual_ss = ARRAY_FIELD_EX32(s->regs, SPICR,
                                      MANUAL_SLAVE_SELECT_ASSERTION_ENABLE);

    /* SPI_SSR is inverted, uninverting it makes bit manipulation easier. */
    ss_noninv = ~ssr;
    if (ss_noninv & s->conf.num_cs_mask) {
        /* Only one SS line should be low at a time */
        if ((ss_noninv & (ss_noninv - 1)) != 0) {
            qemu_log_mask(LOG_GUEST_ERROR,
                          "axiqspi: SPI_SSR has multiple SS lines high. "
                          "SPI_SSR contains 0x%x\n", ssr);
        }

        /*
         * If an SS line is asserted when previously none of them were,
         * the core treats this as the start of a new transaction.
         */
        if ((ssr & s->conf.num_cs_mask) !=
            (s->prev_ss & s->conf.num_cs_mask)) {
            DB_PRINT("axiqspi: New SS asserted\n");
            if (manual_ss) {
                axiqspi_update_cs_lines(s);
            }

            if (ARRAY_FIELD_EX32(s->regs, SPICR, SPE) &&
                !ARRAY_FIELD_EX32(s->regs, SPICR, MASTER_TRANSACTION_INHIBIT) &&
                ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
                DB_PRINT("axiqspi: State changed to transmitting\n");

                s->state = AXIQSPI_STATE_TX_CMD;
            }
        }
    } else {
        /* If an SS was un-asserted, update the CS lines */
        if (ssr != s->prev_ss) {
            DB_PRINT("axiqspi: SS un-asserted\n");
            if (manual_ss) {
                axiqspi_update_cs_lines(s);
            }
        }
        /* All of the SS lines are low, so if the core is transmitting, stop. */
        if (s->state != AXIQSPI_STATE_STOPPED) {
            axiqspi_state_reset(s);
        }
    }
}

static uint64_t axiqspi_spicr_pre_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = val64;
    bool inhib_set = ARRAY_FIELD_EX32(s->regs, SPICR,
                                      MASTER_TRANSACTION_INHIBIT);
    bool will_clear_inhib = FIELD_EX32(val, SPICR,
                                       MASTER_TRANSACTION_INHIBIT) == 0;
    bool inhib_cleared = inhib_set && will_clear_inhib;
    bool spe_set = ARRAY_FIELD_EX32(s->regs, SPICR, SPE);
    bool will_set_spe = FIELD_EX32(val, SPICR, SPE);
    bool spe_now_set = !spe_set && will_set_spe;

    /* FIFO reset */
    if (FIELD_EX32(val, SPICR, RX_FIFO_RESET)) {
        fifo_reset(&s->rx_fifo);
        val = FIELD_DP32(val, SPICR, RX_FIFO_RESET, 0);
    }
    if (FIELD_EX32(val, SPICR, TX_FIFO_RESET)) {
        fifo_reset(&s->tx_fifo);
        val = FIELD_DP32(val, SPICR, TX_FIFO_RESET, 0);
    }
    /*
     * If MASTER_TRANSACTION_INHIBIT is cleared and SPE is set, we start
     * transmitting.
     * This logic is set up to avoid starting a transfer on every SPICR write,
     * unless the write modifies SPE or MASTER_TRANSACTION_INHIBIT.
     */
    if (ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
        if (inhib_cleared && spe_set) {
            axiqspi_bus_tx(s);
        } else if (spe_now_set && !inhib_set) {
            axiqspi_bus_tx(s);
        } else if (spe_now_set && inhib_cleared) {
            axiqspi_bus_tx(s);
        }
    }

    return val;
}

static void axiqspi_spicr_post_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    /* SPI disable */
    if (!ARRAY_FIELD_EX32(s->regs, SPICR, SPE)) {
        axiqspi_state_reset(s);
    }

    if (!ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
        qemu_log_mask(LOG_GUEST_ERROR,
                      "axiqspi: Slave mode is not supported.\n");
    }

    /* Check possible misconfigurations that need action */
    if (s->conf.mode != AXIQSPI_MODE_STD) {
        uint8_t cpol = !!(ARRAY_FIELD_EX32(s->regs, SPICR, CPOL));
        uint8_t cpha = !!(ARRAY_FIELD_EX32(s->regs, SPICR, CPHA));

        if (ARRAY_FIELD_EX32(s->regs, SPICR, LOOP)) {
            ARRAY_FIELD_DP32(s->regs, SPISR, LOOPBACK_ERROR, 1);
            ARRAY_FIELD_DP32(s->regs, IPISR, LOOPBACK_ERROR, 1);
            qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Cannot do loopback in "
                          "dual or quad mode.\n");
        }
        if (ARRAY_FIELD_EX32(s->regs, SPICR, LSB_FIRST)) {
            ARRAY_FIELD_DP32(s->regs, IPISR, MSB_ERROR, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, MSB_ERROR, 1);
            qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Cannot do LSB first in "
                          "dual or quad mode.\n");
        }
        if (!ARRAY_FIELD_EX32(s->regs, SPICR, MASTER)) {
            ARRAY_FIELD_DP32(s->regs, IPISR, SLAVE_MODE_ERROR, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, SLAVE_MODE_ERROR, 1);
            qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: Cannot do slave mode in "
                          "dual or quad mode.\n");
        }
        if ((!cpol && cpha) || (cpol && !cpha)) {
            ARRAY_FIELD_DP32(s->regs, IPISR, CPOL_CPHA_ERROR, 1);
            ARRAY_FIELD_DP32(s->regs, SPISR, CPOL_CPHA_ERROR, 1);
            qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: CPOL and CPHA error.\n");
        }
    }

    axiqspi_update_irq(s);
}

static uint64_t axiqspi_spi_dtr_pre_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);
    uint32_t val = (uint32_t)val64;

    axiqspi_tx_push(s, val);

    /* If SPI and TX are enabled, TX any stored data */
    if (ARRAY_FIELD_EX32(s->regs, SPICR, SPE)) {
        if (!ARRAY_FIELD_EX32(s->regs, SPICR, MASTER) ||
            (ARRAY_FIELD_EX32(s->regs, SPICR, MASTER) &&
            !ARRAY_FIELD_EX32(s->regs, SPICR, MASTER_TRANSACTION_INHIBIT))) {
                axiqspi_bus_tx(s);
        }
    }

    return val;
}

static void axiqspi_dgier_post_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    axiqspi_update_irq(s);
}

static uint64_t axiqspi_spi_rxfifo_or_post_read(RegisterInfo *reg,
                                                uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    if (!s->conf.fifo_depth) {
        qemu_log_mask(LOG_GUEST_ERROR, "Attempted to read RXFIFO_OR when FIFO "
                      "is disabled\n");
        return 0;
    }

    if (fifo_is_empty(&s->rx_fifo)) {
        return 0;
    }

    return fifo_num_used(&s->rx_fifo) - 1;
}

static uint64_t axiqspi_spi_txfifo_or_post_read(RegisterInfo *reg,
                                                uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    if (!s->conf.fifo_depth) {
        qemu_log_mask(LOG_GUEST_ERROR, "Attempted to read TXFIFO_OR when FIFO "
                      "is disabled\n");
        return 0;
    }

    if (fifo_is_empty(&s->tx_fifo)) {
        return 0;
    }

    return fifo_num_used(&s->tx_fifo) - 1;
}

static const RegisterAccessInfo axiqspi_regs_info[] = {
    {   .name = "DGIER",  .addr = A_DGIER,
        .rsvd = 0x7fffffff,
        .post_write = axiqspi_dgier_post_write,
    },{ .name = "IPISR",  .addr = A_IPISR,
        .rsvd = 0xffffc000,
        .pre_write = axiqspi_ipisr_pre_write,
        .post_read = axiqspi_ipisr_post_read,
    },{ .name = "IPIER",  .addr = A_IPIER,
        .rsvd = 0xffffc000,
    },{ .name = "SRR",  .addr = A_SRR,
        .pre_write = axiqspi_srr_pre_write,
    },{ .name = "SPICR",  .addr = A_SPICR,
        .reset = 0x180,
        .rsvd = 0xfffffc00,
        .pre_write = axiqspi_spicr_pre_write,
        .post_write = axiqspi_spicr_post_write,
    },{ .name = "SPISR",  .addr = A_SPISR,
        .reset = 0xa5,
        .rsvd = 0xfffff800,
        .ro = 0x7ff,
    },{ .name = "SPI_DTR",  .addr = A_SPI_DTR,
        .pre_write = axiqspi_spi_dtr_pre_write
    },{ .name = "SPI_DRR",  .addr = A_SPI_DRR,
        .ro = 0xffffffff,
        .post_read = axiqspi_spi_drr_post_read,
    },{ .name = "SPI_SSR",  .addr = A_SPI_SSR,
        .reset = 0xffffffff,
        .pre_write = axiqspi_spi_ssr_pre_write,
        .post_write = axiqspi_spi_ssr_post_write,
    },{ .name = "SPI_TXFIFO_OR",  .addr = A_SPI_TXFIFO_OR,
        .ro = 0xffffffff,
        .post_read = axiqspi_spi_txfifo_or_post_read,
    },{ .name = "SPI_RXFIFO_OR",  .addr = A_SPI_RXFIFO_OR,
        .ro = 0xffffffff,
        .post_read = axiqspi_spi_rxfifo_or_post_read,
    }
};

static bool axiqspi_xip_check_cpol_cpha(XlnxAXIQSPI *s)
{
    bool cpol = !!(ARRAY_FIELD_EX32(s->regs, XIP_CONFIG_REG, CPOL));
    bool cpha = !!(ARRAY_FIELD_EX32(s->regs, XIP_CONFIG_REG, CPHA));

    return (cpol && cpha) || (!cpol && !cpha);
}

static void axiqspi_xip_spicr_post_write(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    if (!axiqspi_xip_check_cpol_cpha(s)) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: CPOL and CPHA error.\n");
        ARRAY_FIELD_DP32(s->regs, XIP_STATUS_REG, CPOL_CPHA_ERROR, 1);
    }
}

static uint64_t axiqspi_xip_spisr_post_read(RegisterInfo *reg, uint64_t val64)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(reg->opaque);

    /* XIP SPISR is brought to reset after every read */
    register_reset(&s->reg_info[R_XIP_STATUS_REG]);

    return val64;
}

static const RegisterAccessInfo axiqspi_xip_regs_info[] = {
    {   .name = "XIP_CONFIG_REG",  .addr = A_XIP_CONFIG_REG,
        .rsvd = 0xfffffffc,
        .post_write = axiqspi_xip_spicr_post_write,
    },{ .name = "XIP_STATUS_REG",  .addr = A_XIP_STATUS_REG,
        .reset = 0x1,
        .rsvd = 0xffffff70,
        .ro = 0x1f,
        .post_read = axiqspi_xip_spisr_post_read,
    }
};

static const MemoryRegionOps axiqspi_ops = {
    .read = register_read_memory,
    .write = register_write_memory,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static void axiqspi_xip_txrx(XlnxAXIQSPI *s, hwaddr addr, uint8_t *val,
                             unsigned size)
{
    qemu_set_irq(s->cs_lines[0], 0);
    ssi_transfer(s->spi_bus, s->cmd);
    DB_PRINT("axiqspi: XIP TX->0x%x\n", s->cmd);

    for (int i = s->addr_bytes - 1; i >= 0; --i) {
        uint8_t shift = i * 8;
        uint8_t addr_byte = (addr >> shift) & 0xFF;
        ssi_transfer(s->spi_bus, addr_byte);
        DB_PRINT("axiqspi: XIP TX->0x%x\n", addr_byte);
    }

    /*
     * If a command has dummy bytes, We TX 8 dummy bytes regardless of
     * command and link state.
     */
    if (s->num_dummies) {
        for (uint8_t i = 0; i < 8; ++i) {
            ssi_transfer(s->spi_bus, 0x00);
            DB_PRINT("axiqspi: XIP TX->0x00 dummy\n");
        }
    }

    for (uint8_t i = 0; i < size; ++i) {
        val[i] = ssi_transfer(s->spi_bus, 0x00);
        DB_PRINT("axiqspi: XIP RX->0x%x\n", val[i]);
    }
    qemu_set_irq(s->cs_lines[0], 1);
}

static void axiqspi_xip_wb_init(XlnxAXIQSPI *s)
{
    axiqspi_parse_cmd(s, HIGH_PERFORMANCE_MODE);

    axiqspi_xip_txrx(s, 0, NULL, 0);
}

static MemTxResult axiqspi_xip_read(void *opaque, hwaddr addr, uint64_t *val,
                                    unsigned size, MemTxAttrs attrs)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(opaque);
    uint8_t cmd;
    bool is_4b_cmd = (s->conf.spi_mem == SPI_MEM_SPANSION ||
                      s->conf.spi_mem == SPI_MEM_MICRON) &&
                      s->conf.xip_addr_bits == 32;

    assert(size <= 64);

    if (!axiqspi_xip_check_cpol_cpha(s)) {
        qemu_log_mask(LOG_GUEST_ERROR, "axiqspi: attempted to read with "
                      "bad CPOL and CPHA.\n");
        return MEMTX_ERROR;
    }

    /* These read commands are universal between all supported flashes */
    switch (s->conf.mode) {
    case AXIQSPI_MODE_STD:
        cmd = is_4b_cmd ? FAST_READ_4B : FAST_READ;
        break;
    case AXIQSPI_MODE_DUAL:
        cmd = is_4b_cmd ? FAST_READ_DUAL_IO_4B : FAST_READ_DUAL_IO;
        break;
    case AXIQSPI_MODE_QUAD:
        cmd = is_4b_cmd ? FAST_READ_QUAD_IO_4B : FAST_READ_QUAD_IO;
        break;
    default:
        g_assert_not_reached();
    }

    if (s->conf.spi_mem == SPI_MEM_WINBOND &&
        s->conf.mode != AXIQSPI_MODE_STD &&
        !s->is_xip_wb_init) {
        axiqspi_xip_wb_init(s);
        s->is_xip_wb_init = true;
    }

    /* Parse to get the number of dummy and addr bytes */
    axiqspi_parse_cmd(s, cmd);

    /*
     * Rather than using the functions we would normally use for AXIQSPI
     * transfers, we will directly transfer things down the bus.
     * This is to avoid having to wrap most of our register accesses with
     * if statements to see if we're in XIP mode or not.
     *
     * Note that we also don't update any RX FIFO status registers.
     * This is because we send data right after we receive it from
     * the SPI flash; meaning the FIFO is always empty.
     */
    axiqspi_xip_txrx(s, addr, (uint8_t *)val, size);

    return MEMTX_OK;
}

static MemTxResult axiqspi_xip_write(void *opaque, hwaddr addr, uint64_t val,
                                    unsigned size, MemTxAttrs attrs)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(opaque);

    /* XIP mode does not allow writes */
    qemu_log_mask(LOG_GUEST_ERROR,
                  "axiqspi: attempted to write in XIP mode\n");
    ARRAY_FIELD_DP32(s->regs, XIP_STATUS_REG, AXI_TRANSACTION_ERROR, 1);

    return MEMTX_ERROR;
}

static const MemoryRegionOps axiqspi_xip_ops = {
    .read_with_attrs = axiqspi_xip_read,
    .write_with_attrs = axiqspi_xip_write,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 1,
        .max_access_size = 4,
    },
};

static bool axiqspi_validate_conf(XlnxAXIQSPI *s, Error **errp)
{
    /* Checks that must be done in any core configuration */
    if (s->conf.mode >= AXIQSPI_MODE_INVALID) {
        error_setg(errp, "axiqspi: Unknown SPI mode %d", s->conf.mode);
        return false;
    }

    if (s->conf.num_cs < AXIQSPI_NUM_CS_MIN ||
        s->conf.num_cs > AXIQSPI_NUM_CS_MAX) {
        error_setg(errp, "axiqspi: Num CS must be between %d and %d.",
                   AXIQSPI_NUM_CS_MIN, AXIQSPI_NUM_CS_MAX);
        return false;
    }

    if (s->conf.fifo_depth) {
        if (s->conf.xip_mode) {
            if (s->conf.fifo_depth != 64) {
                error_setg(errp, "axiqspi: FIFO depth must be 64 in XIP mode, "
                           "but is %d.", s->conf.fifo_depth);
                return false;
            }
        } else {
            if (s->conf.fifo_depth != 16 && s->conf.fifo_depth != 256) {
                error_setg(errp, "axiqspi: FIFO depth can only be 16 or 256, "
                           "but is %d.", s->conf.fifo_depth);
                return false;
            }
        }
    }

    if (s->conf.transaction_width != 8 && s->conf.transaction_width != 16 &&
        s->conf.transaction_width != 32) {
        error_setg(errp, "axiqspi: transaction width must be 8, 16, or "
                   "32, but is %d.", s->conf.transaction_width);
        return false;
    }

    /* Sanity checking when the core is in dual or quad mode */
    if (s->conf.mode == AXIQSPI_MODE_DUAL ||
        s->conf.mode == AXIQSPI_MODE_QUAD) {
        if (s->conf.transaction_width != 8) {
            error_setg(errp, "axiqspi: Transaction width must be 8 in "
                       "dual or quad mode, but is %d.",
                       s->conf.transaction_width);
            return false;
        }

        if (!s->conf.fifo_depth) {
            error_setg(errp, "axiqspi: Dual and quad mode must use FIFO.");
            return false;
        }
    }

    /* If the core is in dual or quad mode, check the memory */
    if (s->conf.mode != AXIQSPI_MODE_STD) {
        if (s->conf.spi_mem >= SPI_MEM_INVALID) {
            error_setg(errp, "axiqspi: Unknown SPI memory %d, "
                       "defaulting to mixed memory", s->conf.spi_mem);
            s->conf.spi_mem = SPI_MEM_MIXED;
        }
    }

    /* Sanity checking when the core is in XIP mode */
    if (s->conf.xip_mode) {
        if (s->conf.transaction_width != 8) {
            error_setg(errp, "axiqspi: Transaction width must be 8 in "
                       "XIP mode, but is %d.", s->conf.transaction_width);
            return false;
        }

        if (s->conf.num_cs != 1) {
            error_setg(errp, "axiqspi: Num CS must be 1 in XIP mode, "
                       "but is %d.", s->conf.num_cs);
            return false;
        }

        if (s->conf.xip_addr_bits != 24 && s->conf.xip_addr_bits != 32) {
            error_setg(errp, "axiqspi: address bits must be 24 or "
                       "32 in XIP mode, but is %d.", s->conf.xip_addr_bits);
            return false;
        }

        if (s->conf.xip_addr_bits == 32) {
            if (s->conf.spi_mem != SPI_MEM_SPANSION &&
                s->conf.spi_mem != SPI_MEM_MICRON) {
                error_setg(errp, "axiqspi: XIP 32-bit addressing is only "
                           "supported on Spansion or Micron memories.");
                return false;

            }
        }
    }

    return true;
}

static void axiqspi_realize(DeviceState *dev, Error **errp)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(dev);
    SysBusDevice *sbd = SYS_BUS_DEVICE(dev);
    RegisterInfoArray *reg_array;

    if (!axiqspi_validate_conf(s, errp)) {
        return;
    }

    s->conf.tx_width_mask = MAKE_32BIT_MASK(s->conf.transaction_width);
    s->conf.num_cs_mask = MAKE_32BIT_MASK(s->conf.num_cs);

    if (s->conf.xip_mode) {
        reg_array =
            register_init_block32(dev, axiqspi_xip_regs_info,
                                  ARRAY_SIZE(axiqspi_xip_regs_info),
                                  s->reg_info, s->regs,
                                  &axiqspi_ops,
                                  XLNX_AXIQSPI_ERR_DEBUG,
                                  XLNX_AXIQSPI_R_MAX * 4);
    } else {
        reg_array =
            register_init_block32(dev, axiqspi_regs_info,
                                  ARRAY_SIZE(axiqspi_regs_info),
                                  s->reg_info, s->regs,
                                  &axiqspi_ops,
                                  XLNX_AXIQSPI_ERR_DEBUG,
                                  XLNX_AXIQSPI_R_MAX * 4);
    }

    memory_region_add_subregion(&s->iomem, 0x0, &reg_array->mem);
    sysbus_init_mmio(sbd, &s->iomem);
    sysbus_init_irq(sbd, &s->irq);

    /*
     * Create FIFOs that hold the largest value possible, rather than
     * separate 8/16/32-bit ones that depend on our configuration.
     * This simplifies function calling at the cost of space.
     */
    if (s->conf.fifo_depth) {
        fifo_create32(&s->tx_fifo, s->conf.fifo_depth);
        fifo_create32(&s->rx_fifo, s->conf.fifo_depth);
    }

    s->spi_bus = ssi_create_bus(dev, "spi0");
    s->cs_lines = g_new0(qemu_irq, s->conf.num_cs);
    ssi_auto_connect_slaves(DEVICE(s), s->cs_lines, s->spi_bus);
    qdev_init_gpio_out(dev, s->cs_lines, s->conf.num_cs);

    DB_PRINT("axiqspi: realized\n");
}

static void axiqspi_init(Object *obj)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(obj);

    /*
     * Even though the XIP registers take up much less space than
     * non-XIP mode registers, the registers array is the same size.
     * Therefore we initialize for the same size in both scenarios.
     */
    memory_region_init(&s->iomem, obj, TYPE_XLNX_AXIQSPI,
                       XLNX_AXIQSPI_R_MAX * 4);
}

static bool axiqspi_parse_reg(FDTGenericMMap *obj, FDTGenericRegPropInfo reg,
                              Error **errp)
{
    XlnxAXIQSPI *s = XLNX_AXIQSPI(obj);
    ObjectClass *klass = object_class_by_name(TYPE_XLNX_AXIQSPI);
    SysBusDevice *sbd = SYS_BUS_DEVICE(obj);
    FDTGenericMMapClass *parent_fmc;

    parent_fmc = FDT_GENERIC_MMAP_CLASS(object_class_get_parent(klass));
    if (s->conf.xip_mode) {
        if (reg.n != 2) {
            error_setg(errp, "axiqspi: XIP mode requires 1 region, but "
                       "device tree specifies %d regions.", reg.n - 1);
            return false;
        }

        memory_region_init_io(&s->xip_mr, OBJECT(obj), &axiqspi_xip_ops,
                              s, "axiqspi-xip-region", reg.s[1]);
        sysbus_init_mmio(sbd, &s->xip_mr);
    }

    return parent_fmc ? parent_fmc->parse_reg(obj, reg, errp) : false;
}

static const VMStateDescription vmstate_axiqspi = {
    .name = TYPE_XLNX_AXIQSPI,
    .version_id = 1,
    .minimum_version_id = 1,
    .fields = (VMStateField[]) {
        VMSTATE_UINT32_ARRAY(regs, XlnxAXIQSPI, XLNX_AXIQSPI_R_MAX),
        VMSTATE_END_OF_LIST(),
    }
};

static Property xlnx_axiqspi_properties[] = {
    /*
     * 0-1.
     * If XIP mode is enabled, the model expects another <addr, size> pair in
     * the reg parameter in the device tree.
     * This pair is used to create the XIP region.
     */
    DEFINE_PROP_BOOL("xip-mode", XlnxAXIQSPI, conf.xip_mode, false),
    /*
     * 24, 32; only used in XIP mode.
     * In non-XIP mode the amount of bits an address has can change, but
     * in XIP mode it must be constant and defined at device creation.
     */
    DEFINE_PROP_UINT8("spi-mem-addr-bits", XlnxAXIQSPI, conf.xip_addr_bits, 24),
    /* 0, 16, 256 */
    DEFINE_PROP_UINT16("fifo-size",  XlnxAXIQSPI, conf.fifo_depth, 256),
    /* 8, 16, 32 */
    DEFINE_PROP_UINT8("bits-per-word", XlnxAXIQSPI,
                      conf.transaction_width, 8),
    /*
     * 0-4:
     * 0 = Mixed
     * 1 = Winbond
     * 2 = Micron (Numonyx)
     * 3 = Spansion
     * 4 = Macronix
     */
    DEFINE_PROP_UINT8("spi-memory", XlnxAXIQSPI, conf.spi_mem, 2),
    /*
     * 0-2:
     * 0 = standard
     * 1 = dual
     * 2 = quad
     */
    DEFINE_PROP_UINT8("spi-mode", XlnxAXIQSPI, conf.mode, 0),
    /* 1-32 */
    DEFINE_PROP_UINT8("num-ss-bits", XlnxAXIQSPI, conf.num_cs, 1),

    DEFINE_PROP_END_OF_LIST(),
};

static void axiqspi_class_init(ObjectClass *klass, void *data)
{
    DeviceClass *dc = DEVICE_CLASS(klass);
    FDTGenericMMapClass *fmc = FDT_GENERIC_MMAP_CLASS(klass);

    dc->reset = axiqspi_reset;
    device_class_set_props(dc, xlnx_axiqspi_properties);
    dc->realize = axiqspi_realize;
    dc->vmsd = &vmstate_axiqspi;
    fmc->parse_reg = axiqspi_parse_reg;
}

static const TypeInfo axiqspi_info = {
    .name               = TYPE_XLNX_AXIQSPI,
    .parent             = TYPE_SYS_BUS_DEVICE,
    .instance_size      = sizeof(XlnxAXIQSPI),
    .class_init         = axiqspi_class_init,
    .instance_init      = axiqspi_init,
    .interfaces         = (InterfaceInfo[]) {
        { TYPE_FDT_GENERIC_MMAP },
        { },
    },
};

static void axiqspi_register_types(void)
{
    type_register_static(&axiqspi_info);
}

type_init(axiqspi_register_types)
