/*
 * QEMU model of the CFRAME_REG Configuration Frame Control
 *
 * Copyright (c) 2022 Xilinx Inc.
 *
 * Partially autogenerated by xregqemu.py 2022-01-10.
 * Written by Francisco Iglesias <francisco.iglesias@xilinx.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "qemu/osdep.h"
#include "hw/sysbus.h"
#include "hw/register.h"
#include "hw/registerfields.h"
#include "qemu/bitops.h"
#include "qemu/log.h"
#include "qapi/error.h"
#include "hw/qdev-properties.h"
#include "migration/vmstate.h"
#include "hw/irq.h"
#include "hw/misc/xlnx-cfi-if.h"

#ifndef XILINX_CFRAME_REG_ERR_DEBUG
#define XILINX_CFRAME_REG_ERR_DEBUG 0
#endif

#define TYPE_XILINX_CFRAME_REG "xlnx.cframe_reg"

#define XILINX_CFRAME_REG(obj) \
     OBJECT_CHECK(CFRAME_REG, (obj), TYPE_XILINX_CFRAME_REG)

#define TYPE_XILINX_CFRAME_BCAST_REG "xlnx.cframe-bcast-reg"

#define XILINX_CFRAME_BCAST_REG(obj) \
     OBJECT_CHECK(CFRAME_BCAST_REG, (obj), TYPE_XILINX_CFRAME_BCAST_REG)

/*
 * The registers in this module are 128 bits wide but it is ok to write
 * and read them through 4 sequential 32 bit accesses (address[3:2] = 0,
 * 1, 2, 3).
 */
REG32(CRC0, 0x0)
    FIELD(CRC, CRC, 0, 32)
REG32(CRC1, 0x4)
REG32(CRC2, 0x8)
REG32(CRC3, 0xc)
REG32(FAR0, 0x10)
    FIELD(FAR0, SEGMENT, 23, 2)
    FIELD(FAR0, BLOCKTYPE, 20, 3)
    FIELD(FAR0, FRAME_ADDR, 0, 20)
REG32(FAR1, 0x14)
REG32(FAR2, 0x18)
REG32(FAR3, 0x1c)
REG32(FAR_SFR0, 0x20)
    FIELD(FAR_SFR0, BLOCKTYPE, 20, 3)
    FIELD(FAR_SFR0, FRAME_ADDR, 0, 20)
REG32(FAR_SFR1, 0x24)
REG32(FAR_SFR2, 0x28)
REG32(FAR_SFR3, 0x2c)
REG32(FDRI0, 0x40)
REG32(FDRI1, 0x44)
REG32(FDRI2, 0x48)
REG32(FDRI3, 0x4c)
REG32(FRCNT0, 0x50)
    FIELD(FRCNT0, FRCNT, 0, 32)
REG32(FRCNT1, 0x54)
REG32(FRCNT2, 0x58)
REG32(FRCNT3, 0x5c)
REG32(CMD0, 0x60)
    FIELD(CMD0, CMD, 0, 5)
REG32(CMD1, 0x64)
REG32(CMD2, 0x68)
REG32(CMD3, 0x6c)
REG32(CR_MASK0, 0x70)
REG32(CR_MASK1, 0x74)
REG32(CR_MASK2, 0x78)
REG32(CR_MASK3, 0x7c)
REG32(CTL0, 0x80)
    FIELD(CTL, PER_FRAME_CRC, 0, 1)
REG32(CTL1, 0x84)
REG32(CTL2, 0x88)
REG32(CTL3, 0x8c)
REG32(CFRM_ISR0, 0x150)
    FIELD(CFRM_ISR0, READ_BROADCAST_ERROR, 21, 1)
    FIELD(CFRM_ISR0, CMD_MISSING_ERROR, 20, 1)
    FIELD(CFRM_ISR0, RW_ROWOFF_ERROR, 19, 1)
    FIELD(CFRM_ISR0, READ_REG_ADDR_ERROR, 18, 1)
    FIELD(CFRM_ISR0, READ_BLK_TYPE_ERROR, 17, 1)
    FIELD(CFRM_ISR0, READ_FRAME_ADDR_ERROR, 16, 1)
    FIELD(CFRM_ISR0, WRITE_REG_ADDR_ERROR, 15, 1)
    FIELD(CFRM_ISR0, WRITE_BLK_TYPE_ERROR, 13, 1)
    FIELD(CFRM_ISR0, WRITE_FRAME_ADDR_ERROR, 12, 1)
    FIELD(CFRM_ISR0, MFW_OVERRUN_ERROR, 11, 1)
    FIELD(CFRM_ISR0, FAR_FIFO_UNDERFLOW, 10, 1)
    FIELD(CFRM_ISR0, FAR_FIFO_OVERFLOW, 9, 1)
    FIELD(CFRM_ISR0, PER_FRAME_SEQ_ERROR, 8, 1)
    FIELD(CFRM_ISR0, CRC_ERROR, 7, 1)
    FIELD(CFRM_ISR0, WRITE_OVERRUN_ERROR, 6, 1)
    FIELD(CFRM_ISR0, READ_OVERRUN_ERROR, 5, 1)
    FIELD(CFRM_ISR0, CMD_INTERRUPT_ERROR, 4, 1)
    FIELD(CFRM_ISR0, WRITE_INTERRUPT_ERROR, 3, 1)
    FIELD(CFRM_ISR0, READ_INTERRUPT_ERROR, 2, 1)
    FIELD(CFRM_ISR0, SEU_CRC_ERROR, 1, 1)
    FIELD(CFRM_ISR0, SEU_ECC_ERROR, 0, 1)
REG32(CFRM_ISR1, 0x154)
REG32(CFRM_ISR2, 0x158)
REG32(CFRM_ISR3, 0x15c)
REG32(CFRM_IMR0, 0x160)
    FIELD(CFRM_IMR0, READ_BROADCAST_ERROR, 21, 1)
    FIELD(CFRM_IMR0, CMD_MISSING_ERROR, 20, 1)
    FIELD(CFRM_IMR0, RW_ROWOFF_ERROR, 19, 1)
    FIELD(CFRM_IMR0, READ_REG_ADDR_ERROR, 18, 1)
    FIELD(CFRM_IMR0, READ_BLK_TYPE_ERROR, 17, 1)
    FIELD(CFRM_IMR0, READ_FRAME_ADDR_ERROR, 16, 1)
    FIELD(CFRM_IMR0, WRITE_REG_ADDR_ERROR, 15, 1)
    FIELD(CFRM_IMR0, WRITE_BLK_TYPE_ERROR, 13, 1)
    FIELD(CFRM_IMR0, WRITE_FRAME_ADDR_ERROR, 12, 1)
    FIELD(CFRM_IMR0, MFW_OVERRUN_ERROR, 11, 1)
    FIELD(CFRM_IMR0, FAR_FIFO_UNDERFLOW, 10, 1)
    FIELD(CFRM_IMR0, FAR_FIFO_OVERFLOW, 9, 1)
    FIELD(CFRM_IMR0, PER_FRAME_SEQ_ERROR, 8, 1)
    FIELD(CFRM_IMR0, CRC_ERROR, 7, 1)
    FIELD(CFRM_IMR0, WRITE_OVERRUN_ERROR, 6, 1)
    FIELD(CFRM_IMR0, READ_OVERRUN_ERROR, 5, 1)
    FIELD(CFRM_IMR0, CMD_INTERRUPT_ERROR, 4, 1)
    FIELD(CFRM_IMR0, WRITE_INTERRUPT_ERROR, 3, 1)
    FIELD(CFRM_IMR0, READ_INTERRUPT_ERROR, 2, 1)
    FIELD(CFRM_IMR0, SEU_CRC_ERROR, 1, 1)
    FIELD(CFRM_IMR0, SEU_ECC_ERROR, 0, 1)
REG32(CFRM_IMR1, 0x164)
REG32(CFRM_IMR2, 0x168)
REG32(CFRM_IMR3, 0x16c)
REG32(CFRM_IER0, 0x170)
    FIELD(CFRM_IER0, READ_BROADCAST_ERROR, 21, 1)
    FIELD(CFRM_IER0, CMD_MISSING_ERROR, 20, 1)
    FIELD(CFRM_IER0, RW_ROWOFF_ERROR, 19, 1)
    FIELD(CFRM_IER0, READ_REG_ADDR_ERROR, 18, 1)
    FIELD(CFRM_IER0, READ_BLK_TYPE_ERROR, 17, 1)
    FIELD(CFRM_IER0, READ_FRAME_ADDR_ERROR, 16, 1)
    FIELD(CFRM_IER0, WRITE_REG_ADDR_ERROR, 15, 1)
    FIELD(CFRM_IER0, WRITE_BLK_TYPE_ERROR, 13, 1)
    FIELD(CFRM_IER0, WRITE_FRAME_ADDR_ERROR, 12, 1)
    FIELD(CFRM_IER0, MFW_OVERRUN_ERROR, 11, 1)
    FIELD(CFRM_IER0, FAR_FIFO_UNDERFLOW, 10, 1)
    FIELD(CFRM_IER0, FAR_FIFO_OVERFLOW, 9, 1)
    FIELD(CFRM_IER0, PER_FRAME_SEQ_ERROR, 8, 1)
    FIELD(CFRM_IER0, CRC_ERROR, 7, 1)
    FIELD(CFRM_IER0, WRITE_OVERRUN_ERROR, 6, 1)
    FIELD(CFRM_IER0, READ_OVERRUN_ERROR, 5, 1)
    FIELD(CFRM_IER0, CMD_INTERRUPT_ERROR, 4, 1)
    FIELD(CFRM_IER0, WRITE_INTERRUPT_ERROR, 3, 1)
    FIELD(CFRM_IER0, READ_INTERRUPT_ERROR, 2, 1)
    FIELD(CFRM_IER0, SEU_CRC_ERROR, 1, 1)
    FIELD(CFRM_IER0, SEU_ECC_ERROR, 0, 1)
REG32(CFRM_IER1, 0x174)
REG32(CFRM_IER2, 0x178)
REG32(CFRM_IER3, 0x17c)
REG32(CFRM_IDR0, 0x180)
    FIELD(CFRM_IDR0, READ_BROADCAST_ERROR, 21, 1)
    FIELD(CFRM_IDR0, CMD_MISSING_ERROR, 20, 1)
    FIELD(CFRM_IDR0, RW_ROWOFF_ERROR, 19, 1)
    FIELD(CFRM_IDR0, READ_REG_ADDR_ERROR, 18, 1)
    FIELD(CFRM_IDR0, READ_BLK_TYPE_ERROR, 17, 1)
    FIELD(CFRM_IDR0, READ_FRAME_ADDR_ERROR, 16, 1)
    FIELD(CFRM_IDR0, WRITE_REG_ADDR_ERROR, 15, 1)
    FIELD(CFRM_IDR0, WRITE_BLK_TYPE_ERROR, 13, 1)
    FIELD(CFRM_IDR0, WRITE_FRAME_ADDR_ERROR, 12, 1)
    FIELD(CFRM_IDR0, MFW_OVERRUN_ERROR, 11, 1)
    FIELD(CFRM_IDR0, FAR_FIFO_UNDERFLOW, 10, 1)
    FIELD(CFRM_IDR0, FAR_FIFO_OVERFLOW, 9, 1)
    FIELD(CFRM_IDR0, PER_FRAME_SEQ_ERROR, 8, 1)
    FIELD(CFRM_IDR0, CRC_ERROR, 7, 1)
    FIELD(CFRM_IDR0, WRITE_OVERRUN_ERROR, 6, 1)
    FIELD(CFRM_IDR0, READ_OVERRUN_ERROR, 5, 1)
    FIELD(CFRM_IDR0, CMD_INTERRUPT_ERROR, 4, 1)
    FIELD(CFRM_IDR0, WRITE_INTERRUPT_ERROR, 3, 1)
    FIELD(CFRM_IDR0, READ_INTERRUPT_ERROR, 2, 1)
    FIELD(CFRM_IDR0, SEU_CRC_ERROR, 1, 1)
    FIELD(CFRM_IDR0, SEU_ECC_ERROR, 0, 1)
REG32(CFRM_IDR1, 0x184)
REG32(CFRM_IDR2, 0x188)
REG32(CFRM_IDR3, 0x18c)
REG32(CFRM_ITR0, 0x190)
    FIELD(CFRM_ITR0, READ_BROADCAST_ERROR, 21, 1)
    FIELD(CFRM_ITR0, CMD_MISSING_ERROR, 20, 1)
    FIELD(CFRM_ITR0, RW_ROWOFF_ERROR, 19, 1)
    FIELD(CFRM_ITR0, READ_REG_ADDR_ERROR, 18, 1)
    FIELD(CFRM_ITR0, READ_BLK_TYPE_ERROR, 17, 1)
    FIELD(CFRM_ITR0, READ_FRAME_ADDR_ERROR, 16, 1)
    FIELD(CFRM_ITR0, WRITE_REG_ADDR_ERROR, 15, 1)
    FIELD(CFRM_ITR0, WRITE_BLK_TYPE_ERROR, 13, 1)
    FIELD(CFRM_ITR0, WRITE_FRAME_ADDR_ERROR, 12, 1)
    FIELD(CFRM_ITR0, MFW_OVERRUN_ERROR, 11, 1)
    FIELD(CFRM_ITR0, FAR_FIFO_UNDERFLOW, 10, 1)
    FIELD(CFRM_ITR0, FAR_FIFO_OVERFLOW, 9, 1)
    FIELD(CFRM_ITR0, PER_FRAME_SEQ_ERROR, 8, 1)
    FIELD(CFRM_ITR0, CRC_ERROR, 7, 1)
    FIELD(CFRM_ITR0, WRITE_OVERRUN_ERROR, 6, 1)
    FIELD(CFRM_ITR0, READ_OVERRUN_ERROR, 5, 1)
    FIELD(CFRM_ITR0, CMD_INTERRUPT_ERROR, 4, 1)
    FIELD(CFRM_ITR0, WRITE_INTERRUPT_ERROR, 3, 1)
    FIELD(CFRM_ITR0, READ_INTERRUPT_ERROR, 2, 1)
    FIELD(CFRM_ITR0, SEU_CRC_ERROR, 1, 1)
    FIELD(CFRM_ITR0, SEU_ECC_ERROR, 0, 1)
REG32(CFRM_ITR1, 0x194)
REG32(CFRM_ITR2, 0x198)
REG32(CFRM_ITR3, 0x19c)
REG32(SEU_SYNDRM00, 0x1a0)
REG32(SEU_SYNDRM01, 0x1a4)
REG32(SEU_SYNDRM02, 0x1a8)
REG32(SEU_SYNDRM03, 0x1ac)
REG32(SEU_SYNDRM10, 0x1b0)
REG32(SEU_SYNDRM11, 0x1b4)
REG32(SEU_SYNDRM12, 0x1b8)
REG32(SEU_SYNDRM13, 0x1bc)
REG32(SEU_SYNDRM20, 0x1c0)
REG32(SEU_SYNDRM21, 0x1c4)
REG32(SEU_SYNDRM22, 0x1c8)
REG32(SEU_SYNDRM23, 0x1cc)
REG32(SEU_SYNDRM30, 0x1d0)
REG32(SEU_SYNDRM31, 0x1d4)
REG32(SEU_SYNDRM32, 0x1d8)
REG32(SEU_SYNDRM33, 0x1dc)
REG32(SEU_VIRTUAL_SYNDRM0, 0x1e0)
REG32(SEU_VIRTUAL_SYNDRM1, 0x1e4)
REG32(SEU_VIRTUAL_SYNDRM2, 0x1e8)
REG32(SEU_VIRTUAL_SYNDRM3, 0x1ec)
REG32(SEU_CRC0, 0x1f0)
REG32(SEU_CRC1, 0x1f4)
REG32(SEU_CRC2, 0x1f8)
REG32(SEU_CRC3, 0x1fc)
REG32(CFRAME_FAR_BOT0, 0x200)
REG32(CFRAME_FAR_BOT1, 0x204)
REG32(CFRAME_FAR_BOT2, 0x208)
REG32(CFRAME_FAR_BOT3, 0x20c)
REG32(CFRAME_FAR_TOP0, 0x210)
REG32(CFRAME_FAR_TOP1, 0x214)
REG32(CFRAME_FAR_TOP2, 0x218)
REG32(CFRAME_FAR_TOP3, 0x21c)
REG32(LAST_FRAME_BOT0, 0x220)
    FIELD(LAST_FRAME_BOT0, BLOCKTYPE1_LAST_FRAME_LSB, 20, 12)
    FIELD(LAST_FRAME_BOT0, BLOCKTYPE0_LAST_FRAME, 0, 20)
REG32(LAST_FRAME_BOT1, 0x224)
    FIELD(LAST_FRAME_BOT1, BLOCKTYPE3_LAST_FRAME_LSB, 28, 4)
    FIELD(LAST_FRAME_BOT1, BLOCKTYPE2_LAST_FRAME, 8, 20)
    FIELD(LAST_FRAME_BOT1, BLOCKTYPE1_LAST_FRAME_MSB, 0, 8)
REG32(LAST_FRAME_BOT2, 0x228)
    FIELD(LAST_FRAME_BOT2, BLOCKTYPE3_LAST_FRAME_MSB, 0, 16)
REG32(LAST_FRAME_BOT3, 0x22c)
REG32(LAST_FRAME_TOP0, 0x230)
    FIELD(LAST_FRAME_TOP0, BLOCKTYPE5_LAST_FRAME_LSB, 20, 12)
    FIELD(LAST_FRAME_TOP0, BLOCKTYPE4_LAST_FRAME, 0, 20)
REG32(LAST_FRAME_TOP1, 0x234)
    FIELD(LAST_FRAME_TOP1, BLOCKTYPE6_LAST_FRAME, 8, 20)
    FIELD(LAST_FRAME_TOP1, BLOCKTYPE5_LAST_FRAME_MSB, 0, 8)
REG32(LAST_FRAME_TOP2, 0x238)
REG32(LAST_FRAME_TOP3, 0x23c)

#define CFRAME_REG_R_MAX (R_LAST_FRAME_TOP3 + 1)

#define KEYHOLE_STREAM_4K 0x1000

#define FRAME_NUM_QWORDS 25
#define FRAME_NUM_WORDS (FRAME_NUM_QWORDS * 4) /* 25 * 128 bits */

#define MAX_BLOCKTYPE 6
#define MAX_BLOCKTYPE_FRAMES 0xFFFFF

typedef enum {
    CFRAME_CMD_WCFG = 1,
    CFRAME_CMD_ROWON = 2,
    CFRAME_CMD_ROWOFF = 3,
    CFRAME_CMD_RCFG = 4,
    CFRAME_CMD_DLPARK = 5
} cframe_cmd;

typedef struct XlnxCFrame {
    uint32_t addr;
    uint32_t idx;
    uint32_t data[FRAME_NUM_WORDS];
} XlnxCFrame;

typedef struct CFRAME_REG {
    SysBusDevice parent_obj;
    MemoryRegion iomem;
    MemoryRegion iomem_fdri;
    qemu_irq irq_cfrm_imr;

    /* 128-bit wfifo.  */
    uint32_t wfifo[4];

    uint32_t regs[CFRAME_REG_R_MAX];
    RegisterInfo regs_info[CFRAME_REG_R_MAX];

    bool rowon;
    bool wcfg;
    bool rcfg;

    GArray *cframes;
    XlnxCFrame new_f;

    struct {
        XlnxCfiIf *cfu_fdro;
        uint32_t blktype_num_frames[7];
    } cfg;
    bool row_configured;

} CFRAME_REG;

typedef struct CFRAME_BCAST_REG {
    SysBusDevice parent_obj;
    MemoryRegion iomem_reg;
    MemoryRegion iomem_fdri;

    /* 128-bit wfifo. */
    uint32_t wfifo[4];

    struct {
        XlnxCfiIf *cframe[15];
    } cfg;

} CFRAME_BCAST_REG;

static void cfrm_imr_update_irq(CFRAME_REG *s)
{
    bool pending = s->regs[R_CFRM_ISR0] & ~s->regs[R_CFRM_IMR0];
    qemu_set_irq(s->irq_cfrm_imr, pending);
}

static void cfrm_isr_postw(RegisterInfo *reg, uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);
    cfrm_imr_update_irq(s);
}

static uint64_t cfrm_ier_prew(RegisterInfo *reg, uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    s->regs[R_CFRM_IMR0] &= ~s->regs[R_CFRM_IER0];
    s->regs[R_CFRM_IER0] = 0;
    cfrm_imr_update_irq(s);
    return 0;
}

static uint64_t cfrm_idr_prew(RegisterInfo *reg, uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    s->regs[R_CFRM_IMR0] |= s->regs[R_CFRM_IDR0];
    s->regs[R_CFRM_IDR0] = 0;
    cfrm_imr_update_irq(s);
    return 0;
}

static uint64_t cfrm_itr_prew(RegisterInfo *reg, uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    s->regs[R_CFRM_ISR0] |= s->regs[R_CFRM_ITR0];
    s->regs[R_CFRM_ITR0] = 0;
    cfrm_imr_update_irq(s);
    return 0;
}

static void cframe_incr_far(CFRAME_REG *s)
{
    uint32_t faddr = ARRAY_FIELD_EX32(s->regs, FAR0, FRAME_ADDR);
    uint32_t blktype = ARRAY_FIELD_EX32(s->regs, FAR0, BLOCKTYPE);

    assert(blktype <= MAX_BLOCKTYPE);

    faddr++;
    if (faddr > s->cfg.blktype_num_frames[blktype]) {
        /* Restart from 0 and increment block type */
        faddr = 0;
        blktype++;

        assert(blktype <= MAX_BLOCKTYPE);

        ARRAY_FIELD_DP32(s->regs, FAR0, BLOCKTYPE, blktype);
    }

    ARRAY_FIELD_DP32(s->regs, FAR0, FRAME_ADDR, faddr);
}

static XlnxCFrame *cframes_get_frame(CFRAME_REG *s, uint32_t addr)
{
    for (int i = 0; i < s->cframes->len; i++) {
        XlnxCFrame *f = &g_array_index(s->cframes, XlnxCFrame, i);

        if (f->addr == addr) {
            return f;
        }
    }
    return NULL;
}

static void cfrm_fdri_post_write(RegisterInfo *reg, uint64_t val)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    if (s->row_configured && s->rowon && s->wcfg) {
        XlnxCFrame *new_f = &s->new_f;

        new_f->data[new_f->idx++] = s->regs[R_FDRI0];
        new_f->data[new_f->idx++] = s->regs[R_FDRI1];
        new_f->data[new_f->idx++] = s->regs[R_FDRI2];
        new_f->data[new_f->idx++] = s->regs[R_FDRI3];

        assert(new_f->idx <= FRAME_NUM_WORDS);

        if (new_f->idx == FRAME_NUM_WORDS) {
            XlnxCFrame *cur_f;

            /* Include block type and frame address */
            new_f->addr = extract32(s->regs[R_FAR0], 0, 23);

            cur_f = cframes_get_frame(s, new_f->addr);

            if (cur_f) {
                /* Overwrite current frame */
                cur_f[0] = new_f[0];
            } else {
                g_array_append_val(s->cframes, new_f[0]);
            }

            cframe_incr_far(s);

            /* Clear out new_f */
            memset(new_f, 0, sizeof(*new_f));
        }
    }
}

static void cfrm_readout_frames(CFRAME_REG *s, uint32_t start_addr,
                                uint32_t end_addr)
{
    for (uint32_t addr = start_addr; addr < end_addr; addr++) {
        XlnxCFrame *f = cframes_get_frame(s, addr);

        for (int i = 0; i < FRAME_NUM_WORDS; i += 4) {
            XlnxCfiPacket pkt = {};

            /* If frame was found transmit the data */
            if (f) {
                pkt.data[0] = f->data[i];
                pkt.data[1] = f->data[i + 1];
                pkt.data[2] = f->data[i + 2];
                pkt.data[3] = f->data[i + 3];
            }

            if (s->cfg.cfu_fdro) {
                xlnx_cfi_transfer_packet(s->cfg.cfu_fdro, &pkt);
            }
        }
    }
}

static void cfrm_frcnt_post_write(RegisterInfo *reg, uint64_t val)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    if (s->row_configured && s->rowon && s->rcfg) {
        uint32_t start_addr = extract32(s->regs[R_FAR0], 0, 23);
        uint32_t end_addr = start_addr + s->regs[R_FRCNT0] / FRAME_NUM_QWORDS;

        cfrm_readout_frames(s, start_addr, end_addr);
    }
}

static void cfrm_cmd_post_write(RegisterInfo *reg, uint64_t val)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    if (s->row_configured) {
        uint8_t cmd = ARRAY_FIELD_EX32(s->regs, CMD0, CMD);

        switch (cmd) {
        case CFRAME_CMD_WCFG:
            s->wcfg = true;
            break;
        case CFRAME_CMD_ROWON:
            s->rowon = true;
            break;
        case CFRAME_CMD_ROWOFF:
            s->rowon = false;
            break;
        case CFRAME_CMD_RCFG:
            s->rcfg = true;
            break;
        case CFRAME_CMD_DLPARK:
            s->wcfg = false;
            s->rcfg = false;
            break;
        default:
            break;
        };
    }
}

static uint64_t cfrm_last_frame_bot_post_read(RegisterInfo *reg,
                                              uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);
    uint64_t val = 0;

    switch (reg->access->addr) {
    case A_LAST_FRAME_BOT0:
        val = FIELD_DP32(val, LAST_FRAME_BOT0, BLOCKTYPE1_LAST_FRAME_LSB,
                         s->cfg.blktype_num_frames[1]);
        val = FIELD_DP32(val, LAST_FRAME_BOT0, BLOCKTYPE0_LAST_FRAME,
                         s->cfg.blktype_num_frames[0]);
        break;
    case A_LAST_FRAME_BOT1:
        val = FIELD_DP32(val, LAST_FRAME_BOT1, BLOCKTYPE3_LAST_FRAME_LSB,
                         s->cfg.blktype_num_frames[3]);
        val = FIELD_DP32(val, LAST_FRAME_BOT1, BLOCKTYPE2_LAST_FRAME,
                         s->cfg.blktype_num_frames[2]);
        val = FIELD_DP32(val, LAST_FRAME_BOT1, BLOCKTYPE1_LAST_FRAME_MSB,
                         (s->cfg.blktype_num_frames[1] >> 12));
        break;
    case A_LAST_FRAME_BOT2:
        val = FIELD_DP32(val, LAST_FRAME_BOT2, BLOCKTYPE3_LAST_FRAME_MSB,
                         (s->cfg.blktype_num_frames[3] >> 4));
        break;
    case A_LAST_FRAME_BOT3:
    default:
        break;
    }

    return val;
}

static uint64_t cfrm_last_frame_top_post_read(RegisterInfo *reg,
                                              uint64_t val64)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);
    uint64_t val = 0;

    switch (reg->access->addr) {
    case A_LAST_FRAME_TOP0:
        val = FIELD_DP32(val, LAST_FRAME_TOP0, BLOCKTYPE5_LAST_FRAME_LSB,
                         s->cfg.blktype_num_frames[5]);
        val = FIELD_DP32(val, LAST_FRAME_TOP0, BLOCKTYPE4_LAST_FRAME,
                         s->cfg.blktype_num_frames[4]);
        break;
    case A_LAST_FRAME_TOP1:
        val = FIELD_DP32(val, LAST_FRAME_TOP1, BLOCKTYPE6_LAST_FRAME,
                         s->cfg.blktype_num_frames[6]);
        val = FIELD_DP32(val, LAST_FRAME_TOP1, BLOCKTYPE5_LAST_FRAME_MSB,
                         (s->cfg.blktype_num_frames[5] >> 12));
        break;
    case A_LAST_FRAME_TOP2:
    case A_LAST_FRAME_BOT3:
    default:
        break;
    }

    return val;
}

static void cfrm_far_sfr_post_write(RegisterInfo *reg, uint64_t val)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(reg->opaque);

    if (s->row_configured && s->rowon && s->rcfg) {
        uint32_t start_addr = extract32(s->regs[R_FAR_SFR0], 0, 23);

        /* Readback 1 frame */
        cfrm_readout_frames(s, start_addr, start_addr + 1);
    }
}

static const RegisterAccessInfo cframe_reg_regs_info[] = {
    {   .name = "CRC0",  .addr = A_CRC0,
        .rsvd = 0x00000000,
    },{ .name = "CRC1",  .addr = A_CRC0,
        .rsvd = 0xffffffff,
    },{ .name = "CRC2",  .addr = A_CRC0,
        .rsvd = 0xffffffff,
    },{ .name = "CRC3",  .addr = A_CRC0,
        .rsvd = 0xffffffff,
    },{ .name = "FAR0",  .addr = A_FAR0,
        .rsvd = 0xfe000000,
    },{ .name = "FAR1",  .addr = A_FAR1,
        .rsvd = 0xffffffff,
    },{ .name = "FAR2",  .addr = A_FAR2,
        .rsvd = 0xffffffff,
    },{ .name = "FAR3",  .addr = A_FAR3,
        .rsvd = 0xffffffff,
    },{ .name = "FAR_SFR0",  .addr = A_FAR_SFR0,
        .rsvd = 0xff800000,
    },{ .name = "FAR_SFR1",  .addr = A_FAR_SFR1,
        .rsvd = 0xffffffff,
    },{ .name = "FAR_SFR2",  .addr = A_FAR_SFR2,
        .rsvd = 0xffffffff,
    },{ .name = "FAR_SFR3",  .addr = A_FAR_SFR3,
        .rsvd = 0xffffffff,
        .post_write = cfrm_far_sfr_post_write,
    },{ .name = "FDRI0",  .addr = A_FDRI0,
    },{ .name = "FDRI1",  .addr = A_FDRI1,
    },{ .name = "FDRI2",  .addr = A_FDRI2,
    },{ .name = "FDRI3",  .addr = A_FDRI3,
        .post_write = cfrm_fdri_post_write,
    },{ .name = "FRCNT0",  .addr = A_FRCNT0,
        .rsvd = 0x00000000,
    },{ .name = "FRCNT1",  .addr = A_FRCNT1,
        .rsvd = 0xffffffff,
    },{ .name = "FRCNT2",  .addr = A_FRCNT2,
        .rsvd = 0xffffffff,
    },{ .name = "FRCNT3",  .addr = A_FRCNT3,
        .rsvd = 0xffffffff,
        .post_write = cfrm_frcnt_post_write
    },{ .name = "CMD0",  .addr = A_CMD0,
        .rsvd = 0xffffffe0,
    },{ .name = "CMD1",  .addr = A_CMD1,
        .rsvd = 0xffffffff,
    },{ .name = "CMD2",  .addr = A_CMD2,
        .rsvd = 0xffffffff,
    },{ .name = "CMD3",  .addr = A_CMD3,
        .rsvd = 0xffffffff,
        .post_write = cfrm_cmd_post_write
    },{ .name = "CR_MASK0",  .addr = A_CR_MASK0,
        .rsvd = 0x00000000,
    },{ .name = "CR_MASK1",  .addr = A_CR_MASK1,
        .rsvd = 0x00000000,
    },{ .name = "CR_MASK2",  .addr = A_CR_MASK2,
        .rsvd = 0x00000000,
    },{ .name = "CR_MASK3",  .addr = A_CR_MASK3,
        .rsvd = 0xffffffff,
    },{ .name = "CTL0",  .addr = A_CTL0,
        .rsvd = 0xfffffff8,
    },{ .name = "CTL1",  .addr = A_CTL1,
        .rsvd = 0xffffffff,
    },{ .name = "CTL2",  .addr = A_CTL2,
        .rsvd = 0xffffffff,
    },{ .name = "CTL3",  .addr = A_CTL3,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_ISR0",  .addr = A_CFRM_ISR0,
        .rsvd = 0xffc04000,
        .w1c = 0x3bfff,
    },{ .name = "CFRM_ISR1",  .addr = A_CFRM_ISR1,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_ISR2",  .addr = A_CFRM_ISR2,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_ISR3",  .addr = A_CFRM_ISR3,
        .rsvd = 0xffffffff,
        .post_write = cfrm_isr_postw,
    },{ .name = "CFRM_IMR0",  .addr = A_CFRM_IMR0,
        .rsvd = 0xffc04000,
        .ro = 0xfffff,
        .reset = 0x3bfff,
    },{ .name = "CFRM_IMR1",  .addr = A_CFRM_IMR1,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IMR2",  .addr = A_CFRM_IMR2,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IMR3",  .addr = A_CFRM_IMR3,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IER0",  .addr = A_CFRM_IER0,
        .rsvd = 0xffc04000,
    },{ .name = "CFRM_IER1",  .addr = A_CFRM_IER1,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IER2",  .addr = A_CFRM_IER2,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IER3",  .addr = A_CFRM_IER3,
        .rsvd = 0xffffffff,
        .pre_write = cfrm_ier_prew,
    },{ .name = "CFRM_IDR0",  .addr = A_CFRM_IDR0,
        .rsvd = 0xffc04000,
    },{ .name = "CFRM_IDR1",  .addr = A_CFRM_IDR1,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IDR2",  .addr = A_CFRM_IDR2,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_IDR3",  .addr = A_CFRM_IDR3,
        .rsvd = 0xffffffff,
        .pre_write = cfrm_idr_prew,
    },{ .name = "CFRM_ITR0",  .addr = A_CFRM_ITR0,
        .rsvd = 0xffc04000,
    },{ .name = "CFRM_ITR1",  .addr = A_CFRM_ITR1,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_ITR2",  .addr = A_CFRM_ITR2,
        .rsvd = 0xffffffff,
    },{ .name = "CFRM_ITR3",  .addr = A_CFRM_ITR3,
        .rsvd = 0xffffffff,
        .pre_write = cfrm_itr_prew,
    },{ .name = "SEU_SYNDRM00",  .addr = A_SEU_SYNDRM00,
    },{ .name = "SEU_SYNDRM01",  .addr = A_SEU_SYNDRM01,
    },{ .name = "SEU_SYNDRM02",  .addr = A_SEU_SYNDRM02,
    },{ .name = "SEU_SYNDRM03",  .addr = A_SEU_SYNDRM03,
    },{ .name = "SEU_SYNDRM10",  .addr = A_SEU_SYNDRM10,
    },{ .name = "SEU_SYNDRM11",  .addr = A_SEU_SYNDRM11,
    },{ .name = "SEU_SYNDRM12",  .addr = A_SEU_SYNDRM12,
    },{ .name = "SEU_SYNDRM13",  .addr = A_SEU_SYNDRM13,
    },{ .name = "SEU_SYNDRM20",  .addr = A_SEU_SYNDRM20,
    },{ .name = "SEU_SYNDRM21",  .addr = A_SEU_SYNDRM21,
    },{ .name = "SEU_SYNDRM22",  .addr = A_SEU_SYNDRM22,
    },{ .name = "SEU_SYNDRM23",  .addr = A_SEU_SYNDRM23,
    },{ .name = "SEU_SYNDRM30",  .addr = A_SEU_SYNDRM30,
    },{ .name = "SEU_SYNDRM31",  .addr = A_SEU_SYNDRM31,
    },{ .name = "SEU_SYNDRM32",  .addr = A_SEU_SYNDRM32,
    },{ .name = "SEU_SYNDRM33",  .addr = A_SEU_SYNDRM33,
    },{ .name = "SEU_VIRTUAL_SYNDRM0",  .addr = A_SEU_VIRTUAL_SYNDRM0,
    },{ .name = "SEU_VIRTUAL_SYNDRM1",  .addr = A_SEU_VIRTUAL_SYNDRM1,
    },{ .name = "SEU_VIRTUAL_SYNDRM2",  .addr = A_SEU_VIRTUAL_SYNDRM2,
    },{ .name = "SEU_VIRTUAL_SYNDRM3",  .addr = A_SEU_VIRTUAL_SYNDRM3,
    },{ .name = "SEU_CRC0",  .addr = A_SEU_CRC0,
    },{ .name = "SEU_CRC1",  .addr = A_SEU_CRC1,
    },{ .name = "SEU_CRC2",  .addr = A_SEU_CRC2,
    },{ .name = "SEU_CRC3",  .addr = A_SEU_CRC3,
    },{ .name = "CFRAME_FAR_BOT0",  .addr = A_CFRAME_FAR_BOT0,
    },{ .name = "CFRAME_FAR_BOT1",  .addr = A_CFRAME_FAR_BOT1,
    },{ .name = "CFRAME_FAR_BOT2",  .addr = A_CFRAME_FAR_BOT2,
    },{ .name = "CFRAME_FAR_BOT3",  .addr = A_CFRAME_FAR_BOT3,
    },{ .name = "CFRAME_FAR_TOP0",  .addr = A_CFRAME_FAR_TOP0,
    },{ .name = "CFRAME_FAR_TOP1",  .addr = A_CFRAME_FAR_TOP1,
    },{ .name = "CFRAME_FAR_TOP2",  .addr = A_CFRAME_FAR_TOP2,
    },{ .name = "CFRAME_FAR_TOP3",  .addr = A_CFRAME_FAR_TOP3,
    },{ .name = "LAST_FRAME_BOT0",  .addr = A_LAST_FRAME_BOT0,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_bot_post_read,
    },{ .name = "LAST_FRAME_BOT1",  .addr = A_LAST_FRAME_BOT1,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_bot_post_read,
    },{ .name = "LAST_FRAME_BOT2",  .addr = A_LAST_FRAME_BOT2,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_bot_post_read,
    },{ .name = "LAST_FRAME_BOT3",  .addr = A_LAST_FRAME_BOT3,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_bot_post_read,
    },{ .name = "LAST_FRAME_TOP0",  .addr = A_LAST_FRAME_TOP0,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_top_post_read,
    },{ .name = "LAST_FRAME_TOP1",  .addr = A_LAST_FRAME_TOP1,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_top_post_read,
    },{ .name = "LAST_FRAME_TOP2",  .addr = A_LAST_FRAME_TOP2,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_top_post_read,
    },{ .name = "LAST_FRAME_TOP3",  .addr = A_LAST_FRAME_TOP3,
        .ro = 0xffffffff,
        .post_read = cfrm_last_frame_top_post_read,
    }
};

static void cframe_reg_cfi_transfer_packet(XlnxCfiIf *cfi_if,
                                           XlnxCfiPacket *pkt)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(cfi_if);
    uint64_t we = MAKE_64BIT_MASK(0, 4 * 8);

    if (!s->row_configured) {
        return;
    }

    switch (pkt->reg_addr) {
    case CFRAME_FAR:
        s->regs[R_FAR0] = pkt->data[0];
        break;
    case CFRAME_SFR:
        s->regs[R_FAR_SFR0] = pkt->data[0];
        register_write(&s->regs_info[R_FAR_SFR3], 0,
                       we, object_get_typename(OBJECT(s)),
                       XILINX_CFRAME_REG_ERR_DEBUG);
        break;
    case CFRAME_FDRI:
    {
        s->regs[R_FDRI0] = pkt->data[0];
        s->regs[R_FDRI1] = pkt->data[1];
        s->regs[R_FDRI2] = pkt->data[2];
        register_write(&s->regs_info[R_FDRI3], pkt->data[3],
                       we, object_get_typename(OBJECT(s)),
                       XILINX_CFRAME_REG_ERR_DEBUG);
        break;
    }
    case CFRAME_CMD:
        ARRAY_FIELD_DP32(s->regs, CMD0, CMD, pkt->data[0]);

        register_write(&s->regs_info[R_CMD3], 0,
                       we, object_get_typename(OBJECT(s)),
                       XILINX_CFRAME_REG_ERR_DEBUG);
        break;
    default:
        break;
    }
}

static uint64_t cframe_reg_fdri_read(void *opaque, hwaddr addr, unsigned size)
{
    qemu_log_mask(LOG_GUEST_ERROR, "%s: Unsupported read from addr=%"
                  HWADDR_PRIx "\n", __func__, addr);
    return 0;
}

static void cframe_reg_fdri_write(void *opaque, hwaddr addr, uint64_t value,
                      unsigned size)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(opaque);
    unsigned int idx;

    /* 4 32bit words. */
    idx = (addr >> 2) & 3;

    s->wfifo[idx] = value;

    /* Writing to the top word triggers the forwarding to the FDRI. */
    if (idx == 3) {
        uint64_t we = MAKE_64BIT_MASK(0, 4 * 8);

        s->regs[R_FDRI0] = s->wfifo[0];
        s->regs[R_FDRI1] = s->wfifo[1];
        s->regs[R_FDRI2] = s->wfifo[2];
        register_write(&s->regs_info[R_FDRI3], s->wfifo[3],
                       we, object_get_typename(OBJECT(s)),
                       XILINX_CFRAME_REG_ERR_DEBUG);

        memset(s->wfifo, 0, 4 * sizeof(uint32_t));
    }
}

static void cframe_reg_reset_enter(Object *obj, ResetType type)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(obj);
    unsigned int i;

    for (i = 0; i < ARRAY_SIZE(s->regs_info); ++i) {
        register_reset(&s->regs_info[i]);
    }
}

static void cframe_reg_reset_hold(Object *obj)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(obj);

    cfrm_imr_update_irq(s);
}

static const MemoryRegionOps cframe_reg_ops = {
    .read = register_read_memory,
    .write = register_write_memory,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static const MemoryRegionOps cframe_reg_fdri_ops = {
    .read = cframe_reg_fdri_read,
    .write = cframe_reg_fdri_write,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static uint64_t cframes_bcast_reg_read(void *opaque, hwaddr addr, unsigned size)
{
    qemu_log_mask(LOG_GUEST_ERROR, "%s: Unsupported read from addr=%"
                  HWADDR_PRIx "\n", __func__, addr);
    return 0;
}

static void cframes_bcast_reg_write(void *opaque, hwaddr addr, uint64_t value,
                      unsigned size)
{
    CFRAME_BCAST_REG *s = XILINX_CFRAME_BCAST_REG(opaque);
    unsigned int idx;

    /* 4 32bit words. */
    idx = (addr >> 2) & 3;

    s->wfifo[idx] = value;

    /* Writing to the top word triggers the transmit onto CFI. */
    if (idx == 3) {
        uint32_t reg_addr = extract32(addr, 4, 6);
        XlnxCfiPacket pkt = {
            .reg_addr = reg_addr,
            .data[0] = s->wfifo[0],
            .data[1] = s->wfifo[1],
            .data[2] = s->wfifo[2],
            .data[3] = s->wfifo[3]
        };

        for (int i = 0; i < ARRAY_SIZE(s->cfg.cframe); i++) {
            if (s->cfg.cframe[i]) {
                xlnx_cfi_transfer_packet(s->cfg.cframe[i], &pkt);
            }
        }

        memset(s->wfifo, 0, 4 * sizeof(uint32_t));
    }
}

static uint64_t cframes_bcast_fdri_read(void *opaque, hwaddr addr,
                                        unsigned size)
{
    qemu_log_mask(LOG_GUEST_ERROR, "%s: Unsupported read from addr=%"
                  HWADDR_PRIx "\n", __func__, addr);
    return 0;
}

static void cframes_bcast_fdri_write(void *opaque, hwaddr addr, uint64_t value,
                      unsigned size)
{
    CFRAME_BCAST_REG *s = XILINX_CFRAME_BCAST_REG(opaque);
    unsigned int idx;

    /* 4 32bit words. */
    idx = (addr >> 2) & 3;

    s->wfifo[idx] = value;

    /* Writing to the top word triggers the transmit onto CFI. */
    if (idx == 3) {
        XlnxCfiPacket pkt = {
            .reg_addr = CFRAME_FDRI,
            .data[0] = s->wfifo[0],
            .data[1] = s->wfifo[1],
            .data[2] = s->wfifo[2],
            .data[3] = s->wfifo[3]
        };

        for (int i = 0; i < ARRAY_SIZE(s->cfg.cframe); i++) {
            if (s->cfg.cframe[i]) {
                xlnx_cfi_transfer_packet(s->cfg.cframe[i], &pkt);
            }
        }

        memset(s->wfifo, 0, 4 * sizeof(uint32_t));
    }
}

static const MemoryRegionOps cframes_bcast_reg_reg_ops = {
    .read = cframes_bcast_reg_read,
    .write = cframes_bcast_reg_write,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static const MemoryRegionOps cframes_bcast_reg_fdri_ops = {
    .read = cframes_bcast_fdri_read,
    .write = cframes_bcast_fdri_write,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static void cframe_reg_realize(DeviceState *dev, Error **errp)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(dev);

    for (int i = 0; i < ARRAY_SIZE(s->cfg.blktype_num_frames); i++) {
        if (s->cfg.blktype_num_frames[i] > MAX_BLOCKTYPE_FRAMES) {
            error_setg(errp,
                       "blktype-frames%d > 0xFFFFF (max frame per block)",
                       i);
            return;
        }
        if (s->cfg.blktype_num_frames[i]) {
            s->row_configured = true;
        }
    }
}

static void cframe_reg_init(Object *obj)
{
    CFRAME_REG *s = XILINX_CFRAME_REG(obj);
    SysBusDevice *sbd = SYS_BUS_DEVICE(obj);
    RegisterInfoArray *reg_array;

    memory_region_init(&s->iomem, obj, TYPE_XILINX_CFRAME_REG,
                       CFRAME_REG_R_MAX * 4);
    reg_array =
        register_init_block32(DEVICE(obj), cframe_reg_regs_info,
                              ARRAY_SIZE(cframe_reg_regs_info),
                              s->regs_info, s->regs,
                              &cframe_reg_ops,
                              XILINX_CFRAME_REG_ERR_DEBUG,
                              CFRAME_REG_R_MAX * 4);
    memory_region_add_subregion(&s->iomem,
                                0x0,
                                &reg_array->mem);
    sysbus_init_mmio(sbd, &s->iomem);
    memory_region_init_io(&s->iomem_fdri, obj, &cframe_reg_fdri_ops, s,
                          TYPE_XILINX_CFRAME_REG "-fdri", KEYHOLE_STREAM_4K);
    sysbus_init_mmio(sbd, &s->iomem_fdri);
    sysbus_init_irq(sbd, &s->irq_cfrm_imr);

    s->cframes = g_array_new(FALSE, FALSE, sizeof(XlnxCFrame));
}

static const VMStateDescription vmstate_cframe_reg = {
    .name = TYPE_XILINX_CFRAME_REG,
    .version_id = 1,
    .minimum_version_id = 1,
    .fields = (VMStateField[]) {
        VMSTATE_UINT32_ARRAY(regs, CFRAME_REG, CFRAME_REG_R_MAX),
        VMSTATE_END_OF_LIST(),
    }
};

static Property cframe_regs_props[] = {
    /* Kept for backwards compatibility */
    DEFINE_PROP_LINK("cfu", CFRAME_REG, cfg.cfu_fdro,
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cfu-fdro", CFRAME_REG, cfg.cfu_fdro,
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_UINT32("blktype0-frames", CFRAME_REG,
                       cfg.blktype_num_frames[0], 0),
    DEFINE_PROP_UINT32("blktype1-frames", CFRAME_REG,
                       cfg.blktype_num_frames[1], 0),
    DEFINE_PROP_UINT32("blktype2-frames", CFRAME_REG,
                       cfg.blktype_num_frames[2], 0),
    DEFINE_PROP_UINT32("blktype3-frames", CFRAME_REG,
                       cfg.blktype_num_frames[3], 0),
    DEFINE_PROP_UINT32("blktype4-frames", CFRAME_REG,
                       cfg.blktype_num_frames[4], 0),
    DEFINE_PROP_UINT32("blktype5-frames", CFRAME_REG,
                       cfg.blktype_num_frames[5], 0),
    DEFINE_PROP_UINT32("blktype6-frames", CFRAME_REG,
                       cfg.blktype_num_frames[6], 0),
    DEFINE_PROP_END_OF_LIST(),
};

static void cframe_bcast_reg_init(Object *obj)
{
    CFRAME_BCAST_REG *s = XILINX_CFRAME_BCAST_REG(obj);
    SysBusDevice *sbd = SYS_BUS_DEVICE(obj);

    memory_region_init_io(&s->iomem_reg, obj, &cframes_bcast_reg_reg_ops, s,
                          TYPE_XILINX_CFRAME_BCAST_REG, KEYHOLE_STREAM_4K);
    memory_region_init_io(&s->iomem_fdri, obj, &cframes_bcast_reg_fdri_ops, s,
                          TYPE_XILINX_CFRAME_BCAST_REG "-fdri",
                          KEYHOLE_STREAM_4K);
    sysbus_init_mmio(sbd, &s->iomem_reg);
    sysbus_init_mmio(sbd, &s->iomem_fdri);
}

static Property cframe_bcast_regs_props[] = {
    DEFINE_PROP_LINK("cframe0", CFRAME_BCAST_REG, cfg.cframe[0],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe1", CFRAME_BCAST_REG, cfg.cframe[1],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe2", CFRAME_BCAST_REG, cfg.cframe[2],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe3", CFRAME_BCAST_REG, cfg.cframe[3],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe4", CFRAME_BCAST_REG, cfg.cframe[4],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe5", CFRAME_BCAST_REG, cfg.cframe[5],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe6", CFRAME_BCAST_REG, cfg.cframe[6],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe7", CFRAME_BCAST_REG, cfg.cframe[7],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe8", CFRAME_BCAST_REG, cfg.cframe[8],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe9", CFRAME_BCAST_REG, cfg.cframe[9],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe10", CFRAME_BCAST_REG, cfg.cframe[10],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe11", CFRAME_BCAST_REG, cfg.cframe[11],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe12", CFRAME_BCAST_REG, cfg.cframe[12],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe13", CFRAME_BCAST_REG, cfg.cframe[13],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_LINK("cframe14", CFRAME_BCAST_REG, cfg.cframe[14],
                     TYPE_XLNX_CFI_IF, XlnxCfiIf *),
    DEFINE_PROP_END_OF_LIST(),
};

static void cframe_reg_class_init(ObjectClass *klass, void *data)
{
    ResettableClass *rc = RESETTABLE_CLASS(klass);
    DeviceClass *dc = DEVICE_CLASS(klass);
    XlnxCfiIfClass *xcic = XLNX_CFI_IF_CLASS(klass);

    dc->vmsd = &vmstate_cframe_reg;
    dc->realize = cframe_reg_realize;
    rc->phases.enter = cframe_reg_reset_enter;
    rc->phases.hold = cframe_reg_reset_hold;
    device_class_set_props(dc, cframe_regs_props);
    xcic->cfi_transfer_packet = cframe_reg_cfi_transfer_packet;
}

static void cframe_bcast_reg_class_init(ObjectClass *klass, void *data)
{
    DeviceClass *dc = DEVICE_CLASS(klass);

    device_class_set_props(dc, cframe_bcast_regs_props);
}

static const TypeInfo cframe_reg_info = {
    .name          = TYPE_XILINX_CFRAME_REG,
    .parent        = TYPE_SYS_BUS_DEVICE,
    .instance_size = sizeof(CFRAME_REG),
    .class_init    = cframe_reg_class_init,
    .instance_init = cframe_reg_init,
    .interfaces = (InterfaceInfo[]) {
        { TYPE_XLNX_CFI_IF },
        { }
    }
};

static const TypeInfo cframe_bcast_reg_info = {
    .name          = TYPE_XILINX_CFRAME_BCAST_REG,
    .parent        = TYPE_SYS_BUS_DEVICE,
    .instance_size = sizeof(CFRAME_BCAST_REG),
    .class_init    = cframe_bcast_reg_class_init,
    .instance_init = cframe_bcast_reg_init,
};

static void cframe_reg_register_types(void)
{
    type_register_static(&cframe_reg_info);
    type_register_static(&cframe_bcast_reg_info);
}

type_init(cframe_reg_register_types)
