/*
 * QEMU model of the AES PMC SEC AES Engine
 *
 * Copyright (c) 2017 Xilinx Inc.
 *
 * Partially Autogenerated by xregqemu.py 2017-10-05.
 *    written by Sai Pavan Boddu <saipava@xilinx.com>
 *               Edgar E. Iglesias <edgari@xilinx.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "qemu/osdep.h"
#include "hw/sysbus.h"
#include "hw/register.h"
#include "hw/hw.h"
#include "qemu/bitops.h"
#include "qemu/log.h"
#include "qapi/error.h"
#include "qapi/qmp/qerror.h"
#include "qemu/error-report.h"
#include "migration/vmstate.h"
#include "hw/qdev-properties.h"

#include "hw/stream.h"
#include "sysemu/dma.h"
#include "hw/zynqmp_aes_key.h"
#include "hw/fdt_generic_util.h"
#include "hw/misc/xlnx-aes.h"

#ifndef XILINX_AES_ERR_DEBUG
#define XILINX_AES_ERR_DEBUG 0
#endif

#define TYPE_XILINX_AES "xlnx,versal-aes"
#define TYPE_XILINX_PMC_KEY_SINK "xlnx.pmc-key-sink"

#define XILINX_AES(obj) \
     OBJECT_CHECK(Zynq3AES, (obj), TYPE_XILINX_AES)

#define XILINX_PMC_KEY_SINK(obj) \
     OBJECT_CHECK(PMCKeySink, (obj), TYPE_XILINX_PMC_KEY_SINK)

#define DPRINT(fmt, args...) do { \
        if (XILINX_AES_ERR_DEBUG) { \
            qemu_log("%s: " fmt, __func__, ## args);\
        } \
    } while (0)

/* Key select values for AES_KEY_SEL register */
enum key_sel {
    KEY_SEL_BBRAM_KEY         = 0xBBDE6600,
    KEY_SEL_BBRAM_RD_KEY      = 0xBBDE8200,
    KEY_SEL_BH_KEY            = 0xBDB06600,
    KEY_SEL_BH_RD_KEY         = 0xBDB08200,
    KEY_SEL_EFUSE_KEY         = 0xEFDE6600,
    KEY_SEL_EFUSE_RED_KEY     = 0xEFDE8200,
    KEY_SEL_EFUSE_USR_KEY0    = 0xEF856601,
    KEY_SEL_EFUSE_USR_KEY1    = 0xEF856602,
    KEY_SEL_EFUSE_USR_RD_KEY0 = 0xEF858201,
    KEY_SEL_EFUSE_USR_RD_KEY1 = 0xEF858202,
    KEY_SEL_KUP_KEY           = 0xBDC98200,
    KEY_SEL_FAMILY_KEY        = 0xFEDE8200,
    KEY_SEL_PUF_KEY           = 0xDBDE8200,
    KEY_SEL_USR_KEY_0         = 0xBD858201,
    KEY_SEL_USR_KEY_1         = 0xBD858202,
    KEY_SEL_USR_KEY_2         = 0xBD858204,
    KEY_SEL_USR_KEY_3         = 0xBD858208,
    KEY_SEL_USR_KEY_4         = 0xBD858210,
    KEY_SEL_USR_KEY_5         = 0xBD858220,
    KEY_SEL_USR_KEY_6         = 0xBD858240,
    KEY_SEL_USR_KEY_7         = 0xBD858280,
};

REG32(AES_STATUS, 0x0)
    FIELD(AES_STATUS, CM_ENABLED, 12, 1)
    FIELD(AES_STATUS, BLACK_KEY_DEC_DONE, 5, 1)
    FIELD(AES_STATUS, KEY_INIT_DONE, 4, 1)
    FIELD(AES_STATUS, GCM_TAG_PASS, 3, 1)
    FIELD(AES_STATUS, DONE, 2, 1)
    FIELD(AES_STATUS, READY, 1, 1)
    FIELD(AES_STATUS, BUSY, 0, 1)
REG32(AES_KEY_SEL, 0x4)
REG32(AES_KEY_LOAD, 0x8)
    FIELD(AES_KEY_LOAD, VAL, 0, 1)
REG32(AES_START_MSG, 0xc)
    FIELD(AES_START_MSG, VAL, 0, 1)
REG32(AES_SOFT_RST, 0x10)
    FIELD(AES_SOFT_RST, VAL, 0, 1)
REG32(AES_KEY_CLEAR, 0x14)
    FIELD(AES_KEY_CLEAR, RESERVED_2, 22, 10)
    FIELD(AES_KEY_CLEAR, PUF_KEY, 21, 1)
    FIELD(AES_KEY_CLEAR, BBRAM_RED_KEY, 20, 1)
    FIELD(AES_KEY_CLEAR, BH_RED_KEY, 19, 1)
    FIELD(AES_KEY_CLEAR, BH_KEY, 18, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_USER_RED_KEY_1, 17, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_USER_RED_KEY_0, 16, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_RED_KEY, 15, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_USER_KEY_1, 14, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_USER_KEY_0, 13, 1)
    FIELD(AES_KEY_CLEAR, EFUSE_KEY, 12, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_7, 11, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_6, 10, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_5, 9, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_4, 8, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_3, 7, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_2, 6, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_1, 5, 1)
    FIELD(AES_KEY_CLEAR, USER_KEY_0, 4, 1)
    FIELD(AES_KEY_CLEAR, RESERVED_1, 2, 2)
    FIELD(AES_KEY_CLEAR, KUP_KEY, 1, 1)
    FIELD(AES_KEY_CLEAR, AES_KEY_ZEROIZE, 0, 1)
REG32(AES_MODE, 0x18)
    FIELD(AES_MODE, ENC_DEC_N, 0, 1)
REG32(AES_KUP_WR, 0x1c)
    FIELD(AES_KUP_WR, IV_SAVE, 1, 1)
    FIELD(AES_KUP_WR, KEY_SAVE, 0, 1)
REG32(AES_IV_0, 0x40)
REG32(AES_IV_1, 0x44)
REG32(AES_IV_2, 0x48)
REG32(AES_IV_3, 0x4c)
REG32(AES_KEY_SIZE, 0x50)
    FIELD(AES_KEY_SIZE, VAL, 0, 2)
REG32(AES_KEY_DEC, 0x58)
REG32(KEY_DEC_TRIG, 0x5c)
    FIELD(KEY_DEC_TRIG, VAL, 0, 1)
REG32(KEY_DEC_SEL, 0x60)
    FIELD(KEY_DEC_SEL, VAL, 0, 3)
REG32(KEY_ZEROED_STATUS, 0x64)
    FIELD(KEY_ZEROED_STATUS, RESERVED_2, 22, 10)
    FIELD(KEY_ZEROED_STATUS, PUF_KEY, 21, 1)
    FIELD(KEY_ZEROED_STATUS, BBRAM_RED_KEY, 20, 1)
    FIELD(KEY_ZEROED_STATUS, BH_RED_KEY, 19, 1)
    FIELD(KEY_ZEROED_STATUS, BH_KEY, 18, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_USER_RED_KEY_1, 17, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_USER_RED_KEY_0, 16, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_RED_KEY, 15, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_USER_KEY_1, 14, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_USER_KEY_0, 13, 1)
    FIELD(KEY_ZEROED_STATUS, EFUSE_KEY, 12, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_7, 11, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_6, 10, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_5, 9, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_4, 8, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_3, 7, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_2, 6, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_1, 5, 1)
    FIELD(KEY_ZEROED_STATUS, USER_KEY_0, 4, 1)
    FIELD(KEY_ZEROED_STATUS, RESERVED_1, 2, 2)
    FIELD(KEY_ZEROED_STATUS, KUP_KEY, 1, 1)
    FIELD(KEY_ZEROED_STATUS, AES_KEY_ZEROED, 0, 1)
REG32(AES_KEY_LOCK_STATUS, 0x68)
    FIELD(AES_KEY_LOCK_STATUS, BBRAM, 1, 1)
    FIELD(AES_KEY_LOCK_STATUS, EFUSE, 0, 1)
REG32(AES_AAD, 0x6c)
    FIELD(AES_AAD, HDR_PAYLOAD_N, 0, 1)
REG32(AES_USER_SEL, 0x70)
    FIELD(AES_USER_SEL, VAL, 0, 3)
REG32(AES_USER_KEY_CRC, 0x74)
REG32(AES_USER_KEY_CRC_STATUS, 0x78)
    FIELD(AES_USER_KEY_CRC_STATUS, DONE, 1, 1)
    FIELD(AES_USER_KEY_CRC_STATUS, PASS, 0, 1)
REG32(AES_CM_EN, 0x7c)
    FIELD(AES_CM_EN, VAL, 0, 1)
REG32(AES_SPLIT_CFG, 0x80)
    FIELD(AES_SPLIT_CFG, KEY_SPLIT, 1, 1)
    FIELD(AES_SPLIT_CFG, DATA_SPLIT, 0, 1)
REG32(AES_DATA_ENDIANNESS_SWAP, 0x84)
    FIELD(AES_DATA_ENDIANNESS_SWAP, VAL, 0, 1)
REG32(BH_KEY_0, 0xf0)
REG32(BH_KEY_1, 0xf4)
REG32(BH_KEY_2, 0xf8)
REG32(BH_KEY_3, 0xfc)
REG32(BH_KEY_4, 0x100)
REG32(BH_KEY_5, 0x104)
REG32(BH_KEY_6, 0x108)
REG32(BH_KEY_7, 0x10c)
REG32(USER_KEY_0_0, 0x110)
REG32(USER_KEY_0_1, 0x114)
REG32(USER_KEY_0_2, 0x118)
REG32(USER_KEY_0_3, 0x11c)
REG32(USER_KEY_0_4, 0x120)
REG32(USER_KEY_0_5, 0x124)
REG32(USER_KEY_0_6, 0x128)
REG32(USER_KEY_0_7, 0x12c)
REG32(USER_KEY_1_0, 0x130)
REG32(USER_KEY_1_1, 0x134)
REG32(USER_KEY_1_2, 0x138)
REG32(USER_KEY_1_3, 0x13c)
REG32(USER_KEY_1_4, 0x140)
REG32(USER_KEY_1_5, 0x144)
REG32(USER_KEY_1_6, 0x148)
REG32(USER_KEY_1_7, 0x14c)
REG32(USER_KEY_2_0, 0x150)
REG32(USER_KEY_2_1, 0x154)
REG32(USER_KEY_2_2, 0x158)
REG32(USER_KEY_2_3, 0x15c)
REG32(USER_KEY_2_4, 0x160)
REG32(USER_KEY_2_5, 0x164)
REG32(USER_KEY_2_6, 0x168)
REG32(USER_KEY_2_7, 0x16c)
REG32(USER_KEY_3_0, 0x170)
REG32(USER_KEY_3_1, 0x174)
REG32(USER_KEY_3_2, 0x178)
REG32(USER_KEY_3_3, 0x17c)
REG32(USER_KEY_3_4, 0x180)
REG32(USER_KEY_3_5, 0x184)
REG32(USER_KEY_3_6, 0x188)
REG32(USER_KEY_3_7, 0x18c)
REG32(USER_KEY_4_0, 0x190)
REG32(USER_KEY_4_1, 0x194)
REG32(USER_KEY_4_2, 0x198)
REG32(USER_KEY_4_3, 0x19c)
REG32(USER_KEY_4_4, 0x1a0)
REG32(USER_KEY_4_5, 0x1a4)
REG32(USER_KEY_4_6, 0x1a8)
REG32(USER_KEY_4_7, 0x1ac)
REG32(USER_KEY_5_0, 0x1b0)
REG32(USER_KEY_5_1, 0x1b4)
REG32(USER_KEY_5_2, 0x1b8)
REG32(USER_KEY_5_3, 0x1bc)
REG32(USER_KEY_5_4, 0x1c0)
REG32(USER_KEY_5_5, 0x1c4)
REG32(USER_KEY_5_6, 0x1c8)
REG32(USER_KEY_5_7, 0x1cc)
REG32(USER_KEY_6_0, 0x1d0)
REG32(USER_KEY_6_1, 0x1d4)
REG32(USER_KEY_6_2, 0x1d8)
REG32(USER_KEY_6_3, 0x1dc)
REG32(USER_KEY_6_4, 0x1e0)
REG32(USER_KEY_6_5, 0x1e4)
REG32(USER_KEY_6_6, 0x1e8)
REG32(USER_KEY_6_7, 0x1ec)
REG32(USER_KEY_7_0, 0x1f0)
REG32(USER_KEY_7_1, 0x1f4)
REG32(USER_KEY_7_2, 0x1f8)
REG32(USER_KEY_7_3, 0x1fc)
REG32(USER_KEY_7_4, 0x200)
REG32(USER_KEY_7_5, 0x204)
REG32(USER_KEY_7_6, 0x208)
REG32(USER_KEY_7_7, 0x20c)
REG32(AES_ISR, 0x214)
    FIELD(AES_ISR, SLVERR, 2, 1)
    FIELD(AES_ISR, ENCDEC_ERR, 1, 1)
    FIELD(AES_ISR, DONE, 0, 1)
REG32(AES_IMR, 0x218)
    FIELD(AES_IMR, SLVERR, 2, 1)
    FIELD(AES_IMR, ENCDEC_ERR, 1, 1)
    FIELD(AES_IMR, DONE, 0, 1)
REG32(AES_IER, 0x21c)
    FIELD(AES_IER, SLVERR, 2, 1)
    FIELD(AES_IER, ENCDEC_ERR, 1, 1)
    FIELD(AES_IER, DONE, 0, 1)
REG32(AES_IDR, 0x220)
    FIELD(AES_IDR, SLVERR, 2, 1)
    FIELD(AES_IDR, ENCDEC_ERR, 1, 1)
    FIELD(AES_IDR, DONE, 0, 1)

#define R_MAX (R_AES_IDR + 1)

typedef struct Zynq3AES Zynq3AES;

typedef struct PMCKeySink {
    Object parent;
    Zynq3AES *tmr;

    uint8_t key[256 / 8];
} PMCKeySink;

struct Zynq3AES {
    SysBusDevice parent_obj;
    MemoryRegion iomem;
    qemu_irq irq_aes_imr;
    bool aes_busy;
    bool aes_done;
    bool key_dec_done;
    bool inSoftRst;

    StreamSink *tx_dev;
    char *family_key_id;
    char *puf_key_id;

    uint32_t regs[R_MAX];
    RegisterInfo regs_info[R_MAX];

    qemu_irq aes_rst;
    XlnxAES *aes;
    uint32_t device_key[8];
    bool key_loaded;

    PMCKeySink bbram_key;
    PMCKeySink bbram_key_red;

    PMCKeySink bh_key_red;

    PMCKeySink efuse_key;
    PMCKeySink efuse_key_red;

    PMCKeySink efuse_user_key[2];
    PMCKeySink efuse_user_key_red[2];

    PMCKeySink puf_key;
    PMCKeySink family_key;

    PMCKeySink kup_key;

    uint8_t *bh_key;
    uint8_t *user_key[8];
    bool     user_key_lock_status[8];

    ZynqMPAESKeySink *key_sync;
    struct {
        uint32_t key[256 / 32];
        uint32_t iv[128 / 32];
    } feedback;

    /* Debug only */
    const char *prefix;

    /* GCM residual handling */
    uint8_t gcm_tag[16];
    uint8_t gcm_pos;
    uint8_t gcm_len;
    bool gcm_push_eop;
};

static void bswap32_buf8(uint8_t *buf, int len)
{
    int i;

    assert((len & 3) == 0);
    for (i = 0; i < len; i += 4) {
        uint8_t v[4];

        v[0] = buf[i];
        v[1] = buf[i + 1];
        v[2] = buf[i + 2];
        v[3] = buf[i + 3];
        buf[i] = v[3];
        buf[i + 1] = v[2];
        buf[i + 2] = v[1];
        buf[i + 3] = v[0];
    }
}

static void wswap128_buf8(uint8_t *buf, int len)
{
    int i, j, copyl;
    int wlen = len / 4;
    uint32_t *buf32 = (uint32_t *) buf;
    for (i = 0; i < wlen; i += 4) {
        uint32_t v[4] = {0};

        copyl = (i + 4 <= wlen) ? 4 : wlen - i;

        memcpy(v, &buf32[i], copyl * 4);
        for (j = 0; j < copyl; j++) {
            buf32[i + j] = v[copyl - 1 - j];
        }
    }
}

static int aes_key_size(Zynq3AES *s)
{
    return s->regs[R_AES_KEY_SIZE] == 0 ? 128 : 256;
}

static void pmc_key_sink_update(ZynqMPAESKeySink *obj, uint8_t *key,
                                      size_t len)
{
    PMCKeySink *ks = XILINX_PMC_KEY_SINK(obj);
    /* We only support MAX 256 bit keys at the moment.  */
    assert(len == 256 / 8);

    memcpy(ks->key, key, len);
}

static void aes_imr_update_irq(Zynq3AES *s)
{
    bool pending = s->regs[R_AES_ISR] & ~s->regs[R_AES_IMR];
    qemu_set_irq(s->irq_aes_imr, pending);
}

static void aes_isr_postw(RegisterInfo *reg, uint64_t val64)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    aes_imr_update_irq(s);
}

static uint64_t aes_ier_prew(RegisterInfo *reg, uint64_t val64)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    uint32_t val = val64;

    s->regs[R_AES_IMR] &= ~val;
    aes_imr_update_irq(s);
    return 0;
}

static uint64_t aes_idr_prew(RegisterInfo *reg, uint64_t val64)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    uint32_t val = val64;

    s->regs[R_AES_IMR] |= val;
    aes_imr_update_irq(s);
    return 0;
}

static int xlx_aes_push_data(Zynq3AES *s,
                             uint8_t *data8x, int len,
                             bool last_word , int lw_len,
                             uint8_t *outbuf, int *outlen);
static void xlx_aes_load_key(Zynq3AES *s, int len)
{
    unsigned int src, i;
    bool efuse_locked = ARRAY_FIELD_EX32(s->regs, AES_KEY_LOCK_STATUS, EFUSE);
    bool bbram_locked = ARRAY_FIELD_EX32(s->regs, AES_KEY_LOCK_STATUS, BBRAM);
    typedef union AESKey {
            uint8_t *u8;
            uint32_t *u32;
    } AESKey;

    AESKey key = { .u8 = 0 };
    uint8_t zerokey[32] = { 0 };
    int be_adj = (len / 32) - 1;

    src = s->regs[R_AES_KEY_SEL];

    switch (src) {
    case KEY_SEL_EFUSE_KEY:
        key.u8 = efuse_locked ? zerokey : s->efuse_key.key;
        be_adj = 0;  /* due to zynqmp compatibility */
        break;
    case KEY_SEL_EFUSE_RED_KEY:
        key.u8 = s->efuse_key_red.key;
        break;
    case KEY_SEL_EFUSE_USR_KEY0:
        key.u8 = efuse_locked ? zerokey : s->efuse_user_key[0].key;
        break;
    case KEY_SEL_EFUSE_USR_KEY1:
        key.u8 = efuse_locked ? zerokey : s->efuse_user_key[1].key;
        break;
    case KEY_SEL_EFUSE_USR_RD_KEY0:
        key.u8 = s->efuse_user_key_red[0].key;
        break;
    case KEY_SEL_EFUSE_USR_RD_KEY1:
        key.u8 = s->efuse_user_key_red[1].key;
        break;
    case KEY_SEL_BBRAM_KEY:
        key.u8 = bbram_locked ? zerokey : s->bbram_key.key;
        be_adj = 0;  /* due to zynqmp compatibility */
        break;
    case KEY_SEL_BBRAM_RD_KEY:
        key.u8 = s->bbram_key_red.key;
        break;
    case KEY_SEL_PUF_KEY:
        key.u8 = s->puf_key.key;
        be_adj = 0;  /* for zynqmp_aes_key_update() compatibility */
        break;
    case KEY_SEL_KUP_KEY:
        key.u8 = s->kup_key.key;
        break;
    case KEY_SEL_USR_KEY_0:
        key.u8 = s->user_key[0];
        break;
    case KEY_SEL_USR_KEY_1:
        key.u8 = s->user_key[1];
        break;
    case KEY_SEL_USR_KEY_2:
        key.u8 = s->user_key[2];
        break;
    case KEY_SEL_USR_KEY_3:
        key.u8 = s->user_key[3];
        break;
    case KEY_SEL_USR_KEY_4:
        key.u8 = s->user_key[4];
        break;
    case KEY_SEL_USR_KEY_5:
        key.u8 = s->user_key[5];
        break;
    case KEY_SEL_USR_KEY_6:
        key.u8 = s->user_key[6];
        break;
    case KEY_SEL_USR_KEY_7:
        key.u8 = s->user_key[7];
        break;
    case KEY_SEL_BH_KEY:
        key.u8 = s->bh_key;
        break;
    case KEY_SEL_BH_RD_KEY:
        key.u8 = s->bh_key_red.key;
        break;
    case KEY_SEL_FAMILY_KEY:
        if (xlnx_aes_k256_is_zero(s->family_key.key)) {
            hw_error("%s: AES Key source FAMILY_KEY: key value missing.\n",
                     s->prefix);
        }
        key.u8 = s->family_key.key;
        be_adj = 0;  /* for zynqmp_aes_key_update() compatibility */
        break;
    default:
        hw_error("%s: Unsupported AES Key source %d\n", s->prefix, src);
        break;
    }

    if (s->regs[R_AES_KEY_DEC] != 0xFFFFFFFF) {
        /* Changing le to be
         * Note: we only reverse the word order,
         *       byte swapping inside word is done by
         *       xlnx_aes core.
         */
        for (i = 0; i < len / 32; i++) {
            xlnx_aes_write_key(s->aes, i , key.u32[i ^ be_adj]);
        }
        xlnx_aes_load_key(s->aes, len);
    } else {
        AESKey dec_key;
        uint32_t *enc_key = g_new0(uint32_t, 8);
        uint8_t *outbuf = g_new0(uint8_t, 32 + 16);
        uint32_t *outbuf32 = (uint32_t *) outbuf;
        int outlen = 0;

        switch (s->regs[R_KEY_DEC_SEL]) {
        case 0:
            dec_key.u8 = s->bbram_key_red.key;
            break;
        case 1:
            dec_key.u8 = s->bh_key_red.key;
            break;
        case 2:
            dec_key.u8 = s->efuse_key_red.key;
            break;
        case 3:
            dec_key.u8 = s->efuse_user_key_red[0].key;
            break;
        case 4:
            dec_key.u8 = s->efuse_user_key_red[1].key;
            break;
        default:
            DPRINT("Invalid KEY_DEC_SEL\n");
            dec_key.u8 = zerokey;
        };
        memcpy(enc_key, key.u8, len / 8);
        /* grey/black key is formated LE for every 128 bit.
         * convert it BE for our purpose
         */
        bswap32_buf8((uint8_t *)enc_key, len / 8);
        wswap128_buf8((uint8_t *)enc_key, len / 8);
        xlx_aes_push_data(s, (uint8_t *)enc_key, len / 8, true, 4, outbuf,
                          &outlen);
        /* Convert the Key to le */
        bswap32_buf8(outbuf, len / 8);
        for (i = 0; i < len / 32; i++) {
            dec_key.u32[i] = outbuf32[len / 32 - i - 1];
        }
        DPRINT("Key Decrypt Done!\n");
        s->key_dec_done = true;
        qemu_irq_pulse(s->aes_rst);
        g_free(outbuf);
        g_free(enc_key);
    }
    s->key_loaded = true;
}


static void aes_key_load_postw(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    if (val) {
        xlx_aes_load_key(s, aes_key_size(s));
    }
}

static void aes_start_msg_postw(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    if (val) {
        s->gcm_len = 0;
        xlnx_aes_start_message(s->aes,
                     s->regs[R_AES_MODE] & R_AES_MODE_ENC_DEC_N_MASK);
    }
}

static void aes_reset(DeviceState *dev);
static void aes_soft_rst_postw(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    if (val && !s->inSoftRst) {
        qemu_irq_pulse(s->aes_rst);
        s->inSoftRst = true;
        aes_reset(DEVICE(s));
    }
}

static void aes_key_clear_postw(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    unsigned int i;
    uint8_t *key = 0;

    for (i = 0; val && (i < R_AES_KEY_CLEAR_RESERVED_2_SHIFT); i++) {
        if (!(val & (1 << i))) {
            continue;
        }

        switch (1 << i) {
        case R_AES_KEY_CLEAR_AES_KEY_ZEROIZE_MASK:
            xlnx_aes_key_zero(s->aes);
            break;
        case R_AES_KEY_CLEAR_KUP_KEY_MASK:
            key = &s->kup_key.key[0];
            break;
        case R_AES_KEY_CLEAR_USER_KEY_0_MASK ...
             R_AES_KEY_CLEAR_USER_KEY_1_MASK:
            key = s->user_key[i - R_AES_KEY_CLEAR_USER_KEY_0_SHIFT];
                break;
        case R_AES_KEY_CLEAR_EFUSE_KEY_MASK:
            key = &s->efuse_key.key[0];
            break;
        case R_AES_KEY_CLEAR_EFUSE_USER_KEY_0_MASK:
        case R_AES_KEY_CLEAR_EFUSE_USER_KEY_1_MASK:
            key = &s->efuse_user_key[i -
                R_AES_KEY_CLEAR_EFUSE_USER_KEY_0_SHIFT].key[0];
            break;
        case R_AES_KEY_CLEAR_EFUSE_RED_KEY_MASK:
            key = &s->efuse_key_red.key[0];
            break;
        case R_AES_KEY_CLEAR_EFUSE_USER_RED_KEY_0_MASK:
        case R_AES_KEY_CLEAR_EFUSE_USER_RED_KEY_1_MASK:
            key = &s->efuse_user_key[i -
                R_AES_KEY_CLEAR_EFUSE_USER_KEY_0_SHIFT].key[0];
            break;
        case R_AES_KEY_CLEAR_BH_KEY_MASK:
            key = s->bh_key;
            break;
        case R_AES_KEY_CLEAR_BH_RED_KEY_MASK:
            key = &s->bh_key_red.key[0];
            break;
        case R_AES_KEY_CLEAR_PUF_KEY_MASK:
            key = &s->puf_key.key[0];
            break;
        case R_AES_KEY_CLEAR_BBRAM_RED_KEY_MASK:
            key = &s->bbram_key_red.key[0];
            break;
        default:
            continue;
        };

        if (key) {
            memset(key, 0, 8 * 4);
        }

        val &= ~(1 << i);
        s->regs[R_KEY_ZEROED_STATUS] |= 1 << i;
    }
}

static uint64_t aes_status_postr(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);
    uint32_t v = 0;
    v |= s->regs[R_AES_CM_EN] ? R_AES_STATUS_CM_ENABLED_MASK : 0;
    v |= s->key_dec_done ? R_AES_STATUS_BLACK_KEY_DEC_DONE_MASK : 0 ;
    v |= s->key_loaded ? R_AES_STATUS_KEY_INIT_DONE_MASK : 0;
    v |= s->aes->tag_ok ? R_AES_STATUS_GCM_TAG_PASS_MASK : 0;
    v |= s->aes_done ? R_AES_STATUS_DONE_MASK : 0;
    v |= s->aes->inp_ready ? R_AES_STATUS_READY_MASK : 0;
    v |= s->aes_busy ? R_AES_STATUS_BUSY_MASK : 0;
    return v;
}

static void key_dec_trig_postw(RegisterInfo *reg, uint64_t val)
{
    Zynq3AES *s = XILINX_AES(reg->opaque);

    if (val & R_KEY_DEC_TRIG_VAL_MASK) {
        if (s->regs[R_AES_KEY_DEC] == 0xFFFFFFFF) {
            DPRINT("Start Key Decrypt..\n");
            xlx_aes_load_key(s, aes_key_size(s));
        } else {
            qemu_log_mask(LOG_GUEST_ERROR, "Key Decrypt triggred before "
                          "AES_KEY_DEC is programmed\n");
        }
    }
}

static RegisterAccessInfo aes_regs_info[] = {
    {   .name = "AES_STATUS",  .addr = A_AES_STATUS,
        .post_read = aes_status_postr,
        .rsvd = 0xfc0,
        .ro = 0x1fff,
    },{ .name = "AES_KEY_SEL",  .addr = A_AES_KEY_SEL,
    },{ .name = "AES_KEY_LOAD",  .addr = A_AES_KEY_LOAD,
        .post_write = aes_key_load_postw,
    },{ .name = "AES_START_MSG",  .addr = A_AES_START_MSG,
        .post_write = aes_start_msg_postw,
    },{ .name = "AES_SOFT_RST",  .addr = A_AES_SOFT_RST,
        .post_write = aes_soft_rst_postw,
        .reset = 0x1,
    },{ .name = "AES_KEY_CLEAR",  .addr = A_AES_KEY_CLEAR,
        .post_write = aes_key_clear_postw,
    },{ .name = "AES_MODE",  .addr = A_AES_MODE,
    },{ .name = "AES_KUP_WR",  .addr = A_AES_KUP_WR,
    },{ .name = "AES_IV_0",  .addr = A_AES_IV_0,
        .ro = 0xffffffff,
    },{ .name = "AES_IV_1",  .addr = A_AES_IV_1,
        .ro = 0xffffffff,
    },{ .name = "AES_IV_2",  .addr = A_AES_IV_2,
        .ro = 0xffffffff,
    },{ .name = "AES_IV_3",  .addr = A_AES_IV_3,
        .ro = 0xffffffff,
    },{ .name = "AES_KEY_SIZE",  .addr = A_AES_KEY_SIZE,
        .reset = 0x2,
    },{ .name = "AES_KEY_DEC",  .addr = A_AES_KEY_DEC,
    },{ .name = "KEY_DEC_TRIG",  .addr = A_KEY_DEC_TRIG,
        .post_write = key_dec_trig_postw,
    },{ .name = "KEY_DEC_SEL",  .addr = A_KEY_DEC_SEL,
    },{ .name = "KEY_ZEROED_STATUS",  .addr = A_KEY_ZEROED_STATUS,
        .ro = 0xffffffff,
    },{ .name = "AES_KEY_LOCK_STATUS",  .addr = A_AES_KEY_LOCK_STATUS,
        .ro = 0x3,
        /* reset value must be the reset states of input GPIOs */
    },{ .name = "AES_AAD",  .addr = A_AES_AAD,
    },{ .name = "AES_USER_SEL",  .addr = A_AES_USER_SEL,
    },{ .name = "AES_USER_KEY_CRC",  .addr = A_AES_USER_KEY_CRC,
    },{ .name = "AES_USER_KEY_CRC_STATUS",  .addr = A_AES_USER_KEY_CRC_STATUS,
        .ro = 0x3,
    },{ .name = "AES_CM_EN",  .addr = A_AES_CM_EN,
        .reset = 0x1,
    },{ .name = "AES_SPLIT_CFG",  .addr = A_AES_SPLIT_CFG,
    },{ .name = "AES_DATA_ENDIANNESS_SWAP",  .addr = A_AES_DATA_ENDIANNESS_SWAP,
    },{ .name = "BH_KEY_0",  .addr = A_BH_KEY_0,
    },{ .name = "BH_KEY_1",  .addr = A_BH_KEY_1,
    },{ .name = "BH_KEY_2",  .addr = A_BH_KEY_2,
    },{ .name = "BH_KEY_3",  .addr = A_BH_KEY_3,
    },{ .name = "BH_KEY_4",  .addr = A_BH_KEY_4,
    },{ .name = "BH_KEY_5",  .addr = A_BH_KEY_5,
    },{ .name = "BH_KEY_6",  .addr = A_BH_KEY_6,
    },{ .name = "BH_KEY_7",  .addr = A_BH_KEY_7,
    },{ .name = "USER_KEY_0_0",  .addr = A_USER_KEY_0_0,
    },{ .name = "USER_KEY_0_1",  .addr = A_USER_KEY_0_1,
    },{ .name = "USER_KEY_0_2",  .addr = A_USER_KEY_0_2,
    },{ .name = "USER_KEY_0_3",  .addr = A_USER_KEY_0_3,
    },{ .name = "USER_KEY_0_4",  .addr = A_USER_KEY_0_4,
    },{ .name = "USER_KEY_0_5",  .addr = A_USER_KEY_0_5,
    },{ .name = "USER_KEY_0_6",  .addr = A_USER_KEY_0_6,
    },{ .name = "USER_KEY_0_7",  .addr = A_USER_KEY_0_7,
    },{ .name = "USER_KEY_1_0",  .addr = A_USER_KEY_1_0,
    },{ .name = "USER_KEY_1_1",  .addr = A_USER_KEY_1_1,
    },{ .name = "USER_KEY_1_2",  .addr = A_USER_KEY_1_2,
    },{ .name = "USER_KEY_1_3",  .addr = A_USER_KEY_1_3,
    },{ .name = "USER_KEY_1_4",  .addr = A_USER_KEY_1_4,
    },{ .name = "USER_KEY_1_5",  .addr = A_USER_KEY_1_5,
    },{ .name = "USER_KEY_1_6",  .addr = A_USER_KEY_1_6,
    },{ .name = "USER_KEY_1_7",  .addr = A_USER_KEY_1_7,
    },{ .name = "USER_KEY_2_0",  .addr = A_USER_KEY_2_0,
    },{ .name = "USER_KEY_2_1",  .addr = A_USER_KEY_2_1,
    },{ .name = "USER_KEY_2_2",  .addr = A_USER_KEY_2_2,
    },{ .name = "USER_KEY_2_3",  .addr = A_USER_KEY_2_3,
    },{ .name = "USER_KEY_2_4",  .addr = A_USER_KEY_2_4,
    },{ .name = "USER_KEY_2_5",  .addr = A_USER_KEY_2_5,
    },{ .name = "USER_KEY_2_6",  .addr = A_USER_KEY_2_6,
    },{ .name = "USER_KEY_2_7",  .addr = A_USER_KEY_2_7,
    },{ .name = "USER_KEY_3_0",  .addr = A_USER_KEY_3_0,
    },{ .name = "USER_KEY_3_1",  .addr = A_USER_KEY_3_1,
    },{ .name = "USER_KEY_3_2",  .addr = A_USER_KEY_3_2,
    },{ .name = "USER_KEY_3_3",  .addr = A_USER_KEY_3_3,
    },{ .name = "USER_KEY_3_4",  .addr = A_USER_KEY_3_4,
    },{ .name = "USER_KEY_3_5",  .addr = A_USER_KEY_3_5,
    },{ .name = "USER_KEY_3_6",  .addr = A_USER_KEY_3_6,
    },{ .name = "USER_KEY_3_7",  .addr = A_USER_KEY_3_7,
    },{ .name = "USER_KEY_4_0",  .addr = A_USER_KEY_4_0,
    },{ .name = "USER_KEY_4_1",  .addr = A_USER_KEY_4_1,
    },{ .name = "USER_KEY_4_2",  .addr = A_USER_KEY_4_2,
    },{ .name = "USER_KEY_4_3",  .addr = A_USER_KEY_4_3,
    },{ .name = "USER_KEY_4_4",  .addr = A_USER_KEY_4_4,
    },{ .name = "USER_KEY_4_5",  .addr = A_USER_KEY_4_5,
    },{ .name = "USER_KEY_4_6",  .addr = A_USER_KEY_4_6,
    },{ .name = "USER_KEY_4_7",  .addr = A_USER_KEY_4_7,
    },{ .name = "USER_KEY_5_0",  .addr = A_USER_KEY_5_0,
    },{ .name = "USER_KEY_5_1",  .addr = A_USER_KEY_5_1,
    },{ .name = "USER_KEY_5_2",  .addr = A_USER_KEY_5_2,
    },{ .name = "USER_KEY_5_3",  .addr = A_USER_KEY_5_3,
    },{ .name = "USER_KEY_5_4",  .addr = A_USER_KEY_5_4,
    },{ .name = "USER_KEY_5_5",  .addr = A_USER_KEY_5_5,
    },{ .name = "USER_KEY_5_6",  .addr = A_USER_KEY_5_6,
    },{ .name = "USER_KEY_5_7",  .addr = A_USER_KEY_5_7,
    },{ .name = "USER_KEY_6_0",  .addr = A_USER_KEY_6_0,
    },{ .name = "USER_KEY_6_1",  .addr = A_USER_KEY_6_1,
    },{ .name = "USER_KEY_6_2",  .addr = A_USER_KEY_6_2,
    },{ .name = "USER_KEY_6_3",  .addr = A_USER_KEY_6_3,
    },{ .name = "USER_KEY_6_4",  .addr = A_USER_KEY_6_4,
    },{ .name = "USER_KEY_6_5",  .addr = A_USER_KEY_6_5,
    },{ .name = "USER_KEY_6_6",  .addr = A_USER_KEY_6_6,
    },{ .name = "USER_KEY_6_7",  .addr = A_USER_KEY_6_7,
    },{ .name = "USER_KEY_7_0",  .addr = A_USER_KEY_7_0,
    },{ .name = "USER_KEY_7_1",  .addr = A_USER_KEY_7_1,
    },{ .name = "USER_KEY_7_2",  .addr = A_USER_KEY_7_2,
    },{ .name = "USER_KEY_7_3",  .addr = A_USER_KEY_7_3,
    },{ .name = "USER_KEY_7_4",  .addr = A_USER_KEY_7_4,
    },{ .name = "USER_KEY_7_5",  .addr = A_USER_KEY_7_5,
    },{ .name = "USER_KEY_7_6",  .addr = A_USER_KEY_7_6,
    },{ .name = "USER_KEY_7_7",  .addr = A_USER_KEY_7_7,
    },{ .name = "AES_ISR",  .addr = A_AES_ISR,
        .w1c = 0x7,
        .post_write = aes_isr_postw,
    },{ .name = "AES_IMR",  .addr = A_AES_IMR,
        .reset = 0x7,
        .ro = 0x7,
    },{ .name = "AES_IER",  .addr = A_AES_IER,
        .pre_write = aes_ier_prew,
    },{ .name = "AES_IDR",  .addr = A_AES_IDR,
        .pre_write = aes_idr_prew,
    }
};

static uint64_t aes_reg_read(void *opaque, hwaddr addr, unsigned size)
{
    uint64_t ret;

    ret = register_read_memory(opaque, addr, size);
    switch (addr) {
    /* Write Only Registers */
    case A_AES_KEY_CLEAR:
    case A_AES_KEY_DEC:
    case A_KEY_DEC_TRIG:
    case A_KEY_DEC_SEL:
    case A_AES_USER_KEY_CRC:
    case A_BH_KEY_0 ... A_USER_KEY_7_7:
    case A_AES_IER:
    case A_AES_IDR:
        ret = 0;
        break;
    default:
        break;
    };

    return ret;
}

static void aes_reg_write(void *opaque, hwaddr addr, uint64_t data,
                          unsigned size)
{
    RegisterInfoArray *reg_array = opaque;
    Zynq3AES *s;
    bool no_reg_update = false;

    if (R_MAX < (addr / 4)) {
        DPRINT("Reg 0x%x not implemented\n", (unsigned int)addr);
        return;
    }
    /* No easy way to find the RegisterInfo of a particular
     * register, so accessing reg0 opaque for Zynq3AES object
     */
    s = XILINX_AES(reg_array->r[0]->opaque);

    switch (addr) {
    case A_USER_KEY_0_0 ... A_USER_KEY_0_7:
        no_reg_update = s->user_key_lock_status[0];
        break;
    case A_USER_KEY_1_0 ... A_USER_KEY_1_7:
        no_reg_update = s->user_key_lock_status[1];
        break;
    case A_USER_KEY_2_0 ... A_USER_KEY_2_7:
        no_reg_update = s->user_key_lock_status[2];
        break;
    case A_USER_KEY_3_0 ... A_USER_KEY_3_7:
        no_reg_update = s->user_key_lock_status[3];
        break;
    case A_USER_KEY_4_0 ... A_USER_KEY_4_7:
        no_reg_update = s->user_key_lock_status[4];
        break;
    case A_USER_KEY_5_0 ... A_USER_KEY_5_7:
        no_reg_update = s->user_key_lock_status[5];
        break;
    case A_USER_KEY_6_0 ... A_USER_KEY_6_7:
        no_reg_update = s->user_key_lock_status[6];
        break;
    case A_USER_KEY_7_0 ... A_USER_KEY_7_7:
        no_reg_update = s->user_key_lock_status[7];
        break;
    default:
        break;
    };

    if (no_reg_update) {
        DPRINT("addr:0x%x: Write reg locked!\n", (uint32_t)addr);
    } else {
        register_write_memory(opaque, addr, data, size);
    }
}

static void aes_reset(DeviceState *dev)
{
    Zynq3AES *s = XILINX_AES(dev);
    unsigned int i;

    for (i = 0; i < ARRAY_SIZE(s->regs_info); ++i) {
        if (s->inSoftRst) {
            switch (i) {
            case R_AES_KEY_LOCK_STATUS:
            case R_KEY_ZEROED_STATUS:
            case R_BH_KEY_0 ... R_BH_KEY_7:
            case R_USER_KEY_0_0 ... R_USER_KEY_7_7:
                continue;
            default:
                break;
            };
        }
        register_reset(&s->regs_info[i]);
    }
    aes_imr_update_irq(s);
    s->key_loaded = false;
    s->gcm_len = 0;
    s->key_dec_done = false;
    s->inSoftRst = false;
}

static const MemoryRegionOps aes_ops = {
    .read = aes_reg_read,
    .write = aes_reg_write,
    .endianness = DEVICE_LITTLE_ENDIAN,
    .valid = {
        .min_access_size = 4,
        .max_access_size = 4,
    },
};

static void aes_busy_update(void *opaque, int n, int level)
{
    Zynq3AES *s = XILINX_AES(opaque);

    s->aes_busy = level;
}

static void aes_done_update(void *opaque, int n, int level)
{
    Zynq3AES *s = XILINX_AES(opaque);

    s->aes_done = level;
}

static void device_key_update(ZynqMPAESKeySink *obj, uint8_t *key, size_t len)
{
    Zynq3AES *s = XILINX_AES(obj);
    /* We only support MAX 256 bit keys at the moment.  */
    assert(len == 256 / 8);

    memcpy(s->device_key, key, len);
}

static int xlx_aes_push_data(Zynq3AES *s,
                             uint8_t *data8x, int len,
                             bool last_word , int lw_len,
                             uint8_t *outbuf, int *outlen)
{
    return xlnx_aes_push_data(s->aes, data8x, len, !!s->regs[R_AES_AAD],
                              last_word, lw_len, outbuf, outlen);
}

static uint32_t shift_in_u32(uint32_t *a, unsigned int size, uint32_t data)
{
    unsigned int i;
    uint32_t r = a[0];

    for (i = 1; i < size; i++) {
        a[i - 1] = a[i];
    }
    a[i - 1] = data;

    return r;
}

static void xlx_aes_feedback(Zynq3AES *s, unsigned char *buf, int len)
{
    bool kup_key_feedback;
    bool iv_feedback;
    int i;
    uint8_t *key8;

    iv_feedback = !!(s->regs[R_AES_KUP_WR] & R_AES_KUP_WR_IV_SAVE_MASK);

    kup_key_feedback = !!(s->regs[R_AES_KUP_WR] & R_AES_KUP_WR_KEY_SAVE_MASK);

    assert((len & 3) == 0);

    for (i = 0; i < len; i += 4) {
        uint32_t data;
        memcpy(&data, buf + i, 4);

        if (iv_feedback) {
            data = shift_in_u32(s->feedback.iv, ARRAY_SIZE(s->feedback.iv),
                               data);
        }
        if (kup_key_feedback) {
            shift_in_u32(s->feedback.key, ARRAY_SIZE(s->feedback.key), data);
        }
    }

    /* feedback the AES output into Key and IV storage.  */
    if (iv_feedback) {
        memcpy(&s->regs[R_AES_IV_0], s->feedback.iv, 16);
        s->regs[R_AES_KUP_WR] &= ~(R_AES_KUP_WR_IV_SAVE_MASK);
    }
    if (kup_key_feedback) {
        key8 = (uint8_t *) s->feedback.key;
        bswap32_buf8(key8, aes_key_size(s) / 8);
        for (i = 0; i < ARRAY_SIZE(s->feedback.key) * 4; i++) {
            s->kup_key.key[(aes_key_size(s) / 8) - i - 1] = key8[i];
        }
        s->regs[R_AES_KUP_WR] &= ~(R_AES_KUP_WR_KEY_SAVE_MASK);
    }
}

static void aes_stream_gcm_push(void *opaque)
{
    Zynq3AES *s = XILINX_AES(opaque);

    while (s->gcm_len && stream_can_push(s->tx_dev, aes_stream_gcm_push, s)) {
        size_t ret;

        ret = stream_push(s->tx_dev, (s->gcm_tag + s->gcm_pos),
                          s->gcm_len, s->gcm_push_eop);
        s->gcm_pos += ret;
        s->gcm_len -= ret;
    }
}

static void aes_stream_dst_push(Zynq3AES *s, uint8_t *outbuf, int outlen,
                                size_t len, bool eop, bool encrypt)
{
    int limit = sizeof(s->gcm_tag);
    size_t pushed;

    pushed = stream_push(s->tx_dev, outbuf, outlen, eop);

    /* Done if there is no generated GCM-tag */
    if (!encrypt || !eop) {
        return;
    }

    /* Done if the entire GCM-tag has been received inline */
    if (pushed >= outlen) {
        return;
    }

    /* GCM-tag is the only allowed residual */
    if (pushed < len) {
        qemu_log_mask(LOG_GUEST_ERROR,
                      "%s: DST channel dropping %zd b of data.\n",
                      s->prefix, (len - pushed));
        return;
    }

    outlen -= pushed;
    if (outlen > limit) {
        qemu_log_mask(LOG_GUEST_ERROR,
                      "%s: Excessive GCM-tag data dropped: %d - %d\n",
                      s->prefix, outlen, limit);
        outlen = limit;
    }

    /*
     * Capture the GCM-tag (or whatever left) for residual push.
     *
     * Receiving the gcm-tag is optional; thus, it is important
     * to discard the residual by reset or a new start-message.
     */
    memcpy(s->gcm_tag, (outbuf + pushed), outlen);
    s->gcm_len = outlen;
    s->gcm_pos = 0;
    s->gcm_push_eop = eop;

    aes_stream_gcm_push(s);
}

static size_t aes_stream_push(StreamSink *obj, uint8_t *buf, size_t len,
                              bool eop)
{
    Zynq3AES *s = XILINX_AES(obj);
    unsigned char outbuf[8 * 1024 + 16];
    int outlen = 0;
    bool feedback;
    bool encrypt;
    size_t ret;

    /* When encrypting, we need to be prepared to receive the 16 byte tag.  */
    encrypt = s->aes->encrypt;
    if (encrypt && len > (sizeof(outbuf) - 16)) {
        len = sizeof(outbuf) - 16;
        eop = false;
    }

    /* TODO: Add explicit eop to the stream interface.  */
    /* As QEMU aes is big endian, we would change the endianess when
     * user dosent request endianess swapp, i.e data is sent le.
     */
    if (!s->regs[R_AES_DATA_ENDIANNESS_SWAP]) {
        wswap128_buf8(buf, len);
    }
    bswap32_buf8(buf, len);
    ret = xlx_aes_push_data(s, buf, len, eop, 4, outbuf, &outlen);
    if (!s->regs[R_AES_DATA_ENDIANNESS_SWAP]) {
        wswap128_buf8(outbuf, outlen);
    }
    bswap32_buf8(outbuf, outlen);
    /* No flow-control on the output.  */
    feedback = !!(s->regs[R_AES_KUP_WR]
                & (R_AES_KUP_WR_IV_SAVE_MASK | R_AES_KUP_WR_KEY_SAVE_MASK));
    if (feedback) {
        xlx_aes_feedback(s, outbuf, outlen);
    } else {
        aes_stream_dst_push(s, outbuf, outlen, ret, eop, encrypt);
    }

    /* printf("%s len=%zd ret=%zd outlen=%d eop=%d\n",
           __func__, len, ret, outlen, eop); */
    return ret;
}

static bool aes_stream_can_push(StreamSink *obj,
                                    StreamCanPushNotifyFn notify,
                                    void *notify_opaque)
{
    Zynq3AES *s = XILINX_AES(obj);
    /* printf("%s: %d\n", __func__, s->aes.inp_ready); */
    return s->aes->inp_ready;
}

static void efuse_key_lock_update(void *opaque, int n, int level)
{
    Zynq3AES *s = XILINX_AES(opaque);

    ARRAY_FIELD_DP32(s->regs, AES_KEY_LOCK_STATUS, EFUSE, level);
}

static void bbram_key_lock_update(void *opaque, int n, int level)
{
    Zynq3AES *s = XILINX_AES(opaque);

    ARRAY_FIELD_DP32(s->regs, AES_KEY_LOCK_STATUS, BBRAM, level);
}

static void user_key_lock_update(void *opaque, int n, int level)
{
    Zynq3AES *s = XILINX_AES(opaque);

    if (0 <= n && n < ARRAY_SIZE(s->user_key_lock_status))
        s->user_key_lock_status[n] = (bool)level;
}

static void aes_realize(DeviceState *dev, Error **errp)
{
    Zynq3AES *s = XILINX_AES(dev);

    s->prefix = object_get_canonical_path(OBJECT(s));
    s->aes->prefix = g_strdup(s->prefix);

    s->bh_key = (uint8_t *) &s->regs[R_BH_KEY_0];
#define USER_KEY(x) { \
    s->user_key[x] = (uint8_t *) &s->regs[R_USER_KEY_ ## x ## _0]; \
    }
    USER_KEY(0)
    USER_KEY(1)
    USER_KEY(2)
    USER_KEY(3)
    USER_KEY(4)
    USER_KEY(5)
    USER_KEY(6)
    USER_KEY(7)
#undef USER_KEY
    qdev_init_gpio_in_named(dev, aes_busy_update, "busy", 1);
    qdev_init_gpio_in_named(dev, aes_done_update, "done", 1);
    qdev_init_gpio_out(dev, &s->aes_rst, 1);
    qdev_init_gpio_in_named(dev, bbram_key_lock_update, "bbram-key-lock", 1);
    qdev_init_gpio_in_named(dev, efuse_key_lock_update, "efuse-key-lock", 1);
    qdev_init_gpio_in_named(dev, user_key_lock_update, "user-key-lock", 8);

    /*
     * Retrieve preset key from 'secret' object and place it in
     * byte order expected by xlx_aes_load_key().  Default to
     * zero to detect missing value.
     */
    xlnx_aes_k256_get_provided(OBJECT(s), "family-key-id",
                               "00000000" "00000000" "00000000" "00000000"
                               "00000000" "00000000" "00000000" "00000000",
                               s->family_key.key, NULL);
    xlnx_aes_k256_swap32(s->family_key.key, s->family_key.key);
}

static void pmc_init_key_sink(Zynq3AES *s,
                                    const char *name, PMCKeySink *ks)
{
    char *ch_name;

    ch_name = g_strdup_printf("zynqmp-aes-key-sink-%s-target", name);
    object_initialize(ks, sizeof(*ks), TYPE_XILINX_PMC_KEY_SINK);
    object_property_add_child(OBJECT(s), ch_name, (Object *)ks);
    free(ch_name);

    /* Back link, non-qom for the moment.  */
    ks->tmr = s;
}

static void aes_init(Object *obj)
{
    Zynq3AES *s = XILINX_AES(obj);
    SysBusDevice *sbd = SYS_BUS_DEVICE(obj);
    RegisterInfoArray *reg_array;

    pmc_init_key_sink(s, "bbram", &s->bbram_key);
    pmc_init_key_sink(s, "efuses", &s->efuse_key);
    pmc_init_key_sink(s, "efuses-user0", &s->efuse_user_key[0]);
    pmc_init_key_sink(s, "efuses-user1", &s->efuse_user_key[1]);
    pmc_init_key_sink(s, "family", &s->family_key);
    pmc_init_key_sink(s, "puf", &s->puf_key);

    if (s->family_key_id == NULL) {
        s->family_key_id = g_strdup("xlnx-aes-family-key");
    }
    if (s->puf_key_id == NULL) {
        s->puf_key_id = g_strdup("xlnx-aes-puf-key");
    }

    memory_region_init(&s->iomem, obj, TYPE_XILINX_AES, R_MAX * 4);
    reg_array =
        register_init_block32(DEVICE(obj), aes_regs_info,
                              ARRAY_SIZE(aes_regs_info),
                              s->regs_info, s->regs,
                              &aes_ops,
                              XILINX_AES_ERR_DEBUG,
                              R_MAX * 4);
    memory_region_add_subregion(&s->iomem,
                                0x0,
                                &reg_array->mem);
    object_property_add_link(obj, "aes-core", TYPE_XLNX_AES,
                             (Object **)&s->aes,
                             qdev_prop_allow_set_link_before_realize,
                             OBJ_PROP_LINK_STRONG);
    object_property_add_link(obj, "stream-connected-aes", TYPE_STREAM_SINK,
                             (Object **)&s->tx_dev,
                             qdev_prop_allow_set_link_before_realize,
                             OBJ_PROP_LINK_STRONG);
    sysbus_init_mmio(sbd, &s->iomem);
    sysbus_init_irq(sbd, &s->irq_aes_imr);
}

static const VMStateDescription vmstate_aes = {
    .name = TYPE_XILINX_AES,
    .version_id = 1,
    .minimum_version_id = 1,
    .fields = (VMStateField[]) {
        VMSTATE_UINT32_ARRAY(regs, Zynq3AES, R_MAX),
        VMSTATE_END_OF_LIST(),
    }
};

static Property aes_properties[] = {
    DEFINE_PROP_STRING("family-key-id", Zynq3AES, family_key_id),
    DEFINE_PROP_STRING("puf-key-id",    Zynq3AES, puf_key_id),

    DEFINE_PROP_END_OF_LIST(),
};

static void aes_class_init(ObjectClass *klass, void *data)
{
    DeviceClass *dc = DEVICE_CLASS(klass);
    ZynqMPAESKeySinkClass *ksc = ZYNQMP_AES_KEY_SINK_CLASS(klass);
    StreamSinkClass *ssc = STREAM_SINK_CLASS(klass);

    dc->reset = aes_reset;
    dc->realize = aes_realize;
    dc->vmsd = &vmstate_aes;
    device_class_set_props(dc, aes_properties);
    ksc->update = device_key_update;

    ssc->push = aes_stream_push;
    ssc->can_push = aes_stream_can_push;
}

static void pmc_key_sink_class_init(ObjectClass *klass, void *data)
{
    ZynqMPAESKeySinkClass *c = ZYNQMP_AES_KEY_SINK_CLASS(klass);
    c->update = pmc_key_sink_update;
}

static const TypeInfo aes_info = {
    .name          = TYPE_XILINX_AES,
    .parent        = TYPE_SYS_BUS_DEVICE,
    .instance_size = sizeof(Zynq3AES),
    .class_init    = aes_class_init,
    .instance_init = aes_init,
    .interfaces = (InterfaceInfo[]) {
        { TYPE_ZYNQMP_AES_KEY_SINK },
        { TYPE_STREAM_SINK },
        { }
    }
};

static const TypeInfo pmc_key_sink_info = {
    .name          = TYPE_XILINX_PMC_KEY_SINK,
    .parent        = TYPE_OBJECT,
    .instance_size = sizeof(PMCKeySink),
    .class_init    = pmc_key_sink_class_init,
    .interfaces    = (InterfaceInfo[]) {
        { TYPE_ZYNQMP_AES_KEY_SINK },
        { }
    }
};


static void aes_register_types(void)
{
    type_register_static(&aes_info);
    type_register_static(&pmc_key_sink_info);
}

type_init(aes_register_types)
