/* SPDX-License-Identifier: BSD-2-Clause */

/**
 * @file
 *
 * @ingroup RTEMSImplCRC
 *
 * @brief This source file contains the implementation of
 *   _CRC24Q_Update() and _CRC24Q_Sequence_update().
 *
 * The CRC-24Q cyclic redundancy checksum is used for example by Qualcomm,
 * RTCM104v3, and LTE-A.
 */

/*
 * Copyright (C) 2023 embedded brains GmbH & Co. KG
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <rtems/crc.h>

static const uint32_t _CRC24Q_Table[256] = {
    UINT32_C(0x000000), UINT32_C(0x864CFB), UINT32_C(0x8AD50D),
    UINT32_C(0x0C99F6), UINT32_C(0x93E6E1), UINT32_C(0x15AA1A),
    UINT32_C(0x1933EC), UINT32_C(0x9F7F17), UINT32_C(0xA18139),
    UINT32_C(0x27CDC2), UINT32_C(0x2B5434), UINT32_C(0xAD18CF),
    UINT32_C(0x3267D8), UINT32_C(0xB42B23), UINT32_C(0xB8B2D5),
    UINT32_C(0x3EFE2E), UINT32_C(0xC54E89), UINT32_C(0x430272),
    UINT32_C(0x4F9B84), UINT32_C(0xC9D77F), UINT32_C(0x56A868),
    UINT32_C(0xD0E493), UINT32_C(0xDC7D65), UINT32_C(0x5A319E),
    UINT32_C(0x64CFB0), UINT32_C(0xE2834B), UINT32_C(0xEE1ABD),
    UINT32_C(0x685646), UINT32_C(0xF72951), UINT32_C(0x7165AA),
    UINT32_C(0x7DFC5C), UINT32_C(0xFBB0A7), UINT32_C(0x0CD1E9),
    UINT32_C(0x8A9D12), UINT32_C(0x8604E4), UINT32_C(0x00481F),
    UINT32_C(0x9F3708), UINT32_C(0x197BF3), UINT32_C(0x15E205),
    UINT32_C(0x93AEFE), UINT32_C(0xAD50D0), UINT32_C(0x2B1C2B),
    UINT32_C(0x2785DD), UINT32_C(0xA1C926), UINT32_C(0x3EB631),
    UINT32_C(0xB8FACA), UINT32_C(0xB4633C), UINT32_C(0x322FC7),
    UINT32_C(0xC99F60), UINT32_C(0x4FD39B), UINT32_C(0x434A6D),
    UINT32_C(0xC50696), UINT32_C(0x5A7981), UINT32_C(0xDC357A),
    UINT32_C(0xD0AC8C), UINT32_C(0x56E077), UINT32_C(0x681E59),
    UINT32_C(0xEE52A2), UINT32_C(0xE2CB54), UINT32_C(0x6487AF),
    UINT32_C(0xFBF8B8), UINT32_C(0x7DB443), UINT32_C(0x712DB5),
    UINT32_C(0xF7614E), UINT32_C(0x19A3D2), UINT32_C(0x9FEF29),
    UINT32_C(0x9376DF), UINT32_C(0x153A24), UINT32_C(0x8A4533),
    UINT32_C(0x0C09C8), UINT32_C(0x00903E), UINT32_C(0x86DCC5),
    UINT32_C(0xB822EB), UINT32_C(0x3E6E10), UINT32_C(0x32F7E6),
    UINT32_C(0xB4BB1D), UINT32_C(0x2BC40A), UINT32_C(0xAD88F1),
    UINT32_C(0xA11107), UINT32_C(0x275DFC), UINT32_C(0xDCED5B),
    UINT32_C(0x5AA1A0), UINT32_C(0x563856), UINT32_C(0xD074AD),
    UINT32_C(0x4F0BBA), UINT32_C(0xC94741), UINT32_C(0xC5DEB7),
    UINT32_C(0x43924C), UINT32_C(0x7D6C62), UINT32_C(0xFB2099),
    UINT32_C(0xF7B96F), UINT32_C(0x71F594), UINT32_C(0xEE8A83),
    UINT32_C(0x68C678), UINT32_C(0x645F8E), UINT32_C(0xE21375),
    UINT32_C(0x15723B), UINT32_C(0x933EC0), UINT32_C(0x9FA736),
    UINT32_C(0x19EBCD), UINT32_C(0x8694DA), UINT32_C(0x00D821),
    UINT32_C(0x0C41D7), UINT32_C(0x8A0D2C), UINT32_C(0xB4F302),
    UINT32_C(0x32BFF9), UINT32_C(0x3E260F), UINT32_C(0xB86AF4),
    UINT32_C(0x2715E3), UINT32_C(0xA15918), UINT32_C(0xADC0EE),
    UINT32_C(0x2B8C15), UINT32_C(0xD03CB2), UINT32_C(0x567049),
    UINT32_C(0x5AE9BF), UINT32_C(0xDCA544), UINT32_C(0x43DA53),
    UINT32_C(0xC596A8), UINT32_C(0xC90F5E), UINT32_C(0x4F43A5),
    UINT32_C(0x71BD8B), UINT32_C(0xF7F170), UINT32_C(0xFB6886),
    UINT32_C(0x7D247D), UINT32_C(0xE25B6A), UINT32_C(0x641791),
    UINT32_C(0x688E67), UINT32_C(0xEEC29C), UINT32_C(0x3347A4),
    UINT32_C(0xB50B5F), UINT32_C(0xB992A9), UINT32_C(0x3FDE52),
    UINT32_C(0xA0A145), UINT32_C(0x26EDBE), UINT32_C(0x2A7448),
    UINT32_C(0xAC38B3), UINT32_C(0x92C69D), UINT32_C(0x148A66),
    UINT32_C(0x181390), UINT32_C(0x9E5F6B), UINT32_C(0x01207C),
    UINT32_C(0x876C87), UINT32_C(0x8BF571), UINT32_C(0x0DB98A),
    UINT32_C(0xF6092D), UINT32_C(0x7045D6), UINT32_C(0x7CDC20),
    UINT32_C(0xFA90DB), UINT32_C(0x65EFCC), UINT32_C(0xE3A337),
    UINT32_C(0xEF3AC1), UINT32_C(0x69763A), UINT32_C(0x578814),
    UINT32_C(0xD1C4EF), UINT32_C(0xDD5D19), UINT32_C(0x5B11E2),
    UINT32_C(0xC46EF5), UINT32_C(0x42220E), UINT32_C(0x4EBBF8),
    UINT32_C(0xC8F703), UINT32_C(0x3F964D), UINT32_C(0xB9DAB6),
    UINT32_C(0xB54340), UINT32_C(0x330FBB), UINT32_C(0xAC70AC),
    UINT32_C(0x2A3C57), UINT32_C(0x26A5A1), UINT32_C(0xA0E95A),
    UINT32_C(0x9E1774), UINT32_C(0x185B8F), UINT32_C(0x14C279),
    UINT32_C(0x928E82), UINT32_C(0x0DF195), UINT32_C(0x8BBD6E),
    UINT32_C(0x872498), UINT32_C(0x016863), UINT32_C(0xFAD8C4),
    UINT32_C(0x7C943F), UINT32_C(0x700DC9), UINT32_C(0xF64132),
    UINT32_C(0x693E25), UINT32_C(0xEF72DE), UINT32_C(0xE3EB28),
    UINT32_C(0x65A7D3), UINT32_C(0x5B59FD), UINT32_C(0xDD1506),
    UINT32_C(0xD18CF0), UINT32_C(0x57C00B), UINT32_C(0xC8BF1C),
    UINT32_C(0x4EF3E7), UINT32_C(0x426A11), UINT32_C(0xC426EA),
    UINT32_C(0x2AE476), UINT32_C(0xACA88D), UINT32_C(0xA0317B),
    UINT32_C(0x267D80), UINT32_C(0xB90297), UINT32_C(0x3F4E6C),
    UINT32_C(0x33D79A), UINT32_C(0xB59B61), UINT32_C(0x8B654F),
    UINT32_C(0x0D29B4), UINT32_C(0x01B042), UINT32_C(0x87FCB9),
    UINT32_C(0x1883AE), UINT32_C(0x9ECF55), UINT32_C(0x9256A3),
    UINT32_C(0x141A58), UINT32_C(0xEFAAFF), UINT32_C(0x69E604),
    UINT32_C(0x657FF2), UINT32_C(0xE33309), UINT32_C(0x7C4C1E),
    UINT32_C(0xFA00E5), UINT32_C(0xF69913), UINT32_C(0x70D5E8),
    UINT32_C(0x4E2BC6), UINT32_C(0xC8673D), UINT32_C(0xC4FECB),
    UINT32_C(0x42B230), UINT32_C(0xDDCD27), UINT32_C(0x5B81DC),
    UINT32_C(0x57182A), UINT32_C(0xD154D1), UINT32_C(0x26359F),
    UINT32_C(0xA07964), UINT32_C(0xACE092), UINT32_C(0x2AAC69),
    UINT32_C(0xB5D37E), UINT32_C(0x339F85), UINT32_C(0x3F0673),
    UINT32_C(0xB94A88), UINT32_C(0x87B4A6), UINT32_C(0x01F85D),
    UINT32_C(0x0D61AB), UINT32_C(0x8B2D50), UINT32_C(0x145247),
    UINT32_C(0x921EBC), UINT32_C(0x9E874A), UINT32_C(0x18CBB1),
    UINT32_C(0xE37B16), UINT32_C(0x6537ED), UINT32_C(0x69AE1B),
    UINT32_C(0xEFE2E0), UINT32_C(0x709DF7), UINT32_C(0xF6D10C),
    UINT32_C(0xFA48FA), UINT32_C(0x7C0401), UINT32_C(0x42FA2F),
    UINT32_C(0xC4B6D4), UINT32_C(0xC82F22), UINT32_C(0x4E63D9),
    UINT32_C(0xD11CCE), UINT32_C(0x575035), UINT32_C(0x5BC9C3),
    UINT32_C(0xDD8538)};

static inline uint32_t _CRC24Q_do_update(uint32_t crc, uint8_t byte) {
  return (crc << 8) ^ _CRC24Q_Table[byte ^ (uint8_t)(crc >> 16)];
}

uint32_t _CRC24Q_Update(uint32_t crc, uint8_t byte) {
  return _CRC24Q_do_update(crc, byte);
}

uint32_t _CRC24Q_Sequence_update(uint32_t crc,
                                 const void* bytes,
                                 size_t size_in_bytes) {
  const uint8_t* the_bytes = bytes;

  for (size_t i = 0; i < size_in_bytes; ++i) {
    crc = _CRC24Q_do_update(crc, the_bytes[i]);
  }

  return crc;
}
