/*
 * Copyright 2017-2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*
 * How to setup clock using clock driver functions:
 *
 * 1. Call CLOCK_InitXXXPLL() to configure corresponding PLL clock.
 *
 * 2. Call CLOCK_InitXXXpfd() to configure corresponding PLL pfd clock.
 *
 * 3. Call CLOCK_SetRootClock() to configure corresponding module clock source and divider.
 *
 */

/* TEXT BELOW IS USED AS SETTING FOR TOOLS *************************************
!!GlobalInfo
product: Clocks v11.0
processor: MIMXRT1166xxxxx
package_id: MIMXRT1166DVM6A
mcu_data: ksdk2_0
processor_version: 13.0.2
 * BE CAREFUL MODIFYING THIS COMMENT - IT IS YAML SETTINGS FOR TOOLS **********/

#include "clock_config.h"
#include "fsl_iomuxc.h"
#include "fsl_dcdc.h"
#include "fsl_pmu.h"
#include "fsl_clock.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 ************************ BOARD_InitBootClocks function ************************
 ******************************************************************************/
void BOARD_InitBootClocks(void)
{
    BOARD_BootClockRUN();
}

/*******************************************************************************
 ********************** Configuration BOARD_BootClockRUN ***********************
 ******************************************************************************/
/* TEXT BELOW IS USED AS SETTING FOR TOOLS *************************************
!!Configuration
name: BOARD_BootClockRUN
called_from_default_init: true
outputs:
- {id: ACMP_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ADC1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ADC2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ARM_PLL_CLK.outFreq, value: 600 MHz, locked: true, accuracy: '0.001'}
- {id: ASRC_CLK_ROOT.outFreq, value: 24 MHz}
- {id: AXI_CLK_ROOT.outFreq, value: 600 MHz}
- {id: BUS_CLK_ROOT.outFreq, value: 198 MHz}
- {id: BUS_LPSR_CLK_ROOT.outFreq, value: 120 MHz}
- {id: CAN1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CAN2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CAN3_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CCM_CLKO1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CCM_CLKO2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CLK_1M.outFreq, value: 1 MHz}
- {id: CSI2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CSI2_ESC_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CSI2_UI_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CSI_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CSSYS_CLK_ROOT.outFreq, value: 24 MHz}
- {id: CSTRACE_CLK_ROOT.outFreq, value: 132 MHz}
- {id: ELCDIF_CLK_ROOT.outFreq, value: 24 MHz}
- {id: EMV1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: EMV2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ENET1_CLK_ROOT.outFreq, value: 50 MHz, locked: true, accuracy: '0.001'}
- {id: ENET2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ENET_1G_TX_CLK.outFreq, value: 24 MHz}
- {id: ENET_25M_CLK_ROOT.outFreq, value: 25 MHz, locked: true, accuracy: '0.001'}
- {id: ENET_REF_CLK.outFreq, value: 50 MHz, locked: true, accuracy: '0.001'}
- {id: ENET_TIMER1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ENET_TIMER2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: ENET_TX_CLK.outFreq, value: 25 MHz, locked: true, accuracy: '0.001'}
- {id: FLEXIO1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: FLEXIO2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: FLEXSPI1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: FLEXSPI2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GC355_CLK_ROOT.outFreq, value: 492.0000125 MHz}
- {id: GPT1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT1_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: GPT2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT2_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: GPT3_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT3_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: GPT4_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT4_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: GPT5_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT5_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: GPT6_CLK_ROOT.outFreq, value: 24 MHz}
- {id: GPT6_ipg_clk_highfreq.outFreq, value: 24 MHz}
- {id: LCDIFV2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C3_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C4_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C5_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPI2C6_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPSPI1_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPSPI2_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPSPI3_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPSPI4_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPSPI5_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPSPI6_CLK_ROOT.outFreq, value: 50 MHz}
- {id: LPUART10_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART11_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART12_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART3_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART4_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART5_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART6_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART7_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART8_CLK_ROOT.outFreq, value: 24 MHz}
- {id: LPUART9_CLK_ROOT.outFreq, value: 24 MHz}
- {id: M4_CLK_ROOT.outFreq, value: 240 MHz}
- {id: M4_SYSTICK_CLK_ROOT.outFreq, value: 24 MHz}
- {id: M7_CLK_ROOT.outFreq, value: 600 MHz}
- {id: M7_SYSTICK_CLK_ROOT.outFreq, value: 24 MHz}
- {id: MIC_CLK_ROOT.outFreq, value: 24 MHz}
- {id: MIPI_DSI_TX_CLK_ESC_ROOT.outFreq, value: 24 MHz}
- {id: MIPI_ESC_CLK_ROOT.outFreq, value: 24 MHz}
- {id: MIPI_REF_CLK_ROOT.outFreq, value: 24 MHz}
- {id: MQS_CLK_ROOT.outFreq, value: 24 MHz}
- {id: MQS_MCLK.outFreq, value: 24 MHz}
- {id: OSC_24M.outFreq, value: 24 MHz}
- {id: OSC_32K.outFreq, value: 32.768 kHz}
- {id: OSC_RC_16M.outFreq, value: 16 MHz}
- {id: OSC_RC_400M.outFreq, value: 400 MHz}
- {id: OSC_RC_48M.outFreq, value: 48 MHz}
- {id: OSC_RC_48M_DIV2.outFreq, value: 24 MHz}
- {id: PLL_VIDEO_CLK.outFreq, value: 984.000025 MHz, locked: true, accuracy: '0.001'}
- {id: SAI1_CLK_ROOT.outFreq, value: 24 MHz}
- {id: SAI1_MCLK1.outFreq, value: 24 MHz}
- {id: SAI1_MCLK3.outFreq, value: 24 MHz}
- {id: SAI2_CLK_ROOT.outFreq, value: 24 MHz}
- {id: SAI2_MCLK1.outFreq, value: 24 MHz}
- {id: SAI2_MCLK3.outFreq, value: 24 MHz}
- {id: SAI3_CLK_ROOT.outFreq, value: 24 MHz}
- {id: SAI3_MCLK1.outFreq, value: 24 MHz}
- {id: SAI3_MCLK3.outFreq, value: 24 MHz}
- {id: SAI4_CLK_ROOT.outFreq, value: 24 MHz}
- {id: SAI4_MCLK1.outFreq, value: 24 MHz}
- {id: SEMC_CLK_ROOT.outFreq, value: 2160/13 MHz}
- {id: SPDIF_CLK_ROOT.outFreq, value: 24 MHz}
- {id: SYS_PLL1_CLK.outFreq, value: 1 GHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL1_DIV2_CLK.outFreq, value: 500 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL1_DIV5_CLK.outFreq, value: 200 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL2_CLK.outFreq, value: 528 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL2_PFD0_CLK.outFreq, value: 352 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL2_PFD1_CLK.outFreq, value: 594 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL2_PFD2_CLK.outFreq, value: 396 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL2_PFD3_CLK.outFreq, value: 396 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_CLK.outFreq, value: 480 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_DIV2_CLK.outFreq, value: 240 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_PFD0_CLK.outFreq, value: 8640/13 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_PFD1_CLK.outFreq, value: 8640/17 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_PFD2_CLK.outFreq, value: 270 MHz, locked: true, accuracy: '0.001'}
- {id: SYS_PLL3_PFD3_CLK.outFreq, value: 4320/11 MHz, locked: true, accuracy: '0.001'}
- {id: USDHC1_CLK_ROOT.outFreq, value: 200 MHz}
- {id: USDHC2_CLK_ROOT.outFreq, value: 24 MHz}
settings:
- {id: CoreBusClockRootsInitializationConfig, value: selectedCore}
- {id: SemcConfigurationPatchConfig, value: disabled}
- {id: ANADIG_OSC_OSC_24M_CTRL_LP_EN_CFG, value: Low}
- {id: ANADIG_OSC_OSC_24M_CTRL_OSC_EN_CFG, value: Enabled}
- {id: ANADIG_PLL.ARM_PLL_POST_DIV.scale, value: '4'}
- {id: ANADIG_PLL.ARM_PLL_VDIV.scale, value: '100'}
- {id: ANADIG_PLL.PLL_AUDIO_BYPASS.sel, value: ANADIG_OSC.OSC_24M}
- {id: ANADIG_PLL.PLL_VIDEO.denom, value: '960000'}
- {id: ANADIG_PLL.PLL_VIDEO.div, value: '41'}
- {id: ANADIG_PLL.PLL_VIDEO.num, value: '1'}
- {id: ANADIG_PLL.SYS_PLL2.denom, value: '268435455'}
- {id: ANADIG_PLL.SYS_PLL2.div, value: '22'}
- {id: ANADIG_PLL.SYS_PLL2.num, value: '0'}
- {id: ANADIG_PLL.SYS_PLL2_PFD3_DIV.scale, value: '24'}
- {id: ANADIG_PLL.SYS_PLL2_SS_DIV.scale, value: '268435455'}
- {id: ANADIG_PLL.SYS_PLL3_PFD3_DIV.scale, value: '22', locked: true}
- {id: ANADIG_PLL.SYS_PLL3_PFD3_MUL.scale, value: '18', locked: true}
- {id: ANADIG_PLL_ARM_PLL_CTRL_POWERUP_CFG, value: Enabled}
- {id: ANADIG_PLL_PLL_AUDIO_CTRL_GATE_CFG, value: Disabled}
- {id: ANADIG_PLL_PLL_VIDEO_CTRL0_POWERUP_CFG, value: Enabled}
- {id: ANADIG_PLL_SYS_PLL1_CTRL_SYS_PLL1_DIV2_CFG, value: Enabled}
- {id: ANADIG_PLL_SYS_PLL1_CTRL_SYS_PLL1_DIV5_CFG, value: Enabled}
- {id: ANADIG_PLL_SYS_PLL2_CTRL_POWERUP_CFG, value: Enabled}
- {id: ANADIG_PLL_SYS_PLL3_CTRL_POWERUP_CFG, value: Enabled}
- {id: ANADIG_PLL_SYS_PLL3_CTRL_SYS_PLL3_DIV2_CFG, value: Enabled}
- {id: CCM.CLOCK_ROOT0.MUX.sel, value: ANADIG_PLL.ARM_PLL_CLK}
- {id: CCM.CLOCK_ROOT1.DIV.scale, value: '2'}
- {id: CCM.CLOCK_ROOT1.MUX.sel, value: ANADIG_PLL.SYS_PLL3_CLK}
- {id: CCM.CLOCK_ROOT2.DIV.scale, value: '2'}
- {id: CCM.CLOCK_ROOT2.MUX.sel, value: ANADIG_PLL.SYS_PLL2_PFD3_CLK}
- {id: CCM.CLOCK_ROOT25.DIV.scale, value: '22'}
- {id: CCM.CLOCK_ROOT25.MUX.sel, value: ANADIG_PLL.SYS_PLL2_CLK}
- {id: CCM.CLOCK_ROOT26.DIV.scale, value: '22'}
- {id: CCM.CLOCK_ROOT26.MUX.sel, value: ANADIG_PLL.SYS_PLL2_CLK}
- {id: CCM.CLOCK_ROOT3.DIV.scale, value: '4'}
- {id: CCM.CLOCK_ROOT3.MUX.sel, value: ANADIG_PLL.SYS_PLL3_CLK}
- {id: CCM.CLOCK_ROOT4.DIV.scale, value: '4'}
- {id: CCM.CLOCK_ROOT4.MUX.sel, value: ANADIG_PLL.SYS_PLL3_PFD0_CLK}
- {id: CCM.CLOCK_ROOT43.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT43.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT44.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT44.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT45.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT45.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT46.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT46.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT47.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT47.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT48.DIV.scale, value: '4', locked: true}
- {id: CCM.CLOCK_ROOT48.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT51.DIV.scale, value: '4'}
- {id: CCM.CLOCK_ROOT51.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT54.DIV.scale, value: '8'}
- {id: CCM.CLOCK_ROOT54.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT55.MUX.sel, value: ANADIG_OSC.OSC_24M}
- {id: CCM.CLOCK_ROOT58.MUX.sel, value: ANADIG_PLL.SYS_PLL1_DIV5_CLK}
- {id: CCM.CLOCK_ROOT6.DIV.scale, value: '4'}
- {id: CCM.CLOCK_ROOT6.MUX.sel, value: ANADIG_PLL.SYS_PLL2_CLK}
- {id: CCM.CLOCK_ROOT68.DIV.scale, value: '2'}
- {id: CCM.CLOCK_ROOT68.MUX.sel, value: ANADIG_PLL.PLL_VIDEO_CLK}
- {id: IOMUXC_GPR.ENET_REF_CLK_SEL.sel, value: CCM.ENET1_CLK_ROOT}
sources:
- {id: IOMUXC_GPR.ENET_REF_CLK_EXT.outFreq, value: 50 MHz, enabled: true}
- {id: IOMUXC_GPR.ENET_TX_CLK_EXT.outFreq, value: 25 MHz, enabled: true}
 * BE CAREFUL MODIFYING THIS COMMENT - IT IS YAML SETTINGS FOR TOOLS **********/

/*******************************************************************************
 * Variables for BOARD_BootClockRUN configuration
 ******************************************************************************/

#if __CORTEX_M == 7
#define BYPASS_LDO_LPSR 1
#endif

const clock_arm_pll_config_t armPllConfig_BOARD_BootClockRUN =
    {
        .postDivider = kCLOCK_PllPostDiv4,        /* Post divider, 0 - DIV by 2, 1 - DIV by 4, 2 - DIV by 8, 3 - DIV by 1 */
        .loopDivider = 200,                       /* PLL Loop divider, Fout = Fin * ( loopDivider / ( 2 * postDivider ) ) */
    };

const clock_sys_pll1_config_t sysPll1Config_BOARD_BootClockRUN =
    {
        .pllDiv2En = 1,                           /* Enable Sys Pll1 divide-by-2 clock or not */
        .pllDiv5En = 1,                           /* Enable Sys Pll1 divide-by-5 clock or not */
        .ss = NULL,                               /* Spread spectrum parameter */
        .ssEnable = false,                        /* Enable spread spectrum or not */
    };

const clock_sys_pll2_config_t sysPll2Config_BOARD_BootClockRUN =
    {
        .mfd = 268435455,                         /* Denominator of spread spectrum */
        .ss = NULL,                               /* Spread spectrum parameter */
        .ssEnable = false,                        /* Enable spread spectrum or not */
    };

const clock_video_pll_config_t videoPllConfig_BOARD_BootClockRUN =
    {
        .loopDivider = 41,                        /* PLL Loop divider, valid range for DIV_SELECT divider value: 27 ~ 54. */
        .postDivider = 0,                         /* Divider after PLL, should only be 1, 2, 4, 8, 16, 32 */
        .numerator = 1,                           /* 30 bit numerator of fractional loop divider, Fout = Fin * ( loopDivider + numerator / denominator ) */
        .denominator = 960000,                    /* 30 bit denominator of fractional loop divider, Fout = Fin * ( loopDivider + numerator / denominator ) */
        .ss = NULL,                               /* Spread spectrum parameter */
        .ssEnable = false,                        /* Enable spread spectrum or not */
    };

/*******************************************************************************
 * Code for BOARD_BootClockRUN configuration
 ******************************************************************************/
void BOARD_BootClockRUN(void)
{
    clock_root_config_t rootCfg = {0};

    /* Set DCDC to DCM mode to improve the efficiency for light loading in run mode and transient performance with a big loading step. */
    DCDC_BootIntoDCM(DCDC);

#if defined(BYPASS_LDO_LPSR) && BYPASS_LDO_LPSR
    PMU_StaticEnableLpsrAnaLdoBypassMode(ANADIG_LDO_SNVS, true);
    PMU_StaticEnableLpsrDigLdoBypassMode(ANADIG_LDO_SNVS, true);
#endif

    /* Config CLK_1M */
    CLOCK_OSC_Set1MHzOutputBehavior(kCLOCK_1MHzOutEnableFreeRunning1Mhz);

    /* Init OSC RC 16M */
    ANADIG_OSC->OSC_16M_CTRL |= ANADIG_OSC_OSC_16M_CTRL_EN_IRC4M16M_MASK;

    /* Init OSC RC 400M */
    CLOCK_OSC_EnableOscRc400M();
    CLOCK_OSC_GateOscRc400M(true);

    /* Init OSC RC 48M */
    CLOCK_OSC_EnableOsc48M(true);
    CLOCK_OSC_EnableOsc48MDiv2(true);

    /* Config OSC 24M */
    ANADIG_OSC->OSC_24M_CTRL |= ANADIG_OSC_OSC_24M_CTRL_OSC_EN(1) | ANADIG_OSC_OSC_24M_CTRL_BYPASS_EN(0) | ANADIG_OSC_OSC_24M_CTRL_BYPASS_CLK(0) | ANADIG_OSC_OSC_24M_CTRL_LP_EN(1) | ANADIG_OSC_OSC_24M_CTRL_OSC_24M_GATE(0);
    /* Wait for 24M OSC to be stable. */
    while (ANADIG_OSC_OSC_24M_CTRL_OSC_24M_STABLE_MASK !=
            (ANADIG_OSC->OSC_24M_CTRL & ANADIG_OSC_OSC_24M_CTRL_OSC_24M_STABLE_MASK))
    {
    }

    /* Swicth both core, M7 Systick and Bus_Lpsr to OscRC48MDiv2 first */
#if __CORTEX_M == 7
    rootCfg.mux = kCLOCK_M7_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M7, &rootCfg);

    rootCfg.mux = kCLOCK_M7_SYSTICK_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M7_Systick, &rootCfg);
#endif

#if __CORTEX_M == 4
    rootCfg.mux = kCLOCK_M4_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M4, &rootCfg);

    rootCfg.mux = kCLOCK_BUS_LPSR_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Bus_Lpsr, &rootCfg);
#endif

    /* Init Arm Pll. */
    CLOCK_InitArmPll(&armPllConfig_BOARD_BootClockRUN);

    /* Init Sys Pll1. */
    CLOCK_InitSysPll1(&sysPll1Config_BOARD_BootClockRUN);

    /* Init Sys Pll2. */
    CLOCK_InitSysPll2(&sysPll2Config_BOARD_BootClockRUN);

    /* Init System Pll2 pfd0. */
    CLOCK_InitPfd(kCLOCK_PllSys2, kCLOCK_Pfd0, 27);

    /* Init System Pll2 pfd1. */
    CLOCK_InitPfd(kCLOCK_PllSys2, kCLOCK_Pfd1, 16);

    /* Init System Pll2 pfd2. */
    CLOCK_InitPfd(kCLOCK_PllSys2, kCLOCK_Pfd2, 24);

    /* Init System Pll2 pfd3. */
    CLOCK_InitPfd(kCLOCK_PllSys2, kCLOCK_Pfd3, 24);

    /* Init Sys Pll3. */
    CLOCK_InitSysPll3();

    /* Init System Pll3 pfd0. */
    CLOCK_InitPfd(kCLOCK_PllSys3, kCLOCK_Pfd0, 13);

    /* Init System Pll3 pfd1. */
    CLOCK_InitPfd(kCLOCK_PllSys3, kCLOCK_Pfd1, 17);

    /* Init System Pll3 pfd2. */
    CLOCK_InitPfd(kCLOCK_PllSys3, kCLOCK_Pfd2, 32);

    /* Init System Pll3 pfd3. */
    CLOCK_InitPfd(kCLOCK_PllSys3, kCLOCK_Pfd3, 22);

    /* Bypass Audio Pll. */
    CLOCK_SetPllBypass(kCLOCK_PllAudio, true);

    /* DeInit Audio Pll. */
    CLOCK_DeinitAudioPll();

    /* Init Video Pll. */
    CLOCK_InitVideoPll(&videoPllConfig_BOARD_BootClockRUN);

    /* Module clock root configurations. */
    /* Configure M7 using ARM_PLL_CLK */
#if __CORTEX_M == 7
    rootCfg.mux = kCLOCK_M7_ClockRoot_MuxArmPllOut;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M7, &rootCfg);
#endif

    /* Configure M4 using SYS_PLL3_CLK */
#if __CORTEX_M == 4
    rootCfg.mux = kCLOCK_M4_ClockRoot_MuxSysPll3Out;
    rootCfg.div = 2;
    CLOCK_SetRootClock(kCLOCK_Root_M4, &rootCfg);
#endif

    /* Configure BUS using SYS_PLL2_PFD3_CLK */
    rootCfg.mux = kCLOCK_BUS_ClockRoot_MuxSysPll2Pfd3;
    rootCfg.div = 2;
    CLOCK_SetRootClock(kCLOCK_Root_Bus, &rootCfg);

    /* Configure BUS_LPSR using SYS_PLL3_CLK */
    rootCfg.mux = kCLOCK_BUS_LPSR_ClockRoot_MuxSysPll3Out;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Bus_Lpsr, &rootCfg);

    /* Configure SEMC using SYS_PLL3_PFD0_CLK */
#ifndef SKIP_SEMC_INIT
    rootCfg.mux = kCLOCK_SEMC_ClockRoot_MuxSysPll3Pfd0;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Semc, &rootCfg);
#endif

    /* Configure CSSYS using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CSSYS_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Cssys, &rootCfg);

    /* Configure CSTRACE using SYS_PLL2_CLK */
    rootCfg.mux = kCLOCK_CSTRACE_ClockRoot_MuxSysPll2Out;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Cstrace, &rootCfg);

    /* Configure M4_SYSTICK using OSC_RC_48M_DIV2 */
#if __CORTEX_M == 4
    rootCfg.mux = kCLOCK_M4_SYSTICK_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M4_Systick, &rootCfg);
#endif

    /* Configure M7_SYSTICK using OSC_RC_48M_DIV2 */
#if __CORTEX_M == 7
    rootCfg.mux = kCLOCK_M7_SYSTICK_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_M7_Systick, &rootCfg);
#endif

    /* Configure ADC1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ADC1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Adc1, &rootCfg);

    /* Configure ADC2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ADC2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Adc2, &rootCfg);

    /* Configure ACMP using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ACMP_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Acmp, &rootCfg);

    /* Configure FLEXIO1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_FLEXIO1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Flexio1, &rootCfg);

    /* Configure FLEXIO2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_FLEXIO2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Flexio2, &rootCfg);

    /* Configure GPT1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt1, &rootCfg);

    /* Configure GPT2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt2, &rootCfg);

    /* Configure GPT3 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT3_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt3, &rootCfg);

    /* Configure GPT4 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT4_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt4, &rootCfg);

    /* Configure GPT5 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT5_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt5, &rootCfg);

    /* Configure GPT6 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_GPT6_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Gpt6, &rootCfg);

    /* Configure FLEXSPI1 using OSC_RC_48M_DIV2 */
#if !(defined(XIP_EXTERNAL_FLASH) && (XIP_EXTERNAL_FLASH == 1) || defined(FLEXSPI_IN_USE))
    rootCfg.mux = kCLOCK_FLEXSPI1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Flexspi1, &rootCfg);
#endif

    /* Configure FLEXSPI2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_FLEXSPI2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Flexspi2, &rootCfg);

    /* Configure CAN1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CAN1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Can1, &rootCfg);

    /* Configure CAN2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CAN2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Can2, &rootCfg);

    /* Configure CAN3 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CAN3_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Can3, &rootCfg);

    /* Configure LPUART1 using SYS_PLL2_CLK */
    rootCfg.mux = kCLOCK_LPUART1_ClockRoot_MuxSysPll2Out;
    rootCfg.div = 22;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart1, &rootCfg);

    /* Configure LPUART2 using SYS_PLL2_CLK */
    rootCfg.mux = kCLOCK_LPUART2_ClockRoot_MuxSysPll2Out;
    rootCfg.div = 22;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart2, &rootCfg);

    /* Configure LPUART3 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART3_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart3, &rootCfg);

    /* Configure LPUART4 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART4_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart4, &rootCfg);

    /* Configure LPUART5 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART5_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart5, &rootCfg);

    /* Configure LPUART6 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART6_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart6, &rootCfg);

    /* Configure LPUART7 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART7_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart7, &rootCfg);

    /* Configure LPUART8 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART8_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart8, &rootCfg);

    /* Configure LPUART9 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART9_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart9, &rootCfg);

    /* Configure LPUART10 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART10_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart10, &rootCfg);

    /* Configure LPUART11 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART11_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart11, &rootCfg);

    /* Configure LPUART12 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPUART12_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpuart12, &rootCfg);

    /* Configure LPI2C1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c1, &rootCfg);

    /* Configure LPI2C2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c2, &rootCfg);

    /* Configure LPI2C3 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C3_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c3, &rootCfg);

    /* Configure LPI2C4 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C4_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c4, &rootCfg);

    /* Configure LPI2C5 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C5_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c5, &rootCfg);

    /* Configure LPI2C6 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LPI2C6_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lpi2c6, &rootCfg);

    /* Configure LPSPI1 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI1_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi1, &rootCfg);

    /* Configure LPSPI2 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI2_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi2, &rootCfg);

    /* Configure LPSPI3 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI3_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi3, &rootCfg);

    /* Configure LPSPI4 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI4_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi4, &rootCfg);

    /* Configure LPSPI5 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI5_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi5, &rootCfg);

    /* Configure LPSPI6 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_LPSPI6_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Lpspi6, &rootCfg);

    /* Configure EMV1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_EMV1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Emv1, &rootCfg);

    /* Configure EMV2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_EMV2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Emv2, &rootCfg);

    /* Configure ENET1 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_ENET1_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 4;
    CLOCK_SetRootClock(kCLOCK_Root_Enet1, &rootCfg);

    /* Configure ENET2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ENET2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Enet2, &rootCfg);

    /* Configure ENET_25M using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_ENET_25M_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 8;
    CLOCK_SetRootClock(kCLOCK_Root_Enet_25m, &rootCfg);

    /* Configure ENET_TIMER1 using OSC_24M */
    rootCfg.mux = kCLOCK_ENET_TIMER1_ClockRoot_MuxOsc24MOut;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Enet_Timer1, &rootCfg);

    /* Configure ENET_TIMER2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ENET_TIMER2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Enet_Timer2, &rootCfg);

    /* Configure USDHC1 using SYS_PLL1_DIV5_CLK */
    rootCfg.mux = kCLOCK_USDHC1_ClockRoot_MuxSysPll1Div5;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Usdhc1, &rootCfg);

    /* Configure USDHC2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_USDHC2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Usdhc2, &rootCfg);

    /* Configure ASRC using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_ASRC_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Asrc, &rootCfg);

    /* Configure MQS using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_MQS_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Mqs, &rootCfg);

    /* Configure MIC using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_MIC_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Mic, &rootCfg);

    /* Configure SPDIF using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_SPDIF_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Spdif, &rootCfg);

    /* Configure SAI1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_SAI1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Sai1, &rootCfg);

    /* Configure SAI2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_SAI2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Sai2, &rootCfg);

    /* Configure SAI3 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_SAI3_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Sai3, &rootCfg);

    /* Configure SAI4 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_SAI4_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Sai4, &rootCfg);

    /* Configure GC355 using PLL_VIDEO_CLK */
    rootCfg.mux = kCLOCK_GC355_ClockRoot_MuxVideoPllOut;
    rootCfg.div = 2;
    CLOCK_SetRootClock(kCLOCK_Root_Gc355, &rootCfg);

    /* Configure LCDIF using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LCDIF_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lcdif, &rootCfg);

    /* Configure LCDIFV2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_LCDIFV2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Lcdifv2, &rootCfg);

    /* Configure MIPI_REF using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_MIPI_REF_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Mipi_Ref, &rootCfg);

    /* Configure MIPI_ESC using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_MIPI_ESC_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Mipi_Esc, &rootCfg);

    /* Configure CSI2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CSI2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Csi2, &rootCfg);

    /* Configure CSI2_ESC using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CSI2_ESC_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Csi2_Esc, &rootCfg);

    /* Configure CSI2_UI using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CSI2_UI_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Csi2_Ui, &rootCfg);

    /* Configure CSI using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CSI_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Csi, &rootCfg);

    /* Configure CKO1 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CKO1_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Cko1, &rootCfg);

    /* Configure CKO2 using OSC_RC_48M_DIV2 */
    rootCfg.mux = kCLOCK_CKO2_ClockRoot_MuxOscRc48MDiv2;
    rootCfg.div = 1;
    CLOCK_SetRootClock(kCLOCK_Root_Cko2, &rootCfg);

    /* Set SAI1 MCLK1 clock source. */
    IOMUXC_SetSaiMClkClockSource(IOMUXC_GPR, kIOMUXC_GPR_SAI1MClk1Sel, 0);
    /* Set SAI1 MCLK2 clock source. */
    IOMUXC_SetSaiMClkClockSource(IOMUXC_GPR, kIOMUXC_GPR_SAI1MClk2Sel, 3);
    /* Set SAI1 MCLK3 clock source. */
    IOMUXC_SetSaiMClkClockSource(IOMUXC_GPR, kIOMUXC_GPR_SAI1MClk3Sel, 0);
    /* Set SAI2 MCLK3 clock source. */
    IOMUXC_SetSaiMClkClockSource(IOMUXC_GPR, kIOMUXC_GPR_SAI2MClk3Sel, 0);
    /* Set SAI3 MCLK3 clock source. */
    IOMUXC_SetSaiMClkClockSource(IOMUXC_GPR, kIOMUXC_GPR_SAI3MClk3Sel, 0);

    /* Set MQS configuration. */
    IOMUXC_MQSConfig(IOMUXC_GPR,kIOMUXC_MqsPwmOverSampleRate32, 0);
    /* Set ENET Tx clock source. */
    IOMUXC_GPR->GPR4 |= IOMUXC_GPR_GPR4_ENET_TX_CLK_SEL_MASK;
    /* Set ENET Ref clock source. */
    IOMUXC_GPR->GPR4 |= IOMUXC_GPR_GPR4_ENET_REF_CLK_DIR_MASK;
    /* Set ENET_1G Tx clock source. */
    IOMUXC_GPR->GPR5 = ((IOMUXC_GPR->GPR5 & ~IOMUXC_GPR_GPR5_ENET1G_TX_CLK_SEL_MASK) | IOMUXC_GPR_GPR5_ENET1G_RGMII_EN_MASK);
    /* Set ENET_1G Ref clock source. */
    IOMUXC_GPR->GPR5 &= ~IOMUXC_GPR_GPR5_ENET1G_REF_CLK_DIR_MASK;
    /* Set GPT1 High frequency reference clock source. */
    IOMUXC_GPR->GPR22 &= ~IOMUXC_GPR_GPR22_REF_1M_CLK_GPT1_MASK;
    /* Set GPT2 High frequency reference clock source. */
    IOMUXC_GPR->GPR23 &= ~IOMUXC_GPR_GPR23_REF_1M_CLK_GPT2_MASK;
    /* Set GPT3 High frequency reference clock source. */
    IOMUXC_GPR->GPR24 &= ~IOMUXC_GPR_GPR24_REF_1M_CLK_GPT3_MASK;
    /* Set GPT4 High frequency reference clock source. */
    IOMUXC_GPR->GPR25 &= ~IOMUXC_GPR_GPR25_REF_1M_CLK_GPT4_MASK;
    /* Set GPT5 High frequency reference clock source. */
    IOMUXC_GPR->GPR26 &= ~IOMUXC_GPR_GPR26_REF_1M_CLK_GPT5_MASK;
    /* Set GPT6 High frequency reference clock source. */
    IOMUXC_GPR->GPR27 &= ~IOMUXC_GPR_GPR27_REF_1M_CLK_GPT6_MASK;

#if __CORTEX_M == 7
    SystemCoreClock = CLOCK_GetRootClockFreq(kCLOCK_Root_M7);
#else
    SystemCoreClock = CLOCK_GetRootClockFreq(kCLOCK_Root_M4);
#endif
}
