/* SPDX-License-Identifier: BSD-2-Clause */

/**
 * @file
 * 
 * @addtogroup RTEMSScoreCPUPowerPC
 * 
 * @brief PowerPc MSR and Registers Access Definitions
 * 
 * This file contains some powerpc MSR and registers access definitions.
 */

/*
 * COPYRIGHT (C) 1999  Eric Valette (valette@crf.canon.fr)
 *                     Canon Centre Recherche France.
 *
 *  Added MPC8260 Andy Dachs <a.dachs@sstl.co.uk>
 *  Surrey Satellite Technology Limited
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _RTEMS_POWERPC_REGISTERS_H
#define _RTEMS_POWERPC_REGISTERS_H

/* Bit encodings for Machine State Register (MSR) */
#define MSR_CM		(1<<31)		/* Computation mode */
#define MSR_GS		(1<<28)		/* Guest state */
#define MSR_UCLE	(1<<26)		/* User-mode cache lock enable (e500) */
#define MSR_VE		(1<<25)		/* Alti-Vec enable (7400+) */
#define MSR_SPE		(1<<25)		/* SPE enable (e500) */
#define MSR_AP          (1<<25)         /* Auxiliary processor available */
#define MSR_APE         (1<<19)         /* APU exception enable */
#define MSR_POW		(1<<18)		/* Enable Power Management */
#define MSR_WE		(1<<18)		/* Wait state enable (e500, 4xx) */
#define MSR_TGPR	(1<<17)		/* TLB Update registers in use */
#define MSR_CE		(1<<17)		/* BookE critical interrupt */
#define MSR_ILE		(1<<16)		/* Interrupt Little-Endian enable */
#define MSR_EE		(1<<15)		/* External Interrupt enable */
#define MSR_PR		(1<<14)		/* Supervisor/User privilege */
#define MSR_FP		(1<<13)		/* Floating Point enable */
#define MSR_ME		(1<<12)		/* Machine Check enable */
#define MSR_FE0		(1<<11)		/* Floating Exception mode 0 */
#define MSR_SE		(1<<10)		/* Single Step */
#define MSR_UBLE	(1<<10)		/* User-mode BTB lock enable (e500) */
#define MSR_DWE         (1<<10)         /* Debug wait enable (4xx) */
#define MSR_BE		(1<<9)		/* Branch Trace */
#define MSR_DE		(1<<9)		/* BookE debug exception */
#define MSR_FE1		(1<<8)		/* Floating Exception mode 1 */
#define MSR_E300_CE	(1<<7)		/* e300 critical interrupt */
#define MSR_IP		(1<<6)		/* Exception prefix 0x000/0xFFF */
#define MSR_IR		(1<<5)		/* Instruction MMU enable */
#define MSR_DR		(1<<4)		/* Data MMU enable */
#define MSR_IS		(1<<5)		/* Instruction address space */
#define MSR_DS		(1<<4)		/* Data address space */
#define MSR_PMM		(1<<2)		/* Performance monitor mark */
#define MSR_RI		(1<<1)		/* Recoverable Exception */
#define MSR_LE		(1<<0)		/* Little-Endian enable */

/* Bit encodings for Hardware Implementation Register (HID0)
   on PowerPC 603, 604, etc. processors (not 601). */

/* WARNING: HID0/HID1 are *truely* implementation dependent!
 *          you *cannot* rely on the same bits to be present,
 *          at the same place or even in the same register
 *          on different CPU familys.
 *          E.g., EMCP isHID0_DOZE is HID0_HI_BAT_EN on the
 *          on the 7450s. IFFT is XBSEN on 7450 and so on...
 */
#define HID0_EMCP	(1<<31)		/* Enable Machine Check pin */
#define HID0_EBA	(1<<29)		/* Enable Bus Address Parity */
#define HID0_EBD	(1<<28)		/* Enable Bus Data Parity */
#define HID0_SBCLK	(1<<27)
#define HID0_TBEN       (1<<26)         /* 7455:this bit must be set
					 * and TBEN signal must be asserted
					 * to enable the time base and
					 * decrementer.
					 */
#define HID0_EICE	(1<<26)
#define HID0_ECLK	(1<<25)
#define HID0_PAR	(1<<24)
#define HID0_DOZE	(1<<23)
/* this HI_BAT_EN only on 7445, 7447, 7448, 7455 & 7457 !!          */
#define HID0_7455_HIGH_BAT_EN (1<<23)

#define HID0_NAP	(1<<22)
#define HID0_SLEEP	(1<<21)
#define HID0_DPM	(1<<20)
#define HID0_ICE	(1<<15)		/* Instruction Cache Enable */
#define HID0_DCE	(1<<14)		/* Data Cache Enable */
#define HID0_ILOCK	(1<<13)		/* Instruction Cache Lock */
#define HID0_DLOCK	(1<<12)		/* Data Cache Lock */
#define HID0_ICFI	(1<<11)		/* Instruction Cache Flash Invalidate */
#define HID0_DCI	(1<<10)		/* Data Cache Invalidate */
/* this bit is XSBSEN (xtended block size enable) on 7447, 7448, 7455 and 7457 only */
#define HID0_7455_XBSEN       (1<<8)
#define HID0_SIED	(1<<7)		/* Serial Instruction Execution [Disable] */
#define HID0_BTIC	(1<<5)          /* Branch Target Instruction Cache [Enable] */
/* S.K. Feng 10/03, added for MPC7455 */
#define HID0_LRSTK	(1<<4)          /* Link register stack enable (7455) */
#define HID0_FOLD	(1<<3)          /* Branch folding enable (7455) */

#define HID0_BHTE	(1<<2)		/* Branch History Table Enable */
#define HID0_BTCD	(1<<1)		/* Branch target cache disable */

/* fpscr settings */
#define FPSCR_FX        (1<<31)
#define FPSCR_FEX       (1<<30)

#define _MACH_prep     1
#define _MACH_Pmac     2 /* pmac or pmac clone (non-chrp) */
#define _MACH_chrp     4 /* chrp machine */
#define _MACH_mbx      8 /* Motorola MBX board */
#define _MACH_apus    16 /* amiga with phase5 powerup */
#define _MACH_fads    32 /* Motorola FADS board */

/* see residual.h for these */
#define _PREP_Motorola 0x01  /* motorola prep */
#define _PREP_Firm     0x02  /* firmworks prep */
#define _PREP_IBM      0x00  /* ibm prep */
#define _PREP_Bull     0x03  /* bull prep */

/* these are arbitrary */
#define _CHRP_Motorola 0x04  /* motorola chrp, the cobra */
#define _CHRP_IBM     0x05   /* IBM chrp, the longtrail and longtrail 2 */

#define _GLOBAL(n)\
	.globl n;\
n:

#define	TBRU	269	/* Time base Upper/Lower (Reading) */
#define	TBRL	268
#define TBWU	285	/* Time base Upper/Lower (Writing) */
#define TBWL	284
#define	PPC_XER	1
#define PPC_LR	8
#define PPC_CTR	9
#define HID0	1008	/* Hardware Implementation 0 */
#define HID1	1009	/* Hardware Implementation 1 */
#define HID2	1011	/* Hardware Implementation 2 */
#define DABR	1013	/* Data Access Breakpoint  */
#define PPC_PVR	287	/* Processor Version */
#define IBAT0U	528	/* Instruction BAT #0 Upper/Lower */
#define IBAT0L	529
#define IBAT1U	530	/* Instruction BAT #1 Upper/Lower */
#define IBAT1L	531
#define IBAT2U	532	/* Instruction BAT #2 Upper/Lower */
#define IBAT2L	533
#define IBAT3U	534	/* Instruction BAT #3 Upper/Lower */
#define IBAT3L	535

/* Only present on 7445, 7447, 7448, 7455 and 7457 (if HID0[HIGH_BAT_EN]) */
#define IBAT4U	560	/* Instruction BAT #4 Upper/Lower */
#define IBAT4L	561
#define IBAT5U	562	/* Instruction BAT #5 Upper/Lower */
#define IBAT5L	563
#define IBAT6U	564	/* Instruction BAT #6 Upper/Lower */
#define IBAT6L	565
#define IBAT7U	566	/* Instruction BAT #7 Upper/Lower */
#define IBAT7L	567

#define DBAT0U	536	/* Data BAT #0 Upper/Lower */
#define DBAT0L	537
#define DBAT1U	538	/* Data BAT #1 Upper/Lower */
#define DBAT1L	539
#define DBAT2U	540	/* Data BAT #2 Upper/Lower */
#define DBAT2L	541
#define DBAT3U	542	/* Data BAT #3 Upper/Lower */
#define DBAT3L	543

/* Only present on 7445, 7447, 7448, 7455 and 7457 (if HID0[HIGH_BAT_EN]) */
#define DBAT4U	568	/* Instruction BAT #4 Upper/Lower */
#define DBAT4L	569
#define DBAT5U	570	/* Instruction BAT #5 Upper/Lower */
#define DBAT5L	571
#define DBAT6U	572	/* Instruction BAT #6 Upper/Lower */
#define DBAT6L	573
#define DBAT7U	574	/* Instruction BAT #7 Upper/Lower */
#define DBAT7L	575

#define DMISS	976	/* TLB Lookup/Refresh registers */
#define DCMP	977
#define HASH1	978
#define HASH2	979
#define IMISS	980
#define ICMP	981
#define PPC_RPA	982
#define SDR1	25	/* MMU hash base register */
#define PPC_DAR	19	/* Data Address Register */
#define DEAR_BOOKE 61
#define DEAR_405 981
#define SPR0	272	/* Supervisor Private Registers */
#define SPRG0   272
#define SPR1	273
#define SPRG1   273
#define SPR2	274
#define SPRG2   274
#define SPR3	275
#define SPRG3   275
#define SPRG4   276
#define SPRG5   277
#define SPRG6   278
#define SPRG7   279
#define USPRG0  256
#define DSISR	18
#define SRR0	26	/* Saved Registers (exception) */
#define SRR1	27
#define IABR	1010	/* Instruction Address Breakpoint */
#define PPC_DEC	22	/* Decrementer */
#define PPC_EAR	282	/* External Address Register */

#define MSSCR0   1014   /* Memory Subsystem Control Register */

#define L2CR	1017    /* PPC 750 and 74xx L2 control register */

#define L2CR_L2E   (1<<31)	/* enable */
#define L2CR_L2I   (1<<21)	/* global invalidate */

/* watch out L2IO and L2DO are different between 745x and 7400/7410 */
/* Oddly, the following L2CR bit defintions in 745x
 * is different from that of 7400 and 7410.
 * Though not used in 7400 and 7410, it is appeded with _745x just
 * to be clarified.
 */
#define L2CR_L2IO_745x  0x100000  /* (1<<20) L2 Instruction-Only  */
#define L2CR_L2DO_745x  0x10000   /* (1<<16) L2 Data-Only */
#define L2CR_LOCK_745x  (L2CR_L2IO_745x|L2CR_L2DO_745x)
#define L2CR_L3OH0      0x00080000 /* 12:L3 output hold 0 */

#define L3CR    1018    /* PPC 7450/7455 L3 control register */
#define L3CR_L3IO_745x  0x400000  /* (1<<22) L3 Instruction-Only */
#define L3CR_L3DO_745x  0x40	  /* (1<<6) L3 Data-Only */

#define L3CR_LOCK_745x  (L3CR_L3IO_745x|L3CR_L3DO_745x)

#define	  L3CR_RESERVED		  0x0438003a /* Reserved bits in L3CR */
#define	  L3CR_L3E		  0x80000000 /* 0: L3 enable */
#define	  L3CR_L3PE		  0x40000000 /* 1: L3 data parity checking enable */
#define	  L3CR_L3APE		  0x20000000 /* 2: L3 address parity checking enable */
#define	  L3CR_L3SIZ		  0x10000000 /* 3: L3 size (0=1MB, 1=2MB) */
#define	   L3SIZ_1M		  0x00000000
#define	   L3SIZ_2M		  0x10000000
#define	  L3CR_L3CLKEN		  0x08000000 /* 4: Enables the L3_CLK[0:1] signals */
#define	  L3CR_L3CLK		  0x03800000 /* 6-8: L3 clock ratio */
#define	   L3CLK_60		  0x00000000 /* core clock / 6   */
#define	   L3CLK_20		  0x01000000 /*            / 2   */
#define	   L3CLK_25		  0x01800000 /*            / 2.5 */
#define	   L3CLK_30		  0x02000000 /*            / 3   */
#define	   L3CLK_35		  0x02800000 /*            / 3.5 */
#define	   L3CLK_40		  0x03000000 /*            / 4   */
#define	   L3CLK_50		  0x03800000 /*            / 5   */
#define	  L3CR_L3IO		  0x00400000 /* 9: L3 instruction-only mode */
#define	  L3CR_L3SPO		  0x00040000 /* 13: L3 sample point override */
#define	  L3CR_L3CKSP		  0x00030000 /* 14-15: L3 clock sample point */
#define	   L3CKSP_2		  0x00000000 /* 2 clocks */
#define	   L3CKSP_3		  0x00010000 /* 3 clocks */
#define	   L3CKSP_4		  0x00020000 /* 4 clocks */
#define	   L3CKSP_5		  0x00030000 /* 5 clocks */
#define	  L3CR_L3PSP		  0x0000e000 /* 16-18: L3 P-clock sample point */
#define	   L3PSP_0		  0x00000000 /* 0 clocks */
#define	   L3PSP_1		  0x00002000 /* 1 clocks */
#define	   L3PSP_2		  0x00004000 /* 2 clocks */
#define	   L3PSP_3		  0x00006000 /* 3 clocks */
#define	   L3PSP_4		  0x00008000 /* 4 clocks */
#define	   L3PSP_5		  0x0000a000 /* 5 clocks */
#define	  L3CR_L3REP		  0x00001000 /* 19: L3 replacement algorithm (0=default, 1=alternate) */
#define	  L3CR_L3HWF		  0x00000800 /* 20: L3 hardware flush */
#define	  L3CR_L3I		  0x00000400 /* 21: L3 global invaregisters.h.orig
lidate */
#define	  L3CR_L3RT		  0x00000300 /* 22-23: L3 SRAM type */
#define	   L3RT_MSUG2_DDR	  0x00000000 /* MSUG2 DDR SRAM */
#define	   L3RT_PIPELINE_LATE	  0x00000100 /* Pipelined (register-register) synchronous late-write SRAM */
#define	   L3RT_PB2_SRAM	  0x00000300 /* PB2 SRAM */
#define	  L3CR_L3NIRCA		  0x00000080 /* 24: L3 non-integer ratios clock adjustment for the SRAM */
#define	  L3CR_L3DO		  0x00000040 /* 25: L3 data-only mode */
#define	  L3CR_PMEN		  0x00000004 /* 29: Private memory enable */
#define	  L3CR_PMSIZ		  0x00000004 /* 31: Private memory size (0=1MB, 1=2MB) */

#define THRM1	1020
#define THRM2	1021
#define THRM3	1022
#define THRM1_TIN (1<<(31-0))
#define THRM1_TIV (1<<(31-1))
#define THRM1_THRES (0x7f<<(31-8))
#define THRM1_TID (1<<(31-29))
#define THRM1_TIE (1<<(31-30))
#define THRM1_V   (1<<(31-31))
#define THRM3_SITV (0x1fff << (31-30))
#define THRM3_E   (1<<(31-31))

/* Segment Registers */
#define PPC_SR0	0
#define PPC_SR1	1
#define PPC_SR2	2
#define PPC_SR3	3
#define PPC_SR4	4
#define PPC_SR5	5
#define PPC_SR6	6
#define PPC_SR7	7
#define PPC_SR8	8
#define PPC_SR9	9
#define PPC_SR10	10
#define PPC_SR11	11
#define PPC_SR12	12
#define PPC_SR13	13
#define PPC_SR14	14
#define PPC_SR15	15

#define BOOKE_DECAR	54

#define PPC405_MCSR     0x23C
#define PPC405_ESR      0x3D4
#define PPC405_DEAR     0x3D5
#define BOOKE_DEAR      61

#define PPC405_TSR	0x3D8
#define BOOKE_TSR	336
#define BOOKE_TSR_ENW		(1<<31)
#define BOOKE_TSR_WIS		(1<<30)
#define BOOKE_TSR_DIS		(1<<27)
#define BOOKE_TSR_FIS		(1<<26)

#define PPC405_TCR	0x3DA
#define BOOKE_TCR	340
#define BOOKE_TCR_WP(x)		(((x)&3)<<30)
#define BOOKE_TCR_WP_MASK	(3<<30)
#define BOOKE_TCR_WRC(x)	(((x)&3)<<28)
#define BOOKE_TCR_WRC_MASK	(3<<28)
#define BOOKE_TCR_WIE		(1<<27)
#define BOOKE_TCR_DIE		(1<<26)
#define BOOKE_TCR_FP(x)		(((x)&3)<<24)
#define BOOKE_TCR_FIE		(1<<23)
#define BOOKE_TCR_ARE		(1<<22)
#define BOOKE_TCR_WPEXT(x)	(((x)&0xf)<<17)
#define BOOKE_TCR_WPEXT_MASK	(0xf<<17)
#define BOOKE_TCR_FPEXT(x)	(((x)&0xf)<<13)
#define BOOKE_TCR_FPEXT_MASK	(0xf<<13)

#define BOOKE_PID       48   /* Process ID                                */
#define BOOKE_CSRR0     58   /* Critical Save/Restore Register 0          */
#define BOOKE_CSRR1     59   /* Critical Save/Restore Register 1          */
#define BOOKE_ESR       62   /* Exception Syndrome Register               */
#define BOOKE_IVPR      63   /* Interrupt Vector Prefix Register          */
#define BOOKE_SPRG4_W  260   /* Special Purpose Register General 4 (WO)   */
#define BOOKE_SPRG5_W  261   /* Special Purpose Register General 5 (WO)   */
#define BOOKE_SPRG6_W  262   /* Special Purpose Register General 6 (WO)   */
#define BOOKE_SPRG7_W  263   /* Special Purpose Register General 7 (WO)   */
#define BOOKE_PIR      286   /* Processor ID Register                     */
#define BOOKE_DBSR     304   /* Debug Status Register                     */

#define BOOKE_EPCR     307   /* Embedded Processor Control Register       */
#define BOOKE_EPCR_EXTGS  (1 << 31)
#define BOOKE_EPCR_DTLBGS (1 << 30)
#define BOOKE_EPCR_ITLBGS (1 << 29)
#define BOOKE_EPCR_DSIGS  (1 << 28)
#define BOOKE_EPCR_ISIGS  (1 << 27)
#define BOOKE_EPCR_DUVD   (1 << 26)
#define BOOKE_EPCR_ICM    (1 << 25)
#define BOOKE_EPCR_GICM   (1 << 24)
#define BOOKE_EPCR_DGTMI  (1 << 23)
#define BOOKE_EPCR_DMIUH  (1 << 22)
#define BOOKE_EPCR_PMGS   (1 << 21)

#define BOOKE_DBCR0    308   /* Debug Control Register 0                  */
#define BOOKE_DBCR1    309   /* Debug Control Register 1                  */
#define BOOKE_DBCR2    310   /* Debug Control Register 2                  */
#define BOOKE_IAC1     312   /* Instruction Address Compare 1             */
#define BOOKE_IAC2     313   /* Instruction Address Compare 2             */
#define BOOKE_IAC3     314   /* Instruction Address Compare 3             */
#define BOOKE_IAC4     315   /* Instruction Address Compare 4             */
#define BOOKE_DAC1     316   /* Data Address Compare 1                    */
#define BOOKE_DAC2     317   /* Data Address Compare 2                    */
#define BOOKE_DVC1     318   /* Data Value Compare 1                      */
#define BOOKE_DVC2     319   /* Data Value Compare 2                      */
#define BOOKE_GSRR0    378   /* Guest Save/Restore Register 0             */
#define BOOKE_GSRR1    379   /* Guest Save/Restore Register 1             */
#define BOOKE_GEPR     380   /* Guest External Proxy Register             */
#define BOOKE_GDEAR    381   /* Guest Data Exception Address Register     */
#define BOOKE_GPIR     382   /* Guest Processor ID Register               */
#define BOOKE_GESR     383   /* Guest Exception Syndrome Register         */
#define BOOKE_IVOR0    400   /* Interrupt Vector Offset Register 0        */
#define BOOKE_IVOR1    401   /* Interrupt Vector Offset Register 1        */
#define BOOKE_IVOR2    402   /* Interrupt Vector Offset Register 2        */
#define BOOKE_IVOR3    403   /* Interrupt Vector Offset Register 3        */
#define BOOKE_IVOR4    404   /* Interrupt Vector Offset Register 4        */
#define BOOKE_IVOR5    405   /* Interrupt Vector Offset Register 5        */
#define BOOKE_IVOR6    406   /* Interrupt Vector Offset Register 6        */
#define BOOKE_IVOR7    407   /* Interrupt Vector Offset Register 7        */
#define BOOKE_IVOR8    408   /* Interrupt Vector Offset Register 8        */
#define BOOKE_IVOR9    409   /* Interrupt Vector Offset Register 9        */
#define BOOKE_IVOR10   410   /* Interrupt Vector Offset Register 10       */
#define BOOKE_IVOR11   411   /* Interrupt Vector Offset Register 11       */
#define BOOKE_IVOR12   412   /* Interrupt Vector Offset Register 12       */
#define BOOKE_IVOR13   413   /* Interrupt Vector Offset Register 13       */
#define BOOKE_IVOR14   414   /* Interrupt Vector Offset Register 14       */
#define BOOKE_IVOR15   415   /* Interrupt Vector Offset Register 15       */
#define BOOKE_IVOR42   436   /* Interrupt Vector Offset Register 42       */
#define BOOKE_IVOR32   528   /* Interrupt Vector Offset Register 32       */
#define BOOKE_IVOR33   529   /* Interrupt Vector Offset Register 33       */
#define BOOKE_IVOR34   530   /* Interrupt Vector Offset Register 34       */
#define BOOKE_IVOR35   531   /* Interrupt Vector Offset Register 35       */
#define BOOKE_IVOR36   532   /* Interrupt Vector Offset Register 36       */
#define BOOKE_IVOR37   533   /* Interrupt Vector Offset Register 37       */
#define BOOKE_IVOR38   432   /* Interrupt Vector Offset Register 38       */
#define BOOKE_IVOR39   433   /* Interrupt Vector Offset Register 39       */
#define BOOKE_IVOR40   434   /* Interrupt Vector Offset Register 40       */
#define BOOKE_IVOR41   435   /* Interrupt Vector Offset Register 41       */
#define BOOKE_GIVOR2   440   /* Guest Interrupt Vector Offset Register 2  */
#define BOOKE_GIVOR3   441   /* Guest Interrupt Vector Offset Register 3  */
#define BOOKE_GIVOR4   442   /* Guest Interrupt Vector Offset Register 4  */
#define BOOKE_GIVOR8   443   /* Guest Interrupt Vector Offset Register 8  */
#define BOOKE_GIVOR13  444   /* Guest Interrupt Vector Offset Register 13 */
#define BOOKE_GIVOR14  445   /* Guest Interrupt Vector Offset Register 14 */
#define BOOKE_GIVPR    446   /* Guest Interrupt Vector Prefix Register    */
#define BOOKE_MCSRR0   570   /* Machine Check Save/Restore Register 0     */
#define BOOKE_MCSRR1   571   /* Machine Check Save/Restore Register 1     */
#define BOOKE_MCSR     572   /* Machine Check Status Register             */
#define BOOKE_DSRR0    574   /* Debug Save/Restore Register 0             */
#define BOOKE_DSRR1    575   /* Debug Save/Restore Register 1             */

#define PPC440_INV0    880   /* Instruction Cache Normal Victim 0         */
#define PPC440_INV1    881   /* Instruction Cache Normal Victim 1         */
#define PPC440_INV2    882   /* Instruction Cache Normal Victim 2         */
#define PPC440_INV3    883   /* Instruction Cache Normal Victim 3         */
#define PPC440_ITV0    884   /* Instruction Cache Transient Victim 0      */
#define PPC440_ITV1    885   /* Instruction Cache Transient Victim 1      */
#define PPC440_ITV2    886   /* Instruction Cache Transient Victim 2      */
#define PPC440_ITV3    887   /* Instruction Cache Transient Victim 3      */
#define PPC440_CCR1    888   /* Core Configuration Register 1             */
#define PPC440_DNV0    912   /* Data Cache Normal Victim 0                */
#define PPC440_DNV1    913   /* Data Cache Normal Victim 1                */
#define PPC440_DNV2    914   /* Data Cache Normal Victim 2                */
#define PPC440_DNV3    915   /* Data Cache Normal Victim 3                */
#define PPC440_DTV0    916   /* Data Cache Transient Victim 0             */
#define PPC440_DTV1    917   /* Data Cache Transient Victim 1             */
#define PPC440_DTV2    918   /* Data Cache Transient Victim 2             */
#define PPC440_DTV3    919   /* Data Cache Transient Victim 3             */
#define PPC440_DVLIM   920   /* Data Cache Victim Limit                   */
#define PPC440_IVLIM   921   /* Instruction Cache Victim Limit            */
#define PPC440_RSTCFG  923   /* Reset Configuration                       */
#define PPC440_DCDBTRL 924   /* Data Cache Debug Tag Register Low         */
#define PPC440_DCDBTRH 925   /* Data Cache Debug Tag Register High        */
#define PPC440_ICDBTRL 926   /* Instruction Cache Debug Tag Register Low  */
#define PPC440_ICDBTRH 927   /* Instruction Cache Debug Tag Register High */
#define PPC440_MMUCR   946   /* Memory Management Unit Control Register   */
#define PPC440_CCR0    947   /* Core Configuration Register 0             */
#define PPC440_ICDBDR  979   /* Instruction Cache Debug Data Register     */
#define PPC440_DBDR   1011   /* Debug Data Register                       */

#define PPC440_TLB0_EPN(n)       ( (((1<<22)-1)&(n)) << (31-21))  /* Etended Page Number    */
#define PPC440_TLB0_EPN_GET(n)   (             ((n)  >> (31-21)) & ((1<<22)-1))
#define PPC440_TLB0_V            (               1   << (31-22))  /* Entry valid            */
#define PPC440_TLB0_TS           (               1   << (31-23))  /* Translation space      */
#define PPC440_TLB0_TSIZE(n)     (       (0xf & (n)) << (31-27))  /* Page size              */
#define PPC440_TLB0_TSIZE_GET(n) (           ((n)  >> (31-27)) & 0xf)
#define PPC440_TLB0_TPAR(n)      (       (0xf & (n)) << (31-31))  /* Tag Parity             */
#define PPC440_TLB0_TPAR_GET(n)  (            ((n)  >> (31-31)) & 0xf)

#define PPC440_PID_TID(n)        (      (0xff & (n)) << (31-31))  /* Translation ID         */
#define PPC440_PID_TID_GET(n)    (             ((n)  >> (31-31)) & 0xff)

#define PPC440_TLB1_RPN(n)       ( (((1<<22)-1)&(n)) << (31-21))  /* Real Page Number       */
#define PPC440_TLB1_RPN_GET(n)   (             ((n)  >> (31-21)) & ((1<<22)-1))
#define PPC440_TLB1_PAR1(n)      (       (0x3 & (n)) << (31-23))  /* Parity for TLB word 1  */
#define PPC440_TLB1_PAR1_GET(n)  (            ((n)  >> (31-23)) & 0x3)
#define PPC440_TLB1_ERPN(n)      (       (0xf & (n)) << (31-31))  /* Extended Real Page No. */
#define PPC440_TLB1_ERPN_GET(n)  (            ((n)  >> (31-31)) & 0xf)

#define PPC440_TLB2_PAR2(n)      (       (0x3 & (n)) << (31- 1))  /* Parity for TLB word 2  */
#define PPC440_TLB2_PAR2_GET(n)  (            ((n)  >> (31- 1)) & 0x3)
#define PPC440_TLB2_U0           (               1   << (31-16))  /* User attr. 0           */
#define PPC440_TLB2_U1           (               1   << (31-17))  /* User attr. 1           */
#define PPC440_TLB2_U2           (               1   << (31-18))  /* User attr. 2           */
#define PPC440_TLB2_U3           (               1   << (31-19))  /* User attr. 3           */
#define PPC440_TLB2_W            (               1   << (31-20))  /* Write-through          */
#define PPC440_TLB2_I            (               1   << (31-21))  /* Cache-inhibited        */
#define PPC440_TLB2_M            (               1   << (31-22))  /* Memory-coherence req.  */
#define PPC440_TLB2_G            (               1   << (31-23))  /* Guarded                */
#define PPC440_TLB2_E            (               1   << (31-24))  /* Little-endian          */
#define PPC440_TLB2_UX           (               1   << (31-26))  /* User  exec.            */
#define PPC440_TLB2_UW           (               1   << (31-27))  /* User  write            */
#define PPC440_TLB2_UR           (               1   << (31-28))  /* User  read             */
#define PPC440_TLB2_SX           (               1   << (31-29))  /* Super exec.            */
#define PPC440_TLB2_SW           (               1   << (31-30))  /* Super write            */
#define PPC440_TLB2_SR           (               1   << (31-31))  /* Super read             */

#define PPC440_TLB2_ATTR(x)      ( ((x) & 0x1ff) << 7 )
#define PPC440_TLB2_ATTR_GET(x)  ( ((x) >> 7) & 0x1ff )

#define PPC440_TLB2_PERM(n)      ( (n) & 0x3f )
#define PPC440_TLB2_PERM_GET(n)  ( (n) & 0x3f )

/* Freescale Book E Implementation Standards (EIS): Branch Operations */

#define FSL_EIS_BUCSR 1013
#define FSL_EIS_BUCSR_STAC_EN (1 << (63 - 39))
#define FSL_EIS_BUCSR_LS_EN (1 << (63 - 41))
#define FSL_EIS_BUCSR_BBFI (1 << (63 - 54))
#define FSL_EIS_BUCSR_BALLOC_ALL (0x0 << (63 - 59))
#define FSL_EIS_BUCSR_BALLOC_FORWARD (0x1 << (63 - 59))
#define FSL_EIS_BUCSR_BALLOC_BACKWARD (0x2 << (63 - 59))
#define FSL_EIS_BUCSR_BALLOC_NONE (0x3 << (63 - 59))
#define FSL_EIS_BUCSR_BPRED_TAKEN (0x0 << (63 - 61))
#define FSL_EIS_BUCSR_BPRED_TAKEN_ONLY_FORWARD (0x1 << (63 - 62))
#define FSL_EIS_BUCSR_BPRED_TAKEN_ONLY_BACKWARD (0x2 << (63 - 62))
#define FSL_EIS_BUCSR_BPRED_NOT_TAKEN (0x3 << (63 - 62))
#define FSL_EIS_BUCSR_BPEN (1 << (63 - 63))

/* Freescale Book E Implementation Standards (EIS): Hardware Implementation-Dependent Registers */

#define FSL_EIS_SVR 1023

/* Freescale Book E Implementation Standards (EIS): Thread Management and Control Registers */

#define FSL_EIS_TENSR 437
#define FSL_EIS_TENS 438
#define FSL_EIS_TENC 439
#define FSL_EIS_PPR32 898

/* Freescale Book E Implementation Standards (EIS): MMU Control and Status */

#define FSL_EIS_MAS0 624
#define FSL_EIS_MAS0_TLBSEL (1 << (63 - 35))
#define FSL_EIS_MAS0_ESEL(n) ((0xfff & (n)) << (63 - 47))
#define FSL_EIS_MAS0_ESEL_GET(m) (((m) >> (63 - 47)) & 0xfff)
#define FSL_EIS_MAS0_NV (1 << (63 - 63))

#define FSL_EIS_MAS1 625
#define FSL_EIS_MAS1_V (1 << (63 - 32))
#define FSL_EIS_MAS1_IPROT (1 << (63 - 33))
#define FSL_EIS_MAS1_TID(n) ((0xff & (n)) << (63 - 47))
#define FSL_EIS_MAS1_TID_GET(n) (((n) >> (63 - 47)) & 0xfff)
#define FSL_EIS_MAS1_TS (1 << (63 - 51))
#define FSL_EIS_MAS1_TSIZE(n) ((0xf & (n)) << (63 - 55))
#define FSL_EIS_MAS1_TSIZE_GET(n) (((n)>>(63 - 55)) & 0xf)

#define FSL_EIS_MAS2 626
#define FSL_EIS_MAS2_EPN(n) ((((1 << 21) - 1)&(n)) << (63-51))
#define FSL_EIS_MAS2_EPN_GET(n) (((n) >> (63 - 51)) & 0xfffff)
#define FSL_EIS_MAS2_EA(n) FSL_EIS_MAS2_EPN((n) >> 12)
#define FSL_EIS_MAS2_EA_GET(n) (FSL_EIS_MAS2_EPN_GET(n) << 12)
#define FSL_EIS_MAS2_X0 (1 << (63 - 57))
#define FSL_EIS_MAS2_X1 (1 << (63 - 58))
#define FSL_EIS_MAS2_W (1 << (63 - 59))
#define FSL_EIS_MAS2_I (1 << (63 - 60))
#define FSL_EIS_MAS2_M (1 << (63 - 61))
#define FSL_EIS_MAS2_G (1 << (63 - 62))
#define FSL_EIS_MAS2_E (1 << (63 - 63))
#define FSL_EIS_MAS2_ATTR(x) ((x) & 0x7f)
#define FSL_EIS_MAS2_ATTR_GET(x) ((x) & 0x7f)

#define FSL_EIS_MAS3 627
#define FSL_EIS_MAS3_RPN(n) ((((1 << 21) - 1) & (n)) << (63-51))
#define FSL_EIS_MAS3_RPN_GET(n) (((n)>>(63 - 51)) & 0xfffff)
#define FSL_EIS_MAS3_RA(n) FSL_EIS_MAS3_RPN((n) >> 12)
#define FSL_EIS_MAS3_RA_GET(n) (FSL_EIS_MAS3_RPN_GET(n) << 12)
#define FSL_EIS_MAS3_U0 (1 << (63 - 54))
#define FSL_EIS_MAS3_U1 (1 << (63 - 55))
#define FSL_EIS_MAS3_U2 (1 << (63 - 56))
#define FSL_EIS_MAS3_U3 (1 << (63 - 57))
#define FSL_EIS_MAS3_UX (1 << (63 - 58))
#define FSL_EIS_MAS3_SX (1 << (63 - 59))
#define FSL_EIS_MAS3_UW (1 << (63 - 60))
#define FSL_EIS_MAS3_SW (1 << (63 - 61))
#define FSL_EIS_MAS3_UR (1 << (63 - 62))
#define FSL_EIS_MAS3_SR (1 << (63 - 63))
#define FSL_EIS_MAS3_PERM(n) ((n) & 0x3ff)
#define FSL_EIS_MAS3_PERM_GET(n) ((n) & 0x3ff)

#define FSL_EIS_MAS4 628
#define FSL_EIS_MAS4_TLBSELD (1 << (63 - 35))
#define FSL_EIS_MAS4_TIDSELD(n) ((0x3 & (n)) << (63 - 47))
#define FSL_EIS_MAS4_TSIZED(n) ((0xf & (n)) << (63 - 55))
#define FSL_EIS_MAS4_X0D FSL_EIS_MAS2_X0
#define FSL_EIS_MAS4_X1D FSL_EIS_MAS2_X1
#define FSL_EIS_MAS4_WD FSL_EIS_MAS2_W
#define FSL_EIS_MAS4_ID FSL_EIS_MAS2_I
#define FSL_EIS_MAS4_MD FSL_EIS_MAS2_M
#define FSL_EIS_MAS4_GD FSL_EIS_MAS2_G
#define FSL_EIS_MAS4_ED FSL_EIS_MAS2_E

#define FSL_EIS_MAS5 629

#define FSL_EIS_MAS6 630
#define FSL_EIS_MAS6_SPID0(n) ((0xff & (n)) << (63 - 55))
#define FSL_EIS_MAS6_SAS (1 << (63 - 63))

#define FSL_EIS_MAS7 944

#define FSL_EIS_MAS8 341

#define FSL_EIS_MMUCFG 1015
#define FSL_EIS_MMUCSR0 1012
#define FSL_EIS_PID0 48
#define FSL_EIS_PID1 633
#define FSL_EIS_PID2 634
#define FSL_EIS_TLB0CFG 688
#define FSL_EIS_TLB1CFG 689

/* Freescale Book E Implementation Standards (EIS): L1 Cache */

#define FSL_EIS_L1CFG0 515
#define FSL_EIS_L1CFG1 516
#define FSL_EIS_L1CSR0 1010
#define FSL_EIS_L1CSR0_CFI (1 << (63 - 62))
#define FSL_EIS_L1CSR1 1011
#define FSL_EIS_L1CSR1_ICFI (1 << (63 - 62))

/* Freescale Book E Implementation Standards (EIS): L2 Cache */

#define FSL_EIS_L2CFG0 519
#define FSL_EIS_L2CSR0 1017
#define FSL_EIS_L2CSR0_L2FI (1 << (63 - 42))
#define FSL_EIS_L2CSR0_L2FL (1 << (63 - 52))
#define FSL_EIS_L2CSR1 1018

/* Freescale Book E Implementation Standards (EIS): Timer */

#define FSL_EIS_ATBL 526
#define FSL_EIS_ATBU 527

/* Freescale Book E Implementation Standards (EIS): Interrupt */

#define FSL_EIS_MCAR 573
#define FSL_EIS_DSRR0 574
#define FSL_EIS_DSRR1 575
#define FSL_EIS_EPR 702

/* Freescale Book E Implementation Standards (EIS): Signal Processing Engine (SPE) */

#define FSL_EIS_SPEFSCR 512

/* Freescale Book E Implementation Standards (EIS): Software-Use SPRs */

#define FSL_EIS_SPRG8 604
#define FSL_EIS_SPRG9 605

/* Freescale Book E Implementation Standards (EIS): Debug */

#define FSL_EIS_DBCR3 561
#define FSL_EIS_DBCR4 563
#define FSL_EIS_DBCR5 564
#define FSL_EIS_DBCR6 603
#define FSL_EIS_DBCNT 562

/**
 * @brief Default value for the interrupt disable mask.
 *
 * The interrupt disable mask is stored in the global symbol
 * _PPC_INTERRUPT_DISABLE_MASK.
 */
#define PPC_INTERRUPT_DISABLE_MASK_DEFAULT MSR_EE

#ifndef ASM

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define _CPU_MSR_GET( _msr_value ) \
  do { \
    _msr_value = 0; \
    __asm__ volatile ("mfmsr %0" : "=&r" ((_msr_value)) : "0" ((_msr_value))); \
  } while (0)

#define _CPU_MSR_SET( _msr_value ) \
{ __asm__ volatile ("mtmsr %0" : "=&r" ((_msr_value)) : "0" ((_msr_value))); }

/**
 * @brief A global symbol used to disable interrupts in the MSR.
 *
 * A one bit means that this bit should be cleared.
 */
#if !defined(PPC_DISABLE_INLINE_ISR_DISABLE_ENABLE)
extern char _PPC_INTERRUPT_DISABLE_MASK[];

static inline uint32_t ppc_interrupt_get_disable_mask( void )
{
  return (uint32_t) (uintptr_t) _PPC_INTERRUPT_DISABLE_MASK;
}

static inline uint32_t ppc_interrupt_disable( void )
{
  uint32_t level;

#if defined(__PPC_CPU_E6500__)
  __asm__ volatile (
    "mfmsr %0;"
    "wrteei 0"
    : "=r" (level)
  );
#else
  uint32_t mask;

  __asm__ volatile (
    "mfmsr %0;"
    "lis %1, _PPC_INTERRUPT_DISABLE_MASK@h;"
    "ori %1, %1, _PPC_INTERRUPT_DISABLE_MASK@l;"
    "andc %1, %0, %1;"
    "mtmsr %1"
    : "=r" (level), "=r" (mask)
  );
#endif

  return level;
}

static inline void ppc_interrupt_enable( uint32_t level )
{
#if defined(__PPC_CPU_E6500__)
  __asm__ volatile (
    "wrtee %0"
    :
    : "r" (level)
  );
#else
  __asm__ volatile (
    "mtmsr %0"
    :
    : "r" (level)
  );
#endif
}

static inline void ppc_interrupt_flash( uint32_t level )
{
  uint32_t current_level;

  __asm__ volatile (
    "mfmsr %0;"
    "mtmsr %1;"
    "mtmsr %0"
    : "=&r" (current_level)
    : "r" (level)
  );
}
#else
uint32_t ppc_interrupt_get_disable_mask( void );
uint32_t ppc_interrupt_disable( void );
void ppc_interrupt_enable( uint32_t level );
void ppc_interrupt_flash( uint32_t level );
#endif /* PPC_DISABLE_INLINE_ISR_DISABLE_ENABLE */

#define _CPU_ISR_Disable( _isr_cookie ) \
  do { \
    _isr_cookie = ppc_interrupt_disable(); \
  } while (0)

/*
 *  Enable interrupts to the previous level (returned by _CPU_ISR_Disable).
 *  This indicates the end of an RTEMS critical section.  The parameter
 *  _isr_cookie is not modified.
 */

#define _CPU_ISR_Enable( _isr_cookie )  \
  ppc_interrupt_enable(_isr_cookie)

/*
 *  This temporarily restores the interrupt to _isr_cookie before immediately
 *  disabling them again.  This is used to divide long RTEMS critical
 *  sections into two or more parts.  The parameter _isr_cookie is not
 *  modified.
 *
 *  NOTE:  The version being used is not very optimized but it does
 *         not trip a problem in gcc where the disable mask does not
 *         get loaded.  Check this for future (post 10/97 gcc versions.
 */

#define _CPU_ISR_Flash( _isr_cookie ) \
  ppc_interrupt_flash(_isr_cookie)

/* end of ISR handler macros */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* ASM */

#endif /* _RTEMS_POWERPC_REGISTERS_H */
