#include "rtems-jffs2-config.h"

/*
 * JFFS2 -- Journalling Flash File System, Version 2.
 *
 * Copyright © 2001-2007 Red Hat, Inc.
 * Copyright © 2004-2010 David Woodhouse <dwmw2@infradead.org>
 *
 * Created by Arjan van de Ven <arjanv@redhat.com>
 *
 * For licensing information, see the file 'LICENCE' in this directory.
 *
 *
 *
 * Very simple lz77-ish encoder.
 *
 * Theory of operation: Both encoder and decoder have a list of "last
 * occurrences" for every possible source-value; after sending the
 * first source-byte, the second byte indicated the "run" length of
 * matches
 *
 * The algorithm is intended to only send "whole bytes", no bit-messing.
 *
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/jffs2.h>
#include "compr.h"

uint16_t rtems_jffs2_compressor_rtime_compress(
	rtems_jffs2_compressor_control *self,
	unsigned char *data_in,
	unsigned char *cpage_out,
	uint32_t *sourcelen,
	uint32_t *dstlen
)
{
	unsigned short positions[256];
	int outpos = 0;
	int pos=0;

	(void) self;

	if (*dstlen <= 3)
		return JFFS2_COMPR_NONE;

	memset(positions,0,sizeof(positions));

	while (pos < (*sourcelen) && outpos <= (*dstlen)-2) {
		int backpos, runlen=0;
		unsigned char value;

		value = data_in[pos];

		cpage_out[outpos++] = data_in[pos++];

		backpos = positions[value];
		positions[value]=pos;

		while ((backpos < pos) && (pos < (*sourcelen)) &&
		       (data_in[pos]==data_in[backpos++]) && (runlen<255)) {
			pos++;
			runlen++;
		}
		cpage_out[outpos++] = runlen;
	}

	if (outpos >= pos) {
		/* We failed */
		return JFFS2_COMPR_NONE;
	}

	/* Tell the caller how much we managed to compress, and how much space it took */
	*sourcelen = pos;
	*dstlen = outpos;
	return JFFS2_COMPR_RTIME;
}


int rtems_jffs2_compressor_rtime_decompress(
	rtems_jffs2_compressor_control *self,
	uint16_t comprtype,
	unsigned char *data_in,
	unsigned char *cpage_out,
	uint32_t srclen,
	uint32_t destlen
)
{
	unsigned short positions[256];
	int outpos = 0;
	int pos=0;

	(void) self;

	if (comprtype != JFFS2_COMPR_RTIME) {
		return -EIO;
	}

	memset(positions,0,sizeof(positions));

	while (outpos<destlen) {
		unsigned char value;
		int backoffs;
		int repeat;

		value = data_in[pos++];
		cpage_out[outpos++] = value; /* first the verbatim copied byte */
		repeat = data_in[pos++];
		backoffs = positions[value];

		positions[value]=outpos;
		if (repeat) {
#ifdef __rtems__
			if ((repeat + outpos) > destlen) {
				return 1;
			}
#endif
			if (backoffs + repeat >= outpos) {
				while(repeat) {
					cpage_out[outpos++] = cpage_out[backoffs++];
					repeat--;
				}
			} else {
				memcpy(&cpage_out[outpos],&cpage_out[backoffs],repeat);
				outpos+=repeat;
			}
		}
	}
	return 0;
}
