/**
 *  @file
 *
 *  @brief LM32 Initialize the ISR Handler
 */

/*
 *  COPYRIGHT (c) 1989-2009.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.org/license/LICENSE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <rtems/score/isr.h>
#include <rtems/score/percpu.h>
#include <rtems/score/threaddispatch.h>

unsigned long *_old_stack_ptr;

void *_exception_stack_frame;

register unsigned long  *stack_ptr __asm__ ("sp");

/*
 *  Prototypes for routines called from assembly that we don't want in
 *  the public name space.
 */
void __ISR_Handler(uint32_t vector, CPU_Interrupt_frame *ifr);

void __ISR_Handler(uint32_t vector, CPU_Interrupt_frame *ifr)
{
  register uint32_t   level;
  _exception_stack_frame = NULL;

  /* Interrupts are disabled upon entry to this Handler */

  _Thread_Dispatch_disable();

  if ( _ISR_Nest_level == 0 ) {
    /* Install irq stack */
    _old_stack_ptr = stack_ptr;
    stack_ptr = _CPU_Interrupt_stack_high - 4;
  }

  _ISR_Nest_level++;

  if ( _ISR_Vector_table[ vector] )
  {
    (*_ISR_Vector_table[ vector ])(vector, ifr);
  };

  /* Make sure that interrupts are disabled again */
  _CPU_ISR_Disable( level );

  _ISR_Nest_level--;

  if( _ISR_Nest_level == 0)
    stack_ptr = _old_stack_ptr;

  _Thread_Dispatch_unnest( _Per_CPU_Get() );

  _CPU_ISR_Enable( level );

  if ( _ISR_Nest_level )
    return;

  if ( _Thread_Dispatch_necessary && _Thread_Dispatch_is_enabled() ) {
    /* save off our stack frame so the context switcher can get to it */
    _exception_stack_frame = ifr;

    _Thread_Dispatch();

    /* and make sure its clear in case we didn't dispatch. if we did, its
     * already cleared */
    _exception_stack_frame = NULL;
  }
}

