/**
 * @file
 * @ingroup stm32f4xxxx_rcc
 * @brief STM32F4XXXX RCC support.
 */

/*
 * Copyright (c) 2012 Sebastian Huber.  All rights reserved.
 *
 * The license and distribution terms for this file may be
 * found in the file LICENSE in this distribution or at
 * http://www.rtems.org/license/LICENSE.
 */

#ifndef LIBBSP_ARM_STM32F4_STM32F4XXXX_RCC_H
#define LIBBSP_ARM_STM32F4_STM32F4XXXX_RCC_H

#include <bsp/utility.h>

/**
 * @defgroup stm32f4xxxx_rcc STM32F4XXXX RCC Support
 * @ingroup stm32f4_rcc
 * @brief STM32F4XXXX RCC Support
 * @{
 */

typedef struct {
  uint32_t cr;
#define STM32F4_RCC_CR_PLLI2SRDY  BSP_BIT32(27) // PLLI2S clock ready flag
#define STM32F4_RCC_CR_PLLI2SON   BSP_BIT32(26) // PLLI2S enable
#define STM32F4_RCC_CR_PLLRDY     BSP_BIT32(25) // Main PLL clock ready flag
#define STM32F4_RCC_CR_PLLON      BSP_BIT32(24) // Main PLL enable
#define STM32F4_RCC_CR_CSSON      BSP_BIT32(19) // Clock security system enable
#define STM32F4_RCC_CR_HSEBYP     BSP_BIT32(18) // HSE clock bypass
#define STM32F4_RCC_CR_HSERDY     BSP_BIT32(17) // HSE clock ready flag
#define STM32F4_RCC_CR_HSEON      BSP_BIT32(16) // HSE clock enable
#define STM32F4_RCC_CR_HSIRDY     BSP_BIT32(1)  // HSI clock ready flag
#define STM32F4_RCC_CR_HSION      BSP_BIT32(0)  // HSI clock enable

  uint32_t pllcfgr;
#define STM32F4_RCC_PLLCFGR_PLLQ(val) BSP_FLD32(val, 24, 27)
#define STM32F4_RCC_PLLCFGR_PLLQ_GET(reg) BSP_FLD32GET(reg, 24, 27)
#define STM32F4_RCC_PLLCFGR_PLLQ_SET(reg, val)  BSP_FLD32SET(reg, val, 24, 27)
#define STM32F4_RCC_PLLCFGR_SRC   BSP_BIT32(22) // PLL entry clock source
#define STM32F4_RCC_PLLCFGR_SRC_HSE STM32F4_RCC_PLLCFGR_SRC
#define STM32F4_RCC_PLLCFGR_SRC_HSI 0
#define STM32F4_RCC_PLLCFGR_PLLP(val) BSP_FLD32(val, 16, 17)
#define STM32F4_RCC_PLLCFGR_PLLP_GET(reg) BSP_FLD32GET(reg, 16, 17)
#define STM32F4_RCC_PLLCFGR_PLLP_SET(reg, val)  BSP_FLD32SET(reg, val, 16, 17)
#define STM32F4_RCC_PLLCFGR_PLLP_2 STM32F4_RCC_PLLCFGR_PLLP(0)
#define STM32F4_RCC_PLLCFGR_PLLP_4 STM32F4_RCC_PLLCFGR_PLLP(1)
#define STM32F4_RCC_PLLCFGR_PLLP_6 STM32F4_RCC_PLLCFGR_PLLP(2)
#define STM32F4_RCC_PLLCFGR_PLLP_8 STM32F4_RCC_PLLCFGR_PLLP(3)
#define STM32F4_RCC_PLLCFGR_PLLN(val) BSP_FLD32(val, 6, 14)
#define STM32F4_RCC_PLLCFGR_PLLN_GET(reg) BSP_FLD32GET(reg, 6, 14)
#define STM32F4_RCC_PLLCFGR_PLLN_SET(reg, val)  BSP_FLD32SET(reg, val, 6, 14)
#define STM32F4_RCC_PLLCFGR_PLLM(val) BSP_FLD32(val, 0, 5)
#define STM32F4_RCC_PLLCFGR_PLLM_GET(reg) BSP_FLD32GET(reg, 0, 5)
#define STM32F4_RCC_PLLCFGR_PLLM_SET(reg, val)  BSP_FLD32SET(reg, val, 0, 5)

  uint32_t cfgr;
#define STM32F4_RCC_CFGR_MCO2(val)  BSP_FLD32(val, 30, 31)  // Microcontroller clock output 2
#define STM32F4_RCC_CFGR_MCO2_GET(reg)  BSP_FLD32GET(reg, 30, 31)
#define STM32F4_RCC_CFGR_MCO2_SET(reg, val) BSP_FLD32SET(reg, val, 30, 31)
#define STM32F4_RCC_CFGR_MCO2_SYSCLK  STM32F4_RCC_CFGR_MCO2(0)
#define STM32F4_RCC_CFGR_MCO2_PLLI2S  STM32F4_RCC_CFGR_MCO2(1)
#define STM32F4_RCC_CFGR_MCO2_HSE     STM32F4_RCC_CFGR_MCO2(2)
#define STM32F4_RCC_CFGR_MCO2_PLL     STM32F4_RCC_CFGR_MCO2(3)
#define STM32F4_RCC_CFGR_MCO2_PRE(val)  BSP_FLD32(val, 27, 29)  // MCO2 prescalar
#define STM32F4_RCC_CFGR_MCO2_PRE_GET(reg)  BSP_FLD32GET(reg, 27, 29)
#define STM32F4_RCC_CFGR_MCO2_PRE_SET(reg, val) BSP_FLD32SET(reg, val, 27, 29)
#define STM32F4_RCC_CFGR_MCO2_DIV1    STM32F4_RCC_CFGR_MCO2_PRE(0)
#define STM32F4_RCC_CFGR_MCO2_DIV2    STM32F4_RCC_CFGR_MCO2_PRE(4)
#define STM32F4_RCC_CFGR_MCO2_DIV3    STM32F4_RCC_CFGR_MCO2_PRE(5)
#define STM32F4_RCC_CFGR_MCO2_DIV4    STM32F4_RCC_CFGR_MCO2_PRE(6)
#define STM32F4_RCC_CFGR_MCO2_DIV5    STM32F4_RCC_CFGR_MCO2_PRE(7)
#define STM32F4_RCC_CFGR_MCO1_PRE(val)  BSP_FLD32(val, 24, 26)  // MCO1 prescalar
#define STM32F4_RCC_CFGR_MCO1_PRE_GET(reg)  BSP_FLD32GET(reg, 24, 26)
#define STM32F4_RCC_CFGR_MCO1_PRE_SET(reg, val) BSP_FLD32SET(reg, val, 24, 26)
#define STM32F4_RCC_CFGR_MCO1_DIV1    STM32F4_RCC_CFGR_MCO1_PRE(0)
#define STM32F4_RCC_CFGR_MCO1_DIV2    STM32F4_RCC_CFGR_MCO1_PRE(4)
#define STM32F4_RCC_CFGR_MCO1_DIV3    STM32F4_RCC_CFGR_MCO1_PRE(5)
#define STM32F4_RCC_CFGR_MCO1_DIV4    STM32F4_RCC_CFGR_MCO1_PRE(6)
#define STM32F4_RCC_CFGR_MCO1_DIV5    STM32F4_RCC_CFGR_MCO1_PRE(7)
#define STM32F4_RCC_CFGR_I2SSCR     BSP_BIT32(23) // I2S clock selection
#define STM32F4_RCC_CFGR_MCO1(val)  BSP_FLD32(val, 21, 22)  // Microcontroller clock output 1
#define STM32F4_RCC_CFGR_MCO1_GET(reg)  BSP_FLD32GET(reg, 21, 22)
#define STM32F4_RCC_CFGR_MCO1_SET(reg, val) BSP_FLD32SET(reg, val, 21, 22)
#define STM32F4_RCC_CFGR_MCO1_HSI     STM32F4_RCC_CFGR_MCO1(0)
#define STM32F4_RCC_CFGR_MCO1_LSE     STM32F4_RCC_CFGR_MCO1(1)
#define STM32F4_RCC_CFGR_MCO1_HSE     STM32F4_RCC_CFGR_MCO1(2)
#define STM32F4_RCC_CFGR_MCO1_PLL     STM32F4_RCC_CFGR_MCO1(3)
#define STM32F4_RCC_CFGR_RTCPRE(val)  BSP_FLD32(val, 16, 20)  // HSE division factor for RTC clock
#define STM32F4_RCC_CFGR_RTCPRE_GET(reg)  BSP_FLD32GET(reg, 16, 20)
#define STM32F4_RCC_CFGR_RTCPRE_SET(reg, val) BSP_FLD32SET(reg, val, 16, 20)
#define STM32F4_RCC_CFGR_PPRE2(val) BSP_FLD32(val, 13, 15)  // APB high-speed prescalar (APB2)
#define STM32F4_RCC_CFGR_PPRE2_GET(reg) BSP_FLD32GET(reg, 13, 15)
#define STM32F4_RCC_CFGR_PPRE2_SET(reg, val)  BSP_FLD32SET(reg, val, 13, 15)
#define STM32F4_RCC_CFGR_PPRE2_DIV1   STM32F4_RCC_CFGR_PPRE2(0)
#define STM32F4_RCC_CFGR_PPRE2_DIV2   STM32F4_RCC_CFGR_PPRE2(4)
#define STM32F4_RCC_CFGR_PPRE2_DIV4   STM32F4_RCC_CFGR_PPRE2(5)
#define STM32F4_RCC_CFGR_PPRE2_DIV8   STM32F4_RCC_CFGR_PPRE2(6)
#define STM32F4_RCC_CFGR_PPRE2_DIV16  STM32F4_RCC_CFGR_PPRE2(7)
#define STM32F4_RCC_CFGR_PPRE1(val) BSP_FLD32(val, 10, 12)  // APB low-speed prescalar (APB1)
#define STM32F4_RCC_CFGR_PPRE1_GET(reg) BSP_FLD32GET(reg, 10, 12)
#define STM32F4_RCC_CFGR_PPRE1_SET(reg, val)  BSP_FLD32SET(reg, val, 10, 12)
#define STM32F4_RCC_CFGR_PPRE1_DIV1   STM32F4_RCC_CFGR_PPRE1(0)
#define STM32F4_RCC_CFGR_PPRE1_DIV2   STM32F4_RCC_CFGR_PPRE1(4)
#define STM32F4_RCC_CFGR_PPRE1_DIV4   STM32F4_RCC_CFGR_PPRE1(5)
#define STM32F4_RCC_CFGR_PPRE1_DIV8   STM32F4_RCC_CFGR_PPRE1(6)
#define STM32F4_RCC_CFGR_PPRE1_DIV16  STM32F4_RCC_CFGR_PPRE1(7)
#define STM32F4_RCC_CFGR_HPRE(val)  BSP_FLD32(val, 4, 15) // AHB prescalar
#define STM32F4_RCC_CFGR_HPRE_GET(reg)  BSP_FLD32GET(reg, 4, 7)
#define STM32F4_RCC_CFGR_HPRE_SET(reg, val) BSP_FLD32SET(reg, val, 4, 7)
#define STM32F4_RCC_CFGR_HPRE_DIV1    STM32F4_RCC_CFGR_HPRE(0)
#define STM32F4_RCC_CFGR_HPRE_DIV2    STM32F4_RCC_CFGR_HPRE(8)
#define STM32F4_RCC_CFGR_HPRE_DIV4    STM32F4_RCC_CFGR_HPRE(9)
#define STM32F4_RCC_CFGR_HPRE_DIV8    STM32F4_RCC_CFGR_HPRE(10)
#define STM32F4_RCC_CFGR_HPRE_DIV16   STM32F4_RCC_CFGR_HPRE(11)
#define STM32F4_RCC_CFGR_HPRE_DIV64   STM32F4_RCC_CFGR_HPRE(12)
#define STM32F4_RCC_CFGR_HPRE_DIV128  STM32F4_RCC_CFGR_HPRE(13)
#define STM32F4_RCC_CFGR_HPRE_DIV256  STM32F4_RCC_CFGR_HPRE(14)
#define STM32F4_RCC_CFGR_HPRE_DIV512  STM32F4_RCC_CFGR_HPRE(15)
#define STM32F4_RCC_CFGR_SWS(val) BSP_FLD32(val, 2, 3)  // System clock switch status
#define STM32F4_RCC_CFGR_SWS_GET(reg) BSP_FLD32GET(reg, 2, 3)
#define STM32F4_RCC_CFGR_SWS_SET(reg, val)  BSP_FLD32SET(reg, val, 2, 3)
#define STM32F4_RCC_CFGR_SWS_HSI  STM32F4_RCC_CFGR_SWS(0)
#define STM32F4_RCC_CFGR_SWS_HSE  STM32F4_RCC_CFGR_SWS(1)
#define STM32F4_RCC_CFGR_SWS_PLL  STM32F4_RCC_CFGR_SWS(2)
#define STM32F4_RCC_CFGR_SW(val)  BSP_FLD32(val, 0, 1)  // System clock switch
#define STM32F4_RCC_CFGR_SW_GET(reg)  BSP_FLD32GET(reg, 0, 1)
#define STM32F4_RCC_CFGR_SW_SET(reg, val) BSP_FLD32SET(reg, val, 0, 1)
#define STM32F4_RCC_CFGR_SW_HSI STM32F4_RCC_CFGR_SW(0)
#define STM32F4_RCC_CFGR_SW_HSE STM32F4_RCC_CFGR_SW(1)
#define STM32F4_RCC_CFGR_SW_PLL STM32F4_RCC_CFGR_SW(2)

  uint32_t cir;

  uint32_t ahbrstr [3];

  uint32_t reserved_1c;

  uint32_t apbrstr [2];

  uint32_t reserved_28 [2];

  uint32_t ahbenr [3];

  uint32_t reserved_3c;

  uint32_t apbenr [2];

  uint32_t reserved_48 [2];

  uint32_t ahblpenr [3];

  uint32_t reserved_5c;

  uint32_t apblpenr [2];

  uint32_t reserved_68 [2];

  uint32_t bdcr;

  uint32_t csr;

  uint32_t reserved_78 [2];

  uint32_t sscgr;

  uint32_t plli2scfgr;

} stm32f4_rcc;

/** @} */

#define RCC_CR_HSION BSP_BIT32( 0 )
#define RCC_CR_HSIRDY BSP_BIT32( 1 )
#define RCC_CR_HSITRIM( val ) BSP_FLD32( val, 3, 7 )
#define RCC_CR_HSITRIM_MSK BSP_MSK32( 3, 7 )
#define RCC_CR_HSICAL( val ) BSP_FLD32( val, 8, 15 )
#define RCC_CR_HSICAL_MSK BSP_MSK32( 8, 15 )
#define RCC_CR_HSEON BSP_BIT32( 16 )
#define RCC_CR_HSERDY BSP_BIT32( 17 )
#define RCC_CR_HSEBYP BSP_BIT32( 18 )
#define RCC_CR_CSSON BSP_BIT32( 19 )
#define RCC_CR_PLLON BSP_BIT32( 24 )
#define RCC_CR_PLLRDY BSP_BIT32( 25 )
#define RCC_CR_PLLI2SON BSP_BIT32( 26 )
#define RCC_CR_PLLI2SRDY BSP_BIT32( 27 )

#define RCC_PLLCFGR_PLLM( val ) BSP_FLD32( val, 0, 5 )
#define RCC_PLLCFGR_PLLM_MSK BSP_MSK32( 0, 5 )
#define RCC_PLLCFGR_PLLN( val ) BSP_FLD32( val, 6, 14 )
#define RCC_PLLCFGR_PLLN_MSK BSP_MSK32( 6, 14 )

#define RCC_PLLCFGR_PLLP 16
#define RCC_PLLCFGR_PLLP_MSK BSP_MSK32( 16, 17 )
#define RCC_PLLCFGR_PLLP_BY_2 0
#define RCC_PLLCFGR_PLLP_BY_4 BSP_FLD32( 1, 16, 17 )
#define RCC_PLLCFGR_PLLP_BY_6 BSP_FLD32( 2, 16, 17 )
#define RCC_PLLCFGR_PLLP_BY_8 BSP_FLD32( 3, 16, 17 )

#define RCC_PLLCFGR_PLLSRC_HSE BSP_BIT32( 22 )
#define RCC_PLLCFGR_PLLSRC_HSI 0

#define RCC_PLLCFGR_PLLQ( val ) BSP_FLD32( val, 24, 27 )
#define RCC_PLLCFGR_PLLQ_MSK BSP_MSK32( 24, 27 )

#define RCC_CFGR_SW 0
#define RCC_CFGR_SW_MSK BSP_MSK32( 0, 1 )
#define RCC_CFGR_SW_HSI 0
#define RCC_CFGR_SW_HSE 1
#define RCC_CFGR_SW_PLL 2

#define RCC_CFGR_SWS 2
#define RCC_CFGR_SWS_MSK BSP_MSK32( 2, 3 )
#define RCC_CFGR_SWS_HSI 0
#define RCC_CFGR_SWS_HSE BSP_FLD32( 1, 2, 3 )
#define RCC_CFGR_SWS_PLL BSP_FLD32( 2, 2, 3 )

#define RCC_CFGR_HPRE 4
#define RCC_CFGR_HPRE_BY_1 0
#define RCC_CFGR_HPRE_BY_2 BSP_FLD32( 8, 4, 7 )
#define RCC_CFGR_HPRE_BY_4 BSP_FLD32( 9, 4, 7 )
#define RCC_CFGR_HPRE_BY_8 BSP_FLD32( 10, 4, 7 )
#define RCC_CFGR_HPRE_BY_16 BSP_FLD32( 11, 4, 7 )
#define RCC_CFGR_HPRE_BY_64 BSP_FLD32( 12, 4, 7 )
#define RCC_CFGR_HPRE_BY_128 BSP_FLD32( 13, 4, 7 )
#define RCC_CFGR_HPRE_BY_256 BSP_FLD32( 14, 4, 7 )
#define RCC_CFGR_HPRE_BY_512 BSP_FLD32( 15, 4, 7 )

#define RCC_CFGR_PPRE1 10
#define RCC_CFGR_PPRE1_BY_1 0
#define RCC_CFGR_PPRE1_BY_2 BSP_FLD32( 4, 10, 12 )
#define RCC_CFGR_PPRE1_BY_4 BSP_FLD32( 5, 10, 12 )
#define RCC_CFGR_PPRE1_BY_8 BSP_FLD32( 6, 10, 12 )
#define RCC_CFGR_PPRE1_BY_16 BSP_FLD32( 7, 10, 12 )

#define RCC_CFGR_PPRE2 13
#define RCC_CFGR_PPRE2 BSP_MSK32( 13, 15 )
#define RCC_CFGR_PPRE2_BY_1 0
#define RCC_CFGR_PPRE2_BY_2 BSP_FLD32( 4, 13, 15 )
#define RCC_CFGR_PPRE2_BY_4 BSP_FLD32( 5, 13, 15 )
#define RCC_CFGR_PPRE2_BY_8 BSP_FLD32( 6, 13, 15 )
#define RCC_CFGR_PPRE2_BY_16 BSP_FLD32( 7, 13, 15 )

#define RCC_CFGR_RTCPRE( val ) BSP_FLD32( val, 16, 20 )
#define RCC_CFGR_RTCPRE_MSK BSP_MSK32( 16, 20 )

#define RCC_CFGR_MCO1 21
#define RCC_CFGR_MCO1_MSK BSP_MSK32( 21, 22 )
#define RCC_CFGR_MCO1_HSI 0
#define RCC_CFGR_MCO1_LSE BSP_FLD32( 1, 21, 22 )
#define RCC_CFGR_MCO1_HSE BSP_FLD32( 2, 21, 22 )
#define RCC_CFGR_MCO1_PLL BSP_FLD32( 3, 21, 22 )

#define RCC_CFGR_I2SSRC BSP_BIT32( 23 )

#define RCC_CFGR_MCO1PRE 24
#define RCC_CFGR_MCO1PRE_MSK BSP_MSK32( 24, 26 )
#define RCC_CFGR_MCO1PRE_BY_1 0
#define RCC_CFGR_MCO1PRE_BY_2 BSP_FLD32( 4, 24, 26 )
#define RCC_CFGR_MCO1PRE_BY_3 BSP_FLD32( 5, 24, 26 )
#define RCC_CFGR_MCO1PRE_BY_4 BSP_FLD32( 6, 24, 26 )
#define RCC_CFGR_MCO1PRE_BY_5 BSP_FLD32( 7, 24, 26 )

#define RCC_CFGR_MCO2PRE 27
#define RCC_CFGR_MCO2PRE_MSK BSP_MSK32( 27, 29 )
#define RCC_CFGR_MCO2PRE_BY_1 0
#define RCC_CFGR_MCO2PRE_BY_2 BSP_FLD32( 4, 27, 29 )
#define RCC_CFGR_MCO2PRE_BY_3 BSP_FLD32( 5, 27, 29 )
#define RCC_CFGR_MCO2PRE_BY_4 BSP_FLD32( 6, 27, 29 )
#define RCC_CFGR_MCO2PRE_BY_5 BSP_FLD32( 7, 27, 29 )

#define RCC_CFGR_MCO2 30
#define RCC_CFGR_MCO2_MSK BSP_MSK32( 30, 31 )
#define RCC_CFGR_MCO2_SYSCLK 0
#define RCC_CFGR_MCO2_PLLI2S BSP_FLD32( 1, 30, 31 )
#define RCC_CFGR_MCO2_HSE BSP_FLD32( 2, 30, 31 )
#define RCC_CFGR_MCO2_PLL BSP_FLD32( 3, 30, 31 )

#endif /* LIBBSP_ARM_STM32F4_STM32F4XXXX_RCC_H */
