///////////////////////////////////////////////////////////////////////////
// $Header: /usr/local/cvs-repository/ssf/tests/ssf_test.cpp,v 1.1 2001/01/20 21:02:46 rosimildo Exp $
//
// Copyright (c) 2000-2001 ConnectTel, Inc. All Rights Reserved.
// by: Rosimildo da Silva
//  
// DESCRIPTION: Test driver for the SSF. It test a connection to the
// MySQL DB.
// 
// The SSF library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
// 02111-1307, USA
//
//  
// NOTE: This has been loosely based on the original work from:
//
// database.cc
// mysql C++ wrapper library
// 
// Author: Roland Haenel <rh@ginster.net>
// 
// This program is in the public domain.
// Distribute and use it freely.
//
// HISTORY:
//
// $Log: ssf_test.cpp,v $
// Revision 1.1  2001/01/20 21:02:46  rosimildo
// Added VC++ projects, and a simple GCC makefile to compile the
// system using CygWin. I do not have the client code for linux/cygwin
// to test the client at this point.
//
// Revision 1.2  2001/01/20 17:30:10  rosimildo
// Changes to add a test directory for the C++ code.
//
// Revision 1.1  2001/01/20 17:19:18  rosimildo
// Moved these files to a new test directory
//
///////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>

#include "ssf_factory.h"
#include "product.h"


const char *MYTABLE          = "cpptest";
const char *URL_MYSQL_SERVER = "mysql://localhost/mysql";

void test_driver_for_raw_interface( ssf::Connection *db )
{
   fprintf( stdout, "SSF: - Raw interface test starting...\n" ); 
   int n;
	n = db->executeUpdateFormat( "CREATE TABLE %s (tname char(100), "
	               "rand_nr int NOT NULL, INDEX(rand_nr))", MYTABLE );

   if( db->success() )
       fprintf( stdout, "SSF: create table = %d\n", n ); 


   fprintf( stdout, "SSF: inserting data\n" ); 
	// Insert random data
	for (int i = 0; i < 1000; i++) 
	{
		db->executeUpdateFormat( 
		   "INSERT INTO %s VALUES( 'User %d', %ld )", MYTABLE, i, rand() );
		if( !db->success() ) 
		{
			fprintf(stderr, "Can't insert into table '%s': %s", MYTABLE, db->errorDescription() );
         delete db;
			exit(1);
		}
	}
   fprintf( stdout, "SSF: insert is complete.\n" ); 

   n = db->executeUpdateFormat( 
		   "UPDATE %s SET  rand_nr = %d WHERE tname = 'User 999'", MYTABLE, 3 );
	if( !db->success() ) 
	{
		fprintf(stderr, "Can't insert into table '%s': %s", MYTABLE, db->errorDescription() );
   }
   fprintf( stdout, "SSF: update is complete = %d\n", n ); 

   // Read sorted data and print it out
	ssf::ResultSet  *res;
	res = db->executeQueryFormat( "SELECT * FROM %s ORDER BY rand_nr DESC", MYTABLE );
   if( res && db->success() )
   {
     res->reset();
     while ( res->next() ) 
     {
	    printf("tname   ==> %10s |  ", res->getString( "tname" ) );
	    printf("rand_nr ==> %10s |  ", res->getString( "rand_nr" ) );
       printf("\n");
     }
	  delete res;
   }
   else
   {
     fprintf(stderr, "Can't query table '%s': %s", MYTABLE, db->errorDescription() );
   }

	db->executeUpdateFormat( "DROP TABLE %s", MYTABLE );
   if( db->success() )
       fprintf( stdout, "SSF: drop table = %d\n", n ); 

	db->executeUpdateFormat( "DROP TABLE %s", "XXXX_NAME" );
   if( !db->success() )
   {
		fprintf( stderr, "ERROR: %s\n", db->errorDescription() );
   }
   fprintf( stdout, "SSF: raw interface test is complete.\n" ); 
}


void test_driver_for_storable( ssf::Connection *db )
{
	db->executeUpdateFormat( "DROP TABLE %s", Product::TABLE_NAME );
   if( db->success() )
   {
     fprintf( stdout, "SSF: dropped previous table\n" ); 
   }

	db->executeUpdate( 
	   "CREATE TABLE Product ( " \
	   "  ID       INTEGER DEFAULT 0 AUTO_INCREMENT PRIMARY KEY," \
      "  NAME     VARCHAR( 80 ) NOT NULL," \
      "  PRICE    FLOAT DEFAULT 0.0,"  \
	   "  QTY      INTEGER DEFAULT 0 )" ); 

   if( db->success() )
   {
      fprintf( stdout, "SSF: Product table created ok\n" ); 
   }

   db->executeUpdateFormat( 
		   "INSERT INTO %s VALUES( %d, 'Coke', %.2f, %d )", 
		            Product::TABLE_NAME , 11, 1.50, 10 );
	if( !db->success() ) 
   {
		fprintf(stderr, "Can't insert : %s", db->errorDescription() );
      return;
   }

   Product p( 11, db );
   fprintf( stdout, "SELECT: %s\n", p.toString().c_str() ); 

   Product     p1;

   p1.setId( 0 );
   p1.setQty( 100 );
   p1.setPrice( 1.99f );
   p1.setName( "BigMac" );
   p1.addNewRow( db );
   fprintf(stdout, "INSERT: %s\n", p1.toString().c_str() );

   p1.setId( 0 );
   p1.setQty( 200 );
   p1.setPrice( 2.99f );
   p1.setName( "Mouse" );
   p1.addNewRow( db );
   fprintf(stdout, "INSERT: %s\n", p1.toString().c_str() );

   ssf::StorableFactory< Product >::StorableList           *l;
   l = Product::getList( db, 0 );
   fprintf(stdout, "Selecting all products... total= %d\n", l->size() );
   for( int k = 0; k < (int)l->size(); k++ )
   {
       Product *t = l->at( k );
       printf( "P=%X, %s\n", t, t->toString().c_str() );
       delete t;
   }
   l->erase( l->begin(), l->end() );
   delete l;


   fprintf(stdout, "Deleting this product...\n" );
   p.deleteRow( db );
   fprintf(stdout, "DELETE: %s\n", p.toString().c_str() );


   fprintf(stdout, "Selecting one product for update...\n" );
   l = Product::getList( db, "NAME=\'Mouse\'" );
   if( l && ( l->size() == 1 ) )
   {
       Product *t = l->at( 0 );
       t->setName( "New Mouse" );
       t->setPrice( 3.99f );
       t->updateRow( db );
       printf( "UPDATE: %s\n", t->toString().c_str() );

       // delete memory object, not the persistent one
       // stored on the DB.
       delete t;
       l->erase( l->begin(), l->end() );
       delete l;
   }

   l = Product::getList( db, 0 );
   fprintf(stdout, "Selecting all products( again )... total= %d\n", l->size() );
   for( int j = 0; j < (int)l->size(); j++ )
   {
       Product *t = l->at( j );
       printf( "%s\n", t->toString().c_str() );
       delete t;
   }
   l->erase( l->begin(), l->end() );
   delete l;
}




int main()
{
	ssf::Connection *db;

   fprintf( stdout, "SSF: - Test driver starting...\n" ); 

   fprintf( stdout, "SSF: - Creating MySQL Driver...\n" ); 
	db = ssf::Factory::createConnection( ssf::Factory::MySQL );

   fprintf( stdout, "SSF: - connecting to MySQL server..\n" ); 
	// Connect to database on localhost
	if( !db->connect( URL_MYSQL_SERVER ) ) 
	{
		fprintf( stderr, "Unable connect to database '%s': %s\n", 
               URL_MYSQL_SERVER, db->errorDescription() );
		exit( 1 );
	}
   fprintf( stdout, "SSF: connect ==> %d\n", db->isConnected() ); 

   test_driver_for_raw_interface( db );


   fprintf( stdout, "SSF: starting storable test.\n" ); 
   test_driver_for_storable( db );


	db->disconnect();
   fprintf( stdout, "SSF: disconnect() ok\n" ); 
   delete db;
   fprintf( stdout, "SSF: test driver is complete\n" ); 
   return 0;
}

