///////////////////////////////////////////////////////////////////////////
// $Header: /usr/local/cvs-repository/ssf/src/mysql/ssf_mysql.cpp,v 1.1 2001/01/20 21:02:46 rosimildo Exp $
//
// Copyright (c) 2000-2001 ConnectTel, Inc. All Rights Reserved.
// by: Rosimildo da Silva
//  
// DESCRIPTION:
// 
// NOTE: This has been loosely based on the original work from:
//
// database.cc
// mysql C++ wrapper library
// 
// Author: Roland Haenel <rh@ginster.net>
// 
// This program is in the public domain.
// Distribute and use it freely.
//
// HISTORY:
//
// $Log: ssf_mysql.cpp,v $
// Revision 1.1  2001/01/20 21:02:46  rosimildo
// Added VC++ projects, and a simple GCC makefile to compile the
// system using CygWin. I do not have the client code for linux/cygwin
// to test the client at this point.
//
// Revision 1.1.1.1  2001/01/13 21:01:18  rosimildo
// Initial checkin for SSF
//
//
///////////////////////////////////////////////////////////////////////////

#include <stdarg.h>
#include <time.h>
#include <stdio.h>
#include <string.h>

#include <libxml/uri.h>
#include "ssf_mysql.h"

namespace ssf
{

// ------------------- MySQLResultSet implementation ------------------
/**
 * Ctor -- setup of data members.
 */
MySQLResultSet::MySQLResultSet( MYSQL_RES *res )
: result( res )
{
  reset();
}


/**
 * Dtor -- free resources associated with the instance..
 */
MySQLResultSet::~MySQLResultSet()
{
	if( result )  // Free memory resources
      mysql_free_result( result );
}


/**
 * Number of "rows" in the result set.
 */
int MySQLResultSet::getRows()
{
	return result ? (int)mysql_num_rows( result ) : 0;
}


/**
 * Number of "columns" on each row..
 */
int MySQLResultSet::getColumns()
{
	return result ? mysql_num_fields( result ) : 0;
}


/**
 * name of the column( Nth index )
 */
char *MySQLResultSet::getColumnName(int n)
{
	MYSQL_FIELD *field = getFieldByIndex( n );
	return field ? field->name : 0;
}


/**
 * size of the column( Nth index )
 */
int MySQLResultSet::getColumnSize(int n)
{
	MYSQL_FIELD *field = getFieldByIndex( n );
	return field ? field->length : 0;
}


/**
 * name of the column( Nth index ) by Name
 */
int MySQLResultSet::getColumnSize(const char *name)
{
	MYSQL_FIELD *field = getFieldByName( name );
	return field ? field->length : 0;
}


/**
 * go to a particular Row
 */
bool MySQLResultSet::seekRow( int row )
{
	if( result ) 
   {
	   mysql_data_seek( result, row );
      return true;
   }
   return false;
}


/**
 * get current row data..
 */
char **MySQLResultSet::getTuple()
{
	return ( char**)( result ? mysql_fetch_row( result ) : 0 );
}


/**
 * get current row data from "tuple" index...
 */
char **MySQLResultSet::getTuple(int tuple)
{
	seekRow( tuple );
	return getTuple();
}

/**
 * reset the iterator for rows... or reset to "moveBegin()...
 */
void MySQLResultSet::reset()
{
   reset_called = true;
	cur_tuple    = 0;
}

/**
 * advances iterator to next row...
 */
bool MySQLResultSet::next()
{
  if( reset_called )
  {
     reset_called = false;
     cur_tuple = getTuple( 0 );
  }
  else
  {
     cur_tuple = getTuple();
  }
  return cur_tuple != 0;
}


/**
 * get column value on current row as string...
 */
const char *MySQLResultSet::getString( const char *name )
{
   int i = getColumnIndex( name );
   if( i < 0 ) return 0;
	return cur_tuple[ i ];
}

/**
 * get column value on current row as string
 */
const char *MySQLResultSet::getString( int columnIndex )
{
   if( columnIndex > ( getColumns() -1 ) )
       return 0;
	return cur_tuple[ columnIndex ];
}

/**
 * get column "FIELD" for Nth index...
 */
MYSQL_FIELD *MySQLResultSet::getFieldByIndex( int n )
{
	if (result == NULL) return 0;
	mysql_field_seek( result, n );
	return mysql_fetch_field( result );
}

/**
 * get internal "FIELD" for column named name.
 */
MYSQL_FIELD *MySQLResultSet::getFieldByName( const char *name )
{
	int i;
	if (result == NULL) return 0;
	int nf = getColumns();
	for (i = 0; i < nf; i++)
   {
      MYSQL_FIELD *field = getFieldByIndex( i );
		if( !STR_COMP_CASE( name, field->name ) )
 			  return field;
   }
	return 0;
}

/**
 * get column Nth index for name...
 */
int MySQLResultSet::getColumnIndex( const char *name )
{
	int i;
	if (result == NULL) return -1;
	int nf = getColumns();
	for (i = 0; i < nf; i++)
   {
      MYSQL_FIELD *field = getFieldByIndex( i );
		if( !STR_COMP_CASE( name, field->name ) )
 			   return i;
   }
	return -1;
}



/**
 * Ctor -- init data members...
 */
MySQLConnection::MySQLConnection()
{
	connected = false;	// No connection yet
	strcpy( error, "No connection established" );
}


/**
 * Dtor -- disconnect from DB, if connected...
 */
MySQLConnection::~MySQLConnection()
{
	disconnect();  // Disconnect if connected to Connection
}

/**
 * description of the error message, if any..
 */
const char *MySQLConnection::errorDescription( int ) const
{
	if( connected == false ) return error;
	return mysql_error( (MYSQL*)&mysql );
}

/**
 * connects to a remote or local mySQL server...
 */
bool MySQLConnection::connect( const char *url, const char *user, const char *passwd )
{
   xmlURIPtr uri = xmlParseURI( url );
   if( strcmp( uri->scheme, "mysql" ) )
   {
     xmlFreeURI( uri );
     printf( "ERROR: wrong URL scheme: %s\n", uri->scheme );
     return false;
   }

   db_err = 0;
	if( mysql_connect(&mysql, uri->server, user, passwd ) == NULL ) 
	{
      printf( "ERROR: wrong server: %s\n", uri->server );
		strcpy( error, "Connect to Database failed" );
      db_err = 1;
	}
   // let's remove the prepending "/" from the DB name..
   else if( mysql_select_db( &mysql, uri->path ) ) 
	{
      printf( "ERROR: wrong DB: %s\n", &uri->path[1] );
		mysql_close( &mysql );
		strcpy( error, "No such Connection");	
      db_err = 1;
	}
   if( uri ) xmlFreeURI( uri );
   connected = db_err ? false : true;
	return connected;
}


/**
 * disconnects from a remote or local mySQL server...
 */
void MySQLConnection::disconnect()
{
	if (connected == false) return;
	mysql_close( &mysql );
	connected = false;
}


/**
 * executes a command. The command is passed as a buffer.
 * No limitation exist to exeute this command.
 */
MYSQL_RES *MySQLConnection::execute( const char *sqlQuery, bool & haveError )
{
	MYSQL_RES *result = 0;
   db_err = 0;  /* assume no errors */
	if (mysql_query( &mysql, sqlQuery ) == 0) 
	{
		result = mysql_store_result( &mysql );
		if( result == NULL ) 
		{
			if (mysql.fields == 0)
				haveError = false;
			else
				haveError = true;
		} 
		else haveError  = false;
	} 
	else haveError  = true;
   return result;
}

ResultSet *MySQLConnection::executeQuery( const char *sqlStatement )
{
  bool haveError = false;
  MYSQL_RES *res = execute( sqlStatement, haveError );

  // fake one error if something went wrong...
  if( haveError )  db_err = 1;

  return res ? new MySQLResultSet( res ) : 0;
}


UINT64 MySQLConnection::executeUpdate(const char *sqlStatement )
{
  bool haveError = false;
  MYSQL_RES *res = execute( sqlStatement, haveError );
  // fake one error if something went wrong...
  if( haveError )  db_err = 1;
  if( res )
  {
    printf( "Update with Result -->, haveError=%d\n", haveError );
    mysql_free_result( res );
    return 0;
  }
  UINT64  n = mysql_affected_rows( &mysql );
//  printf( "Update ==> rows = %d, haveError=%d\n", n, haveError );
  return n;
}

/**
 * ID from last insert...
 */
UINT64 MySQLConnection::idAfterInsert()
{
  UINT64 n = mysql_insert_id( &mysql );
//  printf( "ID after insert = %d, \n", n );
  return n;
}

} // namespace



