/////////////////////////////////////////////////////////////////////////////
// $Header: $
//
// Copyright (c) 2000 ConnectTel, Inc. All Rights Reserved.
//  
// MODULE DESCRIPTION: 
//
// Implementation of the Remote Resources Monitor interface. 
// This is the link between CORBA and RTEMS. This work as the
// Adapter parttern.
//
// All other modules involved are not specific to CORBA.
//
// MODIFICATION/HISTORY:
//
// $Log: $
//
/////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string>

#include <bsp.h>
#include <rtems.h>

#include "rtems_rrm_i.h"
#include "resources.h"

//
// let's not take any chances, and do everything inse of our own namespace...
//
namespace Rtems 
{

const char *errorMessage = "This resource has not been configured.<br>";

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsThreadList * MonitorImpl::getRtemsThreads()
{
  return 0;

}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsSemaphoreList * MonitorImpl::getRtemsSemaphores()
{
  return 0;

}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsTimerList * MonitorImpl::getRtemsTimers()
{
  return 0;

}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsPartitionList * MonitorImpl::getRtemsPartitions()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsRegionList * MonitorImpl::getRtemsRegions()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
RtemsMessageQueueList * MonitorImpl::getRtemsMessageQueues()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixThreadList * MonitorImpl::getPosixThreads()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixSemaphoreList * MonitorImpl::getPosixSemaphores()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixMutexList * MonitorImpl::getPosixMutexes()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixKeyList * MonitorImpl::getPosixKeys()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixCondVariableList * MonitorImpl::getPosixCondVariables()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixInterruptList * MonitorImpl::getPosixInterrupts()
{
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
PosixMessageQueueList * MonitorImpl::getPosixMessageQueues()
{
  return 0;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Formats the header of a html page.
//
/////////////////////////////////////////////////////////////////////////////
static std::string htmlHeader( const char *title )
{
    std::string s = "<html><head><title>";
    if( title )
       s.append( title );
    else
       s.append( "No Title" );
    s.append( "</title></head><body><font face=\"Arial, Verdana, Helvetica\" size=\"4\">" );
    return s;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
/////////////////////////////////////////////////////////////////////////////
static void htmlTail( std::string & s )
{
  s.append( "</font></body></html>" );
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Thread information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlRtemsThreads( std::string & s, const char *h2_title, bool isPosix )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];
   static const char *threadFmt = 
"<h2 text=\"#FF0000\">%d. THREAD ID = 0x%08X</h2>" \
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
  "<TR>" \
     "<TD><b>Name</b></TD><TD>%s</TD>" \
  "</TR>" \
  "<TR>" \
     "<TD><b>Base Priority</b></TD><TD>0x%08X</TD>" \
   "</TR>" \
   "<TR>" \
     "<TD><b>Current Priority</b></TD><TD>0x%08X</TD>" \
   "</TR>" \
   "<TR>" \
     "<TD><b>State</b></TD><TD>0x%08X</TD>" \
   "</TR>" \
   "<TR>" \
     "<TD><b>Wait ID</b></TD><TD>0x%08X</TD>" \
   "</TR>" \
   "<TR>" \
     "<TD><b>Stack Size</b></TD><TD>0x%08X</TD>" \
   "</TR>" \
   "<TR>" \
     "<TD><b>Pending Events</b></TD><TD>0x%08X</TD>" \
   "</TR></TABLE><p><hr noshade size=\"1\" align=\"right\">";
   if( isPosix )
   {
       util::PosixThread::getInstances( list );
       sprintf( buffer, "<h1>POSIX  Threads ==> %d </h1>", list.size()  );
       s.append( buffer );
   }
   else
   {
       util::RtemsThread::getInstances( list );
       sprintf( buffer, "<h1>RTEMS  Threads ==> %d </h1>", list.size()  );
       s.append( buffer );
   }
   int j = 1;
   if( list.size() )
   {
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::RtemsThread & t = *(( util::RtemsThread *)(*i));
       sprintf( buffer, threadFmt, j++, t.id(), t.name(), t.basePriority(), 
                                   t.currentPriority(), t.state(), t.waitId(), 
                                   t.stackSize(),  t.eventsPending() );
       s.append( buffer );
       delete *i;
     }
     list.clear();
   }
   else
   {
     s += "<br><br>*** RTEMS THREADS ***<br><br>";
     s += errorMessage;
   }
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Posix Mutex information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlPosixMutexes( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::PosixMutex::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>POSIX Mutexes ==> %d</H2>", list.size() );
     s.append( buffer );
     s.append( 
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
"<TR><TD><b>ID</b></TD><TD><b>Holder ID</b></TD><TD><b>Lock</b></TD><TD><b>NestCount</b></TD></TR>" );
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::PosixMutex & r = *(( util::PosixMutex *)(*i));
       sprintf( buffer, 
                "<TR><TD>0x%08lX</TD><TD>0x%08X</TD><TD>0x%08X</TD><TD>0x%08lX</TD></TR>",
                 r.id(), r.holderId(), r.isLocked(), r.nestCount() );
       s.append( buffer );
       delete *i;
     }
     s.append( "</TABLE>" );
     list.clear();
   }
   else
   {
     s += "<br><br>*** POSIX MUTEXES ***<br><br>";
     s += errorMessage;
   }
}





/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Posix Semaphores information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlPosixSemaphores( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::PosixSemaphore::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>POSIX Semaphores ==> %d</H2>", list.size() );
     s.append( buffer );
     int j = 1;
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::PosixSemaphore & r = *(( util::PosixSemaphore *)(*i));
       sprintf( buffer, \
  "<h2 text=\"#FF0000\">%d. SEMAPHORE ID = 0x%08X</h2>" \
  "<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
       "<TR><TD><b>Name</b></TD><TD>%4.4s</TD></TR>" \
       "<TR><TD><b>Attributes</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Count</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Is Named</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Is Linked</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Open Count</b></TD><TD>0x%08X</TD></TR>", j++, 
                r.id(), r.name(), r.attributes(), r.count(), r.isNamed(), 
                r.isLinked(), r.openCount() );
       s.append( buffer );
       s.append("</TABLE><p><hr noshade size=\"1\" align=\"right\">" );
       delete *i;
     }
     s.append( "</TABLE>" );
     list.clear();
   }
   else
   {
     s += "<br><br>*** POSIX SEMAPHORES ***<br><br>";
     s += errorMessage;
   }
}




/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Rtems Semaphores information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlRtemsSemaphores( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::RtemsSemaphore::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>RTEMS Semaphores ==> %d</H2>", list.size() );
     s.append( buffer );
     int j = 1;
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::RtemsSemaphore & r = *(( util::RtemsSemaphore *)(*i));

         sprintf( buffer, \
  "<h2 text=\"#FF0000\">%d. SEMAPHORE ID = 0x%08X</h2>" \
  "<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
       "<TR><TD><b>Name</b></TD><TD>%4.4s</TD></TR>" \
       "<TR><TD><b>Attributes</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Count</b></TD><TD>0x%08X</TD></TR>", j++, 
               r.id(), r.name(), r.attributes(), r.count() );
       s.append( buffer );
       s.append("</TABLE><p><hr noshade size=\"1\" align=\"right\">" );
       delete *i;
     }
     list.clear();
   }
   else
   {
     s += "<br><br>*** RTEMS SEMAPHORES ***<br><br>";
     s += errorMessage;
   }
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Regions information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlRtemsRegions( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::RtemsRegion::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>RTEMS Regions ==> %d</H2>", list.size() );
     s.append( buffer );
     int j = 1;
     for( i = list.begin(); i != list.end(); ++i )
     {
         const util::RtemsRegion & r = *(( util::RtemsRegion *)(*i));

         sprintf( buffer, 
  "<h2 text=\"#FF0000\">%d. REGION ID = 0x%08X</h2>" \
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
       "<TR><TD><b>Name</b></TD><TD>%4.4s</TD></TR>" \
       "<TR><TD><b>Attributes</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Starting Address</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Lenght</b></TD></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Page Size</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Max Segment Size</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Used Blocks</b></TD><TD>0x%08X</TD></TR>", j++,
                  r.id(), r.name(), r.attributes(), r.startingAddress(), r.length(), 
                 r.pageSize(), r.maxSegSize(),  r.usedBlocks() );
       s.append( buffer );
       s.append("</TABLE><p><hr noshade size=\"1\" align=\"right\">" );
       delete *i;
     }
     list.clear();
   }
   else
   {
     s += "<br><br>*** RTEMS REGIONS ***<br><br>";
     s += errorMessage;
   }
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Partitions information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlRtemsPartitions( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::RtemsRegion::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>RTEMS Partitions ==> %d</H2>", list.size() );
     s.append( buffer );
     int j = 1;
     for( i = list.begin(); i != list.end(); ++i )
     {
         const util::RtemsPartition & r = *(( util::RtemsPartition *)(*i));
         sprintf( buffer, 
  "<h2 text=\"#FF0000\">%d. PARTITION ID = 0x%08X</h2>" \
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
       "<TR><TD><b>Name</b></TD><TD>%4.4s</TD></TR>" \
       "<TR><TD><b>Attributes</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Address</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Lenth</b></TD></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Buffer Size</b></TD><TD>0x%08X</TD></TR>" \
       "<TR><TD><b>Used Blocks</b></TD><TD>0x%08X</TD></TR>", j++, 
            r.id(), r.name(), r.attributes(), r.startingAddress(), 
            r.length(), r.bufferSize(), r.usedBlocks() );
       s.append( buffer );
       s.append("</TABLE><p><hr noshade size=\"1\" align=\"right\">" );
       delete *i;
     }
     list.clear();
   }
   else
   {
     s += "<br><br>*** RTEMS PARTITIONS ***<br><br>";
     s += errorMessage;
   }
}





/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Posix Keys information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlPosixKeys( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::PosixKey::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>POSIX Keys ==> %d</H2>", list.size() );
     s.append( buffer );
     s.append( 
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
        "<TR><TD><b>ID</b></TD><TD><b>Is Active</b></TD></TR>" );
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::PosixKey & r = *(( util::PosixKey *)(*i));
       sprintf( buffer, "<TR><TD>0x%08lX</TD><TD>0x%08X</TD></TR>", 
                        r.id(), r.isActive() );
       s.append( buffer );
       delete *i;
     }
     s.append( "</TABLE>" );
     list.clear();
   }
   else
   {
     s += "<br><br>*** POSIX KEYS ***<br><br>";
     s += errorMessage;
   }
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Condition variables information as a 
//              HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlPosixCondVariables( std::string & s )
{
   util::ResourcePtrList list;
   util::ResourceIterator i;
   char buffer[ 1024 ];

   util::PosixCondVariable::getInstances( list );
   if( list.size() )
   {
     sprintf( buffer, "<H2>POSIX Cond. Variables ==> %d</H2>", list.size() );
     s.append( buffer );
     s.append( 
"<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
      "<TR><TD><b>ID</b></TD><TD><b>Mutex ID</b></TD></TR>" );
     for( i = list.begin(); i != list.end(); ++i )
     {
       const util::PosixCondVariable & r = *(( util::PosixCondVariable *)(*i));
       sprintf( buffer, "<TR><TD>0x%08lX</TD><TD>0x%08lX</TD></TR>", r.id(), r.mutexId() );
       s.append( buffer );
       delete *i;
     }
     s.append( "</TABLE>" );
     list.clear();
   }
   else
   {
     s += "<br><br>*** CONDITION VARIABLES ***<br><br>";
     s += errorMessage;
   }
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Get and formats the Clock counters information as a HTML page.
//
/////////////////////////////////////////////////////////////////////////////
static void htmlClockCounters( std::string & s )
{
   char buffer[ 1024 ];

   util::ClockInformation c;
   sprintf( buffer, "<H2>Clock Manager</H2>" \
 "<table border=\"1\" cellpadding=\"10\" cellspacing=\"0\" align=\"left\" bgcolor=\"#FFFF70\">" \
     "<TR><TD><b>Microseconds Per Tick</b></TD><TD>%d</TD></TR>" \
     "<TR><TD><b>Ticks Per Timeslice</b></TD><TD>%d</TD></TR>" \
     "<TR><TD><b>Ticks Since Boot</b></TD><TD>%d</TD></TR>", 
                   c.uSecsPerTick(), c.ticksPerTimeslice(),  c.ticksSinceBoot() ); 
   s.append( buffer );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: This function get the instances information and format them
//              as HTML pages to send to the clients. This is a quick and
//              dirty solution to make easier the GUI.
//
/////////////////////////////////////////////////////////////////////////////
char* MonitorImpl::getInstancesAsHTML( CounterType instanceType )
{
   std::string htmlPage;
   switch( instanceType )
   {
     case Rtems::POSIX_THREAD:
       htmlPage = htmlHeader( "Posix Threads" );
       htmlRtemsThreads( htmlPage, "POSIX THREADS", true  );
       break;

     case Rtems::RTEMS_THREAD:
       htmlPage = htmlHeader( "Rtems Threads" );
       htmlRtemsThreads( htmlPage, "RTEMS THREADS", false );
       break;
         
     case Rtems::RTEMS_SEMAPHORE:
       htmlPage = htmlHeader( "Rtems Semaphores" );
       htmlRtemsSemaphores( htmlPage );
       break;

     case Rtems::RTEMS_PARTITION:
       htmlPage = htmlHeader( "Rtems Partitions" );
       htmlRtemsPartitions( htmlPage );
       break;

     case Rtems::RTEMS_REGION:
       htmlPage = htmlHeader( "Rtems Regions" );
       htmlRtemsRegions( htmlPage );
       break;

     case Rtems::POSIX_SEMAPHORE:
       htmlPage = htmlHeader( "Posix Semaphores" );
       htmlPosixSemaphores( htmlPage );
       break;

     case Rtems::POSIX_MUTEX:
       htmlPage = htmlHeader( "Posix Mutexes" );
       htmlPosixMutexes( htmlPage );
       break;

     case Rtems::POSIX_KEY:
       htmlPage = htmlHeader( "Posix Keys" );
       htmlPosixKeys( htmlPage );
       break;

     case Rtems::POSIX_COND_VARIABLE:
       htmlPage = htmlHeader( "Posix Condition Variables" );
       htmlPosixCondVariables( htmlPage );
       printf( "Completed Cond Var: %s\n", htmlPage.c_str() );
       break;


     case Rtems::CLOCK:
       htmlPage = htmlHeader( "Clock Counters" );
       htmlClockCounters( htmlPage );
       break;

     case Rtems::CPU:
     case Rtems::MEMORY:


     case Rtems::RTEMS_MESSAGE_QUEUE:
     case Rtems::RTEMS_TIMER:

     case Rtems::POSIX_MESSAGE_QUEUE:
     case Rtems::POSIX_INTERRUPT:

     default:
       htmlPage = htmlHeader( "Not handled." );
       htmlPage += "Not implemented yet<br>";
       break;
   }
   htmlTail( htmlPage );
   return CORBA::string_dup( htmlPage.c_str() );
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Maps the CORBA counter types to RTEMS class types.
//
/////////////////////////////////////////////////////////////////////////////
Objects_Classes mapToObjectClass( CounterType cntType )
{
   Objects_Classes the_class;
   switch( cntType )
   {
     case Rtems::POSIX_THREAD:
     the_class = OBJECTS_POSIX_THREADS;
     break;

     case Rtems::RTEMS_THREAD:
     the_class = OBJECTS_RTEMS_TASKS;
     break;
         
     case Rtems::RTEMS_SEMAPHORE:
     the_class = OBJECTS_RTEMS_SEMAPHORES;
     break;

     case Rtems::RTEMS_MESSAGE_QUEUE:
     the_class = OBJECTS_RTEMS_MESSAGE_QUEUES;
     break;

     case Rtems::RTEMS_TIMER:
     the_class = OBJECTS_RTEMS_TIMERS;
     break;

     case Rtems::RTEMS_PARTITION:
     the_class = OBJECTS_RTEMS_PARTITIONS;
     break;

     case Rtems::RTEMS_REGION:
     the_class = OBJECTS_RTEMS_REGIONS;
     break;

     case Rtems::POSIX_SEMAPHORE:
     the_class = OBJECTS_POSIX_SEMAPHORES;
     break;

     case Rtems::POSIX_MUTEX:
     the_class = OBJECTS_POSIX_MUTEXES;
     break;

     case Rtems::POSIX_MESSAGE_QUEUE:
     the_class = OBJECTS_POSIX_MESSAGE_QUEUES;
     break;

     case Rtems::POSIX_KEY:
     the_class = OBJECTS_POSIX_KEYS;
     break;

     case Rtems::POSIX_INTERRUPT:
     the_class = OBJECTS_POSIX_INTERRUPTS;
     break;

     case Rtems::POSIX_COND_VARIABLE:
     the_class = OBJECTS_POSIX_CONDITION_VARIABLES;
     break;


     case Rtems::CPU:
     case Rtems::MEMORY:
     case Rtems::CLOCK:
     default:
     the_class = OBJECTS_RTEMS_TASKS;
     break;

   }
   return the_class;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: This function request the information for a counter from
//              local monitor instance.
//
/////////////////////////////////////////////////////////////////////////////
 CounterInfo MonitorImpl::getCounter( CounterType cntType )
 {
   util::u32 mx  = 0;
   util::u32 use = 0;
   CounterInfo counter;

   Objects_Classes cls = mapToObjectClass( cntType );
   switch( cntType )
   {
     case Rtems::RTEMS_THREAD:
          util::RtemsThread::getCounters( mx, use );
          break;
     case Rtems::RTEMS_SEMAPHORE:
          util::RtemsSemaphore::getCounters( mx, use );
          break;
     case Rtems::RTEMS_MESSAGE_QUEUE:
          util::RtemsMessageQueue::getCounters( mx, use );
          break;
     case Rtems::RTEMS_TIMER:
          util::RtemsTimer::getCounters( mx, use );
          break;
     case Rtems::RTEMS_PARTITION:
          util::RtemsPartition::getCounters( mx, use );
          break;
     case Rtems::RTEMS_REGION:
          util::RtemsRegion::getCounters( mx, use );
          break;
     case Rtems::POSIX_THREAD:
          util::PosixThread::getCounters( mx, use );
          break;
     case Rtems::POSIX_SEMAPHORE:
          util::PosixSemaphore::getCounters( mx, use );
          break;
     case Rtems::POSIX_MUTEX:
          util::PosixMutex::getCounters( mx, use );
          break;
     case Rtems::POSIX_MESSAGE_QUEUE:
          break;
     case Rtems::POSIX_KEY:
          util::PosixKey::getCounters( mx, use );
          break;
     case Rtems::POSIX_INTERRUPT:
          break;
     case Rtems::POSIX_COND_VARIABLE:
          util::PosixCondVariable::getCounters( mx, use );
          break;
     case Rtems::MEMORY:
          util::MemoryUsage::getCounters( mx, use );
          break;

     case Rtems::CPU:
     {
        CpuCounters c;
        util::CpuUsage cpu;
        c.total_ticks = cpu.totalTicks(); 
        c.idle_ticks  = cpu.idleTicks();   
        counter.u_cpu( c );
        return counter;
     }
     break;


     case Rtems::CLOCK:
     {
        ClockCounters c;
        util::ClockInformation clk;
        c.usecs_per_tick         = clk.uSecsPerTick();
        c.ticks_per_timeslice    = clk.ticksPerTimeslice();
        c.ticks_since_boot       = clk.ticksSinceBoot(); 
        counter.u_clock( c );
        return counter;
     }
     break;

     default:
     break;
   }

   printf( "CounterForClass() --> Maximum=%d, Inuse=%d\n", mx, use ); 
   DefaultCounters d;
   d.max_available = mx;
   d.cur_inuse = use;
   counter.u_default( d );
   counter._d( cntType );
   return counter;
 }


} // namespace

