/////////////////////////////////////////////////////////////////////////////
// $Header: $
//
// Copyright (c) 2000 ConnectTel, Inc. All Rights Reserved.
//  
// MODULE DESCRIPTION:
//  Implementation of utilities routines to provide the specific 
//  routines that collect information about RTEMS resources. 
//  This code can be used with interface adapters, such CORBA, CGI programs, 
//  etc to provide data locally or remotely. 
//
//  Application programs could use this interface to monitor resources 
//  usage of a running RTEMS system.
//
// MODIFICATION/HISTORY:
//
// $Log: $
//
/////////////////////////////////////////////////////////////////////////////


#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

// RTEMS includes...
#include <bsp.h>
#include <rtems.h>
#include <rtems/posix/mutex.h>
#include <rtems/rtems/config.h>
#include <rtems/posix/semaphore.h>
#include <rtems/posix/cond.h>
#include <rtems/posix/key.h>

#include "resources.h"

extern rtems_configuration_table BSP_Configuration;


// Anything from here belongs to this namespace...
namespace util
{


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: This function scans through the global "object table", 
//              and returns the entry for the class request. 
//              A NULL is returned if the class is invalied or 
//              no objects for the type exist.
//
/////////////////////////////////////////////////////////////////////////////
static Objects_Information *getObjectsForClass
( 
   Objects_Classes the_class 
)
{
  for ( unsigned32 class_index = OBJECTS_CLASSES_FIRST;
        class_index <= OBJECTS_CLASSES_LAST; class_index++ ) 
  {
    Objects_Information *information = _Objects_Information_table[ class_index ];
    if( information && ( information->the_class == the_class ) )
       return information;
  }
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Initializes the ID and NAME from one object control to the
//              users' buffers.
//
/////////////////////////////////////////////////////////////////////////////
void getIdAndName
( 
   const Objects_Control & o, 
   char *name, 
   u32 *id 
)
{
    // FIXME: Let's make a swap here. This might not be the
    // best of doing it.
    char *o_name = ( char *)o.name;
    name[ 0 ] = o_name[ 3 ];
    name[ 1 ] = o_name[ 2 ];
    name[ 2 ] = o_name[ 1 ];
    name[ 3 ] = o_name[ 0 ];

    name[ 4 ] = '\0';

    // Hack: Let's filter some garbage on some names.
    for( int i=0; i <4;i++ )
    {
      if( !isalnum( name[ i ] ) )
          name[ i ] = ' ';
    }

    *id = (u32)o.id;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Scans the "object table" for a class, and counts the current
//              number of instances of that class. The max number configured
//              is also returned.
//
/////////////////////////////////////////////////////////////////////////////
bool getObjectCountersForClass
( 
   u32 & max_available, 
   u32 & cur_inuse, 
   const Objects_Classes the_class 
)
{
    Objects_Information *information;
    int numberInstances = 0;
    bool rc = false;
    max_available = 0;
    cur_inuse = 0;
    switch( the_class )
    {
      case OBJECTS_INTERNAL_THREADS:
      case OBJECTS_RTEMS_TIMERS:
      case OBJECTS_RTEMS_SEMAPHORES:
      case OBJECTS_RTEMS_MESSAGE_QUEUES:
      case OBJECTS_RTEMS_PARTITIONS:
      case OBJECTS_RTEMS_REGIONS:             
      case OBJECTS_RTEMS_TASKS:               
      case OBJECTS_POSIX_THREADS:
      case OBJECTS_POSIX_KEYS:                
      case OBJECTS_POSIX_INTERRUPTS:          
      case OBJECTS_POSIX_MESSAGE_QUEUES:      
      case OBJECTS_POSIX_MUTEXES:
      case OBJECTS_POSIX_SEMAPHORES:          
      case OBJECTS_POSIX_CONDITION_VARIABLES: 
      information = getObjectsForClass( the_class );
      if( information ) 
      {
        for( unsigned32 i=1 ; i <= information->maximum; i++ ) 
        {
           if( information->local_table[ i ] )
           {
              numberInstances++;
           }
        }
        max_available = information->maximum;
        cur_inuse     = numberInstances++;
        rc = true;
      }
      break;

      // Not available at this point...
      case OBJECTS_NO_CLASS:
      case OBJECTS_RTEMS_PORTS:
      case OBJECTS_RTEMS_PERIODS:
      case OBJECTS_RTEMS_EXTENSIONS:          
      default:
      break;

    }
    return rc;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: This method fills in the list of resource instances with the
//              information about the objects of the class requested.
//
// RETURN: true   - at least one object was found.
//         false  - no instances were found for the class requested.
//
/////////////////////////////////////////////////////////////////////////////
bool getInstancesForClass
( 
   ResourcePtrList & list, 
   const Objects_Classes the_class 
)
{
    Objects_Information *information;
    bool rc = false;

    // Check if the class is Ok, and we have information about it...
    information = getObjectsForClass( the_class );
    if( !information ) return false;

    for( unsigned32 i=1 ; i <= information->maximum; i++ ) 
    {
      Resource * info = 0; 
      switch( the_class )
      {
        case OBJECTS_INTERNAL_THREADS:
        case OBJECTS_RTEMS_TASKS:               
        info = RtemsThread::create( information->local_table[ i ] );
        break;

        case OBJECTS_RTEMS_SEMAPHORES:
        info = RtemsSemaphore::create( information->local_table[ i ] );
        break;

        case OBJECTS_RTEMS_TIMERS:
        info = RtemsTimer::create( information->local_table[ i ] );
        break;

        case OBJECTS_RTEMS_MESSAGE_QUEUES:
        info = RtemsMessageQueue::create( information->local_table[ i ] );
        break;

        case OBJECTS_RTEMS_PARTITIONS:
        info = RtemsPartition::create( information->local_table[ i ] );
        break;

        case OBJECTS_RTEMS_REGIONS:             
        info = RtemsRegion::create( information->local_table[ i ] );
        break;

        case OBJECTS_POSIX_THREADS:
        info = PosixThread::create( information->local_table[ i ] );
        break;

        case OBJECTS_POSIX_SEMAPHORES:          
        info = PosixSemaphore::create( information->local_table[ i ] );
        break;

        case OBJECTS_POSIX_KEYS:                
        info = PosixKey::create( information->local_table[ i ] );
        break;

        case OBJECTS_POSIX_MUTEXES:
        info = PosixMutex::create( information->local_table[ i ] );
        break;

        case OBJECTS_POSIX_CONDITION_VARIABLES: 
        info = PosixCondVariable::create( information->local_table[ i ] );
        break;

        // Not available at this point...
        case OBJECTS_POSIX_INTERRUPTS:          
        case OBJECTS_POSIX_MESSAGE_QUEUES:      

        case OBJECTS_NO_CLASS:
        case OBJECTS_RTEMS_PORTS:
        case OBJECTS_RTEMS_PERIODS:
        case OBJECTS_RTEMS_EXTENSIONS:          
        default:
        break;
      }  
      if( info )
      {
         list.push_back( info ); 
         rc = true;
      }
    }   
    return rc;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsThread class. It gets the information
//              from the thread control, and creates an instance for it.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsThread *RtemsThread::create( void *thread_control )
{
   if( !thread_control ) return 0;

   RtemsThread *t     = new RtemsThread;
   Thread_Control *control =  (Thread_Control *)thread_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->ticksExecuted_  = control->ticks_executed;
   t->basePriority_   = control->real_priority;    
   t->currentPriority_ = control->current_priority; 
   t->state_           = control->current_state;           
   t->waitId_          = control->Wait.id;		    
   t->stackSize_       = control->Start.Initial_stack.size;
   t->isPreemptible_   = control->is_preemptible;
   t->eventsPending_   = 0;   
  
   RTEMS_API_Control  *api;
   api = ( RTEMS_API_Control  *)control->API_Extensions[ THREAD_API_RTEMS ];
   if( api )
   {
      t->eventsPending_ = api->pending_events;
   }
   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems & Internal
//              threads. 
//
/////////////////////////////////////////////////////////////////////////////
void RtemsThread::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_INTERNAL_THREADS );
   getInstancesForClass( list, OBJECTS_RTEMS_TASKS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsThread class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsThread::getCounters( u32 & max_available, u32 & inuse )
{
   u32 max_a;
   u32 in_use;
   getObjectCountersForClass( max_a, in_use, OBJECTS_INTERNAL_THREADS );
   max_available = max_a;
   inuse = in_use;
   getObjectCountersForClass( max_a, in_use, OBJECTS_RTEMS_TASKS );
   max_available += max_a;
   inuse += in_use;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsSemaphore class. It gets the 
//              information  from the Semaphore control, and creates an for it.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsSemaphore *RtemsSemaphore::create( void *semaphore_control )
{
   if( !semaphore_control ) return 0;

   RtemsSemaphore *t     = new RtemsSemaphore;
   Semaphore_Control *control =  (Semaphore_Control *)semaphore_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->attributes_ = control->attribute_set;		
   t->count_      = control->Core_control.semaphore.count;

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Semaphores.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsSemaphore::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_RTEMS_SEMAPHORES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsSemaphores class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsSemaphore::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_RTEMS_SEMAPHORES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Ctor - grab the tick number for all threads, and specially
//              the ticks for the idle thread.
//
/////////////////////////////////////////////////////////////////////////////
CpuUsage::CpuUsage()
{
   // FIXME: Big hack, this is the hardcoded ID for the IDLE thread.
   const u32 ID_IDLE = 0x04010001;

   // let's assume that the IDLE thread has run at least one tick.
   // it'll be overwrriten later, this is just in case...
   idleTicks_  = 1;

   totalTicks_ = 0;
   // _Thread_Disable_dispatch();
   for( int class_index = (int)OBJECTS_CLASSES_FIRST;
        (int)class_index <= OBJECTS_CLASSES_LAST; class_index++ ) 
   {
      Objects_Information *information = _Objects_Information_table[ class_index ];
      if( information && information->is_thread ) 
      {
         for ( unsigned32 i=1 ; i <= information->maximum ; i++ ) 
         {
           Thread_Control *the_thread = (Thread_Control *)information->local_table[ i ];
           if ( the_thread )
           {
              if( ID_IDLE == the_thread->Object.id )
              {
                 idleTicks_ = the_thread->ticks_executed;
              }
              totalTicks_ += the_thread->ticks_executed;
              printf( "ID=%X, tk=%d, class=%d\n", 
                      the_thread->Object.id, the_thread->ticks_executed, 
                      information->the_class );
              the_thread->ticks_executed = 0; 
           }
         }
      }
   }
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the MemoryUsage class.
//
/////////////////////////////////////////////////////////////////////////////
void MemoryUsage::getCounters( u32 & max_available, u32 & inuse )
{
  // FIXME: Big hack, for while - returns 16M as total memory and 4 M in use.
  max_available = 0x1000000;  
  inuse         =  0x400000;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Ctor - initializes the clock information.
//
/////////////////////////////////////////////////////////////////////////////
ClockInformation::ClockInformation()
{
  rtems_configuration_table *c  = &BSP_Configuration;
  uSecsPerTick_      = c->microseconds_per_tick;
  ticksPerTimeslice_ = c->ticks_per_timeslice;
  ticksSinceBoot_    = _Watchdog_Ticks_since_boot;
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsTimer class. It gets the the 
//              information  from the Timer control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no semaphore control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsTimer *RtemsTimer::create( void *timer_control )
{
   if( !timer_control ) return 0;

   RtemsTimer *t     = new RtemsTimer;
   Timer_Control *control =  (Timer_Control *)timer_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->t_class_ = control->the_class;
   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Timers.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsTimer::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_RTEMS_TIMERS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsTimer class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsTimer::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_RTEMS_TIMERS );
}




/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsMessageQueue class. It gets the the 
//              information  from the Message queue control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no msg queue control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsMessageQueue *RtemsMessageQueue::create( void *msg_control )
{
   if( !msg_control ) return 0;

   RtemsMessageQueue *t     = new RtemsMessageQueue;
   Message_queue_Control *control =  (Message_queue_Control *)msg_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->attributes_ = control->attribute_set;		

   t->maxMessages_     = control->message_queue.maximum_pending_messages;
   t->pendingMessages_ = control->message_queue.number_of_pending_messages; 

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Message Queues.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsMessageQueue::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_RTEMS_MESSAGE_QUEUES );
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsMessageQueue class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsMessageQueue::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_RTEMS_MESSAGE_QUEUES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsPartition class. It gets the the 
//              information  from the Partition control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no partition control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsPartition *RtemsPartition::create( void *partition_control )
{
   if( !partition_control ) return 0;

   RtemsPartition *t     = new RtemsPartition;
   Partition_Control *control =  (Partition_Control *)partition_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->attributes_ = control->attribute_set;		

   t->startingAddress_  = (u32)control->starting_address;         
   t->length_           = control->length;                   
   t->bufferSize_       = control->buffer_size;             
   t->usedBlocks_       = control->number_of_used_blocks;   

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Partitions.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsPartition::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_RTEMS_PARTITIONS );
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsPartition class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsPartition::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_RTEMS_PARTITIONS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the RtemsRegion class. It gets the the 
//              information  from the Region control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no Region control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
RtemsRegion *RtemsRegion::create( void *region_control )
{
   if( !region_control ) return 0;

   RtemsRegion *t     = new RtemsRegion;
   Region_Control *control =  (Region_Control *)region_control;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->attributes_ = control->attribute_set;		

   t->startingAddress_  = (u32)control->starting_address;         
   t->length_           = control->length;                   
   t->pageSize_         = control->page_size;             
   t->usedBlocks_       = control->number_of_used_blocks;   
   t->maxSegSize_       = control->maximum_segment_size;             
   return t;
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Partitions.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsRegion::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_RTEMS_REGIONS );
}

/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsPartition class.
//
/////////////////////////////////////////////////////////////////////////////
void RtemsRegion::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_RTEMS_REGIONS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the PosixSemaphore class. It gets the the 
//              information  from the Semaphore control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no semaphore control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
PosixSemaphore *PosixSemaphore::create( void *semaphore_control )
{
   if( !semaphore_control ) return 0;

   PosixSemaphore *t     = new PosixSemaphore;
   POSIX_Semaphore_Control *control =  (POSIX_Semaphore_Control *)semaphore_control;

   // overwrite type done by base class
   t->type_ = POSIX_SEMAPHORE;

   // initialize all members of the instance here.
   getIdAndName( control->Object, t->name_, &t->id_ );
   t->attributes_ = 0;		
   t->count_      = control->Semaphore.count;

   t->isNamed_    = control->named;
   t->isLinked_   = control->linked;   
   t->openCount_  = control->open_count; 

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Rtems Semaphores.
//
/////////////////////////////////////////////////////////////////////////////
void PosixSemaphore::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_POSIX_SEMAPHORES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the RtemsSemaphores class.
//
/////////////////////////////////////////////////////////////////////////////
void PosixSemaphore::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_POSIX_SEMAPHORES );
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the PosixThread class. It gets the the information
//              from the thread control, and creates an instance with the 
//              information from the control structure.
//              NULL is returned if no thread control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
PosixThread *PosixThread::create( void *thread_control )
{
   if( !thread_control ) return 0;

   PosixThread *t     = new PosixThread;
   Thread_Control *control =  (Thread_Control *)thread_control;

   // overwrite type done by base class
   t->type_ = POSIX_THREAD;

   // initialize all members of the instance here.
   t->id_              =  (u32)control->Object.id;
   t->ticksExecuted_   = control->ticks_executed;
   t->basePriority_    = control->real_priority;    
   t->currentPriority_ = control->current_priority; 
   t->state_           = control->current_state;           
   t->waitId_          = control->Wait.id; 
   t->stackSize_       = control->Start.Initial_stack.size;
   t->isPreemptible_   = control->is_preemptible;
   t->eventsPending_   = 0;   
   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Posix threads. 
//
/////////////////////////////////////////////////////////////////////////////
void PosixThread::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_POSIX_THREADS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the PosixThread class.
//
/////////////////////////////////////////////////////////////////////////////
void PosixThread::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_POSIX_THREADS );
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the PosixKey class. It gets the the 
//              information  from the PosixKey control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no Key control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
PosixKey *PosixKey::create( void *key_control )
{
   if( !key_control ) return 0;

   PosixKey *t     = new PosixKey;
   POSIX_Keys_Control *control =  (POSIX_Keys_Control *)key_control;

   // initialize all members of the instance here.
   t->id_        =  (u32)control->Object.id;
   t->isActive_  =  (u32)control->is_active;

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all PosixKey keys.
//
/////////////////////////////////////////////////////////////////////////////
void PosixKey::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_POSIX_KEYS );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the PosixKey class.
//
/////////////////////////////////////////////////////////////////////////////
void PosixKey::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_POSIX_KEYS );
}



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the CondVariable class. It gets the the 
//              information  from the CondVariable control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no CondVariable control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
PosixCondVariable *PosixCondVariable::create( void *cond_var_control )
{
   if( !cond_var_control ) return 0;

   PosixCondVariable *t     = new PosixCondVariable;
   POSIX_Condition_variables_Control *control =  
          (POSIX_Condition_variables_Control *)cond_var_control;

   // initialize all members of the instance here.
   t->id_           =  (u32)control->Object.id;
   t->mutexId_      =  (u32)control->Mutex;

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all PosixCondVariable keys.
//
/////////////////////////////////////////////////////////////////////////////
void PosixCondVariable::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_POSIX_CONDITION_VARIABLES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the PosixCondVariable class.
//
/////////////////////////////////////////////////////////////////////////////
void PosixCondVariable::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, 
                              OBJECTS_POSIX_CONDITION_VARIABLES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Factory for the PosixMutex class. It gets the the 
//              information  from the PosixMutex control, and creates an 
//              instance with the  information from the control structure.
//              NULL is returned if no PosixMutex control is passed.
//
//              The caller of this routine is responsible for deleting the
//              instance returned.
//
/////////////////////////////////////////////////////////////////////////////
PosixMutex *PosixMutex::create( void *mutex_control )
{
   if( !mutex_control ) return 0;

   PosixMutex *t     = new PosixMutex;
   POSIX_Mutex_Control *control =  (POSIX_Mutex_Control *)mutex_control;

   // initialize all members of the instance here.
   t->id_        =  (u32)control->Object.id;
   t->holderId_  = control->Mutex.holder_id;    
   t->isLocked_  = control->Mutex.lock;    
   t->nestCount_ = control->Mutex.nest_count;

   return t;
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: Populates the list passed in with all Posix Mutex keys.
//
/////////////////////////////////////////////////////////////////////////////
void PosixMutex::getInstances( ResourcePtrList & list )
{
   getInstancesForClass( list, OBJECTS_POSIX_MUTEXES );
}


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: returns the counters for the PosixMutex class.
//
/////////////////////////////////////////////////////////////////////////////
void PosixMutex::getCounters( u32 & max_available, u32 & inuse )
{
   getObjectCountersForClass( max_available, inuse, OBJECTS_POSIX_MUTEXES );
}


}  // namespace
