/*	event.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	7/17/0 04:54PM
	$Header: /cvsroot/jedi/ui/event.cxx,v 1.12 2000/10/03 07:29:43 tomyeh Exp $
	Purpose:	Event Manager
	Description:
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/mem.h>
#include <jedi/thread.h>
#include <jedi/event.h>
#include <jedi/win.h>

#define MAX_EVENT_NO		256

struct TEventEx: public TEvent {
	TEventEx*	next;
};

static TEventEx *s_queHead=0, *s_queEnd=0;
static volatile int s_evtNo = 0;
static CCond s_condEvtPushed(&g_sysMutex);

__u32	g_hwBtnState=0, g_hwBtnMask=BB_ALL;
TPoint	g_penPos;

static __u8 s_tapCount=0;
static bool s_bPenDown=false, s_bNilSent=false;

__u8 g_winEnterExitState=0;

///////////////////////////////////////////////////////////
static void _STDCALL SetWinEnterExitEvent(TEvent* evt, __s16 eType)
{
	if(evt){
		evtInitEvent(evt);
		evt->eType = eType;
		evt->data.winEnterExit.pWinEnter = g_pWinActive;
		evt->data.winEnterExit.pWinExit  = g_pWinActivePrev;
	}
	TRACE("evtGetEvent: %d, enter %p, exit %p\n", eType, g_pWinActive, g_pWinActivePrev);
}

EXTERN_C bool _AAPI evtGetEvent(TEvent* out_evt, int toTicks)
{
	//Note: EVT_WINENTER/EVT_WINEXIT is sent by checking a flag
	switch(g_winEnterExitState){
	case NOTIFY_WINACTIVECHANGED:
		if(g_pWinActivePrev){
			g_winEnterExitState = NOTIFY_WINEXITSENT;
			SetWinEnterExitEvent(out_evt, EVT_WINEXIT);
			return true;
		}
		//fall through
	case NOTIFY_WINEXITSENT:
		g_winEnterExitState = 0;
		if(g_pWinActive){
			SetWinEnterExitEvent(out_evt, EVT_WINENTER);
			return true;
		}
		break;
	}

	while(!s_queHead){
		if(!s_bNilSent){ //we have to generate EVT_NIL when idle
			s_bNilSent = true;
			if(out_evt) out_evt->eType = EVT_NIL;
			return true;
		}
		if(!s_condEvtPushed.Wait((unsigned)toTicks)){
			if(out_evt) out_evt->eType = EVT_NIL; //denote timeout
			return false;
		}
	}

	s_bNilSent = false;
	--s_evtNo;

	TEventEx* evt = s_queHead;
	s_queHead = s_queHead->next;
	if(!s_queHead){
		ASSERT(!s_evtNo && evt==s_queEnd); //must be the last one
		s_queEnd = 0;
	}

	if(out_evt)
		memcpy(out_evt, evt, sizeof(TEvent));

	TRACE("evtGetEvent: %d, %c%d, %d, %d\n", evt->eType, evt->penDown ? 'T': ' ', evt->tapCount, evt->screenX, evt->screenY);
	_FREE(evt);
	return true;
}

EXTERN_C bool _AAPI evtEventAvail(void)
{
	return s_queHead != 0;
		//no need to lock since only one member is accessed
		//and no temporary inconsistent value
}

EXTERN_C void _AAPI evtPushEvent(const TEvent* in_evt)
{
	ASSERT(in_evt);

	TEventEx* evt = _MALLOC_T(TEventEx);
	if(!evt){
		TRACE("memory not enough for enqueuing event\n");
		//silent error!!

	}else if(s_evtNo > MAX_EVENT_NO){
		//In case that a 'dead' program might not be able to
		//process any event. We drop the event silently
		TRACE("too many events. drop silently\n");

	}else{
		evt->next = 0;
		memcpy(evt, in_evt, sizeof(TEvent));

		if(s_queEnd){
			s_queEnd->next = evt;
			s_queEnd = evt;
		}else{
			ASSERT(!s_queHead && !s_evtNo);
			s_queHead = s_queEnd = evt;
		}
		++s_evtNo;

		s_condEvtPushed.Set(); //signal one of waiting threads
	}
}

EXTERN_C void _AAPI evtPushEventUnique(const TEvent* in_evt, unsigned id, bool bInPlace)
{
	if(!id)
		id = in_evt->eType;

	TEventEx **pp=&s_queHead, *evt=s_queHead;
	for(; evt; pp = &(evt->next), evt = *pp)
		if(evt->eType == (int)id){
			if(!bInPlace && evt->next){//move to end?
				*pp = evt->next;
				evt->next = 0;
				s_queEnd->next = evt;
				s_queEnd = evt;
			}
			memcpy(evt, in_evt, sizeof(TEvent));
			break;
		}

	if(evt)
		evtPushEvent(in_evt);
}

EXTERN_C void _JAPI evtPushEventEx(int eType)
{
	TEvent evt;
	evt.eType = (__s16)eType;
	evtInitEvent(&evt);
	evtPushEvent(&evt);
}

EXTERN_C void _JAPI evtInitEvent(TEvent* evt)
{
	evt->penDown	= s_bPenDown;
	evt->tapCount	= s_tapCount;
	evt->screenX	= g_penPos.x;
	evt->screenY	= g_penPos.y;
}

///////////////////////////////////////////////////////////
EXTERN_C bool _AAPI evtKeyQueEmpty(void)
{
	for(TEventEx* evt=s_queHead; evt; evt=evt->next)
		if(evt->eType == EVT_KEYDOWN)
			return false;

	return true;
}

EXTERN_C _err_t _AAPI evtFlushKeyQue(void)
{
	TEventEx **pp=&s_queHead, *evt=s_queHead;
	for(; evt; evt = *pp)
		if(evt->eType == EVT_KEYDOWN){
			*pp = evt->next;
			_FREE(evt);
		}else
			pp = &(evt->next);

	return 0;
}

EXTERN_C _err_t _AAPI evtPushKey(wchar_t cc, __u16 keyCode, __u16 keyModifiers)
{
	TEvent evt;
	evtInitEvent(&evt);
	evt.eType					= EVT_KEYDOWN;
	evt.data.keyDown.chr		= cc;
	evt.data.keyDown.keyCode	= keyCode;
	evt.data.keyDown.modifiers	= keyModifiers;

	evtPushEvent(&evt);
	return 0;
}

EXTERN_C void _JAPI evtPushKeyEx(wchar_t cc)
{
	__u16 keyModf = 0;
	if(cc >= CC_VIRTUAL_FIRST) keyModf |= KM_COMMAND;

	evtPushKey(cc, 0, keyModf);
}

///////////////////////////////////////////////////////////
EXTERN_C void _JAPI evtPushPen(int eType, int x, int y)
{
	ASSERT(eType==EVT_PENUP || eType==EVT_PENMOVE || eType>=EX_EVT_PENDOWN);

	static TPoint s_start;

	if(eType>=EX_EVT_PENDOWN){
		s_tapCount = (__u8)(eType - (EX_EVT_PENDOWN-1));
		eType = EVT_PENDOWN;

		s_start.x = (__s16)x;
		s_start.y = (__s16)y;
	}

	s_bPenDown = eType!=EVT_PENUP;
	g_penPos.x = (__s16)x;
	g_penPos.y = (__s16)y;

	TEvent evt;
	evtInitEvent(&evt);
		//it depends on g_penX, g_penY, s_tapCount and s_bPenDown

	evt.eType = (__s16)eType;
	if(eType==EVT_PENUP){
		evt.data.penUp.start	= s_start;
		evt.data.penUp.end		= g_penPos;

		s_tapCount = 0; //reset it
	}

	evtPushEvent(&evt);
}

EXTERN_C void _AAPI evtGetPen(__s16* x, __s16* y, bool* bPenDown)
{
	*x = g_penPos.x;
	*y = g_penPos.y;
	*bPenDown = s_tapCount!=0;
}

///////////////////////////////////////////////////////////
EXTERN_C bool _AAPI evtSysEventAvail(bool bIgnorePenUps)
{
	for(TEventEx* evt=s_queHead; evt; evt=evt->next)
		switch(evt->eType){
		case EVT_PENUP:
			if(bIgnorePenUps)
				continue;
			//fall through

		case EVT_PENDOWN:
		case EVT_PENMOVE:
		case EVT_KEYDOWN:
			return true;
		}

	return false;
}

///////////////////////////////////////////////////////////
static void _STDCALL evtOnAppStop(void)
{
	g_bSysStopping = true; //indicate the system is stopping
	for(TAppLocal* app=g_listApp; app; app=app->next)
		evtPushEventEx(EVT_APPSTOP);
}

static void _STDCALL evtOnKey(bool bDown, unsigned key)
{
	unsigned bbf = 0; //BB_xxx
	switch(key){
	case CC_PAGEUP:
		bbf = BB_PAGEUP;
		break;
	case CC_PAGEDOWN:
		bbf = BB_PAGEDOWN;
		break;
	default:
		if(key>=CC_BTN1 && key<=CC_BTN4)
			bbf = 1 << (key - CC_BTN1 + BB_APP1_SHIFT);
		break;
	}

	if(bbf)
		evtSetHwBtnState(bbf, bDown);

	if(bDown && (!bbf || evtTestHwBtnMask(bbf)))
		evtPushKeyEx((__u16)key);
}

EXTERN_C void _CDECL evtOnHwNotify(int eType, unsigned param1, unsigned param2)
{
	AnakinLock();

	switch(eType){
	case EVT_APPSTOP:
		evtOnAppStop();
		break;

	case EVT_KEYDOWN:
	case EX_EVT_KEYUP:
		evtOnKey(eType==EVT_KEYDOWN, param1);
		break;

	default:
		evtPushPen(eType, (int)param1, (int)param2);
		break;
	}

	AnakinUnlock();
}

///////////////////////////////////////////////////////////
EXTERN_C void _JAPI evtSysCleanup(void)
{
	if(s_evtNo)
		TRACE("There are %d unprocessed events\n", s_evtNo);

	while(s_evtNo)
		evtGetEvent(0, 0);
}
