/*	tvmgdi.cxx

{{IS_NOTE

		Authors:	Henri Chen
		Contributors:
		Create Date:	2000/9/26 03:57PM
		$Header: /cvsroot/jedi/tvm/tvmgdi.cxx,v 1.5 2000/10/12 09:00:45 henrichen Exp $
		Purpose:	
		Description:	Implemention dependant part of gdiXXX() for TVM
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/mem.h>
#include <jedi/rct.h>
#include <jedi/fb.h>
#include <jedi/bmp.h>
#include <jedi/fnt.h>
#include <jedi/win.h>
#include <jedi/gdi.h>

class CFbEng {
public:
	CFbEng(int bpp) : m_bpp(bpp) {}
	
	inline int GetBpp(void);
	inline int BeGetPixelBitNo(int x);	//assume big endian framebuffer
	inline int GetPixelBitNo(int x);	//assume cpu's endian
	inline __u32 GetBppMask(void);
	inline __u32 GetPixelMask(int x);
	inline unsigned long GetPixelByte32(int x);
	
private:
	int m_bpp;
};

inline int CFbEng::GetBpp(void)
{
	return m_bpp;
}

//which bit the pixel is located, big endian 
inline int CFbEng::BeGetPixelBitNo(int x)
{
	return -((x+1) * GetBpp()) & 31;
}

//which bit the pixel is located, cpu endian
inline int CFbEng::GetPixelBitNo(int x)
{
	return __be_bitof32_to_cpu(BeGetPixelBitNo(x));
}

inline __u32 CFbEng::GetBppMask(void)
{
	return 0xffffffff >> (32-GetBpp());
}

inline __u32 CFbEng::GetPixelMask(int x)
{
	return GetBppMask() << GetPixelBitNo(x);
}

inline unsigned long CFbEng::GetPixelByte32(int x)
{
	return ((x * GetBpp()) & ~31) >> 3;
}

//////////////////////////////////////////////////////////////////
inline TWin* gdiGetWin(TWin* pWin)
{
	return pWin ? pWin : &g_winScrn;
}

static __u32 _JAPI gdiGetPixelNoClip(TWin *pWin, int x, int y)
{
	TBmp *pBmp= pWin->pBitmap;
	CFbEng eng(pBmp->pixelSize);
	__u8 *pBase= (__u8*) mmLock(*bmpGetMemHandleAddr(pBmp));
	__u32 *pBits= (__u32*) (pBase + y * pBmp->rowBytes + eng.GetPixelByte32(x));
	__u32 clr= (*pBits >> eng.GetPixelBitNo(x)) & eng.GetBppMask();
	mmUnlock(*bmpGetMemHandleAddr(pBmp));
	
	return clr;
}

EXTERN_C __u32 _JAPI gdiGetPixel(TWin *pWin, int x, int y)
{
	//clipping
	pWin= gdiGetWin(pWin);
	TRect rect;
	CopyRect(&rect, &pWin->clippingBounds);
	WinToScreen(pWin, &rect.Left(), &rect.Top());

	if (!rctPtInRect((__s16) x, (__s16) y, &rect)) return 0;
	
	return gdiGetPixelNoClip(pWin, x, y);
}

static void _JAPI gdiSetPixelNoClip(TWin *pWin, int x, int y, TDrawState *pDs)
{
	TBmp *pBmp= pWin->pBitmap;
	CFbEng eng(pBmp->pixelSize);
	__u8 *pBase= (__u8*) mmLock(*bmpGetMemHandleAddr(pBmp));
	__u32 *pBits= (__u32*) (pBase + y * pBmp->rowBytes + eng.GetPixelByte32(x));
	__u32 clr= 0;
	if (!pDs) pDs= winGetDrawState();
	switch(pDs->pattern) {
	case DSPATTERN_BLACK:
		clr= pDs->foreColor;
		break;
	case DSPATTERN_WHITE:
		clr= pDs->backColor;
		break;
	case DSPATTERN_GRAY:
		clr= ((x & 1) ^ (y & 1)) ? 
				pDs->foreColor : pDs->backColor;
		break;
	case DSPATTERN_CUSTOM:
		clr= pDs->patternData[y & 7] & (0x80 >> (x & 7)) ? 
				pDs->foreColor : pDs->backColor;
		break;
	default:
		TRACE("Not supported drawing pattern\n");
		ASSERT(0);
	}
//	TRACE("SetPixel: base(0x%08x), bits(0x%08x), x(%d),y(%d),color(%d)\n", 
//		pBase, pBits, x, y, clr);

	clr<<= eng.GetPixelBitNo(x);

//	TRACE("SetPixel: *pBits(0x%08x), mask(0x%08x), maskcolor(0x%08x)\n",
//		*pBits, eng.GetPixelMask(x), clr);
	
	switch(pDs->transferMode) {
	case DSMODE_PAINT:
		*pBits= *pBits & ~eng.GetPixelMask(x) | clr;
		break;
	case DSMODE_ERASE:
		*pBits= *pBits & (~eng.GetPixelMask(x) | clr);
		break;
	case DSMODE_MASK:
		*pBits= *pBits & (~eng.GetPixelMask(x) | ~clr);
		break;
	case DSMODE_INVERT:
		*pBits^= clr;
		break;
	case DSMODE_OVERLAY:
		*pBits|= clr;
		break;
	case DSMODE_PAINTINVERSE:
		*pBits= *pBits & ~eng.GetPixelMask(x) | ~clr;
		break;
	case DSMODE_SWAP:
		{
		__u32 oldclr= (*pBits >> eng.GetPixelBitNo(x)) & eng.GetBppMask();
		if (oldclr == pDs->foreColor) 
			*pBits= (*pBits & ~eng.GetPixelMask(x)) | 
					(pDs->backColor << eng.GetPixelBitNo(x));
		else if (oldclr == pDs->backColor)
			*pBits= (*pBits & ~eng.GetPixelMask(x)) | 
					(pDs->foreColor << eng.GetPixelBitNo(x));
		break;
		}
	default:
		TRACE("Not supported drawing operation!\n");
		ASSERT(0);
	}
	mmUnlock(*bmpGetMemHandleAddr(pBmp));
}

static void _JAPI gdiSetPixelUpdate(TWin *pWin, int x, int y, TDrawState *pDs, bool bUpdate= false)
{
	//clipping
	pWin= gdiGetWin(pWin);
	TRect rect;
	CopyRect(&rect, &pWin->clippingBounds);
	WinToScreen(pWin, &rect.Left(), &rect.Top());

	if (!rctPtInRect((__s16) x, (__s16) y, &rect)) return;
	
	gdiSetPixelNoClip(pWin, x, y, pDs);
	
	if (bUpdate && !pWin->flags.offscreen) {
		TRect rect={{x,y},{1,1}};
		fbUpdate(&rect);
	}
}

EXTERN_C void _JAPI gdiSetPixel(TWin *pWin, int x, int y, TDrawState *pDs)
{
	gdiSetPixelUpdate(pWin, x, y, pDs, true);
}

#define	CLIP_LEFT	1
#define	CLIP_TOP	2
#define	CLIP_RIGHT	4
#define	CLIP_BOTTOM	8
static int LineGetX(
	int newY, int x, int y, int xDelta, int yDelta)
{
	//special case
	assert(yDelta != 0);	//horizontal line
	if (xDelta == 0)		//vertical line
		return x;
	return (newY - y) * xDelta / yDelta + x;
}

static __s16 LineGetY(
	int newX, int x, int y, int xDelta, int yDelta)
{
	//special case
	assert(xDelta != 0);	//vertical line
	if (yDelta == 0)		//horizontal line
		return y;
	return (newX - x) * yDelta / xDelta + y;
}


//return false means clipping all, no need to do anything!
inline bool _JAPI gdiClipLine(
	int *pX1, int *pY1, int *pX2, int *pY2, TRect *pClip, bool *bDrawLastPt)
{
	*bDrawLastPt= false;
	__s16 left	= pClip->Left();
	__s16 top	= pClip->Top();
	__s16 right	= pClip->Right();
	__s16 bottom= pClip->Bottom();
	
	__u8 region1= 0;
	if (*pX1 < left)
		region1 |= CLIP_LEFT;
	else if (*pX1 >= right)
		region1 |= CLIP_RIGHT;
		
	if (*pY1 < top)
		region1 |= CLIP_TOP;
	else if (*pY1 >= bottom)
		region1 |= CLIP_BOTTOM;
	
	__u8 region2= 0;
	if (*pX2 < left)
		region2 |= CLIP_LEFT;
	else if (*pX2 >= right)
		region2 |= CLIP_RIGHT;
		
	if (*pY2 < top)
		region2 |= CLIP_TOP;
	else if (*pY2 >= bottom)
		region2 |= CLIP_BOTTOM;

	if ((region2 | region1) == 0) return true;	//totally in

	int xDelta= *pX2 - *pX1;
	int yDelta= *pY2 - *pY1;
	int newX1=*pX1, newY1=*pY1, newX2=*pX2, newY2=*pY2;
	
	while (region2 != 0 || region1 != 0) {	//until no clipping needed
		//clip x2,y2 into clip rectangle
		if (region2 & CLIP_LEFT) {		//left
			if (region1 & CLIP_LEFT)	//clipall
				return false;
			newY2= LineGetY(left, *pX1, *pY1, xDelta, yDelta);
			newX2= left;
			region2 &= ~(CLIP_LEFT|CLIP_TOP|CLIP_BOTTOM);
			if (newY2 < top)
				region2 |= CLIP_TOP;
			else if (newY2 >= bottom)
				region2 |= CLIP_BOTTOM;
			*bDrawLastPt= true;
		}
		
		if (region2 & CLIP_TOP) {		//top
			if (region1 & CLIP_TOP)		//clipall
				return false;
			newX2= LineGetX(top, *pX1, *pY1, xDelta, yDelta);
			newY2= top;
			region2 &= ~(CLIP_TOP|CLIP_LEFT|CLIP_RIGHT);
			if (newX2 < left)
				region2 |= CLIP_LEFT;
			else if (newX2 >= right)
				region2 |= CLIP_RIGHT;
			*bDrawLastPt= true;
		}
		
		if (region2 & CLIP_RIGHT) {		//right
			if (region1 & CLIP_RIGHT)	//clipall
				return false;
			newY2= LineGetY(right-1, *pX1, *pY1, xDelta, yDelta);
			newX2= (__s16) (right-1);
			region2 &= ~(CLIP_RIGHT|CLIP_TOP|CLIP_BOTTOM);
			if (newY2 < top)
				region2 |= CLIP_TOP;
			else if (newY2 >= bottom)
				region2 |= CLIP_BOTTOM;
			*bDrawLastPt= true;
		}
		
		if (region2 & CLIP_BOTTOM) {	//bottom
			if (region1 & CLIP_BOTTOM)	//clipall
				return false;
			newX2= LineGetX(bottom-1, *pX1, *pY1, xDelta, yDelta);
			newY2= (__s16)(bottom-1);
			region2 &= ~(CLIP_BOTTOM|CLIP_LEFT|CLIP_RIGHT);
			if (newX2 < left)
				region2 |= CLIP_LEFT;
			else if (newX2 >= right)
				region2 |= CLIP_RIGHT;
			*bDrawLastPt= true;
		}

		//clip x1,y1 into clip rectangle	
		if (region1 & CLIP_LEFT) {		//left
			if (region2 & CLIP_LEFT)	//clipall
				return false;
			newY1= LineGetY(left, *pX1, *pY1, xDelta, yDelta);
			newX1= left;
			region1 &= ~(CLIP_LEFT|CLIP_TOP|CLIP_BOTTOM);
			if (newY1 < top)
				region1 |= CLIP_TOP;
			else if (newY1 >= bottom)
				region1 |= CLIP_BOTTOM;
		}
		
		if (region1 & CLIP_TOP) {		//top
			if (region2 & CLIP_TOP)		//clipall
				return false;
			newX1= LineGetX(top, *pX1, *pY1, xDelta, yDelta);
			newY1= top;
			region1 &= ~(CLIP_TOP|CLIP_LEFT|CLIP_RIGHT);
			if (newX1 < left)
				region1 |= CLIP_LEFT;
			else if (newX1 >= right)
				region1 |= CLIP_RIGHT;
		}
		
		if (region1 & CLIP_RIGHT) {		//right
			if (region2 & CLIP_RIGHT)	//clipall
				return false;
			newY1= LineGetY(right-1, *pX1, *pY1, xDelta, yDelta);
			newX1= (__s16)(right-1);
			region1 &= ~(CLIP_RIGHT|CLIP_TOP|CLIP_BOTTOM);
			if (newY1 < top)
				region1 |= CLIP_TOP;
			else if (newY1 >= bottom)
				region1 |= CLIP_BOTTOM;
		}
		
		if (region1 & CLIP_BOTTOM) {	//bottom
			if (region2 & CLIP_BOTTOM)	//clipall
				return false;
			newX1= LineGetX(bottom-1, *pX1, *pY1, xDelta, yDelta);
			newY1= (__s16)(bottom-1);
			region1 &= ~(CLIP_BOTTOM|CLIP_LEFT|CLIP_RIGHT);
			if (newX1 < left)
				region1 |= CLIP_LEFT;
			else if (newX1 >= right)
				region1 |= CLIP_RIGHT;
		}
	}
	*pX1= newX1; *pY1= newY1; *pX2= newX2; *pY2= newY2;
	return true;
}

inline void gdiBLine(
	TWin *pWin, int x1, int y1, int x2, int y2, TDrawState *pDs, bool bDrawLastPt)
{
	int xDelta= abs(x2-x1);
	int yDelta= abs(y2-y1);
	int xInc= (x2 > x1) ? 1 : -1;
	int yInc= (y2 > y1) ? 1 : -1;
		
	int rem;
	if (xDelta >= yDelta) {	//horizontal style
		rem= xDelta/2;
		while (x1 != x2) {
			gdiSetPixelNoClip(pWin, x1, y1, pDs);
			x1+= xInc;
			rem+= yDelta;
			if (rem >= xDelta) {
				rem-= xDelta;
				y1+= yInc;
			}
		} 
	}
	else {	//vertical style
		rem= yDelta/2;
		while (y1 != y2) {
			gdiSetPixelNoClip(pWin, x1, y1, pDs);
			y1+= yInc;
			rem+= xDelta;
			if (rem >= yDelta) {
				rem-= yDelta;
				x1+= xInc;
			}
		} 
	}
	if (bDrawLastPt) gdiSetPixelNoClip(pWin, x1, y1, pDs);
}

static void _JAPI gdiLineUpdate(
	TWin *pWin, int x1, int y1, int x2, int y2, TDrawState *pDs, bool bUpdate= false)
{
	pWin= gdiGetWin(pWin);
	TRect rect;
	CopyRect(&rect, &pWin->clippingBounds);
	WinToScreen(pWin, &rect.Left(), &rect.Top());
	
	bool bDrawLastPt;
	if (gdiClipLine(&x1, &y1, &x2, &y2, &rect, &bDrawLastPt)) {
		gdiBLine(pWin, x1, y1, x2, y2, pDs, bDrawLastPt);
		if (bUpdate && !pWin->flags.offscreen) {
			if (x1 > x2) Swap(x1, x2);
			if (y1 > y2) Swap(y1, y2);
			TRect r= {{x1, y1}, {x2-x1+1, y2-y1+1}};
			fbUpdate(&r);
		}
	}
}

EXTERN_C void _JAPI gdiLine(
	TWin *pWin, int x1, int y1, int x2, int y2, TDrawState *pDs)
{
	gdiLineUpdate(pWin, x1, y1, x2, y2, pDs, true);
}

EXTERN_C void _JAPI gdiOutlineRect(
	TWin *pWin, TRect *pRect, int radius, TDrawState *pDs)
{
	// Based on the ellipse algorithm in DDJ Graphics Programming, 
	// Aug. 89, by Kent Porter
	if (radius == 1)
		radius= 2;	
	if ((radius << 1) > pRect->Width())
		radius= pRect->Width() >> 1;
	if ((radius << 1) > pRect->Height())
		radius= pRect->Height() >> 1;
	
	int x=0, y= radius;
	int rsq= radius * radius;			//r^2
	int rcb= rsq * radius;				//r^3
	int r2sq= rsq << 1;					//2r^2
	int dd= rsq - rcb + (rsq >> 2);		//r^2 - r^3 + r^2/4
	int dx= 0;
	int dy= rcb << 1;					//2r^3
	
	int l= pRect->Left() + radius;
	int t= pRect->Top() + radius;
	int r= pRect->Right() - radius - 1;
	int b= pRect->Bottom() - radius - 1;

	//loop to draw octant at four corner (assume right-bottom-bottom and mirror)
	//We don't need to draw the first end point, the 4 edges drawing will do it.
	while(dx < dy) {
		if (dd>0L) {		// near the center of the octant circle
			y--;		// move toward center
			dy-= r2sq;	// update control value
			dd-= dy;
		}
		x++;			// next horizontal point
		dx+= r2sq;		// update control value
		dd+= rsq + dx;

		if (x > y)		// cross symetric point, no need to draw once again!
			break;
			
		//right-bottom
		gdiSetPixelUpdate(pWin, r+x, b+y, pDs);
		
		//right-top
		if ((b+y) != (t-y))
			gdiSetPixelUpdate(pWin, r+x, t-y, pDs);
			
		if ((l-x) != (r+x)) {
			//left-bottom
			gdiSetPixelUpdate(pWin, l-x, b+y, pDs);
			//left-top
			if ((b+y) != (t-y))
				gdiSetPixelUpdate(pWin, l-x, t-y, pDs);
		}
		
		if (x == y)		// meet in symetric point, no need to draw once again!
			break;
			
		gdiSetPixelUpdate(pWin, r+y, b+x, pDs);	//symetric on x=y
		gdiSetPixelUpdate(pWin, r+y, t-x, pDs);	//symetric on x=-y
		gdiSetPixelUpdate(pWin, l-y, b+x, pDs);	//symetric on x=-y
		gdiSetPixelUpdate(pWin, l-y, t-x, pDs);	//symetric on x=y
	}

	if (l <= r) {
		gdiLineUpdate(pWin, l, t-radius, r+1, t-radius, pDs);	//top edge
		if ((t-radius) != (b+radius))
			gdiLineUpdate(pWin, l, b+radius, r+1, b+radius, pDs);	//bottom edge
	}
	if (t <= b) {
		if (radius)
			b+= 1;
		else
			t+= 1;
	}
	if (t < b) {
		gdiLineUpdate(pWin, r+radius, t, r+radius, b, pDs);	//right edge
		if ((l-radius) != (r+radius))
			gdiLineUpdate(pWin, l-radius, t, l-radius, b, pDs);	//left edge
	}
	if (!pWin->flags.offscreen) fbUpdate(pRect);
}

EXTERN_C void _JAPI gdiFillRect(
	TWin *pWin, TRect *pRect, int radius, TDrawState *pDs)
{
	// Based on the ellipse algorithm in DDJ Graphics Programming, 
	// Aug. 89, by Kent Porter
	if (radius == 1)
		radius= 2;	
	if ((radius << 1) > pRect->Width())
		radius= pRect->Width() >> 1;
	if ((radius << 1) > pRect->Height())
		radius= pRect->Height() >> 1;
	
	int x=0, y= radius;
	int rsq= radius * radius;			//r^2
	int rcb= rsq * radius;				//r^3
	int r2sq= rsq << 1;					//2r^2
	int dd= rsq - rcb + (rsq >> 2);		//r^2 - r^3 + r^2/4
	int dx= 0;
	int dy= rcb << 1;					//2r^3
	
	int l= pRect->Left() + radius;
	int t= pRect->Top() + radius;
	int r= pRect->Right() - radius - 1;
	int b= pRect->Bottom() - radius - 1;

	//loop to fill in the up and bottom round area (assume right-bottom-bottom and mirror)
	while(dx < dy) {
		if (dd>0L) {		// near the center of the octant circle
			//bottom
			gdiLineUpdate(pWin, l-x, b+y, r+x+1, b+y, pDs);
			//top
			gdiLineUpdate(pWin, l-x, t-y, r+x+1, t-y, pDs);
			y--;		// move toward center
			dy-= r2sq;	// update control value
			dd-= dy;
		}
		x++;			// next horizontal point
		dx+= r2sq;		// update control value
		dd+= rsq + dx;

		if (x == (y+1))	break;	// meet in symetric point, no need to draw once again!
		
		//bottom
		gdiLineUpdate(pWin, l-y, b+x, r+y+1, b+x, pDs);
		//top
		gdiLineUpdate(pWin, l-y, t-x, r+y+1, t-x, pDs);
	}

	//square area
	if ((l-radius) <= (r+radius) && t <= b) {
		for (int y= t; y <= b; y++) {
			gdiLineUpdate(pWin, l-radius, y, r+radius+1, y, pDs);
		}

	}
	if (!pWin->flags.offscreen) fbUpdate(pRect);
}

//for different color tables (might different color depth)
inline void _JAPI gdiPalBltLine(
	TWin *pDstWin, int dx, int dy, 
	TWin *pSrcWin, int sx, int sy, __u8 *transparent,
	int width, int xInc, TPalette *pal, TDrawState *pDs)
{
	//bliting
	//It is ok to modify pDs here, gdiBitBlt() will handle that
	for ( ; width; width--, dx+= xInc, sx+= xInc) {
		__u8 clr= (__u8) gdiGetPixelNoClip(pSrcWin, sx, sy);
		if (transparent && clr == *transparent)
			continue;
		pDs->foreColor= pal[clr].index;
		gdiSetPixelNoClip(pDstWin, dx, dy, pDs);
	}
}

//for simple color table and same color depth
inline void _JAPI gdiBltLine(
	TWin *pDstWin, int dx, int dy, 
	TWin *pSrcWin, int sx, int sy, __u8 *transparent,
	int width, int xInc, TDrawState *pDs)
{
	//bliting
	//It is ok to modify pDs here, gdiBitBlt() will handle that
	for ( ; width; width--, dx+= xInc, sx+= xInc) {
		pDs->foreColor = gdiGetPixelNoClip(pSrcWin, sx, sy);
		if (transparent && pDs->foreColor == *transparent)
			continue;
		gdiSetPixelNoClip(pDstWin, dx, dy, pDs);
	}		
}

EXTERN_C void _JAPI gdiBitBlt(
	TWin *pDstWin, int dstX, int dstY, 
	TWin *pSrcWin, const TRect *pSrcRect, TDrawState *pDs)
{
	pDstWin= gdiGetWin(pDstWin);
	pSrcWin= gdiGetWin(pSrcWin);
	
	//clipping
	TRect rect;
	CopyRect(&rect, &pDstWin->clippingBounds);
	WinToScreen(pDstWin, &rect.Left(), &rect.Top());

	TRect dstRect= {{(__s16)dstX, (__s16)dstY},
						{pSrcRect->Width(), pSrcRect->Height()}};
	rctIntersect(&dstRect, &rect, &dstRect);
	if (dstRect.Width() == 0) return; //totally clipped out!
	
	//adjust source rectangle
	rect.Left()		= (__s16)(pSrcRect->Left() + dstRect.Left() - dstX);
	rect.Top()		= (__s16)(pSrcRect->Top() + dstRect.Top() - dstY);
	rect.Width()	= dstRect.Width();
	rect.Height()	= dstRect.Height();

	//decide bliting direction
	int yInc, sy, dy;
	if (rect.Top() < dstRect.Top()) {
		yInc= -1;
		sy= rect.Bottom()-1;
		dy= dstRect.Bottom()-1;
	} else {
		yInc= 1;
		sy= rect.Top();
		dy= dstRect.Top();
	}
	
	int xInc, sx, dx;
	if (rect.Left() < dstRect.Left()) {
		xInc= -1;
		sx= rect.Right()-1;
		dx= dstRect.Right()-1;
	} else {
		xInc= 1;
		sx= rect.Left();
		dx= dstRect.Left();
	}

	//prepare the color table, slow operation
	TBmp *pSrcBmp= pSrcWin->pBitmap;
	TColorTable *pCt= bmpGetColorTable(pSrcBmp);
	TPalette *pal= 0;
	if (pCt && pSrcBmp != pDstWin->pBitmap) {
		int num= pCt->numEntries;
		pal= bmpGetColorTableEntries(pCt);
		for (int i= 0; i < num; i++) {
			pal[i].index= 
				(__u8)gdiRGBToIndex(pDstWin, gdiIndexToRGB(pSrcWin, i));
		}
	}

	ASSERT(pCt || pDstWin->pBitmap->pixelSize == pSrcBmp->pixelSize);
	//?? 20000926, Henri Chen: we don't support different bpp that without a colortable!
	
	TDrawState ds= 	pDs ? *pDs : *winGetDrawState();
	if (pal) {
		for (int height= rect.Height(); height; height--, dy+= yInc, sy+= yInc)
			gdiPalBltLine(pDstWin, dx, dy, pSrcWin, sx, sy, 
				pSrcBmp->flags.hasTransparency ? &pSrcBmp->transparentIndex : 0,
				rect.Width(), xInc, pal, &ds);
	} else {
		for (int height= rect.Height(); height; height--, dy+= yInc, sy+= yInc)
			gdiBltLine(pDstWin, dx, dy, pSrcWin, sx, sy, 
				pSrcBmp->flags.hasTransparency ? &pSrcBmp->transparentIndex : 0,
				rect.Width(), xInc, &ds);
	}
	if(!pDstWin->flags.offscreen) fbUpdate(&dstRect);
}

inline void gdiDrawUnderline(
	TWin *pWin, int x1, int x2, int y, TDrawState *pDs)
{
	if (x1 == x2) return;
	
	TDrawState ds;
	ds.transferMode= pDs->transferMode;
	ds.foreColor= pDs->foreColor;
	switch(pDs->underlineMode) {
		case DSUNDERLINE_NONE:
			return;
		case DSUNDERLINE_GRAY:
			ds.pattern= DSPATTERN_GRAY;
			ds.backColor= pDs->backColor;
			break;
		default:
			ds.pattern= DSPATTERN_BLACK;
			break;
	}
	gdiLineUpdate(pWin, x1, y, x2, y, &ds);
}

EXTERN_C void _JAPI gdiDrawChars(
	TWin *pWin, const char *chars, int len, int left, int top, TDrawState *pDs)
{
	//?? 20000926, Henri Chen: we don't suport mult-byte fonts yet!
	//?? 20000926, Henri Chen: we now support only one type of font
	if (!pDs) pDs= winGetDrawState();
	TDrawState ds;
	ds.transferMode= pDs->transferMode;
	ds.pattern= pDs->pattern;
	if (pDs->pattern == DSPATTERN_CUSTOM)
		ds.patternData= pDs->patternData;
			
	TFont *pFont= fntGetTFont(pDs->fontId);
	TFontCharInfo *pFci= (TFontCharInfo*)((__u8*)pFont+pFont->ofInfo);
	__u16 *bits= (__u16*)((__u8*)pFont + pFont->ofBits);
	int rowWords= pFont->rowWords;
	int x= left;
	int y= 0;
	while(len--) {
		__u16 theChar= *chars++;
		if (theChar < pFont->firstChar || theChar > pFont->lastChar)
			theChar= pFont->lastChar+1;
		
		__u16 *pScan= &bits[pFci[theChar-pFont->firstChar].offset * pFont->height * rowWords];
//		TRACE("rowWords(%d), offset(%d), pScan=0x%08x, pScan= %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x\n",
//			rowWords, pFci[theChar-pFont->firstChar].offset, 
//			pScan, pScan[0],pScan[1],pScan[2],pScan[3],pScan[4],pScan[5],
//			pScan[6],pScan[7],pScan[8],pScan[9],pScan[10],pScan[11]);

		int width= pFci[theChar-pFont->firstChar].width;
//		TRACE("width(%d), height(%d)\n", width, pFont->height);

		y= top;
		int height= pFont->height;
		while(height--) {
			__u16 pat=0;
			int mask= 0;
			int i= 0;
			for(int j=0; j < width; j++) {
				if (mask == 0) {
					mask= 0x8000;
					pat= *(pScan+i);
					i++;
				}
				if (pat & mask) {
					ds.foreColor= pDs->textColor;
					ds.backColor= pDs->backColor;
				} else {
					ds.foreColor= pDs->backColor;
					ds.backColor= pDs->textColor;
				}
				gdiSetPixelUpdate(pWin, x+j, y, &ds);
				mask >>= 1;
			}
			y++;
			pScan+= rowWords;
		}
		x+= width;
	}
	gdiDrawUnderline(pWin, left, x, y-1, pDs);
	if (!pWin->flags.offscreen)	{
		TRect rect={{left, top},{x-left, y-top}};
		fbUpdate(&rect);
	}
}
