/*	tvmfb.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	6/21/0 11:12AM
	$Header: /cvsroot/jedi/tvm/dll/tvmfb.cxx,v 1.17 2000/10/12 03:07:30 tomyeh Exp $
	Purpose:	Frame Buffer
	Description:
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include "StdAfx.h"
#include <malloc.h>
#include "tvmapp.h"
#include "tvmfb.h"
#include "MainFrm.h"

#pragma warning(disable: 4035)
inline unsigned __swab(unsigned x)
{
	__asm {
		mov eax, x
		bswap eax
	}
}
#pragma warning(default: 4035)

CFb g_tvmFb;

/////////////////////////////////////////////////////////////////////
//CFb

CFb::CFb()
: m_hDIB(0), m_bmi(0), m_fb(0),
  m_view(0), m_hEvtExit(FALSE, TRUE), // manual event
  m_hEvtUpdAvail(FALSE, FALSE), //auto event
  m_bSync(true) //default: synchronous
{
}

CFb::~CFb()
{
	Bind(0);
	Cleanup();
}


HBITMAP CFb::Config(const struct tvm_info* ti,
					 const unsigned colors[], unsigned *baseAddr)
{
	Cleanup();

	//Check ti
	if(!ti){
	l_fail_arg:
		AfxMessageBox(IDP_WRONG_ARGUMENTS);
		return 0;
	}

	switch(ti->bpp){
	case 1: case 4: case 8:
		if(!colors)
			goto l_fail_arg;
		//fall through
	case 16: case 32:
		break;
	case 2:
		AfxMessageBox("TVM does not support 2 bits-per-pixel, due to Windows limitation");
		return 0;
	default:
		goto l_fail_arg;
	}

	//Init m_tvmInfo
	memcpy(&m_tvmInfo, ti, sizeof(m_tvmInfo));
	if(++m_tvmInfo.scale > 3)
		m_tvmInfo.scale = 3;

	//calc m_szLine and m_szTotal
	if((m_tvmInfo.scrn_cx * m_tvmInfo.bpp) & 0x1f){ // n/8/4
		AfxMessageBox("Each scan line must be aligned at 4-byte boundary");
		return 0;
	}

	//Init m_bmi
	unsigned cbColors = m_tvmInfo.bpp <= 8 ?
				(1 << m_tvmInfo.bpp)*sizeof(RGBQUAD) : 0;
	m_bmi=(BITMAPINFO*)malloc(sizeof(BITMAPINFO)-sizeof(RGBQUAD)+cbColors);
	if(!m_bmi){
		AfxMessageBox(IDP_MEMORY_NOT_ENOUGH);
		return 0;
	}

	ZeroMemory(&m_bmi->bmiHeader, sizeof(m_bmi->bmiHeader));
    m_bmi->bmiHeader.biSize         = sizeof(BITMAPINFOHEADER);
    m_bmi->bmiHeader.biWidth        = m_tvmInfo.scrn_cx;
    m_bmi->bmiHeader.biHeight       = -m_tvmInfo.scrn_cy;
    m_bmi->bmiHeader.biPlanes       = 1;
    m_bmi->bmiHeader.biBitCount     = m_tvmInfo.bpp;
    m_bmi->bmiHeader.biCompression  = BI_RGB;

	if(cbColors)
		memcpy(m_bmi->bmiColors, colors, cbColors);

	//DIB section
	HDC hDC = GetDC(0);
	m_hDIB = CreateDIBSection(hDC, m_bmi, DIB_RGB_COLORS,
					(void**)&m_fb, 0, 0);
	ReleaseDC(0, hDC);

	if(!m_hDIB){
		Cleanup();
		AfxMessageBox("Failed to create a DIB section");
		return 0;
	}
	if(baseAddr)
		*baseAddr = (unsigned)m_fb;

	tvmMessage("frame buffer: %p\n", m_fb);
	return m_hDIB;
}

bool CFb::SetPalette(const unsigned colors[],
		int index, int nColors, bool bBigEndian)
{
	if(m_tvmInfo.bpp > 8){
		tvmMessage("Palette is supported, color depth=%d\n", m_tvmInfo.bpp);
		return false;
	}

	int nTotal = 1 << m_tvmInfo.bpp;
	if(index < 0 || index >= nTotal){
		tvmMessage("Palette index, %d, out-of-bound\n", index);
		return false;
	}

	if(nColors == -1) nColors = nTotal - index;
	else if(nColors > nTotal - index) nColors = nTotal - index;

	//update palette in DIB section
	unsigned *cvtColors = (unsigned*)colors;
	if(bBigEndian){
		cvtColors = (unsigned*)_alloca(nColors * sizeof(unsigned));
		for(int j=0; j<nColors; ++j)
			cvtColors[j] = __swab(colors[index+j]);
	}

	HDC hdcMem = CreateCompatibleDC(0);
	HGDIOBJ old = SelectObject(hdcMem, GetDIBSection());
	SetDIBColorTable(hdcMem, index, nColors, (const RGBQUAD*)cvtColors);
	SelectObject(hdcMem, old);
	DeleteObject(hdcMem);

	//update palette stored locally
	memcpy(m_bmi->bmiColors+index, cvtColors, nColors*sizeof(RGBQUAD));

	//update view
	if(m_view){
		m_view->Invalidate(TRUE);
		if(IsSync())
			m_view->UpdateWindow();
	}

	return true;
}

void CFb::Cleanup()
{
	if(m_bmi){
		free(m_bmi);
		m_bmi = 0;
	}
	if(m_hDIB){
		DeleteObject(m_hDIB);
		m_hDIB = 0;
	}
}

void CFb::Update(const RECT* pRect)
{
	CRect rect;
	if(pRect){
		rect = *pRect;
	}else{
		rect.left = rect.top = 0;
		rect.right = m_tvmInfo.scrn_cx;
		rect.bottom = m_tvmInfo.scrn_cy;
	}

	rect.left *= GetScale();
	rect.top *= GetScale();
	rect.right *= GetScale();
	rect.bottom *= GetScale();

	if(m_view && !theApp.IsStopping())
		if(IsSync()){
			CDC* pdc = m_view->GetDC();
			Draw(*pdc, &rect);
			m_view->ReleaseDC(pdc);
		}else{
			m_csUpdList.Lock();
			m_updList.AddTail(rect);
			m_csUpdList.Unlock();
			m_hEvtUpdAvail.SetEvent(); //wake up thread
		}
}

HBITMAP CFb::Duplicate()
{
	HDC hDC = GetDC(0);
	HBITMAP hbmp = CreateDIBitmap(hDC, &m_bmi->bmiHeader, 
			CBM_INIT, m_fb, m_bmi, DIB_RGB_COLORS);
	ReleaseDC(0, hDC);
	return hbmp;
}

//////////////////////////////////////////////////////////////////////////

void CFb::Draw(CDC& dc, const CRect* pRect)
{
	m_csDraw.Lock();

	HDC hdcMem = CreateCompatibleDC(dc.m_hDC);
	HGDIOBJ old = SelectObject(hdcMem, GetDIBSection());

	CRect rc;
	if(!pRect){
		rc.left = rc.top = 0;
		rc.right = WidthScale();
		rc.bottom = HeightScale();
		pRect = &rc;
	}

	StretchBlt(dc.m_hDC,
		pRect->left, pRect->top,
		pRect->Width(), pRect->Height(),
		hdcMem,
		pRect->left/GetScale(), pRect->top/GetScale(),
		pRect->Width()/GetScale(), pRect->Height()/GetScale(),
		SRCCOPY);

	SelectObject(hdcMem, old);
	DeleteObject(hdcMem);

	m_csDraw.Unlock();
}

void CFb::ProcessUpdList()
{
	CRgn rgn;
	for(;;){
		bool bFound = false;
		RECT rc;
		m_csUpdList.Lock();
		if(!m_updList.IsEmpty()){
			bFound = true;
			rc = m_updList.GetHead();
			m_updList.RemoveHead();
		}
		m_csUpdList.Unlock();

		if(!bFound){ //no data?
			if(rgn.m_hObject){
				CDC* pdc = m_view->GetDC();
				pdc->SelectObject(&rgn);
				Draw(*pdc);
				m_view->ReleaseDC(pdc);
			}
			break;
		}

		if(rgn.m_hObject){
			CRgn rgn2;
			rgn2.CreateRectRgnIndirect(&rc);
			rgn.CombineRgn(&rgn, &rgn2, RGN_OR);
		}else
			rgn.CreateRectRgnIndirect(&rc);
	}
}

unsigned __stdcall CFb::UpdThdFunc(void* p)
{
	CFb* fb = (CFb*)p;
	HANDLE syncObjs[2] = {
		fb->m_hEvtExit.m_hObject,
		fb->m_hEvtUpdAvail.m_hObject
	};

	for(;;){
		switch(WaitForMultipleObjects(2, syncObjs, FALSE, INFINITE)){
		case WAIT_OBJECT_0:
			return 0; //exit

		case WAIT_OBJECT_0+1:
			fb->ProcessUpdList();
			break;

		default:
			UNEXPECT_ERROR();
			break;
		}
	}
}

bool CFb::Bind(CWnd* view)
{
	if(view){
		Bind(0); //cleanup

		m_view = view;
		m_hEvtExit.ResetEvent();

		unsigned nId;
		HANDLE hThd = (HANDLE)_beginthreadex(0, 0, 
						UpdThdFunc, this, 0, &nId);

		if(hThd){
			CloseHandle(hThd);
		}else{
			Bind(0);
			UNEXPECT_ERROR();
		}
		return hThd!=0;

	}else{
		if(m_view){
			m_hEvtExit.SetEvent(); //make the thread to terminate
			m_view = 0;
		}
		return true;
	}
}

///////////////////////////////////////////////////////////////////////////
//Global Functions
EXTERN_C TVM_API
HDIB _CDECL tvmFbConfig(const struct tvm_info* ti, 
					 const unsigned colors[], unsigned *baseAddr)
{
	return (HDIB)g_tvmFb.Config(ti, colors, baseAddr);
}

EXTERN_C TVM_API void _CDECL tvmFbUpdate(const TRect* rc)
{
	theApp.WaitIfPause();
	RECT rect;
	if(rc){
		rect.left = rc->leftTop.x;
		rect.right = rect.left + rc->extent.x;
		rect.top = rc->leftTop.y;
		rect.bottom = rect.top + rc->extent.y;
	}
	g_tvmFb.Update(rc ? &rect: 0);
}

EXTERN_C TVM_API void _CDECL tvmFbSync(bool bSync)
{
	g_tvmFb.SetSync(bSync);
}

EXTERN_C TVM_API bool _CDECL tvmFbSetPaletteLE(
	const unsigned colors[], int index, int nColors)
{
	theApp.WaitIfPause();
	return g_tvmFb.SetPalette(colors, index, nColors, false);
}

EXTERN_C TVM_API bool _CDECL tvmFbSetPaletteBE(
	const unsigned colors[], int index, int nColors)
{
	theApp.WaitIfPause();

	return g_tvmFb.SetPalette(colors, index, nColors, true);
}
