/*	thd.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	9/4/0 02:49PM
	$Header: /cvsroot/jedi/sys/thd.cxx,v 1.8 2000/09/26 10:09:26 tomyeh Exp $
	Purpose:	Thread Functions
	Description:
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/thread.h>
#include <jedi/mem.h>
#include <lib/llink.h>

/////////////////////////////////////////////////////////
struct TThdCreateInfo {
	_thread_f	entry;
	void	 	*data;
	TAppLocal	*pAppLocal;
	TThdLocal	*pThdLocal;
	__u32		nId;
	__u32		nIdOwner;
	_handle_t	hThd;

	//output
	bool		*pInited;
};

EXTERN_C void _AAPI AppThdFunc(void* data);

//thread level init/cleanup
EXTERN_C void _JAPI dsThdCleanup(TThdLocal* thd);

/////////////////////////////////////////////////////////
static void _STDCALL ThdSmartCleanup(TThdLocal* thd, bool bExitThread)
{
	if(!thd) thd = GetThdLocal();

	if(thd->IsMainThread())
		AppCleanup(thd->pAppLocal, bExitThread);
	else
		ThdCleanup(thd, true);
}

static void _STDCALL ThdPreInit(TThdCreateInfo* tci)
{
	TAppLocal *app = tci->pAppLocal;
	TThdLocal **pPrevThd;
	if(tci->entry==AppThdFunc){ //a new app thread?
		ASSERT(!app->pThdLocal);
		pPrevThd = &app->pThdLocal;

		app->next	= g_listApp;
		g_listApp	= app;
	}else{
		ASSERT(app->pThdLocal);
		pPrevThd = &app->pThdLocal->next;
	}

	TThdLocal *thd	= tci->pThdLocal;
	thd->next		= *pPrevThd;
	*pPrevThd		= thd;
			//pAppLocal always points to the first create thread

	thd->pAppLocal	= app;
	thd->hThd		= tci->hThd; ASSERT(tci->hThd);
	thd->nId		= tci->nId; ASSERT(tci->nId);
	thd->nIdOwner	= tci->nIdOwner;

	extern unsigned g_thdLocalIndex;
	mtSetTls(g_thdLocalIndex, thd);

	*(tci->pInited)	= true; //inited
	_FREE(tci);
}

static OSTHD_RET_T OSTHD_API ThreadFunc(void* param)
{
	AnakinLock(); //lock first; Note it is a new thread so no lock when enterring

	((TThdCreateInfo*)param)->pThdLocal->stack = &param;
	_thread_f entry	= ((TThdCreateInfo*)param)->entry;
	void* data		= ((TThdCreateInfo*)param)->data;
	ThdPreInit((TThdCreateInfo*)param);
	//don't access param after ThdPreInit

	AnakinUnlock(); //unlock because we are returning to user codes
	exTRY{
		entry(data);
	}exCATCH(errCode){
		TRACE("An uncatched exception is found, %d\n", errCode);
	}exEND
	AnakinLock(); //grant lock again

	ThdSmartCleanup(0, false);

	AnakinUnlock();
	return ((OSTHD_RET_T)0);
}

EXTERN_C __u32 _AAPI mtCreateThread(_thread_f entry, void* data, unsigned stackSize)
{
	TThdCreateInfo* tci = _MALLOC_T(TThdCreateInfo);
	if(!tci)
		return 0;

	tci->pThdLocal = _CALLOC_T(TThdLocal);
	if(!tci->pThdLocal){
	l_fail1:
		_FREE(tci);
		return 0;
	}

	TThdLocal* thd	= GetThdLocal();

	if(entry==AppThdFunc){//a new app?
		tci->pAppLocal = _CALLOC_T(TAppLocal);
		if(!tci->pAppLocal){
		l_fail2:
			_FREE(tci->pThdLocal);
			goto l_fail1;
		}
		if(data){
			strcpy(tci->pAppLocal->prcNext, (char*)data);
			data = 0; //useless in this case
		}
	}else{
		tci->pAppLocal	= thd->pAppLocal;
		ASSERT(tci->pAppLocal);
		ASSERT(tci->pAppLocal->pThdLocal); //the second thread
	}

	tci->nIdOwner	= thd ? thd->nId: 0;
			//note: thd might be zero because the caller might
			//be the first thread (i.e., before any process is created)
	tci->entry		= entry;
	tci->data		= data;
	bool bInited	= false;
	tci->pInited	= &bInited;

	tci->hThd = mtOsCreateThread(ThreadFunc, tci, stackSize, &tci->nId);
		//it is OK to access tci after create, since the thread will
		//wait for the lock. However, you cannot access it after mtJediYield
	__u32 nThdId = tci->nId;
	if(!tci->hThd){
		if(entry==AppThdFunc)
			_FREE(tci->pAppLocal);
		goto l_fail2;
	}

	while(g_listApp && !(volatile bool&)bInited)
		mtJediYield();
		//Make sure the thread is inited correctly; Otherwise, the
		//caller might want to reference the thread immediately
		//The period is very short, so no need to use CCond
		//Note: for the first app, we don't need to wait since
		//the caller is not from applications.

	ASSERT(nThdId);
	return nThdId;
}

/////////////////////////////////////////////////////////
EXTERN_C void _JAPI ThdCleanup(TThdLocal* thd, bool bDetach)
{
	ASSERT(thd); //it doesn't assume zero as GetThdLocal

	//cleanup owned objects
	dsThdCleanup(thd);
	thd->exInfo = 0;

	//terminate the thread
	if(thd != GetThdLocal()) //other thread?
		mtOsKillThread(thd->hThd);

	//cleanup this node
	if(bDetach){ //don't destroy the node
		mtOsDetachThread(thd->hThd);

		sllUnlink((void**)&thd->pAppLocal->pThdLocal,
						thd, OffsetOf(TThdLocal, next));
		_FREE(thd);
	}
}

/////////////////////////////////////////////////////////
EXTERN_C void _AAPI mtExitThread(void)
{
	ThdSmartCleanup(0, false);

	AnakinUnlock(); //release the lock
	mtOsExitThread();
}

/////////////////////////////////////////////////////////
static TThdLocal* _STDCALL Find(TAppLocal* app, __u32 nThdId)
{
	for(TThdLocal* thd=app->pThdLocal; thd; thd=thd->next)
		if(thd->nId == nThdId)
			return thd;
	return 0;
}

static TThdLocal* _STDCALL Find(__u32 nThdId)
{
	//Search the current application first, for better perf.
	TAppLocal* thisApp = GetAppLocal();
	TThdLocal* thd = Find(thisApp, nThdId);
	if(thd)
		return thd;

	for(TAppLocal* app=g_listApp; app; app=app->next)
		if(app != thisApp){
			thd = Find(app, nThdId);
			if(thd)
				return thd;
		}
	return 0;
}

EXTERN_C bool _AAPI mtKillThread(__u32 nThdId)
{
	if(GetThdLocal()->nId == nThdId){
		mtExitThread();
	}else{
		TThdLocal* thd = Find(nThdId);
		if(!thd)
			return false;
		ThdSmartCleanup(thd, true);
			//if thd is a main thread of the calling one,
			//this thread is also terminated
	}
	return true;
}

/////////////////////////////////////////////////////////
EXTERN_C void _JAPI mtJediYield(void)
{
	AnakinUnlock();
	mtAnakinYield();
	AnakinLock();
}

EXTERN_C void _JAPI mtJediSleep(unsigned ticks)
{
	AnakinUnlock();
	mtAnakinSleep(ticks);
	AnakinLock();
}

EXTERN_C __u32 _AAPI mtGetCurrentProcess(void)
{
	TThdLocal* thd =GetThdLocal();
	return thd ? thd->pAppLocal->pThdLocal->nId: 0;
}

/////////////////////////////////////////////////////////
static _handle_t _STDCALL FindHandle(__u32 nThdId)
{
	TThdLocal* thd = Find(nThdId);
	return thd ? thd->hThd: 0;
}

EXTERN_C bool _AAPI mtWaitThread(__u32 nThdId, unsigned toTicks)
{
	_handle_t hThd = FindHandle(nThdId);
	if(!hThd) return true; //assuming done

	AnakinUnlock(); //to let other threads have the chance to stop
	bool bDone = mtOsWaitThread(hThd, toTicks);
	AnakinLock();
	return bDone;
}

EXTERN_C bool _AAPI mtThreadActive(__u32 nThdId)
{
	return FindHandle(nThdId) != 0;
}

/////////////////////////////////////////////////////////
EXTERN_C bool _AAPI mtSetPriority(__u32 nThdId, int priority)
{
	_handle_t hThd = FindHandle(nThdId);
	return hThd ? mtOsSetPriority(hThd, priority): false;
}

EXTERN_C int _AAPI mtGetPriority(__u32 nThdId)
{
	_handle_t hThd = FindHandle(nThdId);
	return hThd ? mtOsGetPriority(hThd): 0;
}
