/*	memdbg.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	7/13/0 11:09AM
	$Header: /cvsroot/jedi/mem/memdbg.cxx,v 1.12 2000/09/26 10:09:26 tomyeh Exp $
	Purpose:	The debug version of memory allocators
	Description:
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/mem.h>

#ifndef NDEBUG
#include <jedi/thread.h>
#include <string.h>

#if defined(__tvm__)
#include <stdio.h>
#undef TRACE
#define TRACE(fmt, args...)		fprintf(stderr, fmt, ##args)
	//Since the trace window is closed, we have to print somewhere else
#endif

///////////////////////////////////////////////
EXTERN_C void* _JAPI _malloc(size_t sz);
EXTERN_C void* _JAPI _realloc(void* p, size_t sz);
EXTERN_C void  _JAPI _free(void* p);
EXTERN_C size_t _JAPI _memsize(void* p);

///////////////////////////////////////////////
struct TDbgAI {
	//Note: sizeof(TDbgAI) must be a multiple of sizeof(int)
	TDbgAI*		next;
	const char* flnm; //we don't duplicate since __FILE__ is const
	int			lnno;

	inline void Attach(TDbgAI** paclist, const char* fn, int ln) {
		next = *paclist;
		*paclist = this;

		flnm = fn;
		lnno = ln;
	}

	inline bool Detach(TDbgAI** paclist, const char* fn, int ln) {
		for(; *paclist; paclist = &(*paclist)->next)
			if(this == *paclist){
				*paclist = next;
				return true;
			}

		TRACE("%p is free-ed twice or invalid at %s, %d\n", this+1, fn, ln);
		return false;
	}

	inline bool IsExist(const TDbgAI* aclist) const {
		for(; aclist; aclist = aclist->next)
			if(this == aclist)
				return true; //found

		TRACE("wrong pointer %p\n", this+1);
		return false;
	}	
};

///////////////////////////////////////////////
static TDbgAI* s_allocList=0; //pointer to a list of allocated blocks
static CMutex s_allocCS; //mutex for accessing s_allocList

static void _STDCALL AddAI(TDbgAI* p, const char* flnm, int lnno)
{
	ASSERT(p && flnm);

	s_allocCS.Lock();
	p->Attach(&s_allocList, flnm, lnno);
	s_allocCS.Unlock();
}

static bool _STDCALL RemoveAI(TDbgAI* p, const char* flnm, int lnno)
{
    ASSERT(p);
	s_allocCS.Lock();
	bool bOk = p->Detach(&s_allocList, flnm, lnno);
	s_allocCS.Unlock();
	return bOk;
}

static bool _STDCALL IsExist(const TDbgAI* p)
{
	s_allocCS.Lock();
	bool bOk = p->IsExist(s_allocList);
	s_allocCS.Unlock();
	return bOk;
}

EXTERN_C void* _JAPI _malloc_dbg(size_t sz, const char* flnm, int lnno)
{
	//zero length is acceptable, see the document of malloc
	TDbgAI* newai = (TDbgAI*)_malloc(sz + sizeof(TDbgAI));
	if(!newai){
		TRACE("memory not enough for %u at %s, %d\n", sz, flnm, lnno);
		return 0;
	}

	AddAI(newai, flnm, lnno);
	return newai+1;
}

EXTERN_C void  _JAPI _free_dbg(void* p, const char* flnm, int lnno)
{
	if(p){
		TDbgAI* oldai = ((TDbgAI*)p)-1;
		if(RemoveAI(oldai, flnm, lnno))
			_free(oldai);
			//we cannot pass to free since free cannot protect itself
	}
}

EXTERN_C size_t _JAPI _memsize_dbg(void* p)
{
	if(!p || !IsExist(((TDbgAI*)p)-1))
		return 0;

	size_t sz = _memsize(((TDbgAI*)p)-1);
	if(sz < sizeof(TDbgAI))
		TRACE("wrong pointer %p, size %d\n", p, sz);
	return sz - sizeof(TDbgAI);
}

EXTERN_C void* _JAPI _calloc_dbg(size_t nm, size_t sz, const char* flnm, int lnno)
{
	void* p = _malloc_dbg(sz *= nm, flnm, lnno);
	if(p)
		memset(p, 0, sz);
	return p;
}

EXTERN_C void* _JAPI _realloc_dbg(void* p, size_t sz, const char* flnm, int lnno)
{
	if(!p)
		return _malloc_dbg(sz, flnm, lnno);

	if(!sz){ //see document of realloc
		_free_dbg(p, flnm, lnno);
		return 0;
	}

	TDbgAI* oldai = ((TDbgAI*)p)-1;

	//we have to remove first since the block might move

	if(!RemoveAI(oldai, flnm, lnno))
		return 0;

	TDbgAI* newai = (TDbgAI*)_realloc(oldai, sz + sizeof(TDbgAI));
	if(newai){
		AddAI(newai, flnm, lnno);
		return newai+1;
	}else{//it is ok to use oldai now, since it is unchanged in this case
		TRACE("memory not enough for %u at %s, %d\n", sz, flnm, lnno);
		AddAI(oldai, oldai->flnm, oldai->lnno);
		return 0;
	}
}

///////////////////////////////////////////////
void* operator new(size_t sz, const char* flnm, int lnno)
{
	void* p = _malloc_dbg(sz, flnm, lnno);
	if(p) return p;

	TRACE("memory not enough for %u at %s, %d\n", sz, flnm, lnno);
	return ::operator new(sz); //let C++ library to handle exception
}

void* operator new[](size_t sz, const char* flnm, int lnno)
{
	void* p = _malloc_dbg(sz, flnm, lnno);
	if(p) return p;

	TRACE("memory not enough for %u at %s, %d\n", sz, flnm, lnno);
	return ::operator new(sz); //let C++ library to handle exception
}

/*	Since we cannot pass extra arguments to delete,
	we have to use two static variables and two functions as follows to
	implement operator delete, see also ualloc.h
*/
static const char* s_flnm;
static int s_lnno;

EXTERN_C void _JAPI _msetdel(const char* flnm, int lnno)
{
	s_flnm = flnm;
	s_lnno = lnno;
}

void _CDECL operator delete(void* p)
{
	_free_dbg(p, s_flnm, s_lnno);
}

void operator delete[](void* p)
{
	_free_dbg(p, s_flnm, s_lnno);
}

EXTERN_C bool _JAPI mmCheck(void)
{
	int nCount = 0;
	s_allocCS.Lock();

	TDbgAI* p = s_allocList;
	if(p){
		TRACE("Memory leakage is detected\n");

		do{
			TRACE("...%d: %s line %d\n", ++nCount, p->flnm, p->lnno);
		}while((p=p->next)!=0);
	}

	s_allocCS.Unlock();
	return !nCount;
}
#endif //NDEBUG

#ifdef NDEBUG
EXTERN_C char* _JAPI __strdup(const char* p)
#else
EXTERN_C char* _JAPI __strdup_dbg(const char* p, const char* flnm, int lnno)
#endif
{
	char* dst = 0;
	if(p){
		int len = strlen(p)+1;
#ifdef NDEBUG
		dst = (char*)_MALLOC(len);
#else
		dst = (char*)_malloc_dbg(len, flnm, lnno);
#endif
		if(dst)
			memcpy(dst, p, len);
	}
	return dst;
}
