/*	mchunk1.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	8/16/0 07:43PM
	$Header: /cvsroot/jedi/mem/mchunk1.cxx,v 1.12 2000/09/05 06:38:33 tomyeh Exp $
	Purpose:	The memory chunk management, implemented on top of malloc
	Description:
		1. Codes are included only if OS_CHUNK_ON_MALLOC is defined
		2. To be really thread-safe, each TMemChunk must has a lock, too.
	
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/mem.h>
#include <jedi/thread.h>

#ifdef OS_CHUNK_ON_MALLOC
#if OS_RAM_CARD_NO!=1
#error The following supports only one card
#endif

#define MMF_MAGIC		0x6A9

static TMemChunk* s_chunkList = 0;

static bool _STDCALL IsValid(const TMemChunk* hmem)
{
	return hmem && hmem->flags.magic==MMF_MAGIC && hmem->data;
}

static void _JAPI Init(_hmem_t hmem, void* p)
{
	hmem->data		= p;
	hmem->cLocks	= 0;
	hmem->prev		= 0;
	hmem->next		= s_chunkList;
	if(s_chunkList)
		s_chunkList->prev = hmem;
	s_chunkList		= hmem;

	hmem->flags.SetZeros();
	hmem->flags.magic	= MMF_MAGIC;
	hmem->owner			= mtGetCurrentProcess();
}

EXTERN_C _hmem_t _JAPI mmAlloc(
size_t sz, unsigned /*cardId*/, bool bStorage, size_t hdrExtraBytes)
{
	void* p = _MALLOC(sz); //note: zero is acceptable
	if(!p)
		return 0;

	//sz==0 is acceptable
	TMemChunk* hmem = (TMemChunk*)_MALLOC(sizeof(TMemChunk) + hdrExtraBytes);
	if(!hmem){
		_FREE(p);
		return 0;
	}

	Init(hmem, p);
	hmem->flags.storageHeap = bStorage;
	return hmem;
}

EXTERN_C _hmem_t _JAPI mmAllocAssoc(void* p)
{
	TMemChunk* hmem = _MALLOC_T(TMemChunk);
	if(!hmem)
		return 0;

	Init(hmem, p);
	hmem->flags.associated = 1;
	return hmem;
}

EXTERN_C size_t _AAPI mmSize(_hmem_t hmem)
{
	if(!IsValid(hmem))
		return 0;

	ASSERT(!hmem->flags.associated);
	return _MSIZE(hmem->data);
}

EXTERN_C _err_t _AAPI mmResize(_hmem_t hmem, size_t sz)
{
	//Note: unlike C, sz==0 doesn't mean free
	if(!IsValid(hmem))
		return ERRM_INVALIDPARAM;

	ASSERT(!hmem->flags.associated);

	if(hmem->cLocks && sz>_MSIZE(hmem->data))
		return ERRM_CHUNKLOCKED;

	void* data = _REALLOC(hmem->data, sz);
	if(!data)
		return ERRM_NOTENOUGHSPACE;

	hmem->data = data;
	return 0;
}

static void _STDCALL Unlink(const TMemChunk* hmem)
{
	TMemChunk* p = hmem->prev;
	if(p){
		p->next = hmem->next;
	}else{
		ASSERT(hmem==s_chunkList);
		s_chunkList = hmem->next;
	}

	p = hmem->next;
	if(p) p->prev = hmem->prev;
}

static void _STDCALL Free(TMemChunk* hmem)
{
	if(!hmem->flags.associated)
		_FREE(hmem->data);

	(int&)(hmem->flags) = 0; //clean it up to detect error
	_FREE(hmem);
}

EXTERN_C _err_t _AAPI mmFree(_hmem_t hmem)
{
	if(!IsValid(hmem))
		return ERRM_INVALIDPARAM;

	Unlink(hmem);
	Free(hmem);
	return 0;
}

EXTERN_C void _JAPI mmFreeByOwner(unsigned /*cardId*/, __u32 owner)
{
	TMemChunk* freeList = 0;

	for(TMemChunk* h = s_chunkList; h; h = h->next)
		if(h->owner==owner){
			Unlink(h);
			h->prev = freeList; //put into the free list
			freeList = h;
		}

	//to save the time in critical section, we free them later
	while(freeList){
		TRACE("mmFreeByOwner: forgotten memory chunk, %p?\n", freeList);
		h = freeList;
		freeList = freeList->prev;

		Free(h);
	}
}

EXTERN_C _hmem_t _AAPI mmGetHandle(void* p)
{
	for(TMemChunk* h = s_chunkList; h; h = h->next)
		if(h->data == p)
			return h;
	return 0;
}

EXTERN_C void* _JAPI mmGetUnsafePtr(_hmem_t hmem)
{
	return IsValid(hmem) ? hmem->data: 0;
}

EXTERN_C void* _JAPI mmGetPtr(_hmem_t hmem, bool bAutoLock)
{
	return	mmGetLockCount(hmem) ? hmem->data: //it calls IsValid implicitly
			bAutoLock ? mmLock(hmem): 0;
}

EXTERN_C void* _AAPI mmLock(_hmem_t hmem)
{
	if(!IsValid(hmem))
		return 0;

	++hmem->cLocks;
	return hmem->data;
}

EXTERN_C _err_t _AAPI mmUnlock(_hmem_t hmem)
{
	if(!IsValid(hmem))
		return ERRM_INVALIDPARAM;

	if(hmem->cLocks)
		--hmem->cLocks;
	return 0;
}

EXTERN_C __u16 _AAPI mmGetLockCount(_hmem_t hmem)
{
	return IsValid(hmem) ? hmem->cLocks: (__u16)0;
}

EXTERN_C __u32 _AAPI mmGetOwner(_hmem_t hmem)
{
	return IsValid(hmem) ? hmem->owner: 0;
}

EXTERN_C _err_t _AAPI mmSetOwner(_hmem_t hmem, __u32 owner)
{
	if(!IsValid(hmem))
		return ERRM_INVALIDPARAM;

	hmem->owner = owner;
	return 0;
}

EXTERN_C bool _JAPI mmIsStorage(_hmem_t hmem)
{
	return IsValid(hmem) ? hmem->flags.storageHeap: false;
}

#endif //OS_CHUNK_ON_MALLOC
