/*	mrbtree.cxx

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:
	Create Date:	2000/9/1 01:52PM
	$Header: /cvsroot/jedi/lib/mrbtree.cxx,v 1.1 2000/10/05 07:22:15 henrichen Exp $
	Purpose:	
	Description:	Implementation of RBTree algorithem

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <lib/mrbtree.h>

//////////////////////////////////////////////////////////////////
void CMRBNode::Set(unsigned mask, bool bSet)
{
	if (bSet) m_flags|= mask;
	else m_flags&= ~mask;
}

//////////////////////////////////////////////////////////////////
void CMRBTree::LeftRotate(CMRBNode *pa, CMRBNode *x)
{
	CMRBNode *y= x->m_right;
	ASSERT(y);
	
	x->m_right= y->m_left;
	
	if (!pa)
		m_root= y;
	else if (x == pa->m_left)
		pa->m_left= y;
	else 
		pa->m_right= y;

	y->m_left= x;
}

void CMRBTree::RightRotate(CMRBNode *pa, CMRBNode *y)
{
	CMRBNode *x= y->m_left;
	ASSERT(x);
	
	y->m_left= x->m_right;
	
	if (!pa)
		m_root= x;
	else if (y == pa->m_left)
		pa->m_left= x;
	else
		pa->m_right= x;
		
	x->m_right= y;
}

void CMRBTree::Attach(CMRBNode *z, void *key)
{
	CMRBNode *nextX;
	Insert(m_root, z, key, NULL, NULL, NULL, &nextX);
	m_root->SetBlack();
}

//recursive function
void CMRBTree::Insert(CMRBNode *root, CMRBNode *z, void *key,
	CMRBNode *ggpa, CMRBNode *gpa, CMRBNode *pa, CMRBNode **nextX)
{
	bool bLeft= false;
	CMRBNode *x= root;
	if (root) {
		//locate insertion point
		if (Comp(key, root) <= 0) {	// to m_left
			root= root->m_left;
			bLeft= true;
		}
		else {
			root= root->m_right;
		}
	}
	
	if (!root) {
		if (!x) {	//empty tree
			m_root= z;
			return;
		}
		else {
			if (bLeft) {
				x->m_left= z;
			}
			else {
				x->m_right= z;
			}
			
			z->SetRed();
			
			//start RB-tree adjustment
			InsertFix(gpa, pa, x, z, nextX);
		}
	}
	else 
		Insert(root, z, key, gpa, pa, x, nextX);
	
	if (x == *nextX) 
		InsertFix(ggpa, gpa, pa, x, nextX);
}

void CMRBTree::InsertFix(
	CMRBNode *ggpa, CMRBNode *gpa, CMRBNode *pa, CMRBNode *x, CMRBNode **nextX)
{
	*nextX= 0;
	if (x != m_root && pa->IsRed()) {
		void (CMRBTree::*Rotate1)(CMRBNode *pa, CMRBNode *x);
		void (CMRBTree::*Rotate2)(CMRBNode *pa, CMRBNode *x);
		CMRBNode *y, *npa;
		if (pa == gpa->m_left) {
			y= gpa->m_right;
			Rotate1= &CMRBTree::LeftRotate;
			Rotate2= &CMRBTree::RightRotate;
			npa= pa->m_right;
		}
		else {
			y= gpa->m_left;
			Rotate1= &CMRBTree::RightRotate;
			Rotate2= &CMRBTree::LeftRotate;
			npa= pa->m_left;
		}
		
		if (y && y->IsRed()) {	//case 1
			y->SetBlack();
			pa->SetBlack();
			gpa->SetRed();
			*nextX= gpa;
		}
		else {				
			if (x == npa) {	//case 2
				(this->*Rotate1)(gpa, pa);
				pa= x;
			}
			pa->SetBlack();	//case 3
			gpa->SetRed();
			(this->*Rotate2)(ggpa, gpa);
		}
	}
}

CMRBNode *CMRBTree::Detach(void *key)
{
	CMRBNode *nextX;
	return Delete(m_root, key, NULL, NULL, &nextX);
}

CMRBNode *CMRBTree::Delete(
	CMRBNode *root, void *key, CMRBNode *gpa, CMRBNode *pa, CMRBNode **nextX)
{
	//cannot locate the key
	if (!root)	return NULL;
	
	CMRBNode *z;
	int compret= key ? Comp(key, root) : (root->m_left ? -1 : 0);
	
	if (compret == 0) {	//found the node
		if (!root->m_left || !root->m_right) {	//case a or b
			CMRBNode *x= root->m_left ? root->m_left : root->m_right;
			*nextX= NULL;
			
			bool bLeft= false;
			if (!pa) {
				m_root= x;
			}
			else {
				if (root == pa->m_left) {
					pa->m_left= x;
					bLeft= true;
				}
				else {
					pa->m_right= x;
				}
			}
			
			if (root->IsBlack())
				DeleteFix(gpa, pa, x, nextX, bLeft);

			return root;
		}

		//case c, find the successor(enforce a successor finding, set key to zero)
		z= Delete(root->m_right, NULL, pa, root, nextX);
		Copy(root, z);
	}
	else {
		z= Delete(compret < 0 ? root->m_left : root->m_right, key, pa, root, nextX);
	}
	
	if (root == *nextX)	 {
		bool bLeft= !pa || root == pa->m_left;
		DeleteFix(gpa, pa, root, nextX, bLeft);
	}
	
	return z;
}

void CMRBTree::DeleteFix(CMRBNode *gpa, CMRBNode *pa, CMRBNode *x, CMRBNode **nextX, bool bLeft)
{
	*nextX= NULL;
	if (x == m_root || (x && x->IsRed())) {
		if (x) x->SetBlack();
		return ;
	}
	
	void (CMRBTree::*Rotate1)(CMRBNode *pa, CMRBNode *x);
	void (CMRBTree::*Rotate2)(CMRBNode *pa, CMRBNode *x);
	
	CMRBNode *w, *y, *z;
	if (bLeft) {
		w= pa->m_right;
		Rotate1= &CMRBTree::LeftRotate;
		Rotate2= &CMRBTree::RightRotate;
		y= w ? w->m_right : NULL;
		z= w ? w->m_left : NULL;
	}
	else {
		w= pa->m_left;
		Rotate1= &CMRBTree::RightRotate;
		Rotate2= &CMRBTree::LeftRotate;
		y= w ? w->m_left : NULL;
		z= w ? w->m_right : NULL;
	}
	
	if (w && w->IsRed()) {					//case 1
		w->SetBlack();
		pa->SetRed();
		(this->*Rotate1)(gpa, pa);
		gpa= w;
		if (bLeft) {
			w= pa->m_right;
			y= w ? w->m_right : NULL;
			z= w ? w->m_left : NULL;
		}
		else {
			w= pa->m_left;
			y= w ? w->m_left : NULL;
			z= w ? w->m_right : NULL;
		}
	}
	
	if ((!z || z->IsBlack()) && (!y || y->IsBlack())) {	//case 2
		if (w) w->SetRed();
		*nextX= pa;
	}
	else {
		if (!y || y->IsBlack()) {				//case 3
			z->SetBlack();
			w->SetRed();
			(this->*Rotate2)(pa, w);
			y= w;
			w= z;
		}
		w->SetColor(pa->IsRed());		//case 4
		pa->SetBlack();
		y->SetBlack();
		(this->*Rotate1)(gpa, pa);
		m_root->SetBlack();
	}
}

CMRBNode *CMRBTree::Search(void *key)
{
	CMRBNode *n= m_root;
	while(n) {
		int ret= Comp(key, n);
		if (ret == 0)
			return n;
		else if (ret < 0)
			n= n->m_left;
		else
			n= n->m_right;
	}
	return n;
}


//////////////////////////////////////////////////////////////////
// For debug only
#ifndef NDEBUG
bool CMRBTree::Check(void)
{
	unsigned bn;
	return CheckNode(NULL, m_root, &bn);
}

bool CMRBTree::CheckNode(CMRBNode *pa, CMRBNode *x, unsigned *bn)
{
	if (!x) { *bn= 1; return true;}
	
	//check rule 3, if my parent is red, I must be black
	if (x->IsRed()) {
		if(!pa) {
			TRACE("Violate RB-tree assumption, "
					"root(0x%08x) must be alway black!\n", x);
			return false;
		}
		else if (pa->IsRed()) {
			TRACE("Violate rule 3, red parent(0x%08x) has red child(0x%08x)\n", pa, x);
			return false;
		}
	}

	unsigned lb= 0;	
	if (!CheckNode(x, x->m_left, &lb)) return false;
	
	unsigned rb= 0;
	if (!CheckNode(x, x->m_right, &rb)) return false;

	//check rule 4, either path from this node should have same number of black
	if (lb != rb) {
		TRACE("Violate rule 4, different number of black nodes on the paths(0x%08x), "
				"left=%d, right=%d\n", x, lb, rb);
		return false;
	}
	
	*bn= lb + (x->IsBlack() ? 1 : 0);
	return true;
}

#endif	//!NDEBUG
