/*	thread.h

{{IS_NOTE

	Authors:	Joel Sherrill <joel@OARcorp.com>
	Contributors:
	Create Date:	12/13/2000
	$Header$
	Purpose:	Thread functions of RTEMS
	Description:	This uses the RTEMS POSIX API as much as possible.
		But some of these services do not map cleanly to POSIX.  For
		those, we are using the Classic API.

}}IS_NOTE

Original File:
	Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.
RTEMS Port:
	Copyright (C) 2000 OAR Corporation.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_os_thread_H
#define _is_os_thread_H

////////////////////////////////////////////////////////
typedef pthread_t	_handle_t;
typedef pthread_mutex_t _mutex_t;
typedef struct {
  bool            bBroadcast;  
  pthread_cond_t  cond;
  _mutex_t       *mutex;
} _cond_t;

#define OSTHD_RET_T		void *
#define OSTHD_API		_CDECL
#define OSTHD_API_O		_CDECL_O
#define OSTHD_API_I		_CDECL_I
typedef OSTHD_RET_T OSTHD_API_O(OSTHD_API_I*_osthread_f)(void*);

////////////////////////////////////////////////////////
EXTERN_C_BEGIN

#include <pthread.h>
#include <sched.h>

inline _handle_t _JAPI mtOsCreateThread(
	_osthread_f entry, void* data, unsigned stackSize, __u32* pThdId)
{
  int rc;
  pthread_attr_t attr;

  // Initialize the attributes to the defaults.  This sets scheduling
  // parameters as inherited and the detachstate to joinable.
  rc = pthread_attr_init( &attr );
  if ( !rc )
    return 0;

  // Override the stack size.  RTEMS ensures it meets minimum requirements.
  rc = pthread_attr_setstacksize( &attr, stackSize );
  if ( !rc )
    return 0;
  
  // Actually create and start the thread.
  rc = pthread_create( pThdId, NULL, entry, data );
  if ( !rc )
    return 0;

  return (_handle_t) pThdId;
}

inline void _JAPI mtOsDetachThread(_handle_t hThd)
{
  (void) pthread_detach( hThd );
}

inline __u32 _JAPI mtOsGetCurrentThreadId(void)
{
  return pthread_self();
}

inline void _JAPI mtOsExitThread(void)
{
  pthread_exit(NULL);
}

// POSIX API does not include a service to delete another thread.  So
// we use the RTEMS Classic API service which can do this. :(
inline void _JAPI mtOsKillThread(_handle_t hThd)
{
  (void)rtems_task_delete(hThd);
}

// XXX do they really mean clock ticks or is ticks in some real unit?
// POSIX delays in time units.  RTEMS Classic API can do this in ticks.
inline void _JAPI mtAnakinSleep(unsigned ticks)
{
  rtems_task_wake_after(ticks);
}

inline void _JAPI mtAnakinYield(void)
{
  sched_yield();
}

// Wait for a thread to complete; return true if its done
// POSIX 1003.1b pthread_join() does not have a timeout capability.
//
// XXX Consider adding POSIX extension routine for pthread_join with timeout?
inline bool _JAPI mtOsWaitThread(_handle_t hThd, unsigned toTicks)
{
  int rc;
  
  rc = pthread_join( hThd, NULL );
  if ( !rc )
    return 1;
  return 0;          // failed to join
}

inline bool _JAPI mtOsSetPriority(_handle_t hThd, int priority)
{
  int rc;
  struct sched_param param;

  param.sched_priority = priority;
  rc = pthread_getschedparam( hThd, NULL, &param );
  if ( !rc )
    return 1;
  return 0;          // failed to set priority
}

// XXX check for error status
inline int _JAPI mtOsGetPriority(_handle_t hThd)
{
  int rc;
  struct sched_param param;

  rc = pthread_getschedparam( hThd, NULL, &param );
  if ( !rc )
    return param.sched_priority;

  return -1;    // failed to get priority 
}

inline void mtInitMutex(_mutex_t* mutex)
{
  int rc;

  ASSERT(mutex);
  rc = pthread_mutex_init( mutex, NULL );
  ASSERT(!rc);
}

inline void mtDestroyMutex(_mutex_t* mutex)
{
  int rc;

  ASSERT(mutex);
  rc = pthread_mutex_destroy( mutex );
  ASSERT(!rc);
}

inline void mtLockMutex(_mutex_t* mutex)
{
  int rc;

  ASSERT(mutex);
  rc = pthread_mutex_lock(mutex);
  ASSERT(!rc);
}

inline void mtUnlockMutex(_mutex_t* mutex)
{
  int rc;

  ASSERT(mutex);
  rc = pthread_mutex_unlock(mutex);
  ASSERT(!rc);
}

// Condition variable in POSIX are a bit different.  The following 
// porting note is from Tom M. Yeh <tomyeh@infoshock.com>.
// 
//  Cond is a little different: whether a cond will broadcast all or signal
//  one of pending threads is determined at the creation, mtInitCond with the
//  bBroadcast argument. On the other hands, pThreads supports
//  pthread_cond_signal and pthread_cond_broadcast at run time. The
//  incompatibility is due to that it is too costly to implement it in Windows
//  32API. Thus, to port to pThread-like OS, you have to maintain a data
//  member to know whether to broadcast or signal when SetCond is called.

inline void _JAPI mtInitCond(_cond_t* cond, _mutex_t* mutex, bool bBroadcast)
{
  int rc;

  ASSERT(cond);
  ASSERT(mutex);
  cond->mutex      = mutex;
  cond->bBroadcast = bBroadcast;
  rc = pthread_cond_init(&cond->cond, NULL);
  ASSERT(!rc);
}

inline void _JAPI mtDestroyCond(_cond_t* cond)
{
  int rc;

  ASSERT(cond);
  rc = pthread_cond_destroy(&cond->cond);
  ASSERT(!rc);
}

inline void _JAPI mtSetCond(_cond_t* cond)
{
  int rc;

  ASSERT(cond);
  if (cond->bBroadcast) rc = pthread_cond_broadcast(&cond->cond);
  else                  rc = pthread_cond_signal(&cond->cond);
  ASSERT(!rc);
}

inline bool _AAPI mtWaitCond(_cond_t* cond, unsigned toTicks)
{
  int rc;
  struct timespec abstime;
  void _POSIX_Interval_to_timespec(unsigned, struct timespec *);

  // XXX toTicks is relative and 0 indicates NOWAIT
  ASSERT(cond);
  _POSIX_Interval_to_timespec(toTicks, &abstime);
  rc = pthread_cond_timedwait(&cond->cond, cond->mutex, &abstime);
  //  XXX add cond_timedwait to RTEMS

  ASSERT(!rc);
  if (!rc)
    return 1;
  return 0;
}

//XXX start here

// Thread Local Storage
// RTEMS implements this as a block of memory associated with a 
//   per task variable.
//
//  Issues Left:  XXX
//    
// XXX where to declare __RTEMS_TLS
// XXX do we need to use the destructor?

typedef struct {
  void *memory;
  bool  in_use;
} __rtems_tls_entry_t;

typedef struct {
  unsigned             count; 
  __rtems_tls_entry_t *Vars;
}  __rtems_tls_t;

extern __rtems_tls_t __RTEMS_TLS;

#ifdef __cplusplus
extern "C" {
#endif
extern void *rtems_realloc_wrapper(void *ptr, size_t size);
#ifdef __cplusplus
} 
#endif

#include <stdlib.h>

inline unsigned _JAPI mtAllocTls(void)
{
  unsigned            i;

  // Since count is initialized to 0, this works out to be automatic.

  // See if any slots are open for reuse
  for ( i=0 ; i<__RTEMS_TLS.count ; i++ ) {
    if ( !__RTEMS_TLS.Vars[i].in_use ) {
      __RTEMS_TLS.Vars[i].in_use = TRUE;
      __RTEMS_TLS.Vars[i].memory = NULL;
      return i;
    }
  }

  // Add a slot
  i = __RTEMS_TLS.count;
  __RTEMS_TLS.count++;
  __RTEMS_TLS.Vars = (__rtems_tls_entry_t *)realloc( __RTEMS_TLS.Vars,
          sizeof(__rtems_tls_entry_t) * __RTEMS_TLS.count );

  __RTEMS_TLS.Vars[i].in_use = TRUE;
  __RTEMS_TLS.Vars[i].memory = NULL;
  return i;
}

inline BOOL _JAPI mtFreeTls(unsigned index)
{
  if ( index >= __RTEMS_TLS.count )
    return 0;

  __RTEMS_TLS.Vars[index].in_use = FALSE;
  return 1;
}

inline void* _JAPI mtGetTls(unsigned index)
{
  if ( index >= __RTEMS_TLS.count )
    return NULL;

  if ( !__RTEMS_TLS.Vars[index].in_use )
    return NULL;

  return __RTEMS_TLS.Vars[index].memory;
}

inline BOOL _JAPI mtSetTls(unsigned index, void* val)
{
  if ( index >= __RTEMS_TLS.count )
    return 0;

  if ( !__RTEMS_TLS.Vars[index].in_use )
    return 0;

  __RTEMS_TLS.Vars[index].memory = val;
  return 1;
}

// called when the last process is exiting
// It does not really depend on OS but different OS might process
// it differently for better UI.

inline void _JAPI mtOsNotifyNoApp(void)
{
  // In RTEMS, we are probably powering off.
  return;
}

EXTERN_C_END

#endif //_is_os_thread_H
