/*	lcllink.h

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	8/30/0 06:53PM
	$Header: /cvsroot/jedi/include/lib/lcllink.h,v 1.4 2000/09/26 10:09:26 tomyeh Exp $
	Purpose:	TAppLink and TThdLink
	Description:
		TAppLink and TThdLink are used as a template to make a struct
		a linked list starting from a member of TAppLocal and TThdLocal.
		Unlike TOwnLink, (1) each node of TAppLink/TThdLocal doesn't store
		the thread Id, (2) each app/thread can only see object it created,
		(3) there is no easy way to change the owner of a node.
	Use:
		1. Add a member to TAppLocal or TThdLocal in jedi/thread.h,
		depending whether the list is local to an application or a thread.
		The following example assumes it is local to a thread.
			struct TThdLocal {
				...
				void*	listMyEx;
		2. In header
			struct TMy {...};
			DECL_THDLINK(TMyEx, TMy, listMyEx)
		3. In source code
			//convert
			TMy* q = ToTMy(p); //assumes TMyEx* p
			TMyEx* p = ToTMyEx(q);
			//create
			TMyEx* p = _MALLOC_T(TMyEx);
			if(p){
				...
				p->Link();
			}
			return ToMy(p); //usually store TMy* instead of TMyEx
			//destroy
			ToTMyEx(p)->Unlink();
			_FREE(ToTMyEx(p));
			//app cleanup
			TMyEx::Apply(myDestroy); //NOTE: myDestroy operators on TMy!!
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_lib_lcllink_H
#define _is_lib_lcllink_H

#include <jedi/thread.h>

//////////////////////////////////////////////////////////
struct TLclLinkNode {
	TLclLinkNode*	_ll_next;
};

EXTERN_C void  _JAPI _tlLink(TLclLinkNode* obj, void* list, int listOfs);
EXTERN_C void  _JAPI _tlUnlink(TLclLinkNode* obj, void* list, int listOfs);
EXTERN_C void  _JAPI _tlApply(void (*pf)(void*, void*), void* list, int listOfs);
EXTERN_C void* _JAPI _tlFind(bool (*pf)(void*, int), int val, void* list, int listOfs);
EXTERN_C void  _JAPI _tlMoveToTop(TLclLinkNode *obj, void *list, int listOfs);
#ifndef NDEBUG
EXTERN_C void _JAPI _tlDump(const char *msg, void* list, int listOfs);
	//If nothing in the list, nothing is shown
#endif

//////////////////////////////////////////////////////////
template<class T, const int listOfs>
struct TAppLink : public TLclLinkNode, public T {

	inline void Link(TAppLocal* app=0) {_tlLink(this, app ? app: GetAppLocal(), listOfs);}
	inline void Unlink(TAppLocal* app=0) {_tlUnlink(this, app ? app: GetAppLocal(), listOfs);}
	inline void MoveToTop() {_tlMoveToTop(this, GetAppLocal(), listOfs);}
		//Move the object that matches val, i.e., pf(obj, val) returns true, to topmost

	static inline void Apply(void (*pf)(T*, TAppLocal*), TAppLocal* app=0)
		{_tlApply((void (*)(void*,void*))pf, app ? app: GetAppLocal(), listOfs);}
		//If app==0, GetAppLocal is assumed

#ifndef NDEBUG
	static inline void Dump(const char *msg) {_tlDump(msg, GetAppLocal(), listOfs);}
#endif

	static inline T* Find(bool (*pf)(T*, int), int val)
	{return (T*)_tlFind((bool (*)(void*, int))pf, val, GetAppLocal(), listOfs);}
		//Find the object that matches val, i.e., pf(obj, val) returns true

	TAppLink* Next() const {return (TAppLink*)_ll_next;}
};

template<class T, const int listOfs>
struct TThdLink : public TLclLinkNode, public T {

	inline void Link(TThdLocal* thd=0) {_tlLink(this, thd ? thd: GetThdLocal(), listOfs);}
	inline void Unlink(TThdLocal* thd=0) {_tlUnlink(this, thd ? thd: GetThdLocal(), listOfs);}
	inline bool MoveToTop() {return _tlMoveToTop(this, GetThdLocal(), listOfs);}

	static inline void Apply(void (*pf)(T*, TThdLocal*), TThdLocal* thd=0)
		{_tlApply((void (*)(void*,void*))pf, thd ? thd: GetThdLocal(), listOfs);}
		//If thd==0, GetThdLocal is assumed

#ifndef NDEBUG
	static inline void Dump(const char *msg) {_tlDump(msg, GetThdLocal(), listOfs);}
#endif

	static inline T* Find(bool (*pf)(T*, int), int val)
	{return (T*)_tlFind((bool (*)(void*, int))pf, val, GetThdLocal(), listOfs);}
		//Find the object that matches val, i.e., pf(obj, val) returns true

	TThdLink* Next() const {return (TThdLink*)_ll_next;}
};

#define DECL_APPLINK(TMyEx, TMy, name) \
	typedef TAppLink<TMy, OffsetOf(TAppLocal, name)> TMyEx; \
	inline TMyEx* To##TMyEx(TMy* p) {return (TMyEx*)(((char*)p) - sizeof(TLclLinkNode));} \
	inline TMy* To##TMy(TMyEx* p) {return (TMy*)(((char*)p) + sizeof(TLclLinkNode));}

#define DECL_THDLINK(TMyEx, TMy, name) \
	typedef TThdLink<TMy, OffsetOf(TThdLocal, name)> TMyEx; \
	inline TMyEx* To##TMyEx(TMy* p) {return (TMyEx*)(((char*)p) - sizeof(TLclLinkNode));} \
	inline TMy* To##TMy(TMyEx* p) {return (TMy*)(((char*)p) + sizeof(TLclLinkNode));}

#ifdef _MSC_VER
#if _MSC_VER <= 1200
#pragma warning(disable: 4042)
	//a bug of MSC 6.0: false alarm when we use DECL_LCLLINK
#endif
#endif
#endif //_is_lib_lcllink_H
