/*	palmfile.h

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:
	Create Date:	2000/8/1 08:42PM
	$Header: /cvsroot/jedi/include/jedi/palmfile.h,v 1.4 2000/10/03 10:37:30 henrichen Exp $
	Purpose:	
	Description:	
		1. Palm's PRC/PDB file related data structure
		2. All info is stored in big-endian format and packed.
		3. Refer http:://web.mit.edu/tytso/www/pilot/prc-format.htm
		4. Refer "Palm file format specification" from Palm computing
		5. Refer CodeWarrior's PalmOS_Startup.c

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_jedi_palmfile_H
#define _is_jedi_palmfile_H

#pragma pack(push, 1)

//DB record for .PDB file
struct TPdbEntry {
	__u32	dataID;	//offset(from file top) to the raw data
	__u32	_JDK_BF2(
			attr: 8, 		//record attributes(high nibble)+category(low nibble)
			unique: 24);	//unique Record Id.
};

//The following data structure must consistent with TPdbEntry. 
//The only different is that they don't have dataID field.
struct TPdbChunk {
	__u32	_JDK_BF2(
		attr: 8, 			//record attributes(high nibble)+category(low nibble)
		unique: 24);		//unique Record Id.
};

//TPdbEntry.attr
#define	DMRECATTR_DELETE	0x80	//delete this record next sync
#define	DMRECATTR_DIRTY		0x40	//archive this record next sync
#define	DMRECATTR_BUSY		0x20	//record currently in use
#define	DMRECATTR_SECRET	0x10	//a password protected record
#define	DMRECCAT_MASK		0x0f	//category mask(low nibble)

//Resource record for .PRC file
struct TPrcEntry {
	__u32	type;	//resource type: TBmp, tAIB, etc.
	__u16	id;			//resource id
	__u32	dataID;		//offset(from file top) to the raw resource data
};

//The following data structure must consistent with TPrcEntry. 
//The only different is that they don't have dataID field.
struct TPrcChunk {
	__u32	type;	//resource type: TBmp, tAIB, etc.
	__u16	id;			//resource id
};
	
//Record list header
struct TEntryList {
	__u32	nextRecordListID;	//offset(from file top) to the next resource list header, 0 to terminate
	__u16	numRecords;			//number of resource record in this list
};

struct TRecordList : public TEntryList {
	__u16	startRecord;		//start of the record array.
								//if (numRecords == 0) { startRecord is the place holder }
								//else { TPrcEntry(for prc file) or TPdbEntry(for pdb file)
								//start from this place }
};

//Basic AppInfo Header (category info)
struct TCategoryInfo {
	__u16	renamedCategories;
	char	categoryLabels[16][16];
	__u8	categoryUniqueIDs[16];
	__u8	lastUniqID;
	__u8	reserved;
};

//PDB/PRC file header
struct TDbHeader {
	__u8	name[32];			//prc/pdb file name
	__u16	attributes;			//prc/pdb file attribute
	__u16	version;			//prc/pdb file version
	__u32	creationDate;		//creation date (seconds since 1/1/1904)
	__u32	modificationDate;	//last modification date (seconds since 1/1/1904)
	__u32	lastBackupDate;		//last backup date (seconds since 1/1/1904)
	__u32	modificationNumber;	//times the prc/pdb file is modified
	__u32	appInfoID;			//offset(from file top) to the application info block (usually 0)
	__u32	sortInfoID;			//offset(from file top) to the sort info black (usually 0)
	__u8	type[4];			//type identifier. (usually 'appl' for prc)
	__u8	creator[4];			//creator identifier.(registered creator name)
	__u32	uniqueIDSeed;		//used to generate unique id on the device. (usually 0)
	TEntryList	recordList;		//A record list associated with this prc/pdb file.
};
//TDbHeader.attributes
#define	DMHDRATTR_RESDB				0x0001	//A resource db
#define	DMHDRATTR_READONLY			0x0002	//Read only db
#define	DMHDRATTR_APPINFODIRTY		0x0004	//Set if AppInfo block is dirty
#define	DMHDRATTR_BACKUP			0x0008	//Set if should backup the db to PC
#define	DMHDRATTR_OKTOINSTALLNEWER	0x0010	//It's ok to install a newer version of the db with a different name if it is opened.
#define	DMHDRATTR_RESETAFTERINSTALL	0x0020	//Reset the device after the db is installed
#define	DMHDRATTR_COPYPREVENTION	0x0040	//The db cannot be copied to
#define	DMHDRATTR_STREAM			0x0080	//This is a stream db
#define	DMHDRATTR_HIDDEN			0x0100	//The db should be hidden from view
#define	DMHDRATTR_LAUNCHABLEDATA	0x0200	//App(with same creator) using sysAppLaunchCmdOpenNamedDB to open this data db
#define	DMHDRATTR_OPEN				0x8000	//The db is currently opened

//Data#0 Resource
// __u32	offset;				//4+n+m, offset(from data resource header) to code1 xfer
// __u8		compressedGlobal[n];//Compressed Global data initilizer (3 blocks)
// __u8		compressedData0[m];	//Comprsssed data#0 xfer
// __u8		compressedCode1[?];	//Compressed code#1 xfer

//One block of Compressed Global Data.
// __s32	a5Offset;			//offset(from A5) to where decompressed data should write to
// __u8		compressedStream[n];//compressed stream
// __u8		0;					//end of block

//Global data compression format
//	byte (0,1,2,3)		meaning
//---------------------------------------------------------
//	(b0 & 0x80) != 0	(b0 & 0x7f)+1 of following raw data bytes
//	(b0 & 0x40) != 0	(b0 & 0x3f)+1 of 0x00
//	(b0 & 0x20) != 0	(b0 & 0x1f)+2 of repeating following raw data bytes
//	(b0 & 0x10) != 0	(b0 & 0x0f)+1 of 0xff
//	b0 == 0x01 			0x00 0x00 0x00 0x00 0xff 0xff b1 b2 
//	b0 == 0x02			0x00 0x00 0x00 0x00 0xff b1 b2 b3 
//	b0 == 0x03			0xA9 0xF0 0x00 0x00 b1 b2 0x00 b3 
//	b0 == 0x04			0xA9 0xF0 0x00 b1 b2 b3 0x00 b4 
//	others				// end of compressed data 

//**Code#0 & Code#1 are special code segment, which has different format from code#2 and after.**

//Jump Entry in a code segment, note that the entries are place at (A5+TCodeSeg.jtlOffset)
//The number of entries is in TCodeSeg.numEntries
//You have to relocate the jumptable's jumpAddress to the segment, too.
struct TJumpEntry {
	__s16	m68kJumpInstruction;	//m68k's jump machine code
	__s32	jumpAddress;			//absolute or relative address of the routine
};

//Code segment format (Code#2 and after)
struct TCodeSeg {
	__s16	jtsOffset;	//offset(from A5) to where jump table was placed(short version)
	__s16	numEntries;	//number of jump entries in this segment
	__s32	jtlOffset;	//offset(from A5) to where jump table was place(long version)
	__s32	xrefOffset;	//offset(from segment top) to compressed xfer data of this code resource
//	__u8	code[1];	//code start from here
};

//There are three blocks of compressed xfer data in a code segment for muticode segment ap,
//first is for data(base on A5), second is related to code#1(base on code#1), third is for 
//self(base on the code segment itself).

// One block of compressed xfer data
//	__u32	numXref;	//number of relocation entries
//	__u8	compressedXref;

// Compressed Xfer data format (each decompressed delta(n) is related to the previous offset)
//	byte (0,1,2,3)		meaning
//---------------------------------------------------------
//	(b_0 & 0x80) !=0	n= (b_0 & 0x7f) << 1;	// 8-bit signed delta to previous xfer offset
//	(b_0 & 0x40) != 0	n= (((b_0 & 0x3f) << 8) + b_1) << 1;	//15-bits delta
//	(b_0 & 0x40) == 0	n= (((b_0 & 0x3f) << 24) + (b_1 << 16) + (b_2 << 8) + b_3) << 1; //31-bits delta

// Code#0 describe the code segment in the memory
//
//                 +-------------------+
//                /|                   |
//               / |  Jump table       |
//              |  +-------------------+  
//A5 world ---> |  |  Appl. Params     |
//              |  +-------------------+  <--- A5 pointer
//               \ |                   |
//                \|  Appl. Globals    |
//       ^         +-------------------+  <--- Stack base
//       |         |                   |
//      big        |     Stack         |
//    address      |      vvv          |
//       ^         +-------------------+  <--- Stack pointer
//       |               ....
//       |         +-------------------+
//     small       |                   |
//    address      |      ^^^          |
//       |         |      Heap         |
//       |         +-------------------+

// However, it looks only the first 4 bytes of "Appl. Params" area is used by the Palm OS.
// That is, *((void**)A5)= &SysAppInfo;
// SysAppInfo is a data structure used to store the control data of each launched 
// application. See <Core/System/SystemMgr.h> of Palm SDK 3.5.

//The jump table is typically used to transfer function control among segments
//If the destination segment is not loaded already, the machine code inside
//the jump table will load that segment and rearrange the jump table contents to
//loaded segment style. 
//*** However, Palm don't use this mechanism ***
//		__u16	funcOffset;
//		m68kLongJumpToAnotherSegment;	//this occupy 6 bytes

struct TMacJumpTable {
	__u16	funcOffset;			//offset (from the segment top) of this routine
	__u16	m68kPush;			//m68K push machine code
	__u16	segNo;				//segment number (arg of the push machine code)
	__u16	loadSegTrap;		//load segment trap
};

//Code#0 Resource: Code#0 is used by loader to setup appropriate control block
// __u32	a5Above;			//size above A5 (jump table + parameters)
// __u32	globalSize;			//size of application globals
// __u32	jtableSize;			//size of jump table
// __u32	a5Offset;			//A5 offset(from A5) to jump table
// TJumpTable	jtEntries[n];		//jump table entry

//Code#1 Resource: It contains machine code only. It's data xfer and code xfer data 
// is stored in Data#0 while the xfer data of Code#2 and after is stored in the code
// segment itself

//Bitmap compression
//	__u16	size;				//n, size of the data after compression
//	__u8	compressedData[n];

//Bitmap scanline compression format
//	__u8	leadmap;			
//		Each leadmap(8-bits) represent eight bytes of the scanline. From msb to lsb
//		if (bit set) { use the byte that follow the leadmap to represent the byte
//		on the scanline}
//		else {use the associated byte of the previous scanline.}
//
//		if the bitmap's rowbytes is larger than 8 bytes (because each leadmap 
//		can only represent 8 bytes), another leadmap and follow-bytes is added
//		in.
//		

#pragma pack(pop)

//////////////////////////////////////////////////////////////////
// class for original palm's PRC/PDB file
class CPalmFile : public CJediFile {
public:
	bool Open(const char *name, int mode);

	unsigned long GetFirstRecordID(void);
	size_t GetAppInfoSize();
	size_t GetSortInfoSize();
	unsigned long ReadPrcChunk(TPrcChunk *pc, unsigned long off);
	unsigned long ReadPdbChunk(TPdbChunk *pc, unsigned long off);

	//data
	TDbHeader m_hdr;
};

#ifdef __LITTLE_ENDIAN__
inline TEntryList* __cpu_to_be_elist(TEntryList *elist)
{
	elist->nextRecordListID= __cpu_to_be32(elist->nextRecordListID);
	elist->numRecords= __cpu_to_be16(elist->numRecords);
	return elist;
}

EXTERN_C TDbHeader* _STDCALL __cpu_to_be_dbhdr(TDbHeader *dbhdr);

inline TDbHeader * __be_dbhdr_to_cpu(TDbHeader *dbhdr)
{
	return __cpu_to_be_dbhdr(dbhdr);
}

inline TPrcChunk* __be_prc_to_cpu(TPrcChunk *prc)
{
	//prc->type				//resource type: TBmp, tAIB, etc.
	prc->id= __be16_to_cpu(prc->id);	//resource id
	return prc;
}

inline TPdbChunk* __be_pdb_to_cpu(TPdbChunk *pdb)
{
	Swap(pdb->uniqueId[0], pdb->uniqueId[2]);
	return pdb;
}

#else	//!__LITTLE_ENDIAN__
	#define	__cpu_to_be_dbhdr(x)	void(0)
	#define __be_dbhdr_to_cpu(x)	void(0)
	#define	__be_prc_to_cpu(x)		void(0)
	#define	__be_pdb_to_cpu(x)		void(0)
#endif	//!__LITTLE_ENDIAN__

#endif //_is_jedi_palmfile_H
