/*	mem.h

{{IS_NOTE

Authors:	Tom M. Yeh
Contributors:
Create Date:	7/11/0 09:57PM
$Header: /cvsroot/jedi/include/jedi/mem.h,v 1.15 2000/09/26 10:09:26 tomyeh Exp $
Purpose:	Memory management of jedi
Description:
	There are two level of memory management. One is mmXxx, which is 
	chunk-based memory, and used by applications. The other is _malloc,
	... and _free, which is used internally by the kernel.

	Some platforms, e.g., Windows CE, support malloc directly,
	and then mmXxx is implemented on top of _malloc.

	On the other hand, some platforms do not support virtual memory and
	memory compacting is necessary, and then we implement mmXxx as
	the low level function while _malloc is implemented on the top.

	The following macros controls how the code is generated.
	OS_CHUNK_ON_MALLOC: The platform supports malloc
	OS_RAM_CARD_NO: # of cards are supported by the platform
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_jedi_mem_H
#define _is_jedi_mem_H

#include <jedi/errmgr.h>
#include <string.h>

////////////////////////////////////////////////////////
#define	ERRM_CHUNKLOCKED			(ERR_MEM | 1)
#define	ERRM_NOTENOUGHSPACE			(ERR_MEM | 2)
#define	ERRM_INVALIDPARAM			(ERR_MEM | 3)
#define	ERRM_CHUNKNOTLOCKED			(ERR_MEM | 4)
#define	ERRM_CARDNOTPRESENT			(ERR_MEM | 5)
#define	ERRM_NOCARDHEADER			(ERR_MEM | 6)
#define	ERRM_INVALIDSTOREHEADER 	(ERR_MEM | 7)
#define	ERRM_RAMONLYCARD			(ERR_MEM | 8)
#define	ERRM_WRITEPROTECT			(ERR_MEM | 9)
#define	ERRM_NORAMONCARD			(ERR_MEM | 10)
#define	ERRM_NOSTORE				(ERR_MEM | 11)
#define	ERRM_ROMONLYCARD			(ERR_MEM | 12)

////////////////////////////////////////////////////////
typedef struct TMemChunk *_hmem_t;

typedef struct TMCFlags {
	__u16 _BF4(
		storageHeap	:1,	//In Jedi, we use MMF_xxxHEAP only as an attribute, no special meaning
		associated	:1,	//associated with a preallocate block
		moveable	:1,
		magic		:13);

	DECL_SETZEROS(__u16)
} TMCFlags;

EXTERN_C _hmem_t _JAPI mmAlloc(size_t sz, unsigned cardId DEFAULTVAL(0),
	bool bStorage DEFAULTVAL(false), size_t hdrExtraBytes DEFAULTVAL(0));
	//bStorage specify whether it is in a storage heap (i.e., whether to set storageHeap)
	//If you want the header to larger than TMemChunk (which you shall
	//derived your class from, and consider it as opaque), you might
	//specify a non-zero to hdrExtraBytes.
EXTERN_C _hmem_t _JAPI mmAllocAssoc(void* p);
	//Allocate a memory chunk by associate with a preallocated block, p
	//Note: MMF_ASSOCIATE will be set and cannot be resize or getsize
EXTERN_C _err_t _AAPI mmResize(_hmem_t hmem, size_t sz);//null if fail; hmem might be changed
EXTERN_C _err_t _AAPI mmFree(_hmem_t hmem); //hmem could be NULL

EXTERN_C size_t _AAPI mmSize(_hmem_t hmem);

EXTERN_C void* _AAPI mmLock(_hmem_t hmem);
EXTERN_C _err_t _AAPI mmUnlock(_hmem_t hmem);
EXTERN_C __u16 _AAPI mmGetLockCount(_hmem_t hmem);

EXTERN_C _hmem_t _AAPI mmGetHandle(void* p);
	//return 0 if not a pointer returned by mmLock
EXTERN_C void* _JAPI mmGetUnsafePtr(_hmem_t hmem);
	//return the current address of data no lock occurs
	//so it is unsafe to access it.
EXTERN_C void* _JAPI mmGetPtr(_hmem_t hmem, bool bAutoLock);
	//Note: if hmem is not locked and bAutoLock is true,
	//mmLock is called automatically. Otherwise, the lock count won't increase.

EXTERN_C __u32 _AAPI mmGetOwner(_hmem_t hmem);
EXTERN_C _err_t _AAPI mmSetOwner(_hmem_t hmem, __u32 owner);
EXTERN_C void _JAPI mmFreeByOwner(unsigned cardId, __u32 owner);
	//owner==0 means system

EXTERN_C unsigned _JAPI mmGetCardNo();
EXTERN_C unsigned _JAPI mmGetCardId(_hmem_t hmem);
EXTERN_C bool _JAPI mmIsStorage(_hmem_t hmem);

struct TCardInfo {
	unsigned szRom; //physical size
	unsigned szRam; //physical size
	unsigned szRamAvail; //available physical size
};
EXTERN_C void _JAPI mmGetInfo(unsigned cardId, TCardInfo* info);

EXTERN_C void _JAPI mmCompact(unsigned cardId);
	
#ifdef NDEBUG //////////////////////////////////////////

EXTERN_C void* _JAPI _malloc(size_t sz);
EXTERN_C void* _JAPI _calloc(size_t nm, size_t sz);
EXTERN_C void* _JAPI _realloc(void* p, size_t sz);
EXTERN_C void  _JAPI _free(void* p);
EXTERN_C size_t _JAPI _memsize(void* p);
EXTERN_C char* _JAPI __strdup(const char* p);

#define _MALLOC(s)			_malloc(s)
#define _CALLOC(s1, s2)		_calloc(s1, s2)
#define _REALLOC(p, s)		_realloc(p, s)
#define _FREE(p)			_free(p)
#define _MSIZE(p)			_memsize(p)

#define strdup(p)		__strdup(p)
#define STRDUP(p)		strdup(p)

#ifdef __cplusplus
#ifndef _NEW
#define _NEW(p)			(new p)
#endif

#ifndef _DEL
#define _DEL(p)			(delete p)
#define _DELV(p)		(delete[] p)
#define _DELV2(p, sz)	(delete[sz] p)
#endif
#endif //__cplusplus

#define	_MCHECK()		(true)

#else //NDEBUG//////////////////////////////////////////

EXTERN_C void* _JAPI _malloc_dbg(size_t sz, const char* flnm, int lnno);
EXTERN_C void* _JAPI _calloc_dbg(size_t nm, size_t sz, const char* flnm, int lnno);
EXTERN_C void* _JAPI _realloc_dbg(void* p, size_t sz, const char* flnm, int lnno);
EXTERN_C void  _JAPI _free_dbg(void* p, const char* flnm, int lnno);
EXTERN_C size_t _JAPI _memsize_dbg(void* p);
EXTERN_C char* _JAPI __strdup_dbg(const char* p, const char* flnm, int lnno);
	// Note:
	// 1. The above functions do not duplicate flnm since they assume
	//    flnm is placed in code segment
	// 2. _realloc_dbg and _free_dbg can process the pointer allocated by malloc

#define _MALLOC(s)		_malloc_dbg(s, __FILE__, __LINE__)
#define _CALLOC(s1, s2)	_calloc_dbg(s1, s2, __FILE__, __LINE__)
#define _REALLOC(p, s)	_realloc_dbg(p, s, __FILE__, __LINE__)
#define _FREE(p)		_free_dbg(p, __FILE__, __LINE__)
#define _MSIZE(p)		_memsize_dbg(p)

#define strdup(p)		__strdup_dbg(p, __FILE__, __LINE__)
#define STRDUP(p)		strdup(p)

#ifdef __cplusplus
extern void* operator new(size_t sz, const char* flnm, int lnno);
extern void* operator new[](size_t sz, const char* flnm, int lnno);

EXTERN_C void _JAPI _msetdel(const char* flnm, int lnno);
extern void _CDECL operator delete(void* p);
extern void operator delete[](void* p);

#ifndef _NEW
#define _NEW(p)			(new (__FILE__, __LINE__) p)
#endif

#ifndef _DEL
#define _DEL(p)			(_msetdel(__FILE__, __LINE__), delete p)
#define _DELV(p)		(_msetdel(__FILE__, __LINE__), delete[] p)
#define _DELV2(p, sz)	(_msetdel(__FILE__, __LINE__), delete[sz] p)
#endif
#endif //__cplusplus

#define _MCHECK()		mmCheck()
EXTERN_C bool _JAPI mmCheck(void);
	//return true if no linkage; the linkage will be dumped, if any

#endif //NDEBUG//////////////////////////////////////////

#define _MALLOC_T(type)		((type*)_MALLOC(sizeof(type)))
#define _CALLOC_T(type)		((type*)_CALLOC(1, sizeof(type)))

////////////////////////////////////////////////////////
#include <os/mem.h>

#ifndef OS_RAM_CARD_NO
#define OS_RAM_CARD_NO		1
	//# of card numbers that RAM resides
#endif

//#ifndef OS_ROM_CARD_NO
//#define OS_ROM_CARD_NO		1
	//# of card numbers that ROM resides
//#endif

////////////////////////////////////////////////////////
EXTERN_C_BEGIN

inline unsigned _JAPI mmGetCardNo() {return OS_RAM_CARD_NO;}

#if OS_RAM_CARD_NO==1
inline unsigned _JAPI mmGetCardId(_hmem_t /*hmem*/) {return 0;}
#endif

#ifdef OS_CHUNK_ON_MALLOC
struct TMemChunk {
	TMemChunk	*next, *prev;
	TMCFlags	flags;
	__u16		cLocks;
	__u32		owner;
	void		*data;
};

inline void _JAPI mmCompact(unsigned /*cardId*/) {}
#endif

#ifdef _MSC_VER
#define	alloca	_alloca
#endif

EXTERN_C_END
#endif //_is_jedi_mem_H
