/*	jdbfile.h

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:
	Create Date:	2000/8/24 03:15PM
	$Header: /cvsroot/jedi/include/jedi/jdbfile.h,v 1.5 2000/09/08 11:58:16 henrichen Exp $
	Purpose:	
	Description:	Jedi's file db system; the data file.

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_jedi_jdbfile_H
#define _is_jedi_jdbfile_H

//////////////////////////////////////////////////////////////////
#pragma pack(push, 1)

#define	JDBFILE_SIG	"ishkjdb"
#define	JDBFILE_VER	0x0001	//version 00.01
// jdb file header
struct TJdbHeader {
	char	signature[sizeof(JDBFILE_SIG)];	//jdb file signature
	__u32	version;	//file version
	__u32	offset;		//offset(from file top) to first record
};

// jdb chunk header
struct TChunkHeader {
	__u32 size;	//size of this record, including this record header
	__u32 flag;	//flag of this record
	
	void SwitchFlag(unsigned flag, bool bSet);
	bool FlagIsSet(unsigned mask);
};
//TChunkHeader.flag
#define	CHUNKFLAG_ALLOC	0x01	//mean this db record(data block) is allocated

#pragma	pack(pop)

//////////////////////////////////////////////////////////////////
class CFreeChunk;
class CArrayChunk;
struct TDbHeader;
class CJdbFile : public CJediFile {
public:
	//open, close
	bool Open(const char* name, int mode);
		//open the jdb file
	bool Create(const char* name);
		//create a jdb file
	bool Shrink(CJdxNode *pn);
		//shrink the jdb file if associated chunk is at the end of the file
		//if cannot shrink, return false
	unsigned long Extend(size_t size);
		//extend the jdb file to give space for new allocated chunk
		//return the offset to the new chunk, 0 if failed.

	//jdb header
	inline bool LoadJdbHeader(void);
		//load the jdb header
	inline bool SaveJdbHeader(void);
		//save the jdb header

	//chunk header	
	inline unsigned long GetFirstChunkOffset(void);
		//get offset of the first chunk in this file
	inline bool SetFirstChunkOffset(unsigned long off);
		//set offset of the first chunk in this file
	inline unsigned long GetNewChunkOffset(void);
		//get offset of the new chunk at the end of the file
	inline bool LoadChunkHeader(unsigned long offset, TChunkHeader *ph);
		//load chunk header
	inline bool SaveChunkHeader(unsigned long offset, TChunkHeader *ph);
		//save chunk header
	inline bool SetChunkSize(unsigned long offset, __u32 size);
		//set new chunk size 
	inline bool SetChunkFlag(unsigned long offset, __u32 flag);
		//set new chunk flag
	inline bool SetChunkSize(CJdxNode *pn);
		//another form of set chunk size
	inline bool SplitChunk(CJdxNode *pn);
		//split the chunk by the given offset and size
	bool SwitchChunkFlag(unsigned long offset, __u32 mask, bool bOn);
		//switch a spcified bit of the chunk flag on/off
	inline bool SetChunkFree(unsigned long offset);
		//switch the chunk to as a free chunk
	inline bool	SetChunkAlloc(unsigned long offset); 
		//switch the chunk to as a allocated chunk

	//contents read,write,copy
	void Copy(CJdxNode *pTgt, CJdxNode *pSrc);
	
#ifndef	NDEBUG
	bool Check(CFreeChunk *freeChunk, CArrayChunk *arrayChunk);
	void Dump(void);
#endif	//!NDEBUG

	//data
	TJdbHeader m_jh;
};

//////////////////////////////////////////////////////////////////
inline void TChunkHeader::SwitchFlag(unsigned mask, bool bSet)
{
	if (bSet)
		flag|= mask;
	else
		flag&= ~mask;
}

inline bool TChunkHeader::FlagIsSet(unsigned mask)
{
	return (flag & mask) != 0;
}


//////////////////////////////////////////////////////////////////
inline bool CJdbFile::LoadJdbHeader(void)
{
	ASSERT(m_hfile!= HFILE_INVALID );
	
	return sizeof(TJdbHeader) == 
		Read(&m_jh, sizeof(TJdbHeader), 0);
}

inline bool CJdbFile::SaveJdbHeader(void)
{
	ASSERT(m_hfile!= HFILE_INVALID );
	
	return sizeof(TJdbHeader) == 
		Write(&m_jh, sizeof(TJdbHeader), 0);
}	

inline unsigned long CJdbFile::GetFirstChunkOffset(void)
{
	ASSERT(m_hfile!= HFILE_INVALID);
	ASSERT(CHUNK_ALIGN(m_jh.offset) == m_jh.offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	return m_jh.offset;
}

inline bool CJdbFile::SetFirstChunkOffset(unsigned long off)
{
	ASSERT(m_hfile!= HFILE_INVALID );
	
	m_jh.offset= CHUNK_ALIGN(off);
	return SaveJdbHeader();
}

inline unsigned long CJdbFile::GetNewChunkOffset(void)
{
	ASSERT(m_hfile!= HFILE_INVALID );

	return jdbSetFilePointer(m_hfile, 0, FILE_END) + sizeof(TChunkHeader);
}
	
inline bool CJdbFile::LoadChunkHeader(unsigned long offset, TChunkHeader *ph)
{
	ASSERT(m_hfile!= HFILE_INVALID  && ph && offset && 
		offset >= GetFirstChunkOffset() && offset < GetNewChunkOffset());
	ASSERT(CHUNK_ALIGN(offset) == offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	
	const size_t sz= sizeof(TChunkHeader);
	return sz == Read(ph, sz, offset-sz);
}

inline bool CJdbFile::SaveChunkHeader(unsigned long offset, TChunkHeader *ph)
{
	ASSERT(m_hfile!= HFILE_INVALID  && ph && offset && 
		offset >= GetFirstChunkOffset() && offset < GetNewChunkOffset());
	ASSERT(CHUNK_ALIGN(offset) == offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	
	const size_t sz= sizeof(TChunkHeader);
	return sz == Write(ph, sz, offset-sz);
}

inline bool CJdbFile::SetChunkSize(unsigned long offset, __u32 size)
{
	ASSERT(m_hfile!= HFILE_INVALID  && offset >= GetFirstChunkOffset() 
		&& offset < GetNewChunkOffset() && size);
	ASSERT(CHUNK_ALIGN(size) == size);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes

	return sizeof(__u32) == 
		Write(&size, sizeof(__u32), 
			offset-sizeof(TChunkHeader)+OffsetOf(TChunkHeader, size));
}

inline bool CJdbFile::SetChunkFlag(unsigned long offset, __u32 flag)
{
	ASSERT(m_hfile!= HFILE_INVALID  && offset >= GetFirstChunkOffset());
	ASSERT(CHUNK_ALIGN(offset) == offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes

	return sizeof(__u32) == 
		Write(&flag, sizeof(__u32), 
			offset-sizeof(TChunkHeader)+OffsetOf(TChunkHeader, flag));
}

inline bool CJdbFile::SetChunkSize(CJdxNode *pn)
{
	ASSERT(m_hfile!= HFILE_INVALID  && pn && pn->m_idx);
	
	return SetChunkSize(pn->GetOffset(), pn->GetSize());
}

inline bool CJdbFile::SetChunkFree(unsigned long offset)
{
	ASSERT(CHUNK_ALIGN(offset) == offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	return SwitchChunkFlag(offset, CHUNKFLAG_ALLOC, false);
}

inline bool CJdbFile::SetChunkAlloc(unsigned long offset)
{
	ASSERT(CHUNK_ALIGN(offset) == offset);
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	return SwitchChunkFlag(offset, CHUNKFLAG_ALLOC, true);
}

inline bool CJdbFile::SplitChunk(CJdxNode *pn)
{
	ASSERT(pn);
	ASSERT(CHUNK_ALIGN(pn->GetOffset()) == pn->GetOffset());
	//In JdxNode, we overload the lower two bits of offset field, 
	//so it must align on 4 bytes
	TChunkHeader ch={pn->GetSize(), 0};
	return SaveChunkHeader(pn->GetOffset(), &ch);
}

#endif //_is_jedi_jdbfile_H
