/*	filedb.h

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:
	Create Date:	2000/8/30 02:44PM
	$Header: /cvsroot/jedi/include/jedi/filedb.h,v 1.3 2000/09/08 11:58:16 henrichen Exp $
	Purpose:	
	Description:	Jedi's file db system

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_jedi_filedb_H
#define _is_jedi_filedb_H

//////////////////////////////////////////////////////////////////
// Jedi's file db system
// Each Jedi DB is composed of two files with same name but different extension.
// 1. The data file (any extension you like. e.g. *.prc or *.pdb in PalmCE)
//		Data file can be deemed as a big chunk of memory
// 2. The index file (*.jdx), which composed of two double-linked-list and an array rb-tree 
//	  structures, packed little endian. Index file can be deemed as memory management 
//	  book-keeping data structures for allocating/freeing the data file, plus a kind of 
//	  sorting mechanism.
//
// What is an array tree? (Array tree is invented by Henri Chen, 20000815)  
// * It is a tree. (Denote the tree height as H.)
// * Each tree node represents an array elements or a segment of array elements.
// * Concepturally, it is a tree with array index as its key.
// * Retrieving of element data is achieved by tree's search algorithm. O(H).
// * However, we don't store array index as tree's key directly; instead, we store 
//   the total number of its left children(leftNum) and number of elements(elmNum) 
//   in this node.(For simple tree, elmNum is always 1, so no need to store it)
// * We can calculate the associated array index(arrayIndex) by traverse the tree
//   from the root.
// * Imaging we chop an array into three segments, root segment(rootSeg), left 
//   segment(leftSeg) and right segment(rightSeg).
// * All nodes in leftSeg has smaller array index than the rootSeg; and all nodes
//   in rightSeg has bigger array index than the rootSeg.
// * Now we denote the first array index of the leftSeg as baseIndex(leftSeg); then
//		baseIndex(rootSeg) = baseIndex(leftSeg) + leftNum(rootSeg).
//		baseIndex(rightSeg) = baseIndex(rootSeg) + elmNum(rootSeg).
// * Now chop each segment to further three sub-segments and recursively, you get an 
//   array-tree. (If each node represent an array elements, it is a simple binary-tree)
// * Each node follows the following three equations:
//		baseIndex(Node)= baseIndex(leftChild) + leftNum(Node);
//		baseIndex(rightChild)= baseIndex(Node) + elmNum(Node);
// * The array index of the i-th element in a node is
//		arrayIndex(Element_i)= baseIndex(Node) + i;
// * Now we can calculate the arrayIndex of each node by traverse the tree from the 
//   root because we know, at root, baseIndex(leftSeg) is 0 in a typical c-array.
// * What is good for array-tree? Why not just use the array directly? Why do we
//   store leftNum and elmNum instead of the arrayIndex itself? 
// * It is good when you need to insert or remove array elements dynamically.
// * It is good if you want to make an array of [n1, ), where n1 can be any number
//   because all you have to do is set baseIndex(leftSeg)@root to n.
// * Let's see. When you insert an array element in the middle of the traditional 
//   array, two things happened: 1) you have to move all elements after the 
//	 insertion point down one element space; 2) the array index after the insertion
//   point is increased one. The removing has similar effects.
// * In an array, thing(1) costs a lot. thing(2) is automatically achieved.
// * In an array-tree, thing(1) cost at most only O(H). thing(2) cost the same O(H) 
//   because we store leftNum and elmNum instead of the arrayIndex itself.
// * Insert(), O(H) is the worst case like any tree. Because only the nodes from 
//	 the root to the insertion point has to modify their leftNum field, it is o(H)
//	 again.
// * Remove(), O(H) is the worst case like any tree. Because only the nodes from
//   the root to the removing point has to modify their leftNum filed, it is o(H)
// * Following is a binary array-tree pseudo code.
//	Search(root, index) {
//		node= root;
//		baseIndex= 0;	//assume c-array
//		while(node != NULL) {
//			nodeIndex= baseIndex + leftNum(node);
//			if (nodeIndex == index)	//found the key
//				return node;
//			else if (nodeIndex < index)
//				node= leftChild(node);
//			else {
//				baseIndex= nodeIndex + 1;
//				node= rightChild(node);
//			}
//		}
//	}
//				
//  Insert(root, node_i) {
//      parent= NULL;
//		node= root;
//		baseIndex= 0;	//assume c-array
//		while(node != NULL) {
//			parent= node;
//			nodeIndex= baseIndex + leftNum(node);
//			if (Index(node_i) <= nodeIndex) {
//				leftNum(node)++;
//				node= leftChild(node);
//			}
//			else {
//				baseIndex= nodeIndex + 1;
//				node= rightChild(node);
//			}
//		}
//		if (parent == NULL) root= node_i;
//		else if (key(node_i) <= key(parent))
//			leftChild(parent)= node_i;
//		else
//			rightChild(parent)= node_i;
//	}
//
//  Remove(root, index) {
//		parent= NULL;
//		node= root;
//		baseIndex= 0;	//assume c-array
//		while(node != NULL) {
//			nodeIndex= baseIndex + leftNum(node);
//			if (Index(node_i) == nodeIndex)
//				break;
//			else {
//				parent= node;
//				if (index(node_i) < nodeIndex) {	
//					leftNum(root)--;
//					node= leftChild(node);
//				}
//				else {
//					baseIndex= nodeIndex + 1;
//					node= rightChild(node);
//				}
//			}
//		}
//		detach(parent, node);	
//			//if (node == NULL) means cannot find it
//			//if (parent == NULL) means root point at the key_i
//	}
//

class CChunkMgr;
class CJdbFile;
class CFileDB {
public:
	//create, open
	bool Create(const char *filename);
	bool Delete(const char *filename);
	bool Open(const char *filename, int mode);
	bool Close(void);
	
	//alloc, free, resize
	_hmem_t NewRecord(size_t size);
		//new a record in this db
	bool DelRecord(_hmem_t hmem);
		//delete the specified record in this db
		//note the pn is invalid after this function.
		//don't delete a record that is not newed from this db. The consequence is unknown.
	bool ResizeRecord(_hmem_t hmem, size_t size);
		//resize the record
	
	//attach, detach
	bool Attach(_hmem_t hmem, unsigned key);
	_hmem_t Detach(unsigned key);
	_hmem_t Search(unsigned key);
	bool Update(_hmem_t hmem, CJdxNode *pn);
	_hmem_t GetDBMemChunk(CJdxNode *pn);
	inline void* GetRecordHeader(_hmem_t h);
	inline void SetRecordHeaderSize(size_t sz);
	
	CFileDB(CJdbFile *jdb) : m_pJdbFile(jdb), m_pChunkMgr(NULL), m_arrayChunk(&m_jdxFile), 
		m_mode(0), m_szRechdr(0){}
	~CFileDB(){Close();}

	inline bool ModeIsSet(int mode);
	inline bool IsOpened(void);

	//data
	CJdbFile	*m_pJdbFile;
	CJdxFile	m_jdxFile;
	CChunkMgr 	*m_pChunkMgr;
	CArrayChunk	m_arrayChunk;
	int			m_mode;
	CDBMemTree	m_dbMem;
	size_t		m_szRechdr;
};

inline bool CFileDB::ModeIsSet(int mask)
{
	return (m_mode & mask) != 0;
}

inline bool CFileDB::IsOpened(void)
{
	return m_jdxFile.m_hfile != HFILE_INVALID  && m_pJdbFile->m_hfile != HFILE_INVALID ;
}

inline void *CFileDB::GetRecordHeader(_hmem_t h)
{
	return (void*)(((TDBMemChunk *)h)+1);
}

inline void CFileDB::SetRecordHeaderSize(size_t sz)
{
	//one file db can have only one time of setting
	if (!m_szRechdr) {
		m_szRechdr= sz;
	}
}

#endif //_is_jedi_filedb_H
