/*	dibeng.h

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:
	Create Date:	2000/6/19 09:19PM
	$Header: /cvsroot/jedi/include/jedi/dibeng.h,v 1.4 2000/08/31 09:50:17 tomyeh Exp $
	Purpose:	
	Description:	
		1. Device-Independent-Bitmap graphic engine
		2. Support 1,2,4,8,16,32bpp dibs.
		3. Optimized at 32bit data bus.

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_dibeng_dibeng_H
#define _is_dibeng_dibeng_H

#include	<jedi/kernel.h>
#include	<jedi/debug.h>
#include	<jedi/swab.h>
#include	<jedi/fb.h>
//#include	<stdlib.h>
//?? 20000719, Henri Chen: include <stdlib.h> give a strange error message
// but if I don't include it, there is no abs functions
inline __s32 abs(__s32 x) {return x < 0 ? -x : x;}

///////////////////////////////////////////////////////////
// data structure
typedef	enum {
	Rop2Copy,
	Rop2And,
	Rop2AndNot,
	Rop2Xor,
	Rop2Or,
	Rop2CopyNot,
	Rop2Swap
} Rop2;

typedef struct TDrawTools {
	__u32	foreColor;
	__u32	backColor;
	__u8	pat[8];
	Rop2	rop2;
	__u16	drawFlags;
	TRect	clip;
	
	TDrawTools():drawFlags(0){}
} TDrawTools;

typedef struct TDrawArea {
	__u8	*pBase;
	__s32	rowBytes;
	const TPalette *pRgb;
	__u16	width;
	__u16	height;
	__u8	bpp;
	__u8	flags;
} TDrawArea;	
//TDrawArea.flags
#define	DRAWAREA_COLOR	0x1

inline void ioSetU32(void* pBits, __u32 val)
{
	*((__u32*)pBits)= val;
}

inline __u32 ioGetU32(void* pBits)
{
	return *((__u32*)pBits);
}

///////////////////////////////////////////////////////////
// TDrawTools get/set
//TDrawTools.drawFlags
#define	DRAWFLAGS_USEPATTERN	0x2

inline __u16 dtGetDrawFlags(TDrawTools *pDt)
{
	return pDt->drawFlags;
}

inline __u32 dtGetForeColor(TDrawTools *pDt)
{
	return pDt->foreColor;
}

inline void dtSetForeColor(TDrawTools *pDt, __u32 val)
{
	pDt->foreColor= val;
}

inline __u32 dtGetBackColor(TDrawTools *pDt)
{
	return pDt->backColor;
}

inline void dtSetBackColor(TDrawTools *pDt, __u32 val)
{
	pDt->backColor= val;
}

inline bool dtUsePattern(TDrawTools *pDt)
{
	return (pDt->drawFlags & DRAWFLAGS_USEPATTERN) != 0;
}

inline void dtSetUsePattern(TDrawTools *pDt)
{
	pDt->drawFlags|= DRAWFLAGS_USEPATTERN;
}

inline void dtClearUsePattern(TDrawTools *pDt)
{
	pDt->drawFlags&= ~DRAWFLAGS_USEPATTERN;
}

inline __u8* dtGetPattern(TDrawTools *pDt)
{
	return pDt->pat;
}

inline void dtSetPattern(TDrawTools *pDt, __u8 *pat)
{
	*((__u32*)(pDt->pat))= *((__u32*)pat);
	*((__u32*)(pDt->pat)+1)= *((__u32*)pat+1);
}

inline void dtSetGrayPattern(TDrawTools *pDt)
{
	*((__u32*)(pDt->pat))= 0xaa55aa55;
	*((__u32*)(pDt->pat)+1)= 0xaa55aa55;
}

inline bool dtGetPatternBit(TDrawTools *pDt, __s16 x, __s16 y)
{
	return (pDt->pat[y&7] & (0x80 >> (x & 7))) != 0;
}

inline Rop2 dtGetRop2(TDrawTools *pDt)
{
	return pDt->rop2;
}

inline void dtSetRop2(TDrawTools *pDt, Rop2 rop2)
{
	pDt->rop2= rop2;
}


inline bool PtInRect(__s16 x, __s16 y, const TRect *pRect)
{
	return (x >= pRect->leftTop.x && x < (pRect->leftTop.x + pRect->extent.x) &&
		y >= pRect->leftTop.y && y < (pRect->leftTop.y + pRect->extent.y));
}

///////////////////////////////////////////////////////////
// Rop2 Operation
inline void CopyU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) & ~mask | 
		(colors & mask));
}

inline void AndU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) & 
		(~mask | (colors & mask)));
}

inline void AndNotU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) &
		(~mask | ~(colors & mask)));
}

inline void XorU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) ^ (colors & mask));
}

inline void OrU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) | (colors & mask));
}

inline void CopyNotU32(void *pBits, __u32 mask, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, ioGetU32(pBits) & ~mask |
		~(colors & mask));
}

inline void AssignU32(void *pBits, __u32 colors)
{
	ASSERT((((__u32)pBits) & 3)==0);	//must aligned on 4bytes boundary
	ioSetU32(pBits, colors);
}

///////////////////////////////////////////////////////////
// Misc functions
// Bit extension to 32bit dword
inline __u32 BitsExtendU32(__u16 bitcount, __u32 colors)
{
	ASSERT(bitcount > 0);
	while (bitcount < 32) {
		colors= (colors << bitcount) | colors;
		bitcount<<= 1;
	}
	return colors;
}

//swap two coordinates
inline void Swap(__s16& a, __s16& b)
{
	a^=b;
	b^=a;
	a^=b;
}

//translate from bpp to bitshift count
inline __u8 BppToBitShift(__u8 bpp)
{
	ASSERT(bpp==1 || bpp==2 || bpp==4 || 
			bpp==8 || bpp==16 || bpp==32);
	return (__u8) ((bpp >= 8) ? ((bpp>>4)+3) : (bpp>>1));
}

///////////////////////////////////////////////////////////
// CDibEng: bpp dependent framebuffer graphic drawing engine
class CDibEng {
public:
	CDibEng() : m_bpp(0), m_bitshift(0) {}
	
	//**general operation to get the pixel value with clipping
	__u32 GetPixel(TDrawArea *pDa, __s16 x, __s16 y, TDrawTools *pDt);

	//**general operation to set pixel with clipping
	void SetPixel(TDrawArea *pDa, __s16 x, __s16 y, TDrawTools *pDt);
	
	//**general operation to draw lines with clpping
	void Line(TDrawArea *pDa,	
		__s16 x1, __s16 y1, __s16 x2, __s16 y2, TDrawTools *pDt);

	//**general operation of PatBlt with clipping
	void PatBlt(TDrawArea *pDa, TRect *pDstRect, TDrawTools *pDt);
			
	//**general operation of BitBlt with clipping
	void BitBlt(TDrawArea *pDstDa, __s16 dstX, __s16 dstY, 
		TDrawArea *pSrcDa, TRect *pSrcRect, TDrawTools *pDt);

	///////////////////////////////////////////////////////////
	// Basic bpp information support
	//set the bpp and bit shift for proper color depth
	inline void SetBpp(__u8 bpp);
	
	//the bit shift of this color depth
	inline __u8 GetBitShift(void);
		
	//the bit-per-pixel(bpp) of this color depth
	inline __u8 GetBpp(void);
	
	//the bit mask of this color depth
	inline __u32 GetBitMask(void);
	
	///////////////////////////////////////////////////////////
	// Pixel support
	
	//which bit the pixel is located in a 32bit dword
	inline __u8 GetPixelBit(__s16 x);
	
	//get the bit mask at x of the value in a 32bit dword
	inline __u32 GetValueMask(__u32 value, __s16 x);
	
	//get the bit mask at x of the pixel in a 32bit dword
	inline __u32 GetPixelMask(__s16 x);
	
	//get the left bit mask of the pixel in a 32 bit dword
	inline __u32 GetPixelLeftMask(__s16 x);
	
	//get the right bit mask of the pixel in a 32 bit dword
	inline __u32 GetPixelRightMask(__s16 x);
	
	//which byte the pixel is located in a scanline
	inline __s16 GetPixelByte(__s16 x);
	
	//which word the pixel is located in a scanline
	inline __s16 GetPixelByte16(__s16 x);
	
	//which dword the pixel is located in a scanline
	inline __s16 GetPixelByte32(__s16 x);
	
	//get the pixel's value at pBits 32byte dword
	inline __u32 GetPixelValue(void *pBits, __s16 x);
	
	//swap foreColor to backColor or vice versa
	inline void SwapPixel(void *pBits, __s16 x, __u32 foreColor, __u32 backColor);
	
	///////////////////////////////////////////////////////////
	// Horizontal & vertical line support
	
	//return the byte used for the pixel width
	inline __u16 PixelWidthToByte(__u16 pixelwidth);
	
	//return the byte used for the pixel width but align on minimum dword boundary
	inline __u16 PixelWidthToByte32(__u16 pixelwidth);
	
	//expand 1bit pattern to specified color
	inline void PatternExpandColor(
		__u8 *buffer, __u32 fg, __u32 bg, __u8 *pPat, __u16 pixelwidth);
	
	//draw a patterned horizontal line
	inline void PatternHLine(
		__u8 *pScan, __s16 y, TDrawTools *pDt, 
		__u32 leftMask, __u32 rightMask, __u16 leftByte32, __u16 rightByte32);
	
	//swap foreColor and backColor on a horizontal line
	inline void SwapHLine(
		__u8 *pScan, __s16 x1, __s16 x2, TDrawTools *pDt);
	
	//**general operation to draw a horizontal line without clipping
	inline void HLine(
		TDrawArea *pDa,	__s16 y, __s16 x1, __s16 x2, TDrawTools *pDt, bool bDrawLastPt);

	//draw a patterned vertical line
	inline void PatternVLine(
		__u8 *pBits, __s32 rowBytes, __s16 x, __s16 y, __u16 height, TDrawTools *pDt);

	//swap foreColor and backColor on a vertical line
	inline void SwapVLine(
		__u8 *pBits, __s32 rowBytes, __s16 x, __s16 height, TDrawTools *pDt);
		
	//**general operation to draw a vertical line without clipping
	inline void VLine(
		TDrawArea *pDa,	__s16 x, __s16 y1, __s16 y2, TDrawTools *pDt, bool bDrawLastPt);

	/////////////////////////////////////////////////
	// Bren. line support
	
	//pixel= pattern ? foreColor : backColor
	void PatternLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel= color
	void CopyLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel&= color
	void AndLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel&=~color
	void AndNotLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel^=color
	void XorLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel|=color
	void OrLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//pixel=~color
	void CopyNotLinePixel(
		void *pBits, __s16 x, __s16 y, TDrawTools *pDt);
	
	//swap foreColor and backColor
	void SwapLinePixel(
		void *pBits, __s16 x,	__s16 y, TDrawTools *pDt);
	
	//**general operation to draw a Bren. line without clipping
	inline void BLine(
		TDrawArea *pDa,	__s16 x1, __s16 y1, __s16 x2, __s16 y2, TDrawTools *pDt, bool bDrawLastPt);
	
	///////////////////////////////////////////////////////////
	//Bitblt/PatBlt support
	
	//swap foreColor and backColor
	inline void SwapBlt(
		__u8 *pScan, __s32 rowBytes,
		__s16 x1, __s16 x2, __u16 height, TDrawTools *pDt);

protected:
	__u8	m_bpp;
	__u8	m_bitshift;
};

///////////////////////////////////////////////////////////
// Basic bpp information support
//set the bit shift of the color depth
inline void CDibEng::SetBpp(__u8 bpp)
{
	if (m_bpp != bpp) {
		m_bpp= bpp;
		m_bitshift= BppToBitShift(m_bpp);
	}
}

//the bit shift of this color depth
inline __u8 CDibEng::GetBitShift(void)
{
	return m_bitshift;
}
	
//the bit-per-pixel(bpp) of this color depth
inline __u8 CDibEng::GetBpp(void)
{
	return m_bpp;
}

//the bit mask of this color depth
inline __u32 CDibEng::GetBitMask(void)
{
	return 0xffffffff >> (32-GetBpp());
}


///////////////////////////////////////////////////////////
// Pixel support

//which bit the pixel is located in a 32bit dword(big endian version)
inline __u8 beGetPixelBit(__s16 x, __u8 bitshift)
{
	ASSERT(x>=0);
	return (__u8)((32-((1+x)<<bitshift)) & 31);
}

//which bit the pixel is located in a 32bit dword
inline __u8 CDibEng::GetPixelBit(__s16 x)
{
	return __be_bitof32_to_cpu(beGetPixelBit(x, GetBitShift()));
}

//get the bit mask at x of the value in a 32bit dword
inline __u32 CDibEng::GetValueMask(__u32 value, __s16 x)
{
	ASSERT(x>=0);
	return value << GetPixelBit(x);
}

//get the bit mask at x of the pixel in a 32bit dword
inline __u32 CDibEng::GetPixelMask(__s16 x)
{
	return GetValueMask(GetBitMask(), x);
}

//get the left bit mask of the pixel in a 32 bit dword
inline __u32 CDibEng::GetPixelLeftMask(__s16 x)
{
	return __be32_to_cpu(((__u32)0xffffffff) >> ((x<<GetBitShift())&31));
}

//get the right bit mask of the pixel in a 32 bit dword
inline __u32 CDibEng::GetPixelRightMask(__s16 x)
{
	return __be32_to_cpu(((__u32)0xffffffff) << beGetPixelBit(x, GetBitShift()));
}

//which byte the pixel is located in a scanline
inline __s16 CDibEng::GetPixelByte(__s16 x)
{
	return (__s16)((x << 2) >> (5-GetBitShift()));
}

//which byte the pixel is located in a scanline (align on 16bit)
inline __s16 CDibEng::GetPixelByte16(__s16 x)
{
	return (__s16)(GetPixelByte(x) & ~1);
}

//which byte the pixel is located in a scanline (align on 32bit)
inline __s16 CDibEng::GetPixelByte32(__s16 x)
{
	return (__s16)(GetPixelByte(x) & ~3);
}

inline __u32 CDibEng::GetPixelValue(void* pBits, __s16 x)
{
	return (ioGetU32(pBits) >> GetPixelBit(x)) & GetBitMask();
}

//swap foreColor to backColor or vice versa if pixel is foreColor or backColor
inline void CDibEng::SwapPixel(void *pBits, __s16 x, __u32 foreColor, __u32 backColor)
{
	__u32 dstColor= GetPixelValue(pBits, x);
	__u32 mask= GetPixelMask(x);
	if (dstColor == foreColor)
		CopyU32(pBits, mask, GetValueMask(backColor,x));
	else if (dstColor == backColor)
		CopyU32(pBits, mask, GetValueMask(foreColor,x));
}

///////////////////////////////////////////////////////////
// line support

//return the byte used for the pixel width
inline __u16 CDibEng::PixelWidthToByte(__u16 pixelwidth)
{
	return (__u16)((pixelwidth << GetBitShift()) >> 3);
}

//return the byte used for the pixel width but align on minimum dword boundary
inline __u16 CDibEng::PixelWidthToByte32(__u16 pixelwidth)
{
	return (__u16)((PixelWidthToByte(pixelwidth)+3) & ~3);
}

//expand 1bit pattern to specified color
inline void CDibEng::PatternExpandColor(
	__u8 *buffer, __u32 fg, __u32 bg, __u8 *pPat, __u16 pixelwidth)
{
//	TRACE("pPat= 0x%08x, *pPat=0x%08x, fg= 0x%08x, bg= 0x%08x, pixelwidth=%d\n", pPat, *pPat, fg, bg, pixelwidth);
	__u32 *pBuf= (__u32*)buffer;
	int size= PixelWidthToByte32(pixelwidth) >> 2;
	for (int i=0; i < size; i++)
		*(pBuf+i)=0;

	__u8 mask= 0x80;
	for (__u16 x=0; x < pixelwidth; x++, mask>>=1) {
		if (mask == 0) 
			mask= 0x80;
		pBuf= (__u32*) (buffer+GetPixelByte32(x));
		*pBuf= *pBuf | GetValueMask((pPat[x>>3] & mask) ? fg : bg, x);
	}
//	TRACE("pBuf= 0x%08x, *pBuf=0x%08x, mask=0x%08x\n", pBuf, *pBuf, mask);
}	

//draw a patterned horizontal line
inline void CDibEng::PatternHLine(
	__u8 *pScan, __s16 y, TDrawTools *pDt, 
	__u32 leftMask, __u32 rightMask, __u16 leftByte32, __u16 rightByte32)
{
//	TRACE("leftMask=0x%08x, rightMask=0x%08x\n", leftMask, rightMask);
	//pattern color expansion first
	//bpp<8, size==1
	//bpp==8, size==2
	//bpp==16, size==4
	//bpp==32, size=8
	__u32 buffer[8];
	PatternExpandColor((__u8*)buffer, 
			pDt->foreColor, pDt->backColor, &(pDt->pat[y&7]), 8);
//	TRACE("before: expanded colors=0x%08x\n", buffer[0]);

	buffer[0]= BitsExtendU32((__u16)(8 << GetBitShift()), buffer[0]);
//	TRACE("after: expanded colors=0x%08x\n", buffer[0]);
	
	//process the left edge
	int indexMask= (PixelWidthToByte32(8) >> 2) - 1;	//index mask of dword
	__u32 colors= buffer[(leftByte32>>2) & indexMask];
	CopyU32(pScan+leftByte32, leftMask, colors);

	//process the middle part
	for (__u16 dw= (__u16)(leftByte32+4); dw < rightByte32; dw+= 4) {
		colors= buffer[(dw>>2) & indexMask];
		AssignU32(pScan+dw, colors);
	}

	//process right edge
	if (leftByte32 < rightByte32) {
		colors= buffer[(rightByte32>>2) & indexMask];
		CopyU32(pScan+rightByte32, rightMask, colors);
	}
}

//swap foreColor and backColor on a horizontal line
inline void CDibEng::SwapHLine(
	__u8 *pScan, __s16 x1, __s16 x2, TDrawTools *pDt)
{
	void *pBits;
	for (__s16 x= x1; x <= x2 ; x++) {
		pBits= pScan + GetPixelByte32(x);
		SwapPixel(pBits, x, pDt->foreColor, pDt->backColor);
	}
}

// horizontal line
inline void CDibEng::HLine(
	TDrawArea *pDa,__s16 y, __s16 x1, __s16 x2, TDrawTools *pDt, bool bDrawLastPt)
{
	//x1 must <= x2
	if (x1 > x2) {
		if (!bDrawLastPt)
			x2++;
		Swap(x1,x2);
	}
	else if (!bDrawLastPt)
		x2--;
		
	__s16 w= (__s16)(x2-x1+1);
	TRect rect={{x1,y},{w,1}};
//	TRACE("x1=%d, x2=%d, w=%d\n", x1, x2, w);
	PatBlt(pDa, &rect, pDt);
}

//swap foreColor and backColor on a vertical line
inline void CDibEng::SwapVLine(
	__u8 *pBits, __s32 rowBytes, __s16 x, __s16 height, TDrawTools *pDt)
{
	while(height--) {
		SwapPixel(pBits, x, pDt->foreColor, pDt->backColor);
		pBits+= rowBytes;
	}
}

//draw a patterned veritcal line
inline void CDibEng::PatternVLine(
	__u8 *pBits, __s32 rowBytes, __s16 x, __s16 y, __u16 height, TDrawTools *pDt)
{
	while(height--) {
		PatternLinePixel(pBits, x, y++, pDt);
		pBits+= rowBytes;
	}
}

//veritcal line
inline void CDibEng::VLine(
	TDrawArea *pDa,	__s16 x, __s16 y1, __s16 y2, TDrawTools *pDt, bool bDrawLastPt)
{
	__s32 rowBytes= pDa->rowBytes;
	__u8 *pBits= pDa->pBase + y1 * rowBytes + GetPixelByte32(x);
	if (y1 > y2) {
		if (!bDrawLastPt)
			y2++;
		Swap(y1,y2);
	}
	else if (!bDrawLastPt)
		y2--;

	__u16 height= (__u16)(y2-y1+1);
	if (dtUsePattern(pDt)) {
		PatternVLine(pBits, rowBytes, x, y1, height, pDt); 
		return;
	}
	else if (dtGetRop2(pDt)==Rop2Swap) {
		SwapVLine(pBits, rowBytes, x, height, pDt);
		return;
	}
	
	void (*XferU32)(void*, __u32, __u32);

	switch(dtGetRop2(pDt)) {
		case Rop2Copy:
			XferU32= &CopyU32;
			break;
		case Rop2And:
			XferU32= &AndU32;
			break;
		case Rop2AndNot:
			XferU32= &AndNotU32;
			break;
		case Rop2Xor:
			XferU32= &XorU32;
			break;
		case Rop2Or:
			XferU32= &OrU32;
			break;
		case Rop2CopyNot:
			XferU32= &CopyNotU32;
			break;
		default:
		{
			//?? 20000605, Henri chen: not supported trasfer mode
			__u8 DIBENG_NOT_SUPPORTED_ROP2=0;
			ASSERT(DIBENG_NOT_SUPPORTED_ROP2);
			return;
		}
	}

	__u32 mask= GetPixelMask(x);
	__u32 color= GetValueMask(dtGetForeColor(pDt), x);
	
	while (height--) {
		(*XferU32)(pBits, mask, color);
		pBits+= rowBytes;
	} 
}

//Bren. line
inline void CDibEng::BLine(
	TDrawArea *pDa, __s16 x1, __s16 y1, __s16 x2, __s16 y2, TDrawTools *pDt, bool bDrawLastPt)
{
	__s32 rowBytes= pDa->rowBytes;
	__u8 *pScan= pDa->pBase + y1 * rowBytes;
	__s16 xDelta= (__s16) abs(x2-x1);
	__s16 yDelta= (__s16) abs(y2-y1);
	__s16 xInc= (__s16)((x2 > x1) ? 1 : -1);
	__s16 yInc;
	if (y2 > y1) {
		yInc= 1;
	}
	else {
		yInc= -1;
		rowBytes= -rowBytes;
	}
		
	void (CDibEng::*XferLinePixel)(void*,__s16,__s16,TDrawTools*);
	if (dtUsePattern(pDt)) {
		XferLinePixel= &CDibEng::PatternLinePixel;
	}
	else {
		switch(dtGetRop2(pDt)) {
			case Rop2Copy:
				XferLinePixel= &CDibEng::CopyLinePixel;
				break;
			case Rop2And:
				XferLinePixel= &CDibEng::AndLinePixel;
				break;
			case Rop2AndNot:
				XferLinePixel= &CDibEng::AndNotLinePixel;
				break;
			case Rop2Xor:
				XferLinePixel= &CDibEng::XorLinePixel;
				break;
			case Rop2Or:
				XferLinePixel= &CDibEng::OrLinePixel;
				break;
			case Rop2CopyNot:
				XferLinePixel= &CDibEng::CopyNotLinePixel;
				break;
			case Rop2Swap:
				XferLinePixel= &CDibEng::SwapLinePixel;
				break;
			default:
				//?? 20000605, henri Chen: not supported rop2
				XferLinePixel= 0;
				ASSERT(0);
		}
	}
	__u8 *pBits= pScan + GetPixelByte32(x1);
	__s16 rem;
	if (xDelta >= yDelta) {	//horizontal style
		rem= (__s16)(xDelta/2);
		while (x1 != x2) {
			(this->*XferLinePixel)(pBits, x1, y1, pDt);
			x1= (__s16) (x1 + xInc);
			rem= (__s16) (rem + yDelta);
			if (rem >= xDelta) {
				rem= (__s16) (rem - xDelta);
				y1= (__s16) (y1 + yInc);
				pScan+= rowBytes;
			}
			pBits= pScan + GetPixelByte32(x1);
		} 
	}
	else {	//vertical style
		rem= (__s16)(yDelta/2);
		while (y1 != y2) {
			(this->*XferLinePixel)(pBits, x1, y1, pDt);
			y1= (__s16) (y1 + yInc);
			pScan+= rowBytes;
			rem= (__s16) (rem+xDelta);
			if (rem >= yDelta) {
				rem= (__s16) (rem-yDelta);
				x1= (__s16) (x1+xInc);
			}
			pBits= pScan + GetPixelByte32(x1);
		} 
	}
	if (bDrawLastPt)
		(this->*XferLinePixel)(pBits, x1, y1, pDt);
}

///////////////////////////////////////////////////////////
//PatBlt support
//swap foreColor and backColor
//And two rectangle
EXTERN_C bool RctAnd(TRect *pDst, const TRect *pSrc1, const TRect *pSrc2);

inline void CDibEng::SwapBlt(
	__u8 *pScan, __s32 rowBytes, 
	__s16 x1, __s16 x2, __u16 height, TDrawTools *pDt)
{
	while (height--) {
		SwapHLine(pScan, x1, x2, pDt);
		pScan+= rowBytes;
	}
}

#endif //_is_fbeng_H
