/*	bmp.h

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:	Tom M. Yeh
	Create Date:	2000/7/24 03:57PM
	$Header: /cvsroot/jedi/include/jedi/bmp.h,v 1.7 2000/09/08 01:48:21 henrichen Exp $
	Purpose:	
	Description:	Implementation Header of Jedi's bmpXXX() function

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#ifndef _is_jedi_bmp_H
#define _is_jedi_bmp_H

#include <jedi/mem.h>
#include <jedi/fb.h>

#pragma pack(push, 1)
//////////////////////////////////////////////////////////////////
typedef struct TBmpFlags {
	__u16 _BF6(
		compressed:1,
		hasColorTable:1,
		hasTransparency:1,
		indirect:1,
		forScreen:1,
		reserved:11);

	DECL_SETZEROS(__u16)
} TBmpFlags;

typedef struct TBmp {
	__s16		width;
	__s16		height;
	__u16		rowBytes;
	TBmpFlags	flags;
	__u8		pixelSize;
	__u8		version;
	__u16		nextDepthOffset;
	__u8		transparentIndex;
	__u8		compressionType;
	__u16		reserved;
} TBmp;

//TBmp.version
#define	BMPVER_ZERO	0
#define	BMPVER_ONE	1
#define	BMPVER_TWO	2

//TBmp.compressionType
#define	BMPCOMPRESS_SCANLINE	0
#define	BMPCOMPRESS_RLE			1
#define	BMPCOMPRESS_NONE		0xff

///////////////////////////////////////////////////////////
typedef struct TColorTable {
	__u16 numEntries;
	__u16 alignpad;		//?? Compatible: added to enforce palette align on 4 bytes
//	TPalette entries[];
} TColorTable;

#pragma pack(pop)

///////////////////////////////////////////////////////////
//Functions match Anakin
EXTERN_C TBmp* _JAPI bmpCreate(__s16 width, __s16 height, __u8 depth, const TColorTable *pCt, _err_t* err=0);
	//Create a bitmap, which will allocate memory for header and memory handle for bits.

EXTERN_C size_t _AAPI bmpGetBitsSize(TBmp *pBmp);
	//Get the bitmap's bits size in byte. 
	//	If a compresed bitmap, return the compressed size.
EXTERN_C TColorTable* _AAPI bmpGetColorTable(TBmp *pBmp);
	//Return the bitmap's color table.
	//	If no colortable at all, return NULL.
EXTERN_C size_t _AAPI bmpGetColorTableSize(TBmp *pBmp);
	//Get the bitmap's color table size in bytes. 
	//	If no colortable at all, return 0.
EXTERN_C size_t _AAPI bmpGetSize(TBmp *pBmp);
	//Get the bitmap's size, if bitmap's bits is stored indirectly,
	//	then only the header and colortable size is counted.
EXTERN_C _err_t _AAPI bmpDestroy(TBmp *pBmp);
	//Delete a bitmap.

///////////////////////////////////////////////////////////
//Functions unique to Jedi
EXTERN_C TBmp* _JAPI bmpCompress(TBmp *pBmp, unsigned type);
	//Compress the bitmap and return the compressed bitmap.
	//	After compression, the pBmp might be at a different location.
EXTERN_C TBmp* _JAPI bmpGetNearest(TBmp *pBmp, unsigned bpp);
	//Return the bitmap from the bitmap family which has the nearest bpp as requested.
EXTERN_C TBmp* _JAPI bmpDecompress(TBmp *pBmp);
	//Decompress the bitmap and return the decompressed bitmap.
	//	After decompression, the pBmp might be at a differnt location.
EXTERN_C void _JAPI bmpGetRect(const TBmp *pBmp, TRect *pRect);
	//Return the bitmap's dimension in the rectangle

EXTERN_C void _JAPI bmpAppCleanup(TAppLocal* app);
	//cleanup per application

///////////////////////////////////////////////////////////
inline bool bmpValidBpp(const TBmp *pBmp)
{
	unsigned bpp= pBmp->pixelSize;
	return bpp == 1 || bpp == 2 || bpp == 4 || bpp == 8 || bpp == 16 || bpp == 32;
}

inline TPalette *bmpGetColorTableEntries(const TColorTable *pCt)
{
	return (TPalette*)(pCt+1);
}

inline bool bmpValidColorTable(const TColorTable *pCt, unsigned bpp)
{
	return pCt->numEntries == (1 << bpp);
}

inline _hmem_t* bmpGetMemHandleAddr(TBmp* pBmp)
{
	return (_hmem_t*)(((__u8*)(pBmp+1)) + bmpGetColorTableSize(pBmp));
}	//Note: it never returns null

///////////////////////////////////////////////////////////
// OS dependent part
#include	<os/bmp.h>

///////////////////////////////////////////////////////////
struct TScrnBmp : public TBmp {
	TColorTable colorTable;
};

extern TScrnBmp *g_pBmpScrn;

#ifdef OS_USEHDIB
inline HDIB* bmpGetHDIBAddr(TBmp* pBmp)
{
	return (HDIB*)(bmpGetMemHandleAddr(pBmp)+1);
}	//Note: HDIB is meanful only if flags.indirect is true
#endif

inline void bmpSysInit(TPalette *pal)
{
	unsigned rowBytes= ((((SCREEN_WIDTH * SCREEN_BPP)+31) >> 5) << 2);
	size_t size= sizeof(TScrnBmp) + (sizeof(TPalette) << SCREEN_BPP) + sizeof(void*);
	
#ifdef OS_USEHDIB
	size += sizeof(HDIB);
#endif
	
	g_pBmpScrn= (TScrnBmp*) _MALLOC(size);
	
#ifdef	OS_USEHDIB
	HDIB hDIB;
	hDIB = fbConfig(pal);
	ASSERT(hDIB);
	*((HDIB*)((__u8*)g_pBmpScrn+size-sizeof(HDIB))) = hDIB;
#else
	DOANDASSERT(fbConfig(pal));
#endif

	g_pBmpScrn->width= (__s16) SCREEN_WIDTH;
	g_pBmpScrn->height= (__s16) SCREEN_HEIGHT;
	g_pBmpScrn->rowBytes= (__u16) rowBytes;
	g_pBmpScrn->flags.SetZeros();
	g_pBmpScrn->flags.hasColorTable= 1;
	g_pBmpScrn->flags.indirect= 1;
	g_pBmpScrn->flags.forScreen= 1;
	g_pBmpScrn->pixelSize= (__u8) SCREEN_BPP;
	g_pBmpScrn->version= BMPVER_TWO;
	g_pBmpScrn->nextDepthOffset= 0;
	g_pBmpScrn->transparentIndex= 0;
	g_pBmpScrn->compressionType= BMPCOMPRESS_NONE;
	g_pBmpScrn->reserved= 0;
	g_pBmpScrn->colorTable.numEntries= (unsigned short)(1 << SCREEN_BPP);

	memcpy(bmpGetColorTableEntries(&g_pBmpScrn->colorTable), pal,
				g_pBmpScrn->colorTable.numEntries * sizeof(TPalette));

	*bmpGetMemHandleAddr(g_pBmpScrn) = mmAllocAssoc((void*)SCREEN_BASEADDR);
	ASSERT(*bmpGetMemHandleAddr(g_pBmpScrn));
}

inline void bmpSysCleanup()
{
	mmFree(*bmpGetMemHandleAddr(g_pBmpScrn));
	_FREE(g_pBmpScrn);
	fbCleanup();
}

#endif //_is_jedi_bmp_H
