/*	bmp.cxx

{{IS_NOTE

	Authors:	Henri Chen
	Contributors:	Tom M. Yeh
	Create Date:	2000/7/21 02:46PM
	$Header: /cvsroot/jedi/gdi/bmp.cxx,v 1.11 2000/09/26 10:09:26 tomyeh Exp $
	Purpose:	
	Description:	Implementation of Jedi's bmpXXX() functions

}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/swab.h>
#include <jedi/mem.h>
#include <jedi/rct.h>
#include <jedi/fb.h>
#include <jedi/bmp.h>
#include <lib/lcllink.h>

//////////////////////////////////////////////////////////////////
DECL_APPLINK(TBmpEx, TBmp, pBmpExList) //declare TBmpEx
TScrnBmp *g_pBmpScrn;

//////////////////////////////////////////////////////////////////
EXTERN_C size_t _AAPI bmpGetSize(TBmp *pBmp)
{
	if(!pBmp) return 0;

	ASSERT(pBmp->flags.indirect);
	return sizeof(TBmp) + bmpGetColorTableSize(pBmp) + sizeof(_hmem_t);
}

static size_t _STDCALL GetColorTableSizePriv(unsigned num)
{
	ASSERT(sizeof(TPalette) == 4);
	//Note that we enforce the size to be aligned on 4 bytes
	//So the memHandle and option HDIB can be accessed without problem
	return (((num << 2)+ sizeof(TColorTable))+3)&~3;
}

EXTERN_C size_t _AAPI bmpGetColorTableSize(TBmp *pBmp)
{
	return pBmp && pBmp->flags.hasColorTable && pBmp->pixelSize <= 8 ? 
			GetColorTableSizePriv(1 << pBmp->pixelSize) : 0;
}

EXTERN_C size_t _AAPI bmpGetBitsSize(TBmp *pBmp)
{
	if(!pBmp) return 0;

#if 0
	if(pBmp->flags.compressed){
		return __be16_to_cpu(*((__u16*)bmpGetBits(pBmp)));
	}
#endif
	return pBmp->rowBytes * pBmp->height;
}

EXTERN_C TBmp* _JAPI bmpGetNearest(TBmp *pBmp, unsigned bpp)
{
	if(!pBmp) return 0;

	//find nearest pixel depth bitmap if a bitmap family
	TBmp *pPreBmp= NULL;
	while(pBmp->nextDepthOffset != 0 && pBmp->pixelSize <= bpp) {
		pPreBmp= pBmp;
		pBmp= (TBmp *)(((__u8*)pBmp)+(pBmp->nextDepthOffset << 2));
	}
	if (pPreBmp != NULL && pBmp->pixelSize > bpp)
		pBmp= pPreBmp;
		
	return pBmp;
}

EXTERN_C TColorTable* _AAPI bmpGetColorTable(TBmp *pBmp)
{
	return pBmp && pBmp->flags.hasColorTable ? (TColorTable*)(pBmp+1) : NULL;
}

//////////////////////////////////////////////////////////////////
#if 0
//not yet: remove bmpGetBits with mmLock(*bmpGetMemHandleAddr(pBmp))
inline TBmp* bmpScanlineCompress(TBmp *pBmp)
{
	//worst case, size
	size_t size= 2 + (pBmp->rowBytes + ((pBmp->rowBytes+7)/8)) * pBmp->height;
	_hmem_t hDstHandle= mmAlloc(size, 0, MMF_DYNAMICHEAP);
	__u8* pSize= (__u8*) mmLock(hDstHandle);		//the first 2 bytes is compressed size
	if (!pSize) return 0;
	
	memset(pSize, 0, size);
	__u8* pDstBits= pSize + 2;
	__u8* pThisScan= (__u8*) bmpGetBits(pBmp);
	__u8* pPreScan= 0;
	
	//do the compression
	__u8* pMap= pDstBits++;
	for (int scan= pBmp->height; scan > 0 ; scan--) {
		__u16 leadmap= 0;
		for (int j= 0; j < pBmp->rowBytes; j++) {
			if (pPreScan == 0 || pPreScan[j] != pThisScan[j]) {
				leadmap|= (0x80 >> (j & 7));
				*pDstBits++= pThisScan[j];
			}
			if (((j&7) == 7 || j == (pBmp->rowBytes-1))) {
				*pMap= (__u8)leadmap;
				pMap= pDstBits++;
				leadmap= 0;
			}
		}
		pPreScan= pThisScan;
		pThisScan+= pBmp->rowBytes;
	}
	
	size= pMap - pSize; 
	if (size > bmpGetBitsSize(pBmp)) {	//compressed bitmap is even bigger!
		mmUnlock(hDstHandle);
		mmFree(hDstHandle);
	}
	else {
		pSize[0]= (__u8)(((size-2) & 0xff00) >> 8);
		pSize[1]= (__u8)(size-2);
		memcpy(bmpGetBits(pBmp), pSize, size);
		mmUnlock(hDstHandle);
		mmFree(hDstHandle);

		pBmp->flags.compressed = 1;
		pBmp->compressionType= BMPCOMPRESS_SCANLINE;
	}
	return pBmp;
}

inline TBmp* bmpRleCompress(TBmp *pBmp)
{
	//?? 20000725, Henri Chen: not yet!
	ASSERT(0);
	return pBmp;
}

EXTERN_C TBmp* _JAPI bmpCompress(TBmp *pBmp, unsigned type)
{
	if(!pBmp) return 0;

	if (type == BMPCOMPRESS_SCANLINE)
		return bmpScanlineCompress(pBmp);
	else if (type == BMPCOMPRESS_RLE) 
		return bmpRleCompress(pBmp);
	return pBmp;
}

inline TBmp* bmpScanlineDecompress(TBmp *pBmp)
{
	__u8* pSrcBits= (__u8*) bmpGetBits(pBmp) + 2;
	size_t size= pBmp->rowBytes * pBmp->height;
	_hmem_t hDstHandle= mmAlloc(size, 0, MMF_DYNAMICHEAP);
	__u8* pDstBits= (__u8*) mmLock(hDstHandle);
	if (!pDstBits) return 0;
	
	memset(pDstBits, 0, size);
	__u8* pThisScan= pDstBits;
	__u8* pPreScan= 0;
	__u8* pEnd= pSrcBits + bmpGetBitsSize(pBmp) - 2;
	__u16 leadmap= 0;
	while(pSrcBits < pEnd) {
		for (int j= 0; j < pBmp->rowBytes; j++) {
			if ((j&7) == 0)
				leadmap= *pSrcBits++;
			pThisScan[j]= (((leadmap<<=1) & 0x100) != 0) ? *pSrcBits++ : pPreScan[j];
		}
		pPreScan= pThisScan;
		pThisScan+= pBmp->rowBytes;
	}
	
	memcpy(bmpGetBits(pBmp), pDstBits, size);
	mmUnlock(hDstHandle);
	mmFree(hDstHandle);

	pBmp->flags.compressed = 0;
	pBmp->compressionType= BMPCOMPRESS_NONE;

	return pBmp;
}

inline TBmp* bmpRleDecompress(TBmp *pBmp)
{
	//?? 20000725, Henri Chen: not yet!
	ASSERT(0);
	return pBmp;
}

EXTERN_C TBmp* _JAPI bmpDecompress(TBmp *pBmp)
{
	if(!pBmp) return 0;

	__u16 type= pBmp->compressionType;
	if (pBmp->flags.compressed) {
		if (type == BMPCOMPRESS_SCANLINE)
			return bmpScanlineCompress(pBmp);
		else if (type == BMPCOMPRESS_RLE) 
			return bmpRleCompress(pBmp);
	}
	return pBmp;
}
#endif

//////////////////////////////////////////////////////////////////
EXTERN_C void bmpGetRect(const TBmp *pBmp, TRect *pRect)
{
	if(pBmp)
		rctSet(pRect, 0, 0, pBmp->width, pBmp->height);
}

//////////////////////////////////////////////////////////////////
EXTERN_C TBmp* _JAPI bmpCreate(__s16 width, __s16 height,
	__u8 depth, const TColorTable *pCt, _err_t* err)
{
	if(width<=0 || height<=0 || (depth!=1 && depth!=2 && depth!=4 && depth!=8)){
		if(err) *err = ERRS_PARAMERR;
		return 0;
	}

	// calculate the total size
	bool bHasCt=0;
	size_t size= sizeof(TBmpEx);
	if (pCt && bmpValidColorTable(pCt, depth)) {
		size+= GetColorTableSizePriv(pCt->numEntries);
		bHasCt=1;
	}
	
	size+= sizeof(void*);

#ifdef OS_USEHDIB
	size += sizeof(HDIB);
#endif

	//allocate memory for the TBmp
	TBmpEx *pBmpEx= (TBmpEx*)_MALLOC(size);
	if(!pBmpEx){
	l_fail0:
		if(err) *err = ERRM_NOTENOUGHSPACE;
		return 0;
	}

	//create associated DIB object
	size_t rowBytes= (((width * depth)+31) >> 5) << 2;	//align on 32bit dwords

#ifdef OS_USEHDIB
	void *fb;
	HDIB hDIB= fbCreateDIB(width, height, depth,
		bHasCt ? bmpGetColorTableEntries(pCt) : 
			bmpGetColorTableEntries(bmpGetColorTable(g_pBmpScrn)), &fb);
	if(!hDIB){
	l_fail1:
		_FREE(pBmpEx);
		goto l_fail0;
	}
	_hmem_t hFb = mmAllocAssoc(fb);
	if(!hFb){
		fbDeleteDIB(hDIB);
		goto l_fail1;
	}
#else
	_hmem_t hFb= mmAlloc(rowBytes * height);
	if(!hFb){
		_FREE(pBmpEx);
		goto l_fail0;
	}
#endif

	// initial the bitmap type
	pBmpEx->Link();
	TBmp* pBmp = ToTBmp(pBmpEx);
	pBmp->width= (__s16) width;
	pBmp->height= (__s16) height;
	pBmp->rowBytes= (__s16) rowBytes;
	pBmp->flags.SetZeros();
	pBmp->flags.hasColorTable = bHasCt;
	pBmp->flags.indirect = 1;
	pBmp->pixelSize= (__u8) depth;
	pBmp->version= BMPVER_TWO;
	pBmp->nextDepthOffset= 0;
	pBmp->transparentIndex= 0;
	pBmp->compressionType= BMPCOMPRESS_NONE;
	pBmp->reserved= 0;
	
	//copy the color table
	if (bHasCt) {
		TColorTable *pBmpCt= bmpGetColorTable(pBmp);
		pBmpCt->numEntries= pCt->numEntries;
		memcpy(bmpGetColorTableEntries(pBmpCt),
			bmpGetColorTableEntries(pCt),
			pCt->numEntries * sizeof(TPalette));
	}
	
	//set the hFb
	*bmpGetMemHandleAddr(pBmp)= hFb;
#ifdef OS_USEHDIB
	*bmpGetHDIBAddr(pBmp) = hDIB;
#endif
	return pBmp;
}

static void DestroyBmp(TBmp *pBmp, TAppLocal* app)
{
	ASSERT(pBmp != g_pBmpScrn);

	if(!pBmp) return;

	ASSERT(pBmp->flags.indirect);
	mmFree(*bmpGetMemHandleAddr(pBmp));

#ifdef OS_USEHDIB
	fbDeleteDIB(*bmpGetHDIBAddr(pBmp));
#endif

	TBmpEx* pBmpEx = ToTBmpEx(pBmp);
	pBmpEx->Unlink(app);
	_FREE(pBmpEx);
}

EXTERN_C _err_t _AAPI bmpDestroy(TBmp *pBmp)
{
	DestroyBmp(pBmp, 0);
	return ERR_NONE;
}

EXTERN_C void _JAPI bmpAppCleanup(TAppLocal* app)
{
#ifndef NDEBUG
	TBmpEx::Dump("Bitmap Leakage");
#endif
	TBmpEx::Apply(DestroyBmp, app);
}
