/*	memmgr.cxx

{{IS_NOTE

	Authors:	Tom M. Yeh
	Contributors:
	Create Date:	8/16/0 03:34PM
	$Header: /cvsroot/jedi/anakin/memmgr.cxx,v 1.9 2000/09/27 04:05:00 tomyeh Exp $
	Purpose:	Memory Management of Anakin
	Description:
}}IS_NOTE

Copyright (C) 2000 Infoshock Corporation. All Rights Reserved.

{{IS_RIGHT
}}IS_RIGHT
*/
#include <jedi/kernel.h>
#include <jedi/debug.h>
#include <jedi/thread.h>
#include <jedi/mem.h>
#include <anakin/types.h>
#include <anakin/api.h>

static const Char s_cardName[]="PalmCard";
static const Char s_manufName[]="Infoshock";

#define CARD_VERSION		1
#define CARD_YEAR			2000
#define CARD_MONTH			8
#define CARD_DAY			17

///////////////////////////////////////////////////////////
EXTERN_C Err _AAPI sysMemCardInfo(UInt16 cardNo,
	Char *cardNameP, Char *manufNameP, UInt16 *versionP, UInt32 *crDateP,
	UInt32 *romSizeP, UInt32 *ramSizeP, UInt32 *freeBytesP)
{
	if(cardNo >= OS_RAM_CARD_NO)
		return ERRM_NOCARDHEADER;

	if(cardNameP) memcpy(cardNameP, s_cardName, sizeof(s_cardName));
	if(manufNameP) memcpy(manufNameP, s_manufName, sizeof(s_manufName));
	if(versionP) *versionP = CARD_VERSION;
	if(crDateP) *crDateP = ToSeconds(CARD_YEAR, CARD_MONTH, CARD_DAY);

	AnakinLock();
	TCardInfo ci;
	mmGetInfo(cardNo, &ci);
	AnakinUnlock();

	if(romSizeP) *romSizeP = ci.szRom;
	if(ramSizeP) *ramSizeP = ci.szRam;
	if(freeBytesP) *freeBytesP = ci.szRamAvail;

	return 0;
}

EXTERN_C Err _AAPI sysMemStoreInfo(UInt16 cardNo, UInt16 storeNumber,  
	UInt16 *versionP, UInt16 *flagsP, Char *nameP, UInt32 *crDateP,
	UInt32 *bkDateP, UInt32 *heapListOffsetP, UInt32 *initCodeOffset1P,
	UInt32 *initCodeOffset2P, LocalID*	databaseDirIDP)
{
	if(cardNo >= OS_RAM_CARD_NO)
		return  ERRM_NOCARDHEADER;
	if(storeNumber >= 2)
		return ERRM_INVALIDSTOREHEADER;

	if(versionP) *versionP = CARD_VERSION;
	if(crDateP) *crDateP = ToSeconds(CARD_YEAR, CARD_MONTH, CARD_DAY);
	if(bkDateP) *bkDateP = 0; //not yet: sync

	AnakinLock();
	static Char s_storeName[]="RAM Store 0";
	if(!storeNumber==0){
		s_storeName[1] = 'O'; s_storeName[_count_of(s_storeName)-3] = 0;
		if(flagsP) *flagsP = 0;
	}else{
		s_storeName[1] = 'A'; s_storeName[_count_of(s_storeName)-3] = ' ';
		if(flagsP) *flagsP = 1;
	}
	if(nameP) memcpy(nameP, s_storeName, _count_of(s_storeName));
	AnakinUnlock();

	if(heapListOffsetP) *heapListOffsetP = 0;
	if(initCodeOffset1P) *initCodeOffset1P = 0;
	if(initCodeOffset2P) *initCodeOffset2P = 0;
	if(databaseDirIDP) *databaseDirIDP = 0; //not yet

	return 0;
}

///////////////////////////////////////////////////////////
#if OS_RAM_CARD_NO!=1
#error The following codes shall be modified
#endif
//The sequence: 0=dynamic heap, 1=storage heap, 2=rom heap
#define HeapId2CardId(id)				((id) >> 2)
#define HeapId2Index(id)				((id) & 3)
#define IsHeapROM(id)					(HeapId2Index(id) >= 2)
#define ToHeapId(cardNo, heapIndex)		(UInt16)(((cardNo) << 2) + ((heapIndex) & 3))

EXTERN_C UInt16 _AAPI sysMemNumCards(void)
{
	return (UInt16)mmGetCardNo();
}

EXTERN_C UInt16 _AAPI sysMemNumHeaps(UInt16 /*cardNo*/)
{
	return 3;
}

EXTERN_C UInt16 _AAPI sysMemNumRAMHeaps(UInt16 /*cardNo*/)
{
	return 2;
}

EXTERN_C UInt16 _AAPI sysMemHeapID(UInt16 cardNo, UInt16 heapIndex)
{
	return ToHeapId(cardNo, heapIndex);
}

EXTERN_C Boolean _AAPI sysMemHeapDynamic(UInt16 heapID)
{
	return HeapId2Index(heapID) ==0; //first one?
}

EXTERN_C Err _AAPI sysMemHeapFreeBytes(UInt16 heapID, UInt32 *freeP, UInt32 *maxP)
{
	AnakinLock();
	TCardInfo ci;
	mmGetInfo(HeapId2CardId(heapID), &ci);
	AnakinUnlock();

	if(IsHeapROM(heapID)){
		if(freeP)	*freeP	= 0;
		if(maxP)	*maxP	= ci.szRom;
	}else{
		if(freeP)	*freeP	= ci.szRamAvail;
		if(maxP)	*maxP	= ci.szRam;
	}
	return 0;
}

EXTERN_C UInt32 _AAPI sysMemHeapSize(UInt16 heapID)
{
	UInt32 sz;
	sysMemHeapFreeBytes(heapID, 0, &sz);
	return sz;
}

EXTERN_C UInt16 _AAPI sysMemHeapFlags(UInt16 heapID)
{
	return (UInt16)(IsHeapROM(heapID) ? 0x2001: 0x2000);
}

EXTERN_C Err _AAPI sysMemHeapFreeByOwnerID(UInt16 heapID, UInt32 ownerID)
{
	AnakinLock();
	if(!IsHeapROM(heapID))
		mmFreeByOwner(HeapId2CardId(heapID), ownerID);
	AnakinUnlock();
	return 0;
}

EXTERN_C Err _AAPI sysMemHeapCompact(UInt16 heapID)
{
	AnakinLock();
	mmCompact(HeapId2CardId(heapID));
	AnakinUnlock();
	return 0;
}

///////////////////////////////////////////////////////////
EXTERN_C MemHandle _AAPI sysMemHandleNew(UInt32 size)
{
	if(!size)
		return 0; //Note: mmAlloc accepts zero but Palm API doesn't

	AnakinLock();
	MemHandle h = mmAlloc(size, 0);
	AnakinUnlock();
	return h;
}

EXTERN_C Boolean _AAPI sysMemHandleDataStorage(MemHandle h)
{
	AnakinLock();
	Boolean b = mmIsStorage(h);
	AnakinUnlock();
	return b;
}

EXTERN_C UInt16 _AAPI sysMemHandleHeapID(MemHandle h)
{
	return ToHeapId(0, (UInt16)sysMemHandleDataStorage(h));
}

DEF_MT_API1(Err, MemHandleFree, mmFree, MemHandle)
DEF_MT_API2(Err, MemHandleResize, mmResize, MemHandle, UInt32)
DEF_MT_API1(UInt32, MemHandleSize, mmSize, MemHandle)

DEF_MT_API1(UInt16, MemHandleLockCount, mmGetLockCount, MemHandle)
DEF_MT_API1(MemPtr, MemHandleLock, mmLock, MemHandle)
DEF_MT_API1(Err, MemHandleUnlock, mmUnlock, MemHandle)

DEF_MT_API1(UInt32, MemHandleOwner, mmGetOwner, MemHandle)
DEF_MT_API2(Err, MemHandleSetOwner, mmSetOwner, MemHandle, UInt32)

///////////////////////////////////////////////////////////
EXTERN_C MemPtr _AAPI sysMemPtrNew(UInt32 size)
{
	return sysMemHandleLock(sysMemHandleNew(size));
}

DEF_MT_API1(MemHandle, MemPtrRecoverHandle, mmGetHandle, MemPtr)

EXTERN_C UInt32 _AAPI sysMemPtrSize(MemPtr p)
{
	return sysMemHandleSize(sysMemPtrRecoverHandle(p));
}

EXTERN_C UInt32 _AAPI sysMemPtrOwner(MemPtr p)
{
	return sysMemHandleOwner(sysMemPtrRecoverHandle(p));
}

EXTERN_C Boolean _AAPI sysMemPtrDataStorage(MemPtr p)
{
	return sysMemHandleDataStorage(sysMemPtrRecoverHandle(p));
}

EXTERN_C UInt16 _AAPI sysMemPtrHeapID(MemPtr p)
{
	return sysMemHandleHeapID(sysMemPtrRecoverHandle(p));
}

EXTERN_C Err _AAPI sysMemPtrSetOwner(MemPtr p, UInt32 owner)
{
	return sysMemHandleSetOwner(sysMemPtrRecoverHandle(p), owner);
}

EXTERN_C Err _AAPI sysMemPtrResize(MemPtr p, UInt32 newSz)
{
	return sysMemHandleResize(sysMemPtrRecoverHandle(p), newSz);
}

EXTERN_C Err _AAPI sysMemPtrUnlock(MemPtr p)
{
	return sysMemHandleUnlock(sysMemPtrRecoverHandle(p));
}

///////////////////////////////////////////////////////////
//Note: LocalID is not visible to Jedi, where only _hmem_t and void* are used
//MemHandle+1 = LocalID
//MemPtr = LocalID
EXTERN_C LocalID _AAPI sysMemPtrToLocalID(MemPtr p)
{
	return (LocalID)p;
}

EXTERN_C LocalID _AAPI sysMemHandleToLocalID(MemHandle h)
{
	return h ? ((LocalID)h) + 1: 0;
}

MemPtr _AAPI sysMemLocalIDToGlobal(LocalID localId, UInt16 cardId)
{
	return cardId >= OS_RAM_CARD_NO ? 0:
		localId & 1 ? (MemPtr)(localId-1): (MemPtr)localId;
}

LocalIDKind _AAPI sysMemLocalIDKind(LocalID localId)
{
	return (LocalIDKind)(localId & 1); //0: ptr, 1: handle
}

static MemPtr _STDCALL sysLocalToPtr(LocalID localId, UInt16 cardId, Boolean bAutoLock)
{
	MemPtr p = 0;
	if(cardId < OS_RAM_CARD_NO)
		if(localId & 1){
			--localId;

			AnakinLock();
			p = mmGetPtr((MemHandle)localId, bAutoLock);
			AnakinUnlock();
		}else
			p = (MemPtr)localId;
	return p;
}

MemPtr _AAPI sysMemLocalIDToPtr(LocalID localId, UInt16 cardId)
{
	return sysLocalToPtr(localId, cardId, false);
}

MemPtr _AAPI sysMemLocalIDToLockedPtr(LocalID localId, UInt16 cardId)
{
	return sysLocalToPtr(localId, cardId, true);
}

///////////////////////////////////////////////////////////
EXTERN_C Err _AAPI sysMemMove(void *dstP, const void *sP, Int32 numBytes)
{
	memmove(dstP, sP, numBytes);
	return 0;
}
					
EXTERN_C Err _AAPI sysMemSet(void *dstP, Int32 numBytes, UInt8 value)
{
	memset(dstP, value, numBytes);
	return 0;
}
