/* This file contains the device dependent part of the driver for the Floppy
 * Disk Controller (FDC) using the NEC PD765 chip.
 * Based on Minix floppy driver
 *
 * Copyright(C) 2001
 * Camilo Alejandro Arboleda
 */

#include <bsp.h>
#include <irq.h>
#include <rtems/libio.h>
/* #include <rtems/libio_.h> */
#include <assert.h>
#include "dma.h"
#include "floppy.h"

/* #define DEBUG_DISK 1 */

#define check_rtems_status(A) \
   if (status != RTEMS_SUCCESSFUL) {\
      printk(A); \
      rtems_semaphore_release(fl_sem_id); \
      return status;\
   }


/*===================================================================
 * Variables.
 */

typedef struct {                /* main drive struct, one entry per drive */
  int fl_curcyl;                /* current cylinder */
  int fl_hardcyl;               /* hardware cylinder, as opposed to: */
  int fl_cylinder;              /* cylinder number addressed */
  int fl_sector;                /* sector addressed */
  int fl_head;	        	/* head number addressed */
  int fl_opened;                /* Device oppened */
  int fl_blocks;                /* Blocks per disk */
  rtems_id fl_timer_id;         /* Auto stop timer id */
  rtems_id fl_irq_sem;              /* Semaphore used for IRQ */
  char fl_motor;                /* motor status */
  char fl_calibration;		/* CALIBRATED or UNCALIBRATED */
  char fl_drive;                /* Drive number */
  char fl_gap;                  /* Gap size */
  char fl_rate;                 /* Transfer rate*/
  char fl_sectors;              /* Sectors per track */
  char fl_steps;                /* Steps per cylinder */
  char fl_test_sector;
  char *dma_buffer;             /* dma_buffer */
} floppy_t;

floppy_t floppy[NR_DRIVES]; /* The floppy */
unsigned char fl_command;   /* Last command executed on drive */
static int need_reset;      /* need reset */
static rtems_id fl_sem_id;  /* Semaphore (mutex) ID */
static rtems_id fl_irq_sem; /* Calling task ID */
char f_results[MAX_RESULTS];/* the controller can give lots of output */

/* Gather transfer data for each sector. */
struct trans {         /* precomputed transfer params */
  int block;      /* block number */
  rtems_libio_t *iop;  /* belongs to this I/O request */
  unsigned char *dma_buf;     /* DMA address */
  unsigned char *usr_buf;     /* User buffer  address */
} ftrans[MAX_SECTORS];

/* Seven combinations of diskette/drive are supported.
 *
 * # Drive  diskette  Sectors  Tracks  Rotation Data-rate  Comment
 * 0  360K    360K      9       40     300 RPM  250 kbps   Standard PC DSDD
 * 1  1.2M    1.2M     15       80     360 RPM  500 kbps   AT disk in AT drive
 * 2  720K    360K      9       40     300 RPM  250 kbps   Quad density PC
 * 3  720K    720K      9       80     300 RPM  250 kbps   Toshiba, et al.
 * 4  1.2M    360K      9       40     360 RPM  300 kbps   PC disk in AT drive
 * 5  1.2M    720K      9       80     360 RPM  300 kbps   Toshiba in AT drive
 * 6  1.44M   1.44M    18	80     300 RPM  500 kbps   PS/2, et al.
 *
 * In addition, 720K diskettes can be read in 1.44MB drives, but that does
 * not need a different set of parameters.  This combination uses
 *
 * X  1.44M   720K	9	80     300 RPM  250 kbps   PS/2, et al.
 */
const char flc_gap[NT] =
	{0x2A, 0x1B, 0x2A, 0x2A, 0x23, 0x23, 0x1B}; /* gap size */
const char flc_rate[NT] =
	{0x02, 0x00, 0x02, 0x02, 0x01, 0x01, 0x00}; /* 2=250,1=300,0=500 kbps*/
const char flc_nr_sectors[NT] =
	{9,    15,   9,    9,    9,    9,    18};   /* sectors/track */
const int flc_nr_blocks[NT] =
	{720,  2400, 720,  1440, 720,  1440, 2880}; /* sectors/diskette*/
const char flc_steps_per_cyl[NT] =
	{1,    1,    2,    1,    2,    1,     1};   /* 2 = dbl step */
const char flc_spec1[NT] =
	{0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF}; /* step rate, etc. */
const char flc_test_sector[NT] =
	{4*9,  14,   2*9,  4*9,  2*9,  4*9,  17};   /* to recognize it */

/*===================================================================
 * Prototypes
 */
/* Part 1: ISR functions */
static void fl_isr_on(const rtems_irq_connect_data *);
static void fl_isr_off(const rtems_irq_connect_data *);
static int  fl_isr_is_on(const rtems_irq_connect_data *);
void        fl_isr(void);
/* Part 2:  Floppy driver general setup */
void inline start_motor(floppy_t *fd);
void        auto_stop_motor(rtems_id id, void *arg);
void        start_stop_motor();
int         recalibrate(floppy_t *fd);
rtems_status_code   fdc_out(unsigned char val);
rtems_status_code   fdc_results(void);
rtems_status_code   fl_seek(floppy_t *fd);
rtems_status_code   fl_reset();
rtems_status_code   fl_read_sect(floppy_t *fd);
/* Part 3: Data transfer */
static inline void  fdc_dma_setup ( unsigned mode, size_t bytes, unsigned char *buffer );
rtems_status_code   fl_transfer (floppy_t *fd, unsigned char command, int count);
rtems_status_code   fl_finish (floppy_t *fd, unsigned *f_count, unsigned char command);
rtems_status_code   fl_read_write( floppy_t *fd, unsigned char command, rtems_libio_rw_args_t *data);
/* Part 4: Device driver entry */
rtems_device_driver fl_initialize( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );
rtems_device_driver fl_open      ( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );
rtems_device_driver fl_close     ( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );
rtems_device_driver fl_read      ( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );
rtems_device_driver fl_write     ( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );
rtems_device_driver fl_ioclt     ( rtems_device_major_number major, rtems_device_minor_number minor, void  *arg );

/*===================================================================
 * Part 1: ISR functions
 */

/* static */
rtems_irq_connect_data floppy_isr_data =
{
   FLOPPY_IRQ,
   fl_isr,
   fl_isr_on,
   fl_isr_off,
   fl_isr_is_on
};

static void
fl_isr_on ( const rtems_irq_connect_data *irq )
{
   BSP_irq_enable_at_i8259s(irq->name);
   return;
}
						   
static void
fl_isr_off ( const rtems_irq_connect_data *irq )
{
   BSP_irq_disable_at_i8259s(irq->name);
   return;
}

static int
fl_isr_is_on ( const rtems_irq_connect_data *irq )
{
   return BSP_irq_enabled_at_i8259s(irq->name);
}

/*-------------------------------------------------------------------
 * fl_isr(void): Interrupt handler for floppy drive
 */
void
fl_isr (void)
{
#ifdef DEBUG_DISK
   printk("Floppy Interrupt!\n");
   if ( rtems_semaphore_release(fl_irq_sem) != RTEMS_SUCCESSFUL )
      printk("Ooops\n");
   else
      printk("Semaphore relased %d\n",fl_irq_sem);
#else
   rtems_semaphore_release(fl_irq_sem);
#endif
   return;
}

/*===================================================================
 * Part 2: Floppy driver general setup
 */


/*-------------------------------------------------------------------
 * int inline fp2offset(floppy_t *fd)
 * Return byte offset from floppy structure H-C-S
 */
int inline
chs2offset(int C, int H, int S, floppy_t *fd)
{
   return ( ( C * NR_HEADS * fd->fl_sectors +
	      H * fd->fl_sectors +
	      S ) << SECTOR_SHIFT
	  );
}

/*-------------------------------------------------------------------
 * void inline offset2fp(int offset, floppy_t *fd)
 * Get floppy structure from offset.
 * Floppy structure MUST be an initialized one.
 */
void inline
offset2chs(int offset, int *C, int *H, int *S, floppy_t *fd)
{
   *S = offset % (fd->fl_sectors);
   *C = offset / (NR_HEADS * fd->fl_sectors);
   *H = (offset % (NR_HEADS * fd->fl_sectors)) / fd->fl_sectors;
}

/*-------------------------------------------------------------------
 * rtems_status_code fdc_out(unsigned char val)
 * Output a byte to the controller.  This is not entirely trivial, since you
 * can only write to it when it is listening, and it decides when to listen.
 * If the controller refuses to listen, the FDC chip is given a hard reset.
 *
 * Input:
 * Output:
 * Return:
 */
rtems_status_code
fdc_out ( unsigned char val )
{
   rtems_interval then, now;
   register unsigned char fl_status;


   if (need_reset) return RTEMS_UNSATISFIED;
   /* if controller is not listening, return */

   /* It may take several tries to get the FDC to accept a command. */

   rtems_clock_get (RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &then);

   do {
      inport_byte(FDC_STATUS,fl_status);
      /* printk ("fl_status = %x\n",fl_status); */
      rtems_clock_get (RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &now);
      if ( (now - then) > TIMEOUT ) {
         need_reset=TRUE;
#ifdef DEBUG_DISK
         printk("fdc_out timed out\n");
#endif
	 return RTEMS_UNSATISFIED;
      }
   } while ( (fl_status & (MASTER | DIRECTION)) != (MASTER | 0));
   outport_byte(FDC_DATA, val);
   return RTEMS_SUCCESSFUL;
}

/*-------------------------------------------------------------------
 * int fdc_results(void)
 * Extract results from the controller after an operation,
 * then allow floppy interrupts again.
 */
rtems_status_code
fdc_results (void)
{

   rtems_interval then, now;
   int status;
   register unsigned char fl_status;
   unsigned int result_nr=0;

   if (need_reset) return RTEMS_UNSATISFIED;
   /* if controller is not listening, return */

   /* It may take several tries to get the FDC to accept a command. */

   rtems_clock_get (RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &then);

   /* Extract bytes from FDC until it says it has no more.  The loop is
    * really an outer loop on result_nr and an inner loop on status.
    */
   for (;;) {
      inport_byte(FDC_STATUS,fl_status);
      status =  fl_status & (MASTER | DIRECTION| CTL_BUSY);
      if (status == (MASTER | DIRECTION | CTL_BUSY)) {
	 if ( result_nr >= MAX_RESULTS) break;	/* too many results */
	 inport_byte(FDC_DATA,f_results[result_nr++]);
	 continue;
      }
      if (status == MASTER) {
#ifdef DEBUG_DISK
         printk("fdc_results OK\n");
#endif
	 return RTEMS_SUCCESSFUL;
      }
      rtems_clock_get (RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &now);
      if ( (TIMEOUT << 2) < (now - then) ){
#ifdef DEBUG_DISK
         printk("fdc_results timed out\n");
#endif
	 need_reset=1;
	 return RTEMS_UNSATISFIED;
      }
   }
   return RTEMS_UNSATISFIED;
}

/*-------------------------------------------------------------------
 * void start_stop_motor()
 * Turns on/off drive motor.
 */
void inline
start_stop_motor()
{
   unsigned int command = 0;
   int fl;

   /* Calculate motor status */
   for ( fl=NR_DRIVES-1; fl>=0; fl--) {
      command <<= 1;
      command |= floppy[fl].fl_motor;
   }
   outport_byte(DOR, (command << MOTOR_SHIFT) | ENABLE_INT );
}

/*-------------------------------------------------------------------
 * void start_motor()
 * Turns on / off drive motor.
 */
void inline
start_motor ( floppy_t *fd )
{
   if (fd->fl_motor == START_MOTOR) {
      /* Prevent motor from being stopped */
      rtems_timer_fire_after(fd->fl_timer_id,TICKS_PER_SEC << 2,
			     auto_stop_motor,(void*)(fd));
      return;
   }
   fd->fl_motor = START_MOTOR;
   start_stop_motor();
   rtems_task_wake_after( TIMEOUT );
   /* Prevent motor from being stopped */
   rtems_timer_fire_after(fd->fl_timer_id,TICKS_PER_SEC << 2,
			  auto_stop_motor,(void*)(fd));
   return;
}

/*-------------------------------------------------------------------
 * void auto_stop_motor(rtems_id id, void *arg)
 * Turns off drive motor.
 */
void
auto_stop_motor ( rtems_id id, void *arg )
{
   floppy_t *fd = arg;

   /* Get device. Wait four times device timeout */
   rtems_semaphore_obtain(fl_sem_id,RTEMS_WAIT,TIMEOUT << 2);

#ifdef DEBUG_DISK
   printk("Auto stop motor\n");
#endif
   fd->fl_motor = STOP_MOTOR;
   start_stop_motor();

   /* Release device */
   rtems_semaphore_release(fl_sem_id);
}

/*-------------------------------------------------------------------
 * int recalibrate(floppy_t *fd)
 * The floppy disk controller has no way of determining its absolute arm
 * position (cylinder).  Instead, it steps the arm a cylinder at a time and
 * keeps track of where it thinks it is (in software).  However, after a
 * SEEK, the hardware reads information from the diskette telling where the
 * arm actually is.  If the arm is in the wrong place, a recalibration is done,
 * which forces the arm to cylinder 0.  This way the controller can get back
 * into sync with reality.
 *
 */
int
recalibrate ( floppy_t *fd )
{
   rtems_status_code status;

   /* Clear events */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_NO_WAIT,RTEMS_NO_TIMEOUT);
#ifdef DEBUG_DISK
   if (status == RTEMS_SUCCESSFUL) printk("Semaphore obtained %d\n",fd->fl_irq_sem);
#endif

   /* Issue the RECALIBRATE command and wait for the interrupt. */
   start_motor(fd);		/* can't recalibrate with motor off */

   fdc_out(FDC_RECALIBRATE);	/* tell drive to recalibrate itself */
   fdc_out(fd->fl_drive);     	/* specify drive */
   if (need_reset) return RTEMS_UNSATISFIED; /* don't wait if controller is sick*/

   /* Wait until interrupt */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_WAIT,TIMEOUT * 200);
   if (status != RTEMS_SUCCESSFUL) {
#ifdef DEBUG_DISK
      printk("Error calibrating device\n");
#endif
      return status;
   }

   /* Determine if the recalibration succeeded. */
#ifdef DEBUG_DISK
   printk("Disk sense\n");
#endif
   fdc_out(FDC_SENSE);	      /* issue SENSE command to request results */
   status = fdc_results();  /* get results of the FDC_RECALIBRATE command */
   fd->fl_curcyl = NO_CYL;    /* force a SEEK next time */
   if ( status != RTEMS_SUCCESSFUL ||   /* controller would not respond */
	(f_results[ST0] & ST0_BITS) != SEEK_ST0 ||
	f_results[ST_PCN] != 0) {
      /* Recalibration failed.  FDC must be reset. */
#ifdef DEBUG_DISK
      printk("%x %x\n",f_results[ST0],f_results[ST_PCN]);
      printk("Error calibrating device\n");
#endif
      need_reset = TRUE;
      return RTEMS_UNSATISFIED;
   } else {
      /* Recalibration succeeded. */
      fd->fl_calibration = CALIBRATED;
      return RTEMS_SUCCESSFUL;
   }
}

/*-------------------------------------------------------------------
 * void fl_seek(floppy_t *fd)
 * Seeks the floppy device.
 */
rtems_status_code
fl_seek ( floppy_t *fd )
{
   rtems_status_code status;

   if ( fd->fl_calibration == UNCALIBRATED ) {
      if ( recalibrate(fd) != RTEMS_SUCCESSFUL) return RTEMS_UNSATISFIED;
   }

   /* Send seek command to floppy */
   start_motor(fd);
   fdc_out(FDC_SEEK);
   fdc_out(fd->fl_head << 2);
   fdc_out(fd->fl_hardcyl);

   /* Wait until interrupt */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_WAIT,TIMEOUT);
   if (status != RTEMS_SUCCESSFUL) {
#ifdef DEBUG_DISK
      printk("Error seeking device\n");
#endif
      return status;
   }

   fdc_out(FDC_SENSE);
   status = fdc_results();
   if (status !=RTEMS_SUCCESSFUL ||
       (f_results[ST0] & ST0_BITS) != SEEK_ST0
       || f_results[ST1] != fd->fl_hardcyl) {
      /* seek failed, may need a recalibrate */
      return RTEMS_UNSATISFIED;
   }
   fd->fl_curcyl = fd->fl_hardcyl;

   return RTEMS_SUCCESSFUL;
}

/*-------------------------------------------------------------------
 * void fl_reset()
 * Device reset
 */
rtems_status_code
fl_reset()
{
   unsigned int i;

   printk("Reset\n");
   BSP_irq_disable_at_i8259s(FLOPPY_IRQ);
   outport_byte(DOR, 0);		/* strobe reset bit low */
   outport_byte(DOR, ENABLE_INT);	/* strobe it high again */
   BSP_irq_enable_at_i8259s(FLOPPY_IRQ);
   for (i = 0; i < 4; i++) {
      fdc_out(FDC_SENSE);	      /* probe FDC to make it return status */
      (void) fdc_results(); /* flush controller */
   }
   need_reset = 0;
   for (i = 0; i < NR_DRIVES; i++)     /* clear each drive */
      floppy[i].fl_calibration = UNCALIBRATED;
   return RTEMS_SUCCESSFUL;
}

/*-------------------------------------------------------------------
 * fl_read_sect ( floppy_t *fd ):
 * Determine current cylinder and sector.
 * Input:
 *    floppy_t *fd: Pointer to the drive struct
 * Output:
 * Returns:
 */
rtems_status_code
fl_read_sect(floppy_t *fd)
{
   rtems_status_code status;

   /* Never attempt a read id if the drive is uncalibrated or motor is off. */
   if (fd->fl_calibration == UNCALIBRATED) return(ERR_READ_ID);
   start_motor(fd);

   /* clear interrupt events */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_NO_WAIT,TIMEOUT);

   /* The command is issued by outputting 2 bytes to the controller chip. */
   fdc_out(FDC_READ_ID);		/* issue the read id command */
   fdc_out( (fd->fl_head << 2) | fd->fl_drive);

   /* Block, waiting for disk interrupt. */
   if (need_reset) return(ERR_READ_ID);	/* if controller is sick, abort op */

   /* Wait until interrupt */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_WAIT,TIMEOUT);
   if (status != RTEMS_SUCCESSFUL) {
      #ifdef DEBUG_DISK
      printk("Error transfering data\n");
#endif
      return status;
   }

   /* Get controller status and check for errors. */
   status = fdc_results();
   if (status != RTEMS_SUCCESSFUL) {
#ifdef DEBUG_DISK
      printk("Error transfering data\n");
#endif
      return status;
   }

   if ((f_results[ST0] & ST0_BITS) != TRANS_ST0) return(ERR_READ_ID);
   if (f_results[ST1] | f_results[ST2]) return(ERR_READ_ID);

   /* The next sector is next for I/O: */
   fd->fl_sector = f_results[ST_SEC] + 1;
   return RTEMS_SUCCESSFUL;
}

/*===================================================================
 * Part 3: Data transfer entry points
 */

/*-------------------------------------------------------------------
 * fdc_dma_setup ( unsigned mode, size_t bytes, void *buffer ):
 * Initialize DMA chip for floppy
 * Input:
 *    mode: DMA_READ or DMA_WRITE
 * Output:
 * Returns:
 */
static inline void
fdc_dma_setup ( unsigned mode, size_t bytes, unsigned char *buffer )
{
   /* Set up the DMA registers.  (The comment on the reset is a bit strong,
    * it probably only resets the floppy channel.)
    */
   outport_byte(DMA_INIT, DMA_RESET_VAL);    /* reset the dma controller */
   outport_byte(DMA_FLIPFLOP, 0);	     /* write anything to reset it */
   outport_byte(DMA_MODE, mode == FDC_READ ? DMA_READ : DMA_WRITE);
   outport_byte(DMA_ADDR, ( (int)buffer >> 0) & 0xff );
   outport_byte(DMA_ADDR, ( (int)buffer >> 8) & 0xff );
   outport_byte(DMA_TOP,    (int)buffer >> 16 );
   outport_byte(DMA_COUNT, (bytes - 1) >> 0 );
   outport_byte(DMA_COUNT, (bytes - 1) >> 8 );
   outport_byte(DMA_INIT, 2);	/* some sort of enable */
}

/*-------------------------------------------------------------------
 * defuse
 */
static inline void
defuse()
{
/* Invalidate leftover requests in the transfer array. */

  struct trans *tp;

  for (tp = ftrans; tp < ftrans + MAX_SECTORS; tp++) tp->block = -1;
}

/*-------------------------------------------------------------------
 * fl_transfer (floppy_t *fd, unsigned char command, int count)
 * Read, write or format 1 block
 * Input:
 *    floppy_t *fd: Pointer to floppy structure.
 *    unsigned char command: what command is going to be performed.
 *                           Can be FDC_WRITE or FDC_READ.
 *    int count: bytes to be transfered.
 * Output:
 * Returns:
 */
rtems_status_code
fl_transfer (floppy_t *fd, unsigned char command, int count)
{
/* The drive is now on the proper cylinder.  Read, write or format 1 block. */

   int r;
   rtems_status_code status;

   /* Never attempt a transfer if the drive is uncalibrated or motor is off. */
   if (fd->fl_calibration == UNCALIBRATED) return(ERR_TRANSFER);
   start_motor(fd);

   /* Clean up interrupts */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_NO_WAIT,TIMEOUT);

   fdc_out(command);
   fdc_out((fd->fl_head << 2) | fd->fl_drive );
   fdc_out(fd->fl_cylinder);
   fdc_out(fd->fl_head);
   fdc_out(fd->fl_sector);
   fdc_out(SECTOR_SIZE_CODE);
   fdc_out(fd->fl_sectors); /* sectors per track */
   fdc_out(fd->fl_gap);	/* sector gap */
   fdc_out(DTL);      	/* data length */

   /* Block, waiting for disk interrupt. */
   if (need_reset) return(ERR_TRANSFER);	/* if controller is sick, abort op */


   /* Wait until interrupt */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_WAIT,TIMEOUT);
   if (status != RTEMS_SUCCESSFUL) {
#ifdef DEBUG_DISK
      printk("Error transfering data\n");
#endif
      return status;
   }

   /* Get controller status and check for errors. */
   r = fdc_results();
   if ( r != RTEMS_SUCCESSFUL ) return(r);

   if (f_results[ST1] & WRITE_PROTECT) {
      printk("Diskette is write protected.\n");
      return(ERR_WR_PROTECT);
   }

#ifdef DEBUG_DISK
   printk("Transfer result: %x %x %x\n",f_results[ST0],f_results[ST1],f_results[ST2]);
#endif
   if ((f_results[ST0] & ST0_BITS) != TRANS_ST0) return(ERR_TRANSFER);
   if (f_results[ST1] | f_results[ST2]) return(ERR_TRANSFER);

   /* Compare actual numbers of sectors transferred with expected number. */
   if ( ( chs2offset(f_results[ST_CYL],f_results[ST_HEAD],f_results[ST_SEC],fd)
	  - chs2offset(fd->fl_cylinder, fd->fl_head, fd->fl_sector, fd) )
        != count ) return(ERR_TRANSFER);

   /* This sector is next for I/O: */
   fd->fl_sector = f_results[ST_SEC];
   return RTEMS_SUCCESSFUL;
}

/*-------------------------------------------------------------------
 * rtems_status_code
 * fl_finish (floppy_t *fd, unsigned f_count);
 * Carry out the I/O requests gathered in ftrans[].
 */
rtems_status_code
fl_finish (floppy_t *fd, unsigned *f_count, unsigned char command)
{
   rtems_status_code status = RTEMS_SUCCESSFUL;
   struct trans *tp;
   int errors;

   if (*f_count == 0) return RTEMS_SUCCESSFUL;	/* Spurious finish. */

   /* Let read_id find out the next sector to read/write if it pays to do so.
    * Note that no read_id is done while formatting if there is one format
    * request per track as there should be.
    */
   fd->fl_sector = BASE_SECTOR;

   do {
      /* See if motor is running; if not, turn it on and wait */
      start_motor(fd);

      /* This loop allows a failed operation to be repeated. */
      errors = 0;
      for (;;) {
         status = RTEMS_SUCCESSFUL;
	 /* First check to see if a reset is needed. */
	 if (need_reset) fl_reset();

	 if (fd->fl_sector == NO_SECTOR) {
	    /* Don't retry read_id too often, we need tp soon */
	    if (errors > 0) fd->fl_sector = BASE_SECTOR;
	 }

	 /* Look for the next job in ftrans[] */
	 if (fd->fl_sector == NO_SECTOR)  return RTEMS_UNSATISFIED;
	 for (;;) {
	    if (fd->fl_sector >= BASE_SECTOR + fd->fl_sectors)
	       fd->fl_sector = BASE_SECTOR;

	    tp = &ftrans[fd->fl_sector - BASE_SECTOR];
#ifdef DEBUG_DISK
            printk("sector %d %d\n",fd->fl_sector, tp->block);
#endif
	    if (tp->block >= 0) break;
	    fd->fl_sector++;
	 }

	 if (status == RTEMS_SUCCESSFUL && command == FDC_WRITE) {
	    /* Copy the bad user buffer to the DMA buffer. */
            memcpy(tp->dma_buf,tp->usr_buf,SECTOR_SIZE);
	 }

	 /* Set up the DMA chip and perform the sector transfer. */
	 if (status == RTEMS_SUCCESSFUL) {
#ifdef DEBUG_DISK
            printk("Transfer\n");
#endif
	    fdc_dma_setup(command, SECTOR_SIZE, tp->dma_buf);
	    status = fl_transfer(fd, command, SECTOR_SIZE);
	 }

	 if (status == RTEMS_SUCCESSFUL && command == FDC_READ) {
#ifdef DEBUG_DISK
            printk("Read\n");
#endif
	    /* Copy the DMA buffer to the bad user buffer. */
            memcpy(tp->usr_buf,tp->dma_buf,SECTOR_SIZE);
	 }

	 if (status == RTEMS_SUCCESSFUL) break;	/* if successful, exit loop */

	 /* Don't retry if write protected or too many errors. */
	 if ( ++errors >= MAX_ERRORS) {
	    /* if (fd->fl_sector != 0) tp->iop->count = 0; */
	    return RTEMS_UNSATISFIED;
	 }

	 /* Recalibrate if halfway, but bail out if optional I/O. */
	 if (errors == MAX_ERRORS / 2) {
	    fd->fl_calibration = UNCALIBRATED;
	 }
      }
      *f_count -= SECTOR_SIZE;
#ifdef DEBUG_DISK
      printk("fcount %d\n",*f_count);
#endif
      tp->block = -1;
      /* tp->iop->io_nbytes -= tp->tr_count; */
   } while (*f_count > 0);

   /* Defuse the leftover partial jobs. */
   defuse(); /**/

   return RTEMS_SUCCESSFUL;
}

/*-------------------------------------------------------------------
 * rtems_status_code
 * fl_read_write(floppy_t *fd, char command, int count, void *buffer)
 * Performs data transfering schedule, and calls fl_finish for
 * actual transfer.
 * Input:
 *    floppy_t *fd: Pointer to floppy structure.
 *    char command: FDC_READ or FDC_WRITE
 *
 */
rtems_status_code
fl_read_write( fd, command, data)
               floppy_t *fd;
               unsigned char command;
               rtems_libio_rw_args_t *data;
{
   rtems_status_code status;
   rtems_libio_t *iop = data->iop;
   unsigned f_count = 0, count, nbytes =  data->count;
   unsigned sector, offset = data->offset;
   unsigned f_nexttrack;
   unsigned char *usr_buf = data->buffer;
   struct trans *tp;

   if (data->count <= 0 ) return RTEMS_UNSATISFIED; /* Bad request */
   if (offset != iop->offset) return RTEMS_UNSATISFIED;
#ifdef DEBUG_DISK
   printk("offset %d\n", offset);
#endif
   start_motor(fd);


   /* Which block on disk and how close to EOF? */
   if (offset >= iop->size) return(RTEMS_SUCCESSFUL); /* At EOF */
   if (offset + nbytes > iop->size) nbytes = iop->size - offset;
   offset >>= SECTOR_SHIFT;
   f_nexttrack = offset;

   /* While there are "unscheduled" bytes in the request: */
   do {
      count = nbytes;

      if (f_count == 0) {
	 offset2chs( offset, &fd->fl_cylinder,
		     &fd->fl_head, &sector, fd );

	 /* See where the next track starts, one is trouble enough */
	 f_nexttrack = (fd->fl_cylinder * NR_HEADS
			+ fd->fl_head + 1) * fd->fl_sectors;
#ifdef DEBUG_DISK
         printk("First step %d\n", sector);
#endif
      }

      if (f_count > 0 && offset >= f_nexttrack) {
	 /* The new job leaves the track, finish all gathered jobs */
	 /* If first byte is not in the current cylinder, reseek */
	 if (fd->fl_cylinder != fd->fl_curcyl) {
	    fd->fl_hardcyl = fd->fl_cylinder;
	    fl_seek(fd);
	 } /**/
#ifdef DEBUG_DISK
         printk("Ready to transfer %d\n", sector);
#endif
	 status = fl_finish(fd, &f_count, command);
	 if (status != RTEMS_SUCCESSFUL) return RTEMS_UNSATISFIED;
      }

      /* Don't do track spanning I/O. */
      if ( offset + (count >> SECTOR_SHIFT) > f_nexttrack )
	 count = (f_nexttrack - offset) << SECTOR_SHIFT;

      /* Store the I/O parameters in the ftrans slots for the sectors to
       * read.  The first slot specifies all sectors, the ones following
       * it each specify one sector less.  This allows I/O to be started
       * in the middle of a block.
       */
      tp = &ftrans[sector];
      nbytes -= count;
      f_count += count;

      do {
	 tp->block   = offset;
	 tp->iop     = iop;
	 tp->dma_buf = fd->dma_buffer;
	 tp->usr_buf = usr_buf;
	 tp++;
	 /* dma_buf += SECTOR_SIZE; */
	 usr_buf += SECTOR_SIZE;
	 count   -= SECTOR_SIZE;
	 offset++;
      } while (count > 0);

   } while (nbytes > 0);

   /* If first byte is not in the current cylinder, reseek */
   if (fd->fl_cylinder != fd->fl_curcyl) {
      fd->fl_hardcyl = fd->fl_cylinder;
      fl_seek(fd);
   } /**/
#ifdef DEBUG_DISK
   printk("Final transfer %d %d\n", f_count, sector);
#endif
   status = fl_finish(fd, &f_count, command);
   if (status != RTEMS_SUCCESSFUL) return RTEMS_UNSATISFIED;

   return RTEMS_SUCCESSFUL;
}

/*===================================================================
 * Part 4: Driver entry points
 */

/*-------------------------------------------------------------------
 * fl_initialize(major, minor, arg)
 * Initialize entry point for floppy device driver. This function
 * installs the IRQ handler and creates a device called fd0.
 *
 * Input:
 * Output:
 * Return:
 */
rtems_device_driver
fl_initialize ( major,minor,arg )
                rtems_device_major_number major;
                rtems_device_minor_number minor;
		void                      *arg;
{
   rtems_status_code status;
   int i;

   /* Create device mutex  */
   status = rtems_semaphore_create(rtems_build_name('F','D','S','0'),
				   1,RTEMS_DEFAULT_ATTRIBUTES,100,&fl_sem_id);
   fl_irq_sem = 0;

   /* Install IRQ handler */
   status = BSP_install_rtems_irq_handler(&floppy_isr_data);
   if (!status){ /* Stop system if error */
      printk("Error installing floppy interrupt handler!\n");
      rtems_fatal_error_occurred(status);
   }

   /* Register the device(s) */
   status = rtems_io_register_name ("/dev/fd0", major, 0);
   if (status != RTEMS_SUCCESSFUL)
   {
      printk("Error registering floppy device!\n");
      rtems_fatal_error_occurred (status);
   }

#ifdef DEBUG_DISK
   printk("Reset\n");
#endif
   fl_reset();
   for (i=0;i<NR_DRIVES;i++) {
      floppy[i].fl_motor = STOP_MOTOR;
      floppy[i].fl_opened = FALSE;
   }
#ifdef DEBUG_DISK
   printk("Stop\n");
#endif
   start_stop_motor();

   /* Specify */
   fdc_out(FDC_SPECIFY);
   fdc_out(SPEC1);
   fdc_out(SPEC2); /**/

   printk("Floppy controller installed\n");
   return RTEMS_SUCCESSFUL;
};

/*-------------------------------------------------------------------
 * fl_open(major, minor, arg)
 * Open the floppy device.
 */
rtems_device_driver
fl_open       ( major,minor,arg )
                rtems_device_major_number major;
                rtems_device_minor_number minor;
                void                      *arg;

{
   rtems_status_code status;
   rtems_libio_open_close_args_t *args = arg;

   if (floppy[minor].fl_opened == TRUE) return RTEMS_UNSATISFIED;

   /* Get device. Wait four times device timeout */
   status = rtems_semaphore_obtain(fl_sem_id,RTEMS_WAIT,TIMEOUT << 2);
   check_rtems_status("Semaphore timed out (device busy)\n");

   status = rtems_semaphore_create(rtems_build_name('F','D','i','0' + minor),
                                   1,
                                   RTEMS_SIMPLE_BINARY_SEMAPHORE | RTEMS_INHERIT_PRIORITY | RTEMS_PRIORITY,
                                   RTEMS_NO_PRIORITY,
                                   &floppy[minor].fl_irq_sem);
   check_rtems_status("Unable to create device semaphore\n");

   /* Get Task ID */
   args->iop->data1 = (void *) (&floppy[minor]);

   /* Get dma buffer */
   status = rtems_partition_get_buffer(dma_buff_id,(void *)&floppy[minor].dma_buffer);
   check_rtems_status("Unable to get dma memory\n");

   /* Create motor off timer */
   status = rtems_timer_create(rtems_build_name('F','D','T','0'+minor),
			       &floppy[minor].fl_timer_id);
   check_rtems_status("Error creating timer\n");

   /* Enable motor auto-stop wait for four seconds */
   status = rtems_timer_fire_after(floppy[minor].fl_timer_id,TICKS_PER_SEC << 2,
				   auto_stop_motor,(void*)(&floppy[minor]));
   check_rtems_status("Error starting motor auto-stop\n");

   /* Allways start motor after device opening */
   floppy[minor].fl_curcyl = 0; floppy[minor].fl_hardcyl     = 0;
   floppy[minor].fl_sector = 0; floppy[minor].fl_cylinder    = 0;
   floppy[minor].fl_head   = 0; floppy[minor].fl_calibration = UNCALIBRATED;
   floppy[minor].fl_motor  = STOP_MOTOR;
   need_reset = 0;

   /* Disk geometry.
    * Driver should try different geometries before choosing one.
    * By this time it works only with 1440k disks.
    */
   floppy[minor].fl_blocks = flc_nr_blocks[6];
   floppy[minor].fl_gap    = flc_gap[6];
   floppy[minor].fl_rate   = flc_rate[6];
   floppy[minor].fl_steps  = flc_steps_per_cyl[6];
   floppy[minor].fl_opened = TRUE;
   floppy[minor].fl_sectors     = flc_nr_sectors[6];
   floppy[minor].fl_test_sector = flc_test_sector[6];
   args->iop->offset = 0;
   args->iop->size = floppy[minor].fl_blocks * SECTOR_SIZE; 

   status = rtems_semaphore_obtain(floppy[minor].fl_irq_sem,RTEMS_NO_WAIT,TIMEOUT);
   fl_irq_sem = floppy[minor].fl_irq_sem;

   recalibrate(&floppy[minor]);
   fl_seek(&floppy[minor]); /* Drive must be recalibrated after opening */
   defuse();

   /* Release device */
   rtems_semaphore_release(fl_sem_id);

   return RTEMS_SUCCESSFUL;
};

/*-------------------------------------------------------------------
 * fl_close(major, minor, arg)
 * Close the floppy device.
 */
rtems_device_driver
fl_close      ( major,minor,arg )
                rtems_device_major_number major;
                rtems_device_minor_number minor;
                void                      *arg;
{
   rtems_status_code status;
/*   rtems_libio_open_close_args_t *args = arg; */

   /* Get device. Wait four times device timeout */
   status = rtems_semaphore_obtain(fl_sem_id,RTEMS_WAIT,TIMEOUT << 2);

   /* Delete auto-stop timer */
   rtems_timer_delete(floppy[minor].fl_timer_id);
   floppy[minor].fl_motor  = STOP_MOTOR;
   floppy[minor].fl_opened = FALSE;
   start_stop_motor();

   rtems_semaphore_release(floppy[minor].fl_irq_sem);

   status = rtems_semaphore_delete(floppy[minor].fl_irq_sem);
   if (status != RTEMS_SUCCESSFUL)
      printk("Unable to delete semaphore at %d\n",minor);

   /* Release device */
   rtems_semaphore_release(fl_sem_id);

   return RTEMS_SUCCESSFUL;
};

/*-------------------------------------------------------------------
 * fl_read(major, minor, arg)
 * Read the floppy device.
 */
rtems_device_driver
fl_read       ( major,minor,arg )
                rtems_device_major_number major;
                rtems_device_minor_number minor;
                void                      *arg;
{
   rtems_status_code status;
   rtems_libio_rw_args_t   *args = arg;
   floppy_t *fd;

#ifdef DEBUG_DISK
   printk("Reading %d %d\n",major,minor);
#endif
   /* Get device. Wait four times device timeout */
   status = rtems_semaphore_obtain( fl_sem_id,RTEMS_WAIT,TIMEOUT << 2 );
   if (status != RTEMS_SUCCESSFUL) return RTEMS_UNSATISFIED;

   fd = (floppy_t *) args->iop->data1;
   if (fd != &floppy[minor]) {
      printk("Driver error!\n");
      rtems_fatal_error_occurred(status);
   }

   /* Clear events */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_NO_WAIT,TIMEOUT);
   fl_irq_sem = fd->fl_irq_sem;

   /* Prevent motor from being stopped */
   start_motor(fd);

   if (fl_read_write(fd, FDC_READ, args) != RTEMS_SUCCESSFUL) {
      rtems_semaphore_release(fl_sem_id);
      return RTEMS_UNSATISFIED;
   } 
   args->bytes_moved = args->count;

   /* Release device */
   rtems_semaphore_release(fl_sem_id);

   return RTEMS_SUCCESSFUL;
};

/*-------------------------------------------------------------------
 * fl_write(major, minor, arg)
 * Write the floppy device.
 */
rtems_device_driver
fl_write      ( major,minor,arg )
                rtems_device_major_number major;
                rtems_device_minor_number minor;
                void                      *arg;
{
   rtems_status_code status;
   rtems_libio_rw_args_t   *args = arg;
   floppy_t *fd;

#ifdef DEBUG_DISK
   printk("Writing %d %d\n",major,minor);
#endif

   /* Get device. Wait four times device timeout */
   status = rtems_semaphore_obtain( fl_sem_id,RTEMS_WAIT,TIMEOUT << 2 );
   if (status != RTEMS_SUCCESSFUL) return RTEMS_UNSATISFIED;

   fd = (floppy_t *) args->iop->data1;
   if (fd != &floppy[minor]) {
      printk("Driver error!\n");
      rtems_fatal_error_occurred(status);
   }

   /* Clear events */
   status = rtems_semaphore_obtain(fd->fl_irq_sem,RTEMS_NO_WAIT,TIMEOUT);
   fl_irq_sem = fd->fl_irq_sem;

   /* Prevent motor from being stopped */
   start_motor(fd);

   if (fl_read_write(fd, FDC_WRITE, args) != RTEMS_SUCCESSFUL) {
      rtems_semaphore_release(fl_sem_id);
      return RTEMS_UNSATISFIED;
   } 
   args->bytes_moved = args->count;

   /* Release device */
   rtems_semaphore_release(fl_sem_id);

   return RTEMS_SUCCESSFUL;
};

/*-------------------------------------------------------------------
 * fl_control(major, minor, arg)
 * Control command the floppy device.
 */
rtems_device_driver
fl_control    ( major,minor,arg )
               rtems_device_major_number major;
               rtems_device_minor_number minor;
               void                      *arg;
{
   rtems_status_code status;
   rtems_libio_ioctl_args_t  *args = arg;
   floppy_t *fd;

   /* Get device. Wait four times device timeout */
   rtems_semaphore_obtain(fl_sem_id,RTEMS_WAIT,TIMEOUT << 2);

   fd = (floppy_t *) args->iop->data1;

   switch (args->command) {
   default:
      status = RTEMS_SUCCESSFUL;
      break;
   }

   /* Release device */
   rtems_semaphore_release(fl_sem_id);
   return status;
};

#undef check_rtems_status
