#ifndef storable_h__
#define storable_h__
///////////////////////////////////////////////////////////////////////////
// $Header: /usr/local/cvs-repository/ssf/src/storable.h,v 1.1.1.1 2001/01/13 21:01:17 rosimildo Exp $
//
// Copyright (c) 2000-2001 ConnectTel, Inc. All Rights Reserved.
// by: Rosimildo da Silva
//  
// The SSF library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
// 02111-1307, USA
//
//
// DESCRIPTION: Framework to map classes to relational database tables.
//
// HISTORY:
//
// $Log: storable.h,v $
// Revision 1.1.1.1  2001/01/13 21:01:17  rosimildo
// Initial checkin for SSF
//
//
///////////////////////////////////////////////////////////////////////////

#include <string>
#include <vector>

#include "ssf_base.h"

namespace ssf
{

/**
 * Abstract base class for any subclass that needs to provide a persistence 
 * mechanism unsing a relational database table.
 */
class Storable
{

protected:
   /** 
    * Returns the table of name associated with this class.
    */
    virtual const char *getTableName() = 0;

   /** 
    * Returns the string representing all column names that are necessary
    * to be mapped to the instance.
    *
    * Example: "col1, col2, ..., colx"
    */
    virtual const char *getColumnList() = 0;


   /** 
    * Returns the string representing all column names that are necessary
    * during an INSERT call. It usually differs from the regular column list
    * because for inserts, in some cases the primary keys( PKs ) might be
    * generated automatically by the DB engine.
    *
    * Example: "col1, col2, ..., colx"
    */
    virtual const char *getColumnListForInsert() = 0;


   /** 
    * Returns the string representing the "where clause" that
    * matches to the instance ID.
    *
    * Example:  "WHERE ID = 1"
    *
    */
    virtual const char *getWhereClause() = 0;

   /** 
    * Returns the string representing all "values" that need to be
    *	present during the insertion of a new "row" to the table.
    */
    virtual const char *getInsertValues() = 0;

 
   /** 
    * Returns string representing the "SET" of values of an 
    * update statement.
    */
    virtual const char * getUpdateValues() = 0;


   /** 
    * This function is called after one updated is performed. It allows
    * the instance to fetch a possible new "ID" for the object.
    */
    virtual void setPKAfterInsert( Connection * ) {}

public:

   /** 
    * This method is implemented by each "Storable" descendents to map the 
    * fields returned from a ResultSet and maps it to the data members of 
    * the instance.
    */
    virtual bool mapping( ResultSet *rs ) = 0;

   /** 
    * Unique ID represeting the current instance.
    */
    virtual long getUniqueId() = 0;


   /** 
    * inserts a new "row" representing this instance to the
    * corresponding table in the DB..
    */
    virtual bool addNewRow( Connection *c );


   /** 
    * updates the "row" that this instance represents.
    */
    virtual bool updateRow( Connection *c );


   /** 
    * Deletes the "row" that the instance represents on the DB.
    */
    virtual bool deleteRow( Connection *c );

   /** 
    * Refreshes the current instance from the DB.
    * NOTE: It is important that the PK for the table is 
    * valid whenever you call this function.
    */
    virtual bool queryRow( Connection *c );
};


/**
 * Generic factory class to create "Storables" instances from
 * query matching a set of rows in a particular DB table.
 */
template< class T >
class StorableFactory
{
public:
    typedef std::vector< T * > StorableList;

   /** 
    * This method returns a list of instances that "matches" the
    * query defined by the "where clause". For each row, in the 
    * match, the "mapping" method is called on the instance to
    * fill out the values.
    */
    static StorableList *getList( Connection *c, const char *where );
};


/** 
 * Returns list of instances ... or an empty list if no matches...
 */
template< class T >
StorableFactory< T >::StorableList *StorableFactory< T >::getList
( 
   Connection *c,
   const char *where
)
{
   StorableList *l = 0;
   std::string s = std::string( "SELECT * FROM " ) + T::TABLE_NAME;
   if( where )
   {
      s += std::string( " WHERE " ) + where;
   }
   ResultSet *res = c->executeQuery( s.c_str() );
   if( res && c->success() )
   {
      l = new StorableList;
      while( res->next() )
      {
         T *storable = new T;
         storable->mapping( res );
         l->push_back( storable );
      }
   }
   delete res;
   return l;
}

};

#endif //  storable_h__

