/////////////////////////////////////////////////////////////////////////////
// $Header: /usr/local/cvs-repository/ssf/java/Storable.java,v 1.1.1.1 2001/01/13 21:01:17 rosimildo Exp $
//
// Copyright (c) 2000-2001 ConnectTel, Inc. All Rights Reserved.
//  
// The SSF library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
// 02111-1307, USA
//  
// MODULE DESCRIPTION: Java implmentation of the SSF -
// Simple Storable Framework.  
// SSF is a persistency framework that maps java/C++ objects to
// DB tables. The idea is to have a simple, and generic mechanism
// to allows java classes to map to DB tables in a very simple way.
//
// MODIFICATION/HISTORY:
//
// $Log: Storable.java,v $
// Revision 1.1.1.1  2001/01/13 21:01:17  rosimildo
// Initial checkin for SSF
//
//
/////////////////////////////////////////////////////////////////////////////

package ssf;

import java.sql.*;
import java.util.*;

/**
 * Base class for classes that need to provide persistence unsing a relational
 * database table.
 */
public abstract class Storable
{
   /** 
    * Returns the table of name associated with this class.
    */
   protected abstract String getTableName();

   /** 
    * Returns the string representing all column names that are necessary
    * to be mapped to the instance.
    *
    * Example: "col1, col2, ..., colx"
    */
   protected  abstract String getColumnList();


   /** 
    * Returns the string representing all column names that are necessary
    * during an INSERT call. It usually differs from the regular column list
    * because for inserts, in some cases the primary keys( PKs ) might be
    * generated automatically by the DB engine.
    *
    * Example: "col1, col2, ..., colx"
    */
   protected  abstract String getColumnListForInsert();


   /** 
    * Returns the string representing the "where clause" that
    * matches to the instance ID.
    *
    * Example:  "WHERE ID = 1"
    *
    */
   protected  abstract String getWhereClause();

   /** 
    * Returns the string representing all "values" that need to be
    *	present during the insertion of a new "row" to the table.
    */
   protected  abstract String getInsertValues();

 
   /** 
    * Returns string representing the "SET" of values of an 
    * update statement.
    */
   protected  abstract String getUpdateValues();


   /** 
    * This function is called after one updated is performed. It allows
    * the instance to fetch a possible new "ID" for the object.
    */
   public void setPKAfterInsert( Connection c, Statement s ) {}

   /** 
    * This method is implemented by each "Storable" descendents to map the 
    * fields returned from a ResultSet and maps it to the data members of 
    * the instance.
    */
   public abstract boolean mapping( ResultSet rs ) throws SQLException; 

   /** 
    * Unique ID represeting the current instance.
    */
   public abstract long getUniqueId();


   /** 
    * inserts a new "row" representing this instance to the
    * corresponding table in the DB..
    */
   public  boolean addNewRow( Connection c ) throws SQLException 
    {
      String s;
      s = "INSERT INTO " + getTableName();
      String fields = getColumnListForInsert();
      String values = getInsertValues();
      if( fields != null )
      {
        s =  s + " (" + fields + " )";
      }
      s = s + " VALUES(" + values + " )";
      Statement stmt = null;
      try 
      { 
	    stmt = c.createStatement();
	    stmt.executeUpdate( s );
        setPKAfterInsert( c, stmt );
	    stmt.close();
	    return true;
      } 
	  catch(SQLException ex) 
	  {
	    try 
	    {
	      if( stmt != null ) stmt.close();
	    }
        catch(SQLException ex2) { throw ex2; } 
	    throw ex; 
	  }
   }

   /** 
    * updates the "row" that this instance represents.
    */
   public  boolean updateRow( Connection c ) throws SQLException 
    {
       String s;
       String fields = getUpdateValues();
       s = "UPDATE " + getTableName() + " SET " + fields;
       String where = getWhereClause();
       if( where != null  )
	   {  
	     s = s + " WHERE " + where;
	   }
       Statement stmt = null;
       try 
       { 
	    stmt = c.createStatement();
	    stmt.executeUpdate( s );
	    stmt.close();
	    return true;
       } 
	   catch(SQLException ex) 
	   {
	    try 
	    {
	      if( stmt != null ) stmt.close();
	    }
        catch(SQLException ex2) { throw ex2; } 
	    throw ex; 
	   }
    }


   /** 
    * Deletes the "row" that the instance represents on the DB.
    */
   public  boolean deleteRow( Connection c ) throws SQLException 
    {
       String s;
       s = "DELETE FROM " + getTableName();
       String where = getWhereClause();
	   if( where != null )
 	   {
	     s = s+ " WHERE " + where;
 	   }
       Statement stmt = null;
       try 
       { 
	    stmt = c.createStatement();
	    stmt.executeUpdate( s );
	    stmt.close();
	    return true;
       } 
	   catch(SQLException ex) 
	   {
	    try 
	    {
	      if( stmt != null ) stmt.close();
	    }
        catch(SQLException ex2) { throw ex2; } 
	    throw ex; 
	   }
    }

   /** 
    * Refreshes the current instance from the DB.
    * NOTE: It is important that the PK for the table is 
    * valid whenever you call this function.
    */
   public  boolean queryRow( Connection c ) throws SQLException 
    {
       String s;
       String fields = getColumnList();
       s = "SELECT " + fields + " FROM " + getTableName();
       String where = getWhereClause();
       if( where != null )
	   {
	     s =  s + " WHERE " + where;
	   }
       Statement stmt = null;
       try 
       {
         stmt = c.createStatement();
         ResultSet rs = stmt.executeQuery( s );
         if( rs.next() )
         {
            mapping( rs );
         }
         stmt.close();
         return true;
       }
       catch( SQLException e ) 
       {
         e.printStackTrace();
         try 
         {
            if( stmt != null )
               stmt.close();
         }
         catch( SQLException e2 )
         {
            e2.printStackTrace();
            throw e2;
         }
         throw e;
       }
    }
}