/*
 * $Id: openeth.c,v 1.1 2005-08-11 15:29:52+02 rolf Exp rolf $
 *
 * lwIP ethernet driver for the Open-Ethernet core (used in the Leon2 processor).
 * By Rolf.Schroedter@dlr.de, August 2005
 *
 * Heavily based on open_eth driver for RTEMS by Jiri Gaisler, Gaisler Research.
 */
/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

#include <bsp.h>                /* Defines LEON_REG */
#include <string.h>

#include "lwip/opt.h"
#include "lwip/def.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include <lwip/stats.h>
#include <netif/etharp.h>

/* define some debug output macros */
#ifdef OPENETH_DEBUG
#define DEBUG_OUTPUT(...) { fprintf(stderr, __VA_ARGS__); fflush(stderr); }
#else /* OPENETH_DEBUG */
#define DEBUG_OUTPUT(...)
#endif /* OPENETH_DEBUG */


/* Define those to better describe your network interface. */
#define IFNAME0 'o'
#define IFNAME1 'e'

/*******************************************************************
* OPEN ETHERNET hardware registers
********************************************************************/
#define OETH_BASE_ADDRESS    0xB0000000      /* OPENETH registers base address */
#define OETH_VECTOR          0x1C            /* Default interrupt vector */

#define OETH_TOTAL_BUFFERS   128             /* Total maximum of TX/RX buffer descriptors */
#define OETH_TX_BUFFERS      16              /* Number of Transmit Buffers/Descriptor Areas */
#define OETH_RX_BUFFERS      16              /* Number of Receive Buffers/Descriptor Areas */
#define OETH_MAX_BUFLEN      0x610           /* Maximum buffer size */
#define OETH_MIN_BUFLEN      64              /* Minimum message length */

#include "openeth_def.h"

#define OETH_RX_EVENT        RTEMS_EVENT_1   /* Interrupt receiver event */

/*******************************************************************
* Access OPEN ETHERNET registers inside Leon registers
********************************************************************/
static struct openeth_regs *regs = (struct openeth_regs *)OETH_BASE_ADDRESS;

static struct {
  int           enable100MHz;           /* 10 or 100 MHz interface speed */

  rtems_id      rxDaemonTid;            /* Task id of receiver daemon */

  /* Interface statistics */
  unsigned long rxInterrupts;           /* Number of RX interrupts */
  unsigned long rxPackets;
  unsigned long rxLengthError;
  unsigned long rxNonOctet;
  unsigned long rxBadCRC;
  unsigned long rxOverrun;
  unsigned long rxMiss;
  unsigned long rxCollision;

  unsigned long txInterrupts;
  unsigned long txDeferred;
  unsigned long txHeartbeat;
  unsigned long txLateCollision;
  unsigned long txRetryLimit;
  unsigned long txUnderrun;
  unsigned long txLostCarrier;
  unsigned long txRawWait;

  /* Transmit & Receive buffers */
  int           txIdx, rxIdx;                   /* Index to TX/RX buffers */
  unsigned8     *txBuf[OETH_TX_BUFFERS];     /* Pointers to allocated TX buffers */
  unsigned8     *rxBuf[OETH_RX_BUFFERS];     /* Pointers to allocated RX buffers */
} oeth;

/*******************************************************************
* OPEN ETHERNET declarations
********************************************************************/
struct ethernetif {
  struct eth_addr *ethaddr;
  /* Add whatever per-interface state that is needed here. */
};

static const struct eth_addr ethbroadcast = {{0xff,0xff,0xff,0xff,0xff,0xff}};

/* Forward declarations. */
static void  openeth_input(struct netif *netif);
static err_t openeth_output(struct netif *netif, struct pbuf *p,
             struct ip_addr *ipaddr);

/*******************************************************************
* OPEN ETHERNET interrupt service
********************************************************************/
static rtems_isr
openeth_rxInterrupt (rtems_vector_number v)
{
  unsigned32 status;

  /*
  * Read & Clear interrupt cause
  */
  status = regs->int_src;
  regs->int_src = status;

  /*
  * Frame received?
  */
  if (status & (OETH_INT_RXF | OETH_INT_RXE)) {
    oeth.rxInterrupts++;
    rtems_event_send (oeth.rxDaemonTid, OETH_RX_EVENT);
  }
}

/*******************************************************************
* OPEN ETHERNET background task
********************************************************************/
static
void openeth_rxDaemon( void *arg )
{
  rtems_event_set events;

  for (;;) {
    DEBUG_OUTPUT("openeth: Waiting for RX event...\n");

    rtems_event_receive( OETH_RX_EVENT, RTEMS_WAIT | RTEMS_EVENT_ANY,
                         RTEMS_NO_TIMEOUT, &events);

    DEBUG_OUTPUT("openeth: RX event received\n");

    /*
    * More than one packet could have been received for this interrupt:
    * Read out ALL available packets at one time.
    * Loop HERE, because low_level_input() can read ONLY ONE PACKET.
    */
    while (! (regs->xd[oeth.rxIdx + OETH_TX_BUFFERS].len_status & OETH_RX_BD_EMPTY) ) {

      DEBUG_OUTPUT("openeth:::rxDaemon Read 1 packet rxIdx=%d\n", oeth.rxIdx);

      openeth_input( (struct netif *)arg );
      oeth.rxIdx = (oeth.rxIdx + 1) % OETH_RX_BUFFERS;
    }
  }
  printf("\nopeneth: fatal - exit rxDaemon\n");
}

/*******************************************************************
* Read/write MII (Media Independent Interface) registers
********************************************************************/
static unsigned32 read_mii(unsigned32 addr)
{
    while (regs->miistatus & OETH_MIISTATUS_BUSY) {}
    regs->miiaddress = addr << 8;
    regs->miicommand = OETH_MIICOMMAND_RSTAT;
    while (regs->miistatus & OETH_MIISTATUS_BUSY) {}
    if (!(regs->miistatus & OETH_MIISTATUS_NVALID))
        return(regs->miirx_data);
    else {
        printf("\nopeneth: failed to read mii\n");
        return (0);
    }
}

static void write_mii(unsigned32 addr, unsigned32 data)
{
    while (regs->miistatus & OETH_MIISTATUS_BUSY) {}
    regs->miiaddress = addr << 8;
    regs->miitx_data = data;
    regs->miicommand = OETH_MIICOMMAND_WCTRLDATA;
    while (regs->miistatus & OETH_MIISTATUS_BUSY) {}
}

/*******************************************************************
* TCP packet dum, stolen from lwIP's tcpdump.c
********************************************************************/
#ifdef OPENETH_DEBUG

#include <stdio.h>

static void rawdump(void *p, int size, char *indent)
{
    int i;
    unsigned8 *addr=p;

    for (i=0; i<size; i++) {
        if( i%16 == 0 ) printf("%s", indent);
        printf("%02x ", addr[i]);
        if( i%16 == 15 ) printf("\n");
    }
    if( size%16 != 0 ) printf("\n");
}

#endif  /* OPENETH_DEBUG */

/*******************************************************************
* LWIP low-level functions
********************************************************************/
/*
* low_level_init();
*
* Should initialize the ethernet hardware & driver
*/
static void
low_level_init(struct netif *netif)
{
  int mii_cr = 0;
  int i;

  DEBUG_OUTPUT("openeth::low_level_init() BEGIN\n");

  /*
  * Reset interface settings and statistics.
  * Should be called only once, otherwise memory-leaks (alloc's)
  */
  memset( &oeth, 0, sizeof(oeth) );

  /*
  * Enable 100 MHz Ethernet if Leon clock >= 50 MHz
  * Clock detected from Leon's Scaler Reload Register
  */
  oeth.enable100MHz = (LEON_REG.Scaler_Reload >= 49);

  /*
  * Start the receiver daemon
  */
  oeth.rxDaemonTid = sys_thread_new( openeth_rxDaemon, netif, 0 );

  /*
  * Check whether the MAC address has been set
  *   before calling openeth_init(&netif).
  * Otherwise set some default PRIVATE MAC address (AC-DE-48-xx-xx-xx):
  */
  if( netif->hwaddr_len != 6 ) {
    netif->hwaddr_len = 6;
    netif->hwaddr[0] = 0xAC;
    netif->hwaddr[1] = 0xDE;
    netif->hwaddr[2] = 0x48;
    netif->hwaddr[3] = 0x01;
    netif->hwaddr[4] = 0x02;
    netif->hwaddr[5] = 0x03;
  }

  /* maximum transfer unit */
  netif->mtu = 1500;

  /* broadcast capability */
  netif->flags = NETIF_FLAG_BROADCAST;

  /* Do whatever else is needed to initialize interface. */

  /*
  * Reset the controller
  */
  regs->ctrlmoder = 0;
  regs->moder = OETH_MODER_RST;    /* Reset ON */
  regs->moder = 0;                    /* Reset OFF */

  /*
  * Clear interrupts
  */
  LEON_REG.Interrupt_Clear  = (1 << (OETH_VECTOR - 0x10));
  LEON_REG.Interrupt_Mask  |= (1 << (OETH_VECTOR - 0x10));

  /*
  * Reset PHY and wait for completion
  */
  mii_cr = oeth.enable100MHz ? 0x3300 : 0x0000;
  write_mii(0, mii_cr | 0x8000);
  while (read_mii(0) & 0x8000) {}
  if (!oeth.enable100MHz) write_mii(0, 0);
  mii_cr = read_mii(0);
  printf("openeth: driver attached, (%d Mbit) PHY config : 0x%04x\n", oeth.enable100MHz ? 100:10, mii_cr);

  /*
  * Setup Transmit Descriptor Areas
  */
  regs->tx_bd_num = OETH_TX_BUFFERS;

  /*
  * Setup min/max packet length
  * Really a minimum packet length ?
  */
  regs->packet_len = (OETH_MIN_BUFLEN << 16) + OETH_MAX_BUFLEN;    /* 0x00400600 */

  /*
  * Setup IPGT register to recomended values
  */

  /* Set IPGT,IPGR1,IPGR2,COLLCONF register to recomended value */
  regs->ipgt = 0x00000015;
  regs->ipgr1 = 0x0000000c;
  regs->ipgr2 = 0x00000012;
  regs->collconf = 0x000f003f;

  /*
  * Alloc & Initialize TX buffers/descriptors
  * Last TX buffer wraps arounds.
  * Options:    PAD: Short packets are padded
  *             CRC: CRC is added to each packet
  */
  for (i = 0; i < OETH_TX_BUFFERS; i++) {
    regs->xd[i].len_status = /* OETH_TX_BD_PAD | */ OETH_TX_BD_CRC;
    oeth.txBuf[i] = calloc(1, OETH_MAX_BUFLEN);
  }
  regs->xd[OETH_TX_BUFFERS - 1].len_status |= OETH_TX_BD_WRAP;

  /*
  * Alloc & Initialize RX buffers/descriptors
  * Last RX buffer wraps arounds.
  */
  for (i = 0; i < OETH_RX_BUFFERS; i++) {
    int k = i + OETH_TX_BUFFERS;

    regs->xd[k].len_status = OETH_RX_BD_EMPTY | OETH_RX_BD_IRQ;
    oeth.rxBuf[i] = (unsigned char *)calloc(1, OETH_MAX_BUFLEN);
    regs->xd[k].addr = (unsigned32 *)oeth.rxBuf[i];
  }
  regs->xd[OETH_TX_BUFFERS + OETH_RX_BUFFERS - 1].len_status |= OETH_RX_BD_WRAP;

  /*
  * Setup ethernet hardware address.
  */
  regs->mac_addr1 = netif->hwaddr[0] << 8  | netif->hwaddr[1];
  regs->mac_addr0 = netif->hwaddr[2] << 24 | netif->hwaddr[3] << 16
                  | netif->hwaddr[4] << 8  | netif->hwaddr[5];

  /*
  * Install interrupt vector
  */
  set_vector (openeth_rxInterrupt, OETH_VECTOR, 1);

  /*
  * clear all pending interrupts
  */
  regs->int_src = 0xffffffff;

  /*
  * MAC mode register: PAD, IFG, CRCEN
  *     OETH_MODER_CRCEN: Add CRC word (32 bits) to each received frame
  *                       RoS: Seems OETH doesn't work well without OETH_MODER_CRCEN
  */
  regs->moder = OETH_MODER_PAD | OETH_MODER_CRCEN | ((mii_cr & 0x100) << 2);

  /*
  * Enable interrupts & transmitter/receiver
  */
  regs->int_mask = OETH_INT_MASK_RXF | OETH_INT_MASK_RXE | OETH_INT_MASK_RXC;
  regs->moder |= OETH_MODER_RXEN | OETH_MODER_TXEN;

  DEBUG_OUTPUT("openeth::low_level_init() END\n");
}

/*******************************************************************
* LWIP low-level functions
********************************************************************/
/*
 * low_level_output():
 *
 * Should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 */
static err_t
low_level_output(struct netif *netif, struct pbuf *p)
{
/*  struct ethernetif *ethernetif = netif->state; */
  struct pbuf *q;

  unsigned32 len_status, size;
  unsigned8  *addr;

  DEBUG_OUTPUT("openeth::low_level_output() BEGIN\n");

  /*
  * We have 16 TX buffers.
  * If the current buffer isn't transmitted, we should wait here.
  */
  while (regs->xd[oeth.txIdx].len_status & OETH_TX_BD_READY) {
    rtems_task_wake_after(RTEMS_YIELD_PROCESSOR);
  }

#if ETH_PAD_SIZE
  pbuf_header(p, -ETH_PAD_SIZE);                        /* drop the padding word */
#endif

  size = 0;
  addr = oeth.txBuf[oeth.txIdx];

  for(q = p; q != NULL; q = q->next) {
    /* Send the data from the pbuf to the interface, one pbuf at a
       time. The size of the data in each pbuf is kept in the ->len
       variable. */
    if( size + q->len > OETH_MAX_BUFLEN) {
      fprintf( stderr, "openeth: error - output buffer too large\n");
      return ERR_BUF;
    }
    memmove( addr + size, q->payload, q->len);
    size += q->len;
  }
  regs->xd[oeth.txIdx].addr = (unsigned32 *)addr;

  /*
  * Clear all of the status flags.
  * If the frame is short, tell CPM to pad it.
  */
  len_status = regs->xd[oeth.txIdx].len_status & ~OETH_TX_BD_STATS;
  if (size < OETH_MIN_BUFLEN) {
    len_status |= OETH_TX_BD_PAD;
    size = OETH_MIN_BUFLEN;
  } else {
    len_status &= ~OETH_TX_BD_PAD;
  }

#ifdef OPENETH_DEBUG
  printf("openeth::low_level_output() packet txIdx=%d, addr=%p, size=%d\n", oeth.txIdx, addr, size );
  rawdump(addr, size, "  <=  ");
#endif

  /*
  * Write buffer descriptor length and status.
  * Packet will be sent.
  */
  len_status &= 0x0000ffff;
  len_status |= (size << 16) | (OETH_TX_BD_READY | OETH_TX_BD_CRC);
  regs->xd[oeth.txIdx].len_status = len_status;
  oeth.txIdx = (oeth.txIdx + 1) % OETH_TX_BUFFERS;

#if ETH_PAD_SIZE
  pbuf_header(p, ETH_PAD_SIZE);                 /* reclaim the padding word */
#endif

#if LINK_STATS
  lwip_stats.link.xmit++;
#endif /* LINK_STATS */

  DEBUG_OUTPUT("openeth::low_level_output() END\n");
  return ERR_OK;
}

/*******************************************************************
* LWIP low-level functions
********************************************************************/
/*
 * low_level_input():
 *
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 */

static struct pbuf *
low_level_input(struct netif *netif)
{
/*  struct ethernetif *ethernetif = netif->state; */
  struct pbuf *p, *q;

  unsigned32 size, ofs, len_status;
  unsigned8  *addr;
  int        bad;

  DEBUG_OUTPUT("openeth::low_level_input() BEGIN\n");

  while (! ((len_status = regs->xd[oeth.rxIdx + OETH_TX_BUFFERS].len_status) & OETH_RX_BD_EMPTY) ) {
    bad = 0;
    if (len_status & (OETH_RX_BD_TOOLONG | OETH_RX_BD_SHORT)) {
      oeth.rxLengthError++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- too long or too short\n");
    }
    if (len_status & OETH_RX_BD_DRIBBLE) {
      oeth.rxNonOctet++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- dribble nibble\n");
    }
    if (len_status & OETH_RX_BD_CRCERR) {
      oeth.rxBadCRC++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- CRC error\n");
    }
    if (len_status & OETH_RX_BD_OVERRUN) {
      oeth.rxOverrun++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- RX overrun\n");
    }
    if (len_status & OETH_RX_BD_MISS) {
      oeth.rxMiss++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- M MISS\n");
    }
    if (len_status & OETH_RX_BD_LATECOL) {
      oeth.rxCollision++;
      bad = 1;
      DEBUG_OUTPUT("openeth::low_level_input() -- late collision\n");
    }
    if ( ! bad ) {
      /*
      * Obtain the size of the packet and put it into the
      * "size" variable.
      */
      size = (len_status >> 16) - 4;   /* OETH_MODER_CRCEN: 4 additional CRC bytes */

#if ETH_PAD_SIZE
      size += ETH_PAD_SIZE;       /* allow room for Ethernet padding */
#endif

      /*
      * We allocate a pbuf chain of pbufs from the pool.
      */
      p = pbuf_alloc(PBUF_RAW, size, PBUF_POOL);

      if (p != NULL) {

        DEBUG_OUTPUT( "openeth: low_level_input(): allocated pbuf=%08x\n", p);

#if ETH_PAD_SIZE
        pbuf_header(p, -ETH_PAD_SIZE);                      /* drop the padding word */
#endif

        /*
        * We iterate over the pbuf chain until we have read the entire
        * packet into the pbuf.
        */
        ofs = 0;
        addr = oeth.rxBuf[oeth.rxIdx];

#ifdef OPENETH_DEBUG
        printf("openeth::low_level_input() packet rxIdx=%d, addr=%p, size=%d\n", oeth.rxIdx, addr, size );
        rawdump(addr, size, "  =>  ");
#endif

        for(q = p; q != NULL; q = q->next) {
          /*
          * Read enough bytes to fill this pbuf in the chain. The
          * available data in the pbuf is given by the q->len
          * variable.
          */
          memmove(q->payload, addr + ofs, q->len);
          ofs += q->len;
        }
        ++oeth.rxPackets;

#if ETH_PAD_SIZE
        pbuf_header(p, ETH_PAD_SIZE);   /* reclaim the padding word */
#endif

#if LINK_STATS
        lwip_stats.link.recv++;
#endif /* LINK_STATS */
      } else {
        /* if p==NULL the drop packet */
        fprintf(stderr, "openeth: low_level_input(): --- pbuf allocation failed ---\n", p);
#if LINK_STATS
        lwip_stats.link.memerr++;
        lwip_stats.link.drop++;
#endif /* LINK_STATS */
      }
    } /* if bad */

    /*
    * Acknowledge that packet has been read.
    */
    regs->xd[oeth.rxIdx + OETH_TX_BUFFERS].addr = (unsigned32 *)oeth.rxBuf[oeth.rxIdx];
    regs->xd[oeth.rxIdx + OETH_TX_BUFFERS].len_status =
      (regs->xd[oeth.rxIdx + OETH_TX_BUFFERS].len_status &
        ~OETH_TX_BD_STATS) | OETH_TX_BD_READY;

  } /* while */

  DEBUG_OUTPUT("openeth::low_level_input() END\n");
  return p;
}

/*******************************************************************
* LWIP high-level functions
********************************************************************/
/*
 * openeth_output():
 *
 * This function is called by the TCP/IP stack when an IP packet
 * should be sent. It calls the function called low_level_output() to
 * do the actual transmission of the packet.
 *
 */

static err_t
openeth_output(struct netif *netif, struct pbuf *p,
      struct ip_addr *ipaddr)
{

 /* resolve hardware address, then send (or queue) packet */
  return etharp_output(netif, ipaddr, p);

}

/*******************************************************************
* LWIP high-level functions
********************************************************************/
/*
 * openeth_input():
 *
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface.
 *
 */

static void
openeth_input(struct netif *netif)
{
  struct ethernetif *ethernetif;
  struct eth_hdr *ethhdr;
  struct pbuf *p;

  ethernetif = netif->state;

  /* move received packet into a new pbuf */
  p = low_level_input(netif);
  /* no packet could be read, silently ignore this */
  if (p == NULL) return;
  /* points to packet payload, which starts with an Ethernet header */
  ethhdr = p->payload;

#if LINK_STATS
  lwip_stats.link.recv++;
#endif /* LINK_STATS */

  ethhdr = p->payload;

  switch (htons(ethhdr->type)) {
  /* IP packet? */
  case ETHTYPE_IP:
DEBUG_OUTPUT("openeth_input: Received IP packet\n");
    /* update ARP table */
    etharp_ip_input(netif, p);
    /* skip Ethernet header */
    pbuf_header(p, -sizeof(struct eth_hdr));
    /* pass to network layer */
    netif->input(p, netif);
    break;

    case ETHTYPE_ARP:
DEBUG_OUTPUT("openeth_input: Received ARP packet\n");
      /* pass p to ARP module  */
      etharp_arp_input(netif, ethernetif->ethaddr, p);
      break;
    default:
DEBUG_OUTPUT("openeth_input: Received UNKNOWN packet\n");
      pbuf_free(p);
      p = NULL;
      break;
  }
}

/*******************************************************************
* LWIP high-level functions
********************************************************************/
static void
arp_timer(void *arg)
{
  etharp_tmr();
  sys_timeout(ARP_TMR_INTERVAL, arp_timer, NULL);
}

/*******************************************************************
* LWIP high-level functions
********************************************************************/
/*
 * openeth_init():
 *
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 */

err_t
openeth_init(struct netif *netif)
{
  struct ethernetif *ethernetif;

  ethernetif = mem_malloc(sizeof(struct ethernetif));

  if (ethernetif == NULL)
  {
        LWIP_DEBUGF(NETIF_DEBUG, ("openeth_init: out of memory\n"));
        return ERR_MEM;
  }

  netif->state = ethernetif;
  netif->name[0] = IFNAME0;
  netif->name[1] = IFNAME1;
  netif->output = openeth_output;
  netif->linkoutput = low_level_output;

  ethernetif->ethaddr = (struct eth_addr *)&(netif->hwaddr[0]);

  low_level_init(netif);

  etharp_init();

  sys_timeout(ARP_TMR_INTERVAL, arp_timer, NULL);

  return ERR_OK;
}

