/////////////////////////////////////////////////////////////////////////////
// $Header: $
//
// Copyright (c) 2000 ConnectTel, Inc. All Rights Reserved.
// by: Rosimildo da Silva.
//  
// MODULE DESCRIPTION:
//
// Panel to show the counters information retrieved from the RTEMS monitor.
//
// MODIFICATION/HISTORY:
//
// $Log: $
//
/////////////////////////////////////////////////////////////////////////////

// Swing packages
import javax.swing.JTabbedPane;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.border.*;

// AWT packages
import java.awt.*;
import java.awt.event.*;

/**
 * This class is responsible for displaying the "Counters" information
 * retrieved from the Rtems Spy monitor.
 * It is a "panel" based control that all interface for the counters.
 */
public class CountersControl extends JPanel implements java.util.Observer,
             java.awt.event.ActionListener {
  
    // Instance variables...
    private BaseMonitor       upMonitor;
    private BaseMonitor       centerMonitor;
    private BaseMonitor       downMonitor;
    private CounterSelection  cs;
    private Rtems.CounterType upCounterType;
    private Rtems.CounterType centerCounterType;
    private Rtems.CounterType downCounterType;


    //
    // Ctor - Initializes the panel controlling the counters for the RTEMS spy
    //
    public CountersControl() 
    {

     // Let's set the default value to something that is not used for monitoring..
     upCounterType     = Rtems.CounterType.CLOCK;
     centerCounterType = Rtems.CounterType.CLOCK;
     downCounterType   = Rtems.CounterType.CLOCK;

     cs = new CounterSelection();
     upMonitor = new BaseMonitor( "Up" );
     centerMonitor = new BaseMonitor( "Center" );
     downMonitor = new BaseMonitor( "Down" );

     GridLayout grid;

     //Add the tabbed pane to this panel.
     grid = new GridLayout( 4, 1 ); 
     setLayout( grid ); 

     upMonitor.mc.observable.addObserver( this );
     centerMonitor.mc.observable.addObserver( this );
     downMonitor.mc.observable.addObserver( this );

     cs.addListener( this );

     add( cs );
     add( upMonitor );
     add( centerMonitor );
     add( downMonitor );

    }


    //
    // This is called by the "Observable" object when it is time to
    // get the infromation to display on one of the "monitors".
    //
    public void update( java.util.Observable o, java.lang.Object arg )
    {
      BaseMonitor.MonitorComponent mc = ( BaseMonitor.MonitorComponent )arg;
      Rtems.CounterInfo c;

      if( mc == upMonitor.mc )
      {
         refreshMonitor( upCounterType, upMonitor );
      }
      else if( mc == centerMonitor.mc )
      {
         refreshMonitor( centerCounterType, centerMonitor );
      }
      else
      {
         refreshMonitor( downCounterType, downMonitor );
      }
      
    }

    //
    // Called internally to refresh the values of a particular counter..
    //
    private void refreshMonitor( Rtems.CounterType theCounter, BaseMonitor monitor )
    {
         Rtems.CounterInfo c;

         double percent = 0.3;
         String top  = null;
         String bottom = null;

         c = RtemsORBWrapper.monitorRef.getCounter( theCounter );
         switch( theCounter.value() )
         {
            case Rtems.CounterType._CPU:
            percent = 1.0 - (double)c.u_cpu().idle_ticks / (double)c.u_cpu().total_ticks;
            bottom = "Idle Ticks= " + c.u_cpu().idle_ticks;
            top = "Total Ticks= " + c.u_cpu().total_ticks;
            break;

            case Rtems.CounterType._CLOCK:
            percent = 0.0;
            bottom = "Ticks= ";
            top = "Ticks= ";
            break;

            case Rtems.CounterType._MEMORY:
            break;


            // All other cases should be handled here...
            default:    
            percent = (double)c.u_default().cur_inuse/(double)c.u_default().max_available;
            bottom  = "Current Inuse: " + c.u_default().cur_inuse;
            top     = "Max Available: " + c.u_default().max_available;
            break;
         }

         percent = percent*100.0;
         monitor.mc.setCurrentValue( ( float )percent );
         if( bottom != null )
         {
            monitor.mc.setBottomMsg( bottom );
         }
         if( top != null )
         {
            monitor.mc.setTopMsg( top );
         }

    }


    //
    // maps the option selected by the user ( string based ) to a 
    // enum that defines the type of counter selected.
    //
    private Rtems.CounterType mapCounterType()
    {
      String item = (String)cs.combo.getSelectedItem();

      if( item.equals( CounterSelection.OPTION_CPU ) )
      {
         return Rtems.CounterType.CPU;
      } 
      if( item.equals( CounterSelection.OPTION_MEMORY ) )
      {
         return Rtems.CounterType.MEMORY;
      } 

      // RTEMS API counters....
      if( item.equals( CounterSelection.OPTION_RTEMS_THREAD ) )
      {
         return Rtems.CounterType.RTEMS_THREAD;
      }
      if( item.equals( CounterSelection.OPTION_RTEMS_SEMAPHORE ) )
      {
         return Rtems.CounterType.RTEMS_SEMAPHORE;
      } 
      if( item.equals( CounterSelection.OPTION_RTEMS_TIMER ) )
      {
         return Rtems.CounterType.RTEMS_TIMER;
      }
      if( item.equals( CounterSelection.OPTION_RTEMS_REGION ) )
      {
         return Rtems.CounterType.RTEMS_REGION;
      }
      if( item.equals( CounterSelection.OPTION_RTEMS_PARTITION ) )
      {
         return Rtems.CounterType.RTEMS_PARTITION;
      }

      // POSIX API counters....
      if( item.equals( CounterSelection.OPTION_POSIX_THREAD ) )
      {
         return Rtems.CounterType.POSIX_THREAD;
      }
      if( item.equals( CounterSelection.OPTION_POSIX_SEMAPHORE ) )
      {
         return Rtems.CounterType.POSIX_SEMAPHORE;
      }
      if( item.equals( CounterSelection.OPTION_POSIX_MUTEX ) )
      {
         return Rtems.CounterType.POSIX_MUTEX;
      }
      if( item.equals( CounterSelection.OPTION_POSIX_COND_VAR ) )
      {
         return Rtems.CounterType.POSIX_COND_VARIABLE;
      }
      if( item.equals( CounterSelection.OPTION_POSIX_KEY ) )
      {
         return Rtems.CounterType.POSIX_KEY;
      }
      if( item.equals( CounterSelection.OPTION_POSIX_INTERRUPT ) )
      {
         return Rtems.CounterType.POSIX_INTERRUPT;
      }

      // default value... should never get here...
      return Rtems.CounterType.CPU;
    }


    //
    // Event handler called when one of the buttons on the panel
    // is clicked. It is responsible for selecting the proper
    // counter to get its graph displayed on the appropriated
    // panel.
    //
    public synchronized void actionPerformed( ActionEvent e ) 
    {
      String item = (String)cs.combo.getSelectedItem();
      if( e.getActionCommand().equals( CounterSelection.UP_ACTION ) )
      {
         System.out.println( "Up Action called" );
         Rtems.CounterType tempType = mapCounterType();
         if( tempType != upCounterType )
         {
            upCounterType = tempType;
            System.out.println( "Up - restting all" );
            upMonitor.mc.resetAll();
            upMonitor.setMonitorTitle( item );
            System.out.println( "Up - Starting monitor" );
            upMonitor.mc.start();
            System.out.println( "Up - is done" );
         }
      } 
      else if (e.getActionCommand().equals(CounterSelection.CENTER_ACTION ) ) 
      {
         System.out.println( "Center Action called" );
         Rtems.CounterType tempType = mapCounterType();
         if( tempType != centerCounterType )
         {
            centerCounterType = tempType;
            System.out.println( "Center - restting all" );
            centerMonitor.mc.resetAll();
            centerMonitor.setMonitorTitle( item );
            System.out.println( "Center - Starting monitor" );
            centerMonitor.mc.start();
            System.out.println( "Center - is done" );
         }
      }
      else if (e.getActionCommand().equals(CounterSelection.DOWN_ACTION ) ) 
      {
         System.out.println( "Down Action called" );
         Rtems.CounterType tempType = mapCounterType();
         if( tempType != downCounterType )
         {
            downCounterType = tempType;
            System.out.println( "Down - restting all" );
            downMonitor.mc.resetAll();
            downMonitor.setMonitorTitle( item );
            System.out.println( "Down - Starting monitor" );
            downMonitor.mc.start();
            System.out.println( "Down - is done" );
         }
      }
    }


    //
    // Test driver for the Counters Panel ---
    //
    public static void main(String[] args) 
    {
      try
      {
          RtemsORBWrapper.getORB( args, "rps1", "6000" );  // create and initialize the ORB
          if( !RtemsORBWrapper.setupConnection() )
		  {
             System.out.println("Could not set remote connection." );
			 System.exit( 1 );
          }
          System.out.println("Connected successfully." );
      } 
      catch (Exception e) 
      {
          System.out.println("ERROR : " + e) ;
          e.printStackTrace(System.out);
          System.exit( 1 );
      }

        JFrame frame = new JFrame("Counters Demo");
        frame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
        });

        frame.getContentPane().add(new CountersControl(), 
                                   BorderLayout.CENTER);
        frame.setSize(400, 125);
        frame.setVisible(true);
    }
}
