/* $Id: xsyms.c,v 1.13 2004/05/21 18:20:32 till Exp $ */
/* utility to remove the data from all sections
 * except .symtab, .strtab, .shstrtab.
 * The "stripped" output generated by this program
 * may then be used as a vxWorks or CEXP "sym" file.
 * NOTE: simply deleting the respective sections does not
 * work.
 *
 * Author: Till Straumann <strauman@slac.stanford.edu>, 2001
 *
 * This program is based on an example found in the solaris
 * manpage for elf_begin().
 *
 */
#include  <fcntl.h>
#include  <sys/stat.h>
#include  <stdio.h>
/* #include  <elf.h> */
#include  <libelf/libelf.h>
#include  <stdlib.h>
#include  <string.h>
#include  <unistd.h>

static void myfailure();

/* this option controls whether an existing
 * file is modified or whether to create a
 * new file and copy the relevant parts of
 * the old one.
 *
 * Both solutions are "quick hacks" and maybe
 * not really clean solutions as I am not too
 * familiar with the ELF format and the library.
 * NOTE: as of now (2/2002), the 'purging' version
 * which recreates a new ELF file is maybe cleaner
 * than the older approach...
 */

/*
 * Copyright 2002, Stanford University and
 * 		Till Straumann <strauman@slac.stanford.edu>
 * 
 * Stanford Notice
 * ***************
 * 
 * Acknowledgement of sponsorship
 * * * * * * * * * * * * * * * * *
 * This software was produced by the Stanford Linear Accelerator Center,
 * Stanford University, under Contract DE-AC03-76SFO0515 with the Department
 * of Energy.
 * 
 * Government disclaimer of liability
 * - - - - - - - - - - - - - - - - -
 * Neither the United States nor the United States Department of Energy,
 * nor any of their employees, makes any warranty, express or implied,
 * or assumes any legal liability or responsibility for the accuracy,
 * completeness, or usefulness of any data, apparatus, product, or process
 * disclosed, or represents that its use would not infringe privately
 * owned rights.
 * 
 * Stanford disclaimer of liability
 * - - - - - - - - - - - - - - - - -
 * Stanford University makes no representations or warranties, express or
 * implied, nor assumes any liability for the use of this software.
 * 
 * This product is subject to the EPICS open license
 * - - - - - - - - - - - - - - - - - - - - - - - - - 
 * Consult the LICENSE file or http://www.aps.anl.gov/epics/license/open.php
 * for more information.
 * 
 * Maintenance of notice
 * - - - - - - - - - - -
 * In the interest of clarity regarding the origin and status of this
 * software, Stanford University requests that any recipient of it maintain
 * this notice affixed to any distribution by the recipient that contains a
 * copy or derivative of this software.
 */
#define DOWRITE (argc>optind+1 || ofd>=0)

#define I_OPENFLAGS	O_RDONLY
#define O_FLGS ( O_RDWR | O_TRUNC | O_CREAT )
#define O_MODE (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH )
#define ELF_BEGINFLAGS	ELF_C_READ

static void
usage(char *nm)
{
char *chpt=strrchr(nm,'/');
if (chpt)
	nm=chpt+1;
fprintf(stderr,"usage: %s [-p] [-z] [-h] [<infile> [<outfile>] ]\n", nm);
fprintf(stderr,"       %s implementation using LIBELF\n",nm);
fprintf(stderr,"       $Id: xsyms.c,v 1.13 2004/05/21 18:20:32 till Exp $\n\n");
fprintf(stderr,"       strip an ELF file leaving only the symbol table\n");
fprintf(stderr,"       - if there's no <infile>, operate on stdin/out as a filter\n");
fprintf(stderr,"       - if there's no <outfile>, just list section headers\n");
fprintf(stderr,"       -h this info\n");
fprintf(stderr,"       -p really 'purge' other sections rather than setting\n");
fprintf(stderr,"          their size to zero (default)\n");
fprintf(stderr,"       -z dont 'purge' other sections; just set their size to zero\n");
fprintf(stderr,"       -C generate a  C-source file for building into the executable\n");
fprintf(stderr,"       -a (UNSUPPORTED) determine BFD target CPU architecture\n");
}

static Elf_Scn *
copyscn(Elf *eout, Elf_Scn *from)
{
Elf_Data    *s, *d;
Elf_Scn     *to;
Elf32_Shdr  *nshdr, *shdr=elf32_getshdr(from);

	to=elf_newscn(eout);
	nshdr=elf32_getshdr(to);
	nshdr->sh_name = shdr->sh_name;
	nshdr->sh_type = shdr->sh_type;
	nshdr->sh_size = shdr->sh_size;
	nshdr->sh_info = shdr->sh_info;
	nshdr->sh_addralign = shdr->sh_addralign;
	nshdr->sh_entsize = shdr->sh_entsize;

	for (s=0; (s=elf_getdata(from,s)); ) {
		d=elf_newdata(to);
		*d=*s;
		elf_flagdata(d,ELF_C_SET,ELF_F_DIRTY);
	}
	return to;
}

#ifndef xsyms_main
#define xsyms_main main
#endif

int
xsyms_main(int argc, char ** argv)
{
Elf32_Shdr     *shdr, *nshdr=0;
Elf32_Ehdr     *ehdr, *nehdr=0;
Elf            *elf, *eout=0;
Elf_Scn        *scn=0, *nscn=0, *strtab=0, *symtab=0;
Elf_Data       *data=0;
int            fd, ofd=-1, ch;
unsigned int   cnt;
char           *optstr="phzC";
int            purge=1;
int            gensrc = 0;
int            dumparch=0;

	while ((ch=getopt(argc, argv, optstr))>0) {
		switch (ch) {
			default:  /* should never get here */
			case 'h':
				usage(argv[0]); exit(0);
			case 'p':
				purge=1; break;
			case 'z':
				purge=0; break;
			case 'C':
				gensrc=1; break;
			case 'a':
				dumparch=1; break;
		}
	}

	if ( dumparch ) {
		printf("unknown");
		return 0;
	}

	if (argc>optind) {
	/* Open the input file */
	if ((fd = open(argv[optind], I_OPENFLAGS )) == -1) {
	       perror("unable to open infile");
               exit(1);
	}
	if (DOWRITE && (ofd = open(argv[optind+1], O_FLGS, O_MODE)) == -1) {
	       perror("unable to open outfile");
               exit(1);
	}
	} else {
		/* filter stdin to stdout */
		fd = 0; ofd=1;
	}

	/* Obtain the ELF descriptor */
	(void) elf_version(EV_CURRENT);
	if ((elf = elf_begin(fd, ELF_BEGINFLAGS, NULL)) == NULL)
		myfailure(argv[0]);
	if (DOWRITE) {
		if ((eout= elf_begin(ofd,  ELF_C_WRITE, NULL)) == NULL)
			myfailure(argv[0]);
		if (!(nehdr=elf32_newehdr(eout)))
			myfailure(argv[0]);
	}
	/* Obtain the .shstrtab data buffer */
	if (((ehdr = elf32_getehdr(elf)) == NULL) ||
		((scn = elf_getscn(elf, ehdr->e_shstrndx)) == NULL) ||
		((data = elf_getdata(scn, NULL)) == NULL))
			myfailure(argv[0]);
	if (DOWRITE) {
		if (purge) {
			nehdr->e_type=ehdr->e_type;
			nehdr->e_machine=ehdr->e_machine;
			nehdr->e_version=ehdr->e_version;
			nehdr->e_shstrndx=elf_ndxscn(copyscn(eout,scn));
			/* vxWorks also needs a copy of the ident... */
			memcpy(nehdr->e_ident,ehdr->e_ident,EI_NIDENT);
		} else {
			/* just copy the header */
			*nehdr = *ehdr;
		}
		/* vxWorks wants a program header, so give them a dummy */
		elf32_newphdr(eout,1/*ehdr->e_phnum*/);
	}

	/* Traverse input filename, printing each section */
	for (cnt = 1, scn = NULL; (scn = elf_nextscn(elf, scn)); cnt++) {
		if ((shdr = elf32_getshdr(scn)) == NULL)
                    myfailure(argv[0]);
		(void) fprintf(stderr,"[%d] %s (size: %i, offset %i)\n", cnt,
				(char *)data->d_buf + shdr->sh_name,
				shdr->sh_size, shdr->sh_offset);
		if (DOWRITE) {
			if (purge) {
				if (0==strcmp((char *)data->d_buf + shdr->sh_name,
					   ".symtab"))
					symtab=copyscn(eout, scn);
				if (0==strcmp((char *)data->d_buf + shdr->sh_name,
					   ".strtab"))
					strtab=copyscn(eout, scn);
			} else {

				if (strcmp((char *)data->d_buf + shdr->sh_name,
					   ".shstrtab") &&
				    strcmp((char *)data->d_buf + shdr->sh_name,
					   ".symtab") &&
				    strcmp((char *)data->d_buf + shdr->sh_name,
					   ".strtab")) {
					nscn=elf_newscn(eout);
					nshdr=elf32_getshdr(nscn);
					nshdr->sh_size=0;
				} else {
					nscn=copyscn(eout, scn);
					*elf32_getshdr(nscn)=*shdr;
					elf_flagshdr(nscn,ELF_C_SET,ELF_F_DIRTY);
				}
			}
		}
	}

	/* recalculate the link for symtab */
	if (symtab)
		elf32_getshdr(symtab)->sh_link = elf_ndxscn(strtab);

/*TODO*/
#if 0
	nehdr->e_ident[5]=2;
#endif
	elf_update(eout,ELF_C_WRITE);
	elf_end(eout);
	elf_end(elf);
	return 0;
}         /* end main */

static void
myfailure(char *nm)
{
	(void) fprintf(stderr, "%s\n\n", elf_errmsg(elf_errno()));
	usage(nm);
	exit(1);
}
