#ifndef __util_resources__
#define __util_resources__
/////////////////////////////////////////////////////////////////////////////
// $Header: $
//
// Copyright (c) 2000 ConnectTel, Inc. All Rights Reserved.
//  
// MODULE DESCRIPTION:
//  Generic utility package to collect resource information from a
//  RTEMS running target.
//
// MODIFICATION/HISTORY:
//
// $Log: $
//
/////////////////////////////////////////////////////////////////////////////


#include <vector>


// let's 
namespace util
{

typedef unsigned long u32;
typedef char RtemsName[ 5 ];

class Resource
{
 public:
    // type of resources to be monitored.
    enum Type 
    {
	  // MISC resources
      CPU, 
      MEMORY,
      CLOCK,

      // RTEMS based resources
      RTEMS_THREAD,
      RTEMS_SEMAPHORE,
      RTEMS_MESSAGE_QUEUE,
      RTEMS_TIMER,
      RTEMS_PARTITION,
      RTEMS_REGION,

      // POSIX - based resources
      POSIX_THREAD,
      POSIX_SEMAPHORE,
      POSIX_MUTEX,
      POSIX_MESSAGE_QUEUE,
      POSIX_KEY,
      POSIX_INTERRUPT,
      POSIX_COND_VARIABLE

      // iTRON - based resources
    }; 

    // returns the name of the resource
    const char *name() const { return name_; }

    // returns the resource type for the instance
    Type type() const { return type_; }

    // return the unique ID identifing the resource
    u32 id() const { return id_; }

protected:
    // Ctor is called by the "factory" method only.
    Resource( Type t ) : type_( t ), id_( 0 ), attributes_( 0 ) 
    { name_[ 0 ] = '\0'; }

    // data members common to any resource..
    Type       type_;
    u32        id_;
    u32        attributes_;
    RtemsName  name_;
};

// General containers..
typedef std::vector< Resource * >  ResourcePtrList;
typedef ResourcePtrList::iterator  ResourceIterator;



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION: 
//
/////////////////////////////////////////////////////////////////////////////
class RtemsThread: public Resource
{
public:

//
// access fucntions per instance
//

  // Initial priority of the task.
  short  basePriority() const { return basePriority_; }

  // Current priority of the task.
  short  currentPriority() const { return currentPriority_; }

  // current state of this thread
  u32    state() const { return state_; }

  // ID of the object that this thread  is waiting on
  u32    waitId() const { return waitId_; }

  // size of the stack for this task.
  u32    stackSize() const { return stackSize_; }

  // number of ticks uxecuted by this task since the system started.
  u32    ticksExecuted() const { return ticksExecuted_; }

  // boolean indicating whether or not this task is preemptible.
  bool   isPreemptible() const { return isPreemptible_; }

  // events pending on this task.
  u32    eventsPending() const { return eventsPending_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static RtemsThread *create( void *thread_control );


protected:
  // do not let this class be created without using the factory..
  RtemsThread() : Resource( RTEMS_THREAD ) {}

  short      basePriority_;    
  short      currentPriority_; 
  u32        state_;           
  u32        waitId_;          
  u32        stackSize_;       
  u32        ticksExecuted_;   
  boolean    isPreemptible_;   
  u32        eventsPending_;   

};


class RtemsSemaphore: public Resource
{
public:

   // counter of the semaphore.
   u32  count() const { return  count_; }

   // attributes of the instance
   u32  attributes() const { return  attributes_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static RtemsSemaphore *create( void *semaphore_control );

protected:
  // do not let this class be created without using the factory..
  RtemsSemaphore() : Resource( RTEMS_SEMAPHORE ) {}

  u32   count_;           // counter of the semaphore.
};


class CpuUsage
{
public:
   CpuUsage();

   // total number of ticks executed so far..
   u32  totalTicks() const { return totalTicks_; }
   
   // Number of ticks executed by the IDLE thread.
   u32  idleTicks() const { return idleTicks_;  }

private:
   u32  totalTicks_;
   u32  idleTicks_;
};

class MemoryUsage
{
public:
  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

private:
   // no isntances of this class should be created at this time.
   MemoryUsage();
};

class ClockInformation
{
public:
   ClockInformation();

    // number of microseconds per tick		
   u32  uSecsPerTick() const { return uSecsPerTick_; }

   // number of ticks per timeslice
   u32  ticksPerTimeslice() const { return ticksPerTimeslice_; }

    // total number of ticks since start
   u32  ticksSinceBoot() const { return ticksSinceBoot_;  }

private:

   u32  uSecsPerTick_;
   u32  ticksPerTimeslice_; 
   u32  ticksSinceBoot_;    
};


class RtemsTimer: public Resource
{
public:

  // time of day, time interval, etc.
  u32   t_class() const { return t_class_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static RtemsTimer *create( void *timer_control );

private:
  // do not let this class be created without using the factory..
  RtemsTimer() : Resource( RTEMS_TIMER ) {}

  u32   t_class_;  
};


class RtemsMessageQueue: public Resource
{
public:

   // maximum number of messages( size of queue )
   u32 maxMessages() const { return maxMessages_; }

   // current number of messages pending
   u32 pendingMessages() const { return pendingMessages_; }


   // attributes of the instance
   u32  attributes() const { return  attributes_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static RtemsMessageQueue *create( void *msg_control );

protected:
  // do not let this class be created without using the factory..
  RtemsMessageQueue() : Resource( RTEMS_SEMAPHORE ) {}

  u32   maxMessages_;
  u32   pendingMessages_;

};



class RtemsPartition: public Resource
{
public:

   // attributes of the instance
   u32  attributes() const { return  attributes_; }

   // initial address of the partition
   u32  startingAddress() const { return startingAddress_; }

   // size in bytes of the partition
   u32  length() const { return length_; }

   // number of blocks used in this partition
   u32  usedBlocks() const { return usedBlocks_; }

   // size of each buffer in this partition
   u32  bufferSize() const { return bufferSize_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a partition control, it creates an instance of this class
  static RtemsPartition *create( void *partition_control );

private:
  // do not let this class be created without using the factory..
  RtemsPartition() : Resource( RTEMS_PARTITION ) {}

  u32  startingAddress_;
  u32  length_;
  u32  usedBlocks_;
  u32  bufferSize_;
};



class RtemsRegion: public Resource
{
public:

   // attributes of the instance
   u32  attributes() const { return  attributes_; }

   // initial address of the partition
   u32  startingAddress() const { return startingAddress_; }

   // size in bytes of the partition
   u32  length() const { return length_; }

   // number of blocks used in this partition
   u32  usedBlocks() const { return usedBlocks_; }

   // size of each page in this partition
   u32  pageSize() const { return pageSize_; }

  // maximum size of a segment on this region
  u32  maxSegSize() const { return maxSegSize_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a region control, it creates an instance of this class
  static RtemsRegion *create( void *region_control );

private:
  // do not let this class be created without using the factory..
  RtemsRegion() : Resource( RTEMS_REGION ) {}

  u32  startingAddress_;
  u32  length_;
  u32  usedBlocks_;
  u32  pageSize_;
  u32  maxSegSize_; 
};


class PosixThread: public RtemsThread
{
public:

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all POSIX threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static PosixThread *create( void *thread_control );

private:
  // do not let this class be created without using the factory..
  PosixThread() {}
};


class PosixSemaphore: public RtemsSemaphore
{
public:

   // counter of the semaphore.
   u32  openCount() const { return openCount_; }

   // whether this instance is named or not
   bool isNamed() const { return isNamed_; }

   // whether this instance is linked or not
   bool isLinked() const { return isLinked_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static PosixSemaphore *create( void *semaphore_control );

private:
  // do not let this class be created without using the factory..
  PosixSemaphore() {}

  bool isNamed_;
  bool isLinked_;
  u32  openCount_;

};


class PosixKey: public Resource
{
public:

  // boolean indicating whether or not the is active.
  bool isActive() const { return isActive_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static PosixKey *create( void *key_control );

private:
  // do not let this class be created without using the factory..
  PosixKey() : Resource( POSIX_KEY ) {}

  bool isActive_; 
};



class PosixCondVariable: public Resource
{
public:

  // ID of the mutex controling this Cond. Var.
  u32 mutexId() const { return mutexId_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static PosixCondVariable *create( void *cond_var_control );

private:
  // do not let this class be created without using the factory..
  PosixCondVariable() : Resource( POSIX_COND_VARIABLE ) {}

  u32  mutexId_;
};



class PosixMutex: public Resource
{
public:

   // ID of the instance holding this mutex.
   u32  holderId() const { return holderId_; }


   // whether the instance is holded.
   bool  isLocked() const { return isLocked_; }

   u32   nestCount() const { return nestCount_; }

//
// per class interface
//

  // get counters
  static void getCounters( u32 & max_available, u32 & inuse );

  // returns a list with all RTEMS threads 
  static void getInstances( ResourcePtrList & list );

  // given a thread control, it creates an instance of this class
  static PosixMutex *create( void *cond_var_control );

private:
  // do not let this class be created without using the factory..
  PosixMutex() : Resource( POSIX_MUTEX ) {}

  u32  holderId_; 
  bool isLocked_; 
  u32  nestCount_;

};



#if 0
// THIS IS LEFT HERE TO HELP IMPLEMENTING THESE LATER...
struct RtemsMutex
{
   u32          id;              // ID for this instance.
   RtemsName    name;            // Name of the instance. This is an up to 4 digits name.
   u32          attributes;	   // Bitwise desired attribute - See API
   u32          holderId;        // ID of the instance holding this mutex.
   boolean      isLocked;        // whether the instance is holded.
};

struct PosixMessageQueue
{
  RtemsMessageQueue m;   
   boolean    isNamed;        // whether this instance is named
   boolean    isLinked;       // whether this instance is linked
   boolean    isBlocking;     // whether this instance is blocking
   u32        openCount;      // reference counting for opens..
};

struct PosixInterrupt
{
   u32      id;               // ID for this instance.
   boolean  isActive;         // boolean indicating whether or not the is active.
   u32		numberInstalled;  // ???
   u32		lockCount; 		   // ???
   u32		deferredCount;    // ???
};
#endif


}; // namespace


#endif  // __util_resources__


