#ifndef   CALLBACK_H
#define   CALLBACK_H
/////////////////////////////////////////////////////////////////////////////
// $Header: d:/archive/cvs/master/callback/callback.h,v 1.6 2000/02/08 00:53:03 rosimildo Exp $
//
// Copyright (c) 1999 - Rosimildo da Silva
//  
// MODULE DESCRIPTION:
// This module implements a simple Callback framkework for C++ programs. It is very
// portable and should work with any ANSI C++ compiler.
//
// MODIFICATION/HISTORY:
//
// $Log: callback.h,v $
// Revision 1.6  2000/02/08 00:53:03  rosimildo
//
// Fixed bug pointed by Hendrik Schober about a missing return
// statement in the removeFirst() method.
//
// Revision 1.5  1999/11/14 18:00:49  rosimildo
// Fixed missing include when compiling with GCC.
// Added command line options for the test
// when compiling with different C++ compilers.
//
// Revision 1.4  1999/11/14 17:38:02  rosimildo
// A few modifications to the readme file,
// and fixed some spelling on the comments
// of the callback.h.
//
// Revision 1.3  1999/11/14 17:28:55  rosimildo
// Added the copying.lib file with the LGPL license.
// Included the readme.txt file.
//
// Revision 1.2  1999/11/14 17:09:46  rosimildo
// Added comments to describe each class.
// Added headers to the classes.
//
//
/////////////////////////////////////////////////////////////////////////////


#include <vector>      // std::vector<>
#include <algorithm>   // std::find().. 


/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
// This is an abstract class that defines the interface for the Callback
// framework for C++. It is templated on the message "Parameter" to be
// used by the parties involved.
//
/////////////////////////////////////////////////////////////////////////////
template< class Parameter > 
class Callback
{
public:
   // Contract between the clients and the subsystems
   virtual int notify( Parameter * ) = 0;

   virtual ~Callback() { }

   virtual bool operator==( const Callback& c ) { return( this == &c); }
   virtual bool operator<( const Callback &  )  { return 1; }

   // Creates a copy of the instance...
   virtual Callback< Parameter > *clone() = 0;

protected:
   Callback(){}
   // Copy and assignment are not public at this level...
   Callback( const Callback & ){}
   Callback & operator = ( const Callback & ){}
};




/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
// This is a concrete class that implements the interface for the Callback
// framework for C++ for Global functions. This class should be used with
// global functions or static methods of a particular class.
//
/////////////////////////////////////////////////////////////////////////////
template < class Parameter >
class FunctionCallback : public Callback< Parameter >
{
public:
   FunctionCallback() : funcPtr( 0 ) { }
   FunctionCallback( int ( *fPtr ) ( Parameter * ) ) : funcPtr( fPtr ) { }
   virtual ~FunctionCallback() { }

   virtual bool operator==( const FunctionCallback & c ) { return( funcPtr == c.funcPtr ); }

   virtual int notify( Parameter *param ) { return funcPtr ? funcPtr( param ) : 0; }
   void setFunction( int (*fPtr) (Parameter *) )  { funcPtr = fPtr; }

   // Make a copy of this object
   Callback< Parameter > *clone() 
   { return new FunctionCallback< Parameter >( *this ); }

private:
   int ( *funcPtr )( Parameter * );
};





/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
//
// This is a concrete class that implements the interface for the Callback
// framework for C++ for member functions of a class. This class should be 
// used with methods of a class.
//
/////////////////////////////////////////////////////////////////////////////
template < class Parameter, class ClassType >
class MemberCallback : public Callback< Parameter >
{
public:
   MemberCallback() : memberPtr( 0 ), instance( 0 ) { }
   MemberCallback( int (ClassType::*f) (Parameter *), ClassType* o ) :
     memberPtr( f ), instance( o ) {}
   virtual ~MemberCallback() { }

   virtual bool operator==( const MemberCallback& c ) 
   { 
     return( ( memberPtr == c.memberPtr ) && ( instance == c.instance ) ); 
   }

   virtual int notify( Parameter* param )  
   { return  instance ? ( instance->*memberPtr )( param ) : 0; }
   void setMember( int (ClassType::*f) (Parameter* ), ClassType* o )
   {  memberPtr= f;  instance = o;  }

   // Make a copy of this object
   Callback< Parameter > *clone() 
   { return new MemberCallback< Parameter, ClassType >( *this ); }

private:
   int (ClassType::*memberPtr) (Parameter *);
   ClassType  *instance;
};



/////////////////////////////////////////////////////////////////////////////
//  
// DESCRIPTION:
// 
// This class is a container based on STL vector, implemented here as a helper
// class for multi-casting of notifications of the callback framework.
//
/////////////////////////////////////////////////////////////////////////////
template < class Parameter >
class CallbackContainer 
{
  typedef std::vector< Callback< Parameter > * > CallbackVector;

public:
   CallbackContainer() { }
   ~CallbackContainer() { }

   void add( Callback< Parameter > *c )
   {
     v.push_back( c );
   }

   bool find( Callback< Parameter > *c )
   {
      return(  std::find( v.begin(), v.end(), c ) != v.end() ) ? true : false;
   }

   Callback< Parameter > *remove( Callback< Parameter > *c )
   {
     CallbackVector::iterator i = std::find( v.begin(), v.end(), c );
     if( i != v.end() )
     {
        v.erase( i );
        return *i;
     }
     return 0;
   }

   Callback< Parameter > *removeFirst()
   {
     if( v.size() )
     {
        // get element from the fron the of the array, and
        // return to the caller.
        Callback< Parameter > *temp = *v.begin();
        v.erase( v.begin() );
        return temp;
     }
     return 0;
   }

   // not inline because of warnings about inline functions
   // with for() loops.
   void notify( Parameter* param );


   int size() { return v.size(); }

private:
  // Container to hold the callbacks
  std::vector< Callback< Parameter > * > v;

};

template < class Parameter >
void CallbackContainer< Parameter >::notify( Parameter* param )
{
  for( CallbackVector::iterator i = v.begin(); i != v.end(); ++i )
  {
     (*i)->notify( param );
  }
}  

#endif // CALLBACK_H




