/*
 * Copyright (c) 2001-2003 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

#include <unistd.h>
#include <fcntl.h>

#include "lwip/opt.h"

#include "lwip/mem.h"
#include "lwip/memp.h"
#include "lwip/sys.h"

#include "lwip/stats.h"


#include "lwip/tcpip.h"

#include "netif/openeth.h"
#include "netif/loopif.h"
#include "netif/tcpdump.h"

#include <termios.h>

#include "lwip/ip_addr.h"

#include "arch/perf.h"

#include "httpd.h"
#include "udpecho.h"
#include "tcpecho.h"
#include "shell.h"

#if LWIP_RAW
#include "lwip/icmp.h"
#include "lwip/raw.h"
#include "lwip/sockets.h"
#endif

/*-----------------------------------------------------------------------------------*/
static void
tcp_timeout(void *data)
{
#if TCP_DEBUG
  tcp_debug_print_pcbs();
#endif /* TCP_DEBUG */
  sys_timeout(5000, tcp_timeout, NULL);
}
/*-----------------------------------------------------------------------------------*/
static void
tcpip_init_done(void *arg)
{
  sys_sem_t *sem;
  sem = arg;
  sys_sem_signal(*sem);
}

/*-----------------------------------------------------------------------------------*/
#if LWIP_RAW

static int seq_num;

#if 0
/* Ping using the raw ip */
static int
ping_recv(void *arg, struct raw_pcb *pcb, struct pbuf *p, struct ip_addr *addr)
{
  printf("ping recv\n");
  return 1; /* eat the event */
}

static void
ping_send(struct raw_pcb *raw, struct ip_addr *addr)
{
  struct pbuf *p;
  struct icmp_echo_hdr *iecho;

  p = pbuf_alloc(PBUF_IP,sizeof(struct icmp_echo_hdr),PBUF_RAM);
  if (!p) return;

  iecho = p->payload;
  ICMPH_TYPE_SET(iecho,ICMP_ECHO);
  iecho->chksum = 0;
  iecho->seqno = htons(seq_num);

  iecho->chksum = inet_chksum(iecho, p->len);
  raw_send_to(raw,p,addr);

  pbuf_free(p);

  seq_num++;
}

static void
ping_thread(void *arg)
{
  struct raw_pcb *raw;
  struct ip_addr dest_addr;

  if (!(raw = raw_new(IP_PROTO_ICMP))) return;

  raw_recv(raw,ping_recv,NULL);

  IP4_ADDR(&dest_addr,192,168,0,1);

  while (1)
  {
    printf("ping send\n");
    ping_send(raw,&dest_addr);
    sleep(1);
  }
  /* Never reaches this */
  raw_remove(raw);
}
#else
/* Ping using the socket ip */

static void
ping_send(int s, struct ip_addr *addr)
{
  struct icmp_echo_hdr *iecho;
  struct sockaddr_in to;

  if (!(iecho = malloc(sizeof(struct icmp_echo_hdr))))
    return;

  ICMPH_TYPE_SET(iecho,ICMP_ECHO);
  iecho->chksum = 0;
  iecho->seqno = htons(seq_num);
  iecho->chksum = inet_chksum(iecho, sizeof(*iecho));

  to.sin_len = sizeof(to);
  to.sin_family = AF_INET;
  to.sin_addr.s_addr = addr->addr;

  lwip_sendto(s,iecho,sizeof(*iecho),0,(struct sockaddr*)&to,sizeof(to));

  free(iecho);
  seq_num++;
}

static void
ping_recv(int s, struct ip_addr *addr)
{
  char buf[200];
  int fromlen,len;
  struct sockaddr_in from;

  while (1) {
    len = lwip_recvfrom(s, buf,sizeof(buf), MSG_DONTWAIT,(struct sockaddr*)&from,&fromlen);
    if ( len <= 0 ) break;
    printf("simhost: Received %d bytes from %lX\n", len, ntohl(from.sin_addr.s_addr));
  }
}

static void
ping_thread(void *arg)
{
  int s;
  struct ip_addr dest_addr;

  if ((s = lwip_socket(AF_INET, SOCK_RAW, IP_PROTO_ICMP)) < 0) {
    return;
  }

  IP4_ADDR(&dest_addr,192,168,0,1);

  while (1) {
    printf("simhost: Sending ping\n");
    ping_send(s,&dest_addr);
    ping_recv(s,&dest_addr);
    sleep(3);
  }
}
#endif

#endif

struct netif netif;

#if LWIP_HAVE_LOOPIF
struct netif loopif;
#endif

/*-----------------------------------------------------------------------------------*/
static void
main_thread(void *arg)
{
  struct ip_addr ipaddr, netmask, gw;
  sys_sem_t sem;

  netif_init();

  sem = sys_sem_new(0);
  tcpip_init(tcpip_init_done, &sem);
  sys_sem_wait(sem);
  sys_sem_free(sem);
  printf("TCP/IP initialized.\n");

  IP4_ADDR(&gw, 192,168,0,1);
  IP4_ADDR(&ipaddr, 192,168,0,10);
  IP4_ADDR(&netmask, 255,255,255,0);
  netif_add(&netif, &ipaddr, &netmask, &gw, NULL, openeth_init, tcpip_input);
  netif_set_default(&netif);
  netif_set_up(&netif); /* new step from lwip 1.0.0 */


#if LWIP_HAVE_LOOPIF
  IP4_ADDR(&gw, 127,0,0,1);
  IP4_ADDR(&ipaddr, 127,0,0,1);
  IP4_ADDR(&netmask, 255,0,0,0);

  netif_add(&loopif, &ipaddr, &netmask, &gw, NULL, loopif_init, tcpip_input);
  netif_set_up(&loopif); /* new step from lwip 1.0.0 */
#endif

#if LWIP_TCP
puts("=== init TCP ECHO ===");
  tcpecho_init();
puts("=== init SHELL ===");
  shell_init();
puts("=== init HTTPD ===");
  httpd_init();
#endif
#if LWIP_UDP
puts("=== init UDP ECHO ===");
  udpecho_init();
#endif

#if LWIP_RAW
  sys_thread_new(ping_thread, NULL, DEFAULT_THREAD_PRIO);
#endif

  printf("Applications started.\n");

  /*  sys_timeout(5000, tcp_timeout, NULL);*/

#ifdef MEM_PERF
  mem_perf_init("/tmp/memstats.client");
#endif /* MEM_PERF */
  /*    stats_display(); */
  /* Block for ever. */
  sem = sys_sem_new(0);
  sys_sem_wait(sem);
}
/*-----------------------------------------------------------------------------------*/
int
simhost_main(int argc, char **argv)
{
#ifdef PERF
  perf_init("/tmp/simhost.perf");
#endif /* PERF */
#if LWIP_STATS
  stats_init();
#endif /* STATS */

  sys_init();
  mem_init();
  memp_init();
  pbuf_init();

#ifdef LWIP_TCPDUMP
  tcpdump_init();
#endif

  printf("System initialized.\n");

  sys_thread_new(main_thread, NULL, 0);

  return 0;
}
/*-----------------------------------------------------------------------------------*/








