/*
 * $Id: sys_arch.c,v 1.1 2005-08-11 15:29:52+02 rolf Exp rolf $
 *
 * The system dependent part of the light-weight TCP/IP stack.
 * Implementation for the RTEMS real time operating system.
 *
 * Initial RTEMS port by Karl Scheibelhofer (Karl.Scheibelhofer@iaik.tugraz.at)
 * Socware 2002, IAIK, Graz University of Technology
 *
 * Update to lwIP 1.1.0 (CVS July 2005) by Rolf.Schroedter@dlr.de (RoS).
 *
 * Changes:
 *
 * 2005-Aug-11 <Rolf.Schroedter@dlr.de>:
 *     sys_arch_sem_wait()
 *         - lwIP has changed timeout return values for sys_arch_sem_wait()
 *           somewhere between lwIP 0.5.x ... 0.6.2 ~Jan/May 2003.
 *         - Now sys_arch_sem_wait() should return SYS_ARCH_TIMEOUT(-1)
 *         - And waitTime=0 is a valid return value.
 *      sys_arch_mbox_fetch()
 *         - Same SYS_ARCH_TIMEOUT(-1) changes.
 *         - Return *msg==NULL on timeout
 *      sys_arch_mbox_post()
 *         - If the message queue is full, there is no other way than to block.
 */

#include <rtems.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "lwip/debug.h"
#include "lwip/sys.h"
#include "lwip/opt.h"
#include "lwip/stats.h"

/* define some debug output macros */
#ifdef SYSARCH_DEBUG
#define DEBUG_OUTPUT(...) { fprintf(stderr, __VA_ARGS__); fflush(stderr); }
#else /* SYSARCH_DEBUG */
#define DEBUG_OUTPUT(...)
#endif /* SYSARCH_DEBUG */

static rtems_interval ticksPerSecond;

#define MESSAGE_QUEUE_LENGTH 32
#define MESSAGE_QUEUE_RETRY  100    /* Retry interval (msec) for message posting if queue is full */

/* the timeout list, this will be made a thread private global variable,
   i.e. each thread will see its own instance of this variable */
struct sys_timeouts *timeoutList;

/* the stack size for the tasks (thread) created by the IP stack */
#define TASK_STACK_SIZE (RTEMS_MINIMUM_STACK_SIZE * 2)

/*-----------------------------------------------------------------------------------*/

/**
 * This function is called when the thread is deleted.
 * Here we cleanup the given per-thread variable.
 */
void cleanupThreadVariable(void *threadVariable)
{
  DEBUG_OUTPUT("cleanupThreadVariable BEGIN ptr=%p\n", threadVariable);

  /* free the allocated memory */
  free(threadVariable); /* FIXXXME, causes a SIGSEGV, why? */

  DEBUG_OUTPUT("cleanupThreadVariable END\n");
}

/*-----------------------------------------------------------------------------------*/

/**
 * Is called to initialize the system architectur (sys_arch) layer.
 */
void sys_init(void)
{
  rtems_status_code statusCode;

  DEBUG_OUTPUT("sys_init BEGIN\n" );

  /* get the number of ticks per second - used for time calculations */
  statusCode = rtems_clock_get(RTEMS_CLOCK_GET_TICKS_PER_SECOND, &ticksPerSecond);
  assert(statusCode == RTEMS_SUCCESSFUL);

  /* we set the shared global timeoutList to NULL. any thread that tries to get
   its timeoutList knows that it needs to make it per-thread global and
   initialize it, if it is still NULL. */
  timeoutList = NULL;

  DEBUG_OUTPUT("sys_init END\n");
}

/*-----------------------------------------------------------------------------------*/

/**
 * Creates and returns a new semaphore. The "count" argument specifies
 * the initial state of the semaphore.
 *
 * sys_sem_t is defined as rtems_id in sys_arch.h
 */
sys_sem_t sys_sem_new(u8_t count)
{
  static int objCount=0;
  rtems_id semaphore;
  rtems_name semaphoreName;
  rtems_status_code statusCode;

  DEBUG_OUTPUT("*** sys_sem_new name='ips%c'\n", '0'+objCount);

  semaphoreName = rtems_build_name('i','p','s','0'+(objCount++));

  statusCode = rtems_semaphore_create(semaphoreName,
                                      (rtems_unsigned32) count,
                                      RTEMS_FIFO
                                        | RTEMS_COUNTING_SEMAPHORE
                                        | RTEMS_NO_INHERIT_PRIORITY
                                        | RTEMS_NO_PRIORITY_CEILING
                                        | RTEMS_LOCAL,
                                      RTEMS_NO_PRIORITY,
                                      &semaphore);

  assert(statusCode == RTEMS_SUCCESSFUL);


  DEBUG_OUTPUT("sys_sem_new END, id=%08X\n", semaphore);
  return semaphore;
}

/*-----------------------------------------------------------------------------------*/

/**
 * Deallocates (destroys) a semaphore.
 *
 * sys_sem_t is defined as rtems_id in sys_arch.h
 */
void sys_sem_free(sys_sem_t sem)
{
  rtems_status_code statusCode;

  DEBUG_OUTPUT("--- sys_sem_free BEGIN id=%08X\n", sem);

  statusCode = rtems_semaphore_delete(sem);
  assert(statusCode == RTEMS_SUCCESSFUL);

  DEBUG_OUTPUT("sys_sem_free END id=%08X\n", sem);
}

/*-----------------------------------------------------------------------------------*/

/**
 * Signals a semaphore.
 *
 * sys_sem_t is defined as rtems_id in sys_arch.h
 */
void sys_sem_signal(sys_sem_t sem)
{
  rtems_status_code statusCode;

  DEBUG_OUTPUT("sys_sem_signal BEGIN id=%08X\n", sem);

  statusCode = rtems_semaphore_release(sem);
  assert(statusCode == RTEMS_SUCCESSFUL);

  DEBUG_OUTPUT("sys_sem_signal END id=%08X\n", sem);
}

/*-----------------------------------------------------------------------------------*/

/**
 * Blocks the thread while waiting for the semaphore to be
 * signaled. If the "timeout" argument is non-zero, the thread should
 * only be blocked for the specified time (measured in
 * milliseconds).
 *
 * If the timeout argument is non-zero, the return value is the number of
 * milliseconds spent waiting for the semaphore to be signaled. If the
 * semaphore wasn't signaled within the specified time, the return value is
 * SYS_ARCH_TIMEOUT. If the thread didn't have to wait for the semaphore
 * (i.e., it was already signaled), the function may return zero.
 *
 * Notice that lwIP implements a function with a similar name,
 * sys_sem_wait(), that uses the sys_arch_sem_wait() function.
 *
 * sys_sem_t is defined as rtems_id in sys_arch.h
 */
u32_t sys_arch_sem_wait(sys_sem_t sem, u32_t timeout)
{
  rtems_status_code statusCode, statusCodeSemaphore;
  rtems_interval rtemsTimeout;
  rtems_interval waitStartTime;
  rtems_interval waitEndTime;
  u32_t waitingTime;

  DEBUG_OUTPUT("sys_arch_sem_wait BEGIN id=%08X\n", sem);

  if (timeout != 0) {
    rtemsTimeout = ((timeout*ticksPerSecond)/1000); /* milliseconds to ticks */
    if (rtemsTimeout == 0) {
      rtemsTimeout = 1;
    }
  } else {
    rtemsTimeout = RTEMS_NO_TIMEOUT;
  }

  /* take the current time (ticks) before going for the semaphore */
  statusCode = rtems_clock_get(RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &waitStartTime);
  assert(statusCode == RTEMS_SUCCESSFUL);

  statusCodeSemaphore = rtems_semaphore_obtain(sem, RTEMS_WAIT, rtemsTimeout);
  assert((statusCodeSemaphore == RTEMS_SUCCESSFUL)
         || (statusCodeSemaphore == RTEMS_TIMEOUT));

  /* take the current time (ticks) after going for the semaphore */
  statusCode = rtems_clock_get(RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &waitEndTime);
  assert(statusCode == RTEMS_SUCCESSFUL);

  if (statusCodeSemaphore == RTEMS_TIMEOUT) {
    waitingTime = SYS_ARCH_TIMEOUT;
  } else {
    /* calculate the waiting time in milliseconds */
    waitingTime = (waitEndTime - waitStartTime) * 1000 / ticksPerSecond;
  }

  DEBUG_OUTPUT("sys_arch_sem_wait END id=%08X\n", sem);

  return waitingTime ;
}

/*-----------------------------------------------------------------------------------*/

/**
 * Creates an empty mailbox.
 *
 * sys_mbox_t is defined as rtems_id in sys_arch.h
 */
sys_mbox_t sys_mbox_new(void)
{
  static int objCount=0;
  rtems_id messageQueue;
  rtems_name messageQueueName;
  rtems_status_code statusCode;

  DEBUG_OUTPUT("*** sys_mbox_new name='ipm%c'\n", '0'+objCount);

  messageQueueName = rtems_build_name('i','p','m','0'+(objCount++));

  statusCode = rtems_message_queue_create(messageQueueName,
                                          MESSAGE_QUEUE_LENGTH,
                                          sizeof(void *),
                                          RTEMS_DEFAULT_ATTRIBUTES,
                                          &messageQueue);
  assert(statusCode == RTEMS_SUCCESSFUL);


  DEBUG_OUTPUT("sys_mbox_new END id=%08X\n", messageQueue);

  return messageQueue ;
}

/*-----------------------------------------------------------------------------------*/

/**
 * Deallocates a mailbox. If there are messages still present in the
 * mailbox when the mailbox is deallocated, it is an indication of a
 * programming error in lwIP and the developer should be notified.
 *
 * sys_mbox_t is defined as rtems_id in sys_arch.h
 */
void sys_mbox_free(sys_mbox_t mbox)
{
  rtems_unsigned32 messageCount;
  rtems_status_code statusCode;

  DEBUG_OUTPUT("--- sys_mbox_free BEGIN id=%08X\n", mbox);

  statusCode = rtems_message_queue_get_number_pending(mbox, &messageCount);
  assert(statusCode == RTEMS_SUCCESSFUL);

  statusCode = rtems_message_queue_delete(mbox);
  assert(statusCode == RTEMS_SUCCESSFUL);

  /* if there were messages left in the queue, report a warning */
  if (messageCount != 0) {
    fprintf(stderr, "Warning! Deleted non-empty message queue in TCP/IP stack.");
  }

  DEBUG_OUTPUT("sys_mbox_free END id=%08X\n", mbox);
}

/*-----------------------------------------------------------------------------------*/

/**
 * Posts the "msg" to the mailbox.
 *
 * sys_mbox_t is defined as rtems_id in sys_arch.h
 */
void sys_mbox_post(sys_mbox_t mbox, void *data)
{
  rtems_status_code statusCode;

  DEBUG_OUTPUT("sys_mbox_post BEGIN id=%08X\n", mbox);

  while (1) {
    statusCode = rtems_message_queue_send(mbox, (void *) &data, sizeof(void *));
    assert( (statusCode == RTEMS_SUCCESSFUL) || (statusCode == RTEMS_TOO_MANY) );

    if ( statusCode == RTEMS_TOO_MANY ) {
        rtems_interval retryTime;

        /* Retry to post the same message */
        retryTime = ((MESSAGE_QUEUE_RETRY*ticksPerSecond)/1000); /* milliseconds to ticks */
        rtems_task_wake_after(retryTime);
    } else {
        /* okay: we are done */
        break;
    }
  }

  DEBUG_OUTPUT("sys_mbox_post END id=%08X\n", mbox);
}

/*-----------------------------------------------------------------------------------*/

/**
 * Blocks the thread until a message arrives in the mailbox, but does
 * not block the thread longer than "timeout" milliseconds (similar to
 * the sys_arch_sem_wait() function). The "msg" argument is a result
 * parameter that is set by the function (i.e., by doing "*msg =
 * ptr"). The "msg" parameter maybe NULL to indicate that the message
 * should be dropped.
 *
 * The return values are the same as for the sys_arch_sem_wait() function:
 * Number of milliseconds spent waiting or SYS_ARCH_TIMEOUT if there was a
 * timeout.
 *
 * Note that a function with a similar name, sys_mbox_fetch(), is
 * implemented by lwIP.
 *
 * sys_mbox_t is defined as rtems_id in sys_arch.h
 */
u32_t sys_arch_mbox_fetch(sys_mbox_t mbox, void **msg, u32_t timeout)
{
  void *localMessage;
  rtems_status_code statusCode, statusCodeMessageQueue;
  rtems_unsigned32 messageSize;
  rtems_interval rtemsTimeout;
  rtems_interval waitStartTime;
  rtems_interval waitEndTime;
  u32_t waitingTime;

  DEBUG_OUTPUT("sys_arch_mbox_fetch BEGIN id=%08X\n", mbox);

  if (timeout != 0) {
    rtemsTimeout = ((timeout*ticksPerSecond)/1000); /* milliseconds to ticks */
    if (rtemsTimeout == 0) {
      rtemsTimeout = 1;
    }
  } else {
    rtemsTimeout = RTEMS_NO_TIMEOUT;
  }

  /* take the current time (ticks) before going for a message */
  statusCode = rtems_clock_get(RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &waitStartTime);
  assert(statusCode == RTEMS_SUCCESSFUL);

  statusCodeMessageQueue = rtems_message_queue_receive(mbox,
                                                       &localMessage,
                                                       &messageSize,
                                                       RTEMS_WAIT,
                                                       rtemsTimeout);

  /* take the current time (ticks) after going for a message
     only needed if we got a message (RTEMS_SUCCESSFUL) */
  if (statusCodeMessageQueue == RTEMS_SUCCESSFUL) {
    statusCode = rtems_clock_get(RTEMS_CLOCK_GET_TICKS_SINCE_BOOT, &waitEndTime);
    assert(statusCode == RTEMS_SUCCESSFUL);
  }

  /* if we got a message (RTEMS_SUCCESSFUL), we should have got a
     void pointer, because we only send void pointers,
     or we could have run into a timeout */
  assert(((statusCodeMessageQueue == RTEMS_SUCCESSFUL)
          && (messageSize == sizeof(void *)))
         || (statusCodeMessageQueue == RTEMS_TIMEOUT));

  if (statusCodeMessageQueue == RTEMS_TIMEOUT) {
    waitingTime = SYS_ARCH_TIMEOUT;
    *msg = NULL;
  } else { /* statusCodeMessageQueue == RTEMS_SUCCESSFUL */
    /* if msg == NULL, drop message */
    if ((msg != NULL)) {
      *msg = localMessage;
    }

    /* calculate the waiting time in milliseconds */
    waitingTime = (waitEndTime - waitStartTime) * 1000 / ticksPerSecond;
  }

  DEBUG_OUTPUT("sys_arch_mbox_fetch END id=%08X\n", mbox);

  return waitingTime ;
}

/*-----------------------------------------------------------------------------------*/

/**
 * Returns a pointer to the per-thread sys_timeouts structure. In lwIP,
 * each thread has a list of timeouts which is repressented as a linked
 * list of sys_timeout structures. The sys_timeouts structure holds a
 * pointer to a linked list of timeouts. This function is called by
 * the lwIP timeout scheduler and must not return a NULL value.
 *
 * In a single thread sys_arch implementation, this function will
 * simply return a pointer to a global sys_timeouts variable stored in
 * the sys_arch module.
 */
struct sys_timeouts * sys_arch_timeouts(void)
{
  rtems_status_code statusCode;

  DEBUG_OUTPUT("sys_arch_timeouts BEGIN\n");

  /* check, if we need to already made the timeoutList per-thread global
     for the calling thread. we do this here, because we cannot ensure
     that all threads that call this method have been created using
     the thread creation method of this file */
  if (timeoutList == NULL) {
    /* we must make it per-thread global and initialize it */
    statusCode = rtems_task_variable_add(RTEMS_SELF, (void **) &timeoutList, &cleanupThreadVariable);
    assert(statusCode == RTEMS_SUCCESSFUL);
    timeoutList = (struct sys_timeouts *) malloc(sizeof(struct sys_timeouts));
    timeoutList->next = NULL; /* initialize to emtpy, do not forget! */
  }

  DEBUG_OUTPUT("sys_arch_timeouts END\n");

  /* we added the timeoutList to the thread's private global variables,
     thus we can return it directly */
  return timeoutList ;
}

/*-----------------------------------------------------------------------------------*/

/* this structure is used to pass the arguments of the ip stack to the
 * thread's entry function
 */
struct sys_thread_arg {
  void (* threadEntryPoint)(void *);
  void *threadArgument;
};

/**
 * Helper function that calls the entry point given by the ip stack.
 */
static rtems_task taskEntryPoint(rtems_task_argument argument)
{
  struct sys_thread_arg *rtemsThreadArgument;
  rtems_status_code statusCode;

  DEBUG_OUTPUT("taskEntryPoint BEGIN\n");

#ifdef SYSARCH_DEBUG
  {
    rtems_id id;
    rtems_task_ident(RTEMS_SELF, RTEMS_SEARCH_ALL_NODES, &id);
    printf("taskEntryPoint, task ID %x\n", id);
  }
#endif /* SYSARCH_DEBUG */

  /* now call the thread real entry function as provided by lwIP */
  rtemsThreadArgument = (struct sys_thread_arg *) argument;
  rtemsThreadArgument->threadEntryPoint(rtemsThreadArgument->threadArgument);

  /* free the argument helper object */
  free(rtemsThreadArgument);

  /* if the ip stack finished, there is nothing left to do, delete this task */
  DEBUG_OUTPUT("taskEntryPoint DELETE task\n");
  statusCode = rtems_task_delete(RTEMS_SELF);
  assert(statusCode == RTEMS_SUCCESSFUL);

  DEBUG_OUTPUT("taskEntryPoint END\n");
}

/**
 * Starts a new thread that will begin its execution in the function
 * "thread()". The "arg" argument will be passed as an argument to the
 * thread() function.
 */
sys_thread_t sys_thread_new(void (* threadFunction)(void *), void *arg, int prio)
{
  static int objCount=0;

  rtems_id task;
  rtems_status_code statusCode;
  struct sys_thread_arg *rtemsThreadArgument;

  /*
  * Check for valid task priority
  */
  if( (prio <= 0) || (prio > 255) ) {
    prio = DEFAULT_THREAD_PRIO;
  }

  DEBUG_OUTPUT("*** sys_thread_new, task name='ipt%c', priority=%d\n", '0'+objCount, prio);

  /*
  * Create the task.
  * Although different tasks may have the same name, we count them
  */
  statusCode = rtems_task_create(rtems_build_name('i','p','t','0'+(objCount++)),
                                 prio,
                                 TASK_STACK_SIZE,
                                 RTEMS_PREEMPT | RTEMS_TIMESLICE | RTEMS_NO_ASR | RTEMS_INTERRUPT_LEVEL(0),
                                 RTEMS_DEFAULT_ATTRIBUTES,
                                 &task);
  assert(statusCode == RTEMS_SUCCESSFUL);

  /* start thread */
  rtemsThreadArgument = (struct sys_thread_arg *) malloc(sizeof(struct sys_thread_arg));
  rtemsThreadArgument->threadEntryPoint = threadFunction;
  rtemsThreadArgument->threadArgument = arg;

  /* we must start in our own dummy entry function taskEntryPoint, because
   RTEMS requires a special function signature that is incompatible with lwIP.
   the real entry function is called in taskEntryPoint.
   do not forget to pass the argument to the entry function. */
  statusCode = rtems_task_start(task, taskEntryPoint, (rtems_task_argument) rtemsThreadArgument);
  assert(statusCode == RTEMS_SUCCESSFUL);

  DEBUG_OUTPUT("sys_thread_new END id=%08X\n", task);

  return task;
}

