/* floppy.h
 * Some defines for floppy device driver
 *
 * Copyright(C) 2001
 * Camilo Alejandro Arboleda
 */
#ifndef __FLOPPY_H
#define __FLOPPY_H

#define FLOPPY_IRQ     6
/* I/O Ports used by floppy disk task. */
#define DOR            0x3F2	/* motor drive control bits */
#define FDC_STATUS     0x3F4	/* floppy disk controller status register */
#define FDC_DATA       0x3F5	/* floppy disk controller data register */
#define FDC_RATE       0x3F7	/* transfer rate register */

/* Status registers returned as result of operation. */
#define ST0             0x00	/* status register 0 */
#define ST1             0x01	/* status register 1 */
#define ST2             0x02	/* status register 2 */
#define ST3             0x00	/* status register 3 (return by DRIVE_SENSE) */
#define ST_CYL          0x03	/* slot where controller reports cylinder */
#define ST_HEAD         0x04	/* slot where controller reports head */
#define ST_SEC          0x05	/* slot where controller reports sector */
#define ST_PCN          0x01	/* slot where controller reports present cyl */

/* Fields within the I/O ports. */
/* Main status register. */
#define CTL_BUSY        0x10	/* bit is set when read or write in progress */
#define DIRECTION       0x40	/* bit is set when reading data reg is valid */
#define MASTER          0x80	/* bit is set when data reg can be accessed */

/* Digital output port (DOR). */
#define MOTOR_SHIFT        4	/* high 4 bits control the motors in DOR */
#define ENABLE_INT      0x0C	/* used for setting DOR port */
#define START_MOTOR        1
#define STOP_MOTOR         0

/* ST0. */
#define ST0_BITS        0xF8	/* check top 5 bits of seek status */
#define TRANS_ST0       0x00	/* top 5 bits of ST0 for READ/WRITE */
#define SEEK_ST0        0x20	/* top 5 bits of ST0 for SEEK */

/* ST1. */
#define BAD_SECTOR      0x05	/* if these bits are set in ST1, recalibrate */
#define WRITE_PROTECT   0x02	/* bit is set if diskette is write protected */

/* ST2. */
#define BAD_CYL         0x1F	/* if any of these bits are set, recalibrate */

/* ST3 (not used). */
#define ST3_FAULT       0x80	/* if this bit is set, drive is sick */
#define ST3_WR_PROTECT  0x40	/* set when diskette is write protected */
#define ST3_READY       0x20	/* set when drive is ready */

/* Floppy disk controller command bytes. */
#define FDC_SEEK        0x0F	/* command the drive to seek */
#define FDC_READ        0xE6	/* command the drive to read */
#define FDC_WRITE       0xC5	/* command the drive to write */
#define FDC_SENSE       0x08	/* command the controller to tell its status */
#define FDC_RECALIBRATE 0x07	/* command the drive to go to cyl 0 */
#define FDC_SPECIFY     0x03	/* command the drive to accept params */
#define FDC_READ_ID     0x4A	/* command the drive to read sector identity */
#define FDC_FORMAT      0x4D	/* command the drive to format a track */
#define DMA_RESET_VAL   0x06

/* DMA channel commands. */
#define DMA_READ        0x46	/* DMA read opcode */
#define DMA_WRITE       0x4A	/* DMA write opcode */

/* Parameters for the disk drive. */
#define HC_SIZE         2880	/* # sectors on largest legal disk (1.44MB) */
#define NR_HEADS        0x02	/* two heads (i.e., two tracks/cylinder) */
#define MAX_SECTORS	  18	/* largest # sectors per track */
#define DTL             0xFF	/* determines data length (sector size) */
#define SPEC1           0xdf    /* */
#define SPEC2           0x02	/* second parameter to SPECIFY */
#define MOTOR_OFF       3*HZ	/* how long to wait before stopping motor */
#define WAKEUP		2*HZ	/* timeout on I/O, FDC won't quit. */

/* Error codes */
#define ERR_SEEK         (-1)	/* bad seek */
#define ERR_TRANSFER     (-2)	/* bad transfer */
#define ERR_STATUS       (-3)	/* something wrong when getting status */
#define ERR_READ_ID      (-4)	/* bad read id */
#define ERR_RECALIBRATE  (-5)	/* recalibrate didn't work properly */
#define ERR_DRIVE        (-6)	/* something wrong with a drive */
#define ERR_WR_PROTECT   (-7)	/* diskette is write protected */
#define ERR_TIMEOUT      (-8)	/* interrupt timeout */

/* No retries on some errors. */
#define err_no_retry(err)	((err) <= ERR_WR_PROTECT)

/* Encoding of drive type in minor device number. */
#define DEV_TYPE_BITS   0x7C	/* drive type + 1, if nonzero */
#define DEV_TYPE_SHIFT     2	/* right shift to normalize type bits */
#define FORMAT_DEV_BIT  0x80	/* bit in minor to turn write into format */

/* Miscellaneous. */
#define MAX_ERRORS         6	/* how often to try rd/wt before quitting */
#define MAX_RESULTS        7	/* max number of bytes controller returns */
#define NR_DRIVES          1	/* maximum number of drives */
#define DIVISOR          128	/* used for sector size encoding */
#define SECTOR_SIZE_CODE   2	/* code to say "512" to the controller */
#define TIMEOUT		  50	/* milliseconds waiting for FDC */
#define MS_TIMEOUT	 500	/* milliseconds waiting for FDC */
#define TICKS_PER_SEC    100    /* Ticks per second */
#define NT                 7	/* number of diskette/drive combinations */
#define UNCALIBRATED       0	/* drive needs to be calibrated at next use */
#define CALIBRATED         1	/* no calibration needed */
#define BASE_SECTOR        1	/* sectors are numbered starting at 1 */
#define NO_SECTOR          0	/* current sector unknown */
#define NO_CYL		 (-1)	/* current cylinder unknown, must seek */
#define NO_DENS		 100	/* current media unknown */
#define BSY_IDLE	   0	/* busy doing nothing */
#define BSY_IO		   1	/* doing I/O */
#define BSY_WAKEN	   2	/* got a wakeup call */
#define GAP_SIZE        0x1b    /* I have no idea what it is :-) */
#define SECTORS_PER_TRACK 18    /* 1.44 Mb floppy */
#define TRACKS_PER_SIDE   80    /* 1.44 Mb floppy */
#define BYTES_PER_FLOPPY  1474560 /* 512*18*80*2 = 1440 Kbytes */

/* Events */
#define FDC_INTERRUPT   0x00000001 /* Interrupt */

/* Parameters for the disk drive. */
#ifndef SECTOR_SIZE
#define SECTOR_SIZE      512	/* physical sector size in bytes */
#endif
#ifndef SECTOR_SHIFT
#define SECTOR_SHIFT       9	/* for division */
#endif
#ifndef SECTOR_MASK
#define SECTOR_MASK      511	/* and remainder */
#endif

#ifdef __cplusplus
extern "C" {
#endif

rtems_device_driver fl_initialize(rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);
rtems_device_driver fl_open      (rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);
rtems_device_driver fl_close     (rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);
rtems_device_driver fl_read      (rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);
rtems_device_driver fl_write     (rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);
rtems_device_driver fl_control   (rtems_device_major_number major,
				  rtems_device_minor_number minor,
				  void                      *arg);

/* Entry points to this driver. */
#define FLOPPY_DRIVER_TABLE_ENTRY { \
  fl_initialize,              \
  fl_open,                    \
  fl_close,                   \
  fl_read,                    \
  fl_write,                   \
  fl_control                  \
}

#ifdef __cplusplus
}
#endif

#endif