/*
 * RTEMS BOOTP/TFTP Bootstrap
 *
 * This program may be distributed and used for any purpose.
 * I ask only that you:
 *      1. Leave this author information intact.
 *      2. Document any changes you make.
 *
 * W. Eric Norum
 * Saskatchewan Accelerator Laboratory
 * University of Saskatchewan
 * Saskatoon, Saskatchewan, CANADA
 * eric@skatter.usask.ca
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <sys/fcntl.h>
#include "load.h"

/*
 * Number of bytes that must be read to be able to
 * determine the format of the executable file.
 */
#define FORMAT_CHECK_SIZE	4

/*
 * Header of an a.out executable
 */
struct aout_exec {
	unsigned long   a_magic;        /* magic number */
	unsigned long   a_text;         /* size of text segment */
	unsigned long   a_data;         /* size of initialized data */
	unsigned long   a_bss;          /* size of uninitialized data */
	unsigned long   a_syms;         /* size of symbol table */
	unsigned long   a_entry;        /* entry point */
	unsigned long   a_trsize;       /* size of text relocation */
	unsigned long   a_drsize;       /* size of data relocation */
};

/*
 * File header of a COFF executable
 */
struct coff_filehdr {
	char f_magic[2];	/* magic number			*/
	char f_nscns[2];	/* number of sections		*/
	char f_timdat[4];	/* time & date stamp		*/
	char f_symptr[4];	/* file pointer to symtab	*/
	char f_nsyms[4];	/* number of symtab entries	*/
	char f_opthdr[2];	/* sizeof(optional hdr)		*/
	char f_flags[2];	/* flags			*/
};

/*
 * File header of an ELF executable
 */
struct elf_filehdr {
	unsigned char	e_ident[16];	/* ELF "magic number" */
	unsigned char	e_type[2];	/* Identifies object file type */
	unsigned char	e_machine[2];	/* Specifies required architecture */
	unsigned char	e_version[4];	/* Identifies object file version */
	unsigned char	e_entry[4];	/* Entry point virtual address */
	unsigned char	e_phoff[4];	/* Program header table file offset */
	unsigned char	e_shoff[4];	/* Section header table file offset */
	unsigned char	e_flags[4];	/* Processor-specific flags */
	unsigned char	e_ehsize[2];	/* ELF header size in bytes */
	unsigned char	e_phentsize[2];	/* Program header table entry size */
	unsigned char	e_phnum[2];	/* Program header table entry count */
	unsigned char	e_shentsize[2];	/* Section header table entry size */
	unsigned char	e_shnum[2];	/* Section header table entry count */
	unsigned char	e_shstrndx[2];	/* Section header string table index */
};

/*
 * Storage for executable header information
 */
union {
	unsigned char		cbuf[FORMAT_CHECK_SIZE];
	unsigned long		lbuf[FORMAT_CHECK_SIZE / sizeof (long)];
	struct aout_exec	aout_exec;
	struct coff_filehdr	coff_filehdr;
	struct elf_filehdr	elf_filehdr;
} header;

/*
 * Read the specified number of bytes
 */
static void
mustRead (int fd, char *buffer, int length)
{
	int nread;

	/*
	 * Work around stupid I/O routines that refuse to read to location 0
	 */
	if (buffer == 0) {
		char ctmp;
		mustRead (fd, &ctmp, 1);
		*buffer++ = ctmp;
		if (--length == 0)
			return;
	}
	nread = read (fd, buffer, length);
	if (nread < 0) {
		printf ("Read failed: %s\n", strerror (errno));
		delayExit ();
	}
	if (nread != length) {
		printf ("Read %d, expected to read %d\n", nread, length);
		delayExit ();
	}
}

/*
 * Skip over the specified number of bytes
 */
static void
skipBytes (int fd, int nskip)
{
	char jnk[256];
	int nread;

	while (nskip) {
		if (nskip <= (int)sizeof jnk)
			nread = nskip;
		else
			nread = sizeof jnk;
		mustRead (fd, jnk, nread);
		nskip -= nread;
	}
}

/*
 * Reallocate memory -- don't accept `no' as an answer
 */
static void *
mustRealloc (void *p, int nbytes)
{
	p = realloc (p, nbytes);

	if (p == NULL) {
		printf ("Can't allocate %d bytes of memory.\n", nbytes);
		exit (4);
	}
	return p;
}

/*
 **********************************************************************
 *                   LOAD AND RUN A.OUT EXECUTABLE                    *
 **********************************************************************
 */
static void
loadaout (int fd)
{
	/*
	 * Read remainder of header
	 */
	mustRead (fd, &header.cbuf[FORMAT_CHECK_SIZE],
				sizeof header.aout_exec - FORMAT_CHECK_SIZE);

	/*
	 * Read the text and data segments
	 */
	printf ("Load %lu bytes of text+data at 0x%lx.\n",
			header.aout_exec.a_text + header.aout_exec.a_data,
			header.aout_exec.a_entry);
	mustRead (fd, (char *)header.aout_exec.a_entry,
			header.aout_exec.a_text + header.aout_exec.a_data);
	close (fd);

	/*
	 * Start the program
	 */
	startProgram ((void *)header.aout_exec.a_entry);
}


/*
 **********************************************************************
 *                   LOAD AND RUN COFF EXECUTABLE                     *
 **********************************************************************
 */

/*
 * Section header
 */
struct coff_scnhdr {
	char		s_name[8];	/* section name			*/
	char		s_paddr[4];	/* physical address, aliased s_nlib */
	char		s_vaddr[4];	/* virtual address		*/
	char		s_size[4];	/* section size			*/
	char		s_scnptr[4];	/* file ptr to raw data for section */
	char		s_relptr[4];	/* file ptr to relocation	*/
	char		s_lnnoptr[4];	/* file ptr to line numbers	*/
	char		s_nreloc[2];	/* number of relocation entries	*/
	char		s_nlnno[2];	/* number of line number entries*/
	char		s_flags[4];	/* flags			*/
};

/*
 * Convert 2-byte COFF value
 */
static unsigned short
coff2 (const char *ap)
{
	const unsigned char *cp = (const unsigned char *)ap;
	return (cp[0] << 8) | cp[1];
}

/*
 * Convert 4-byte COFF value
 */
static unsigned long
coff4 (const char *ap)
{
	const unsigned char *cp = (const unsigned char *)ap;
	return (cp[0] << 24) | (cp[1] << 16) | (cp[2] << 8) | cp[3];
}

static void
loadcoff (int fd)
{
	int i;
	int nsections;
	struct coff_scnhdr *sp, *spbuf;
	int spcount;
	unsigned long scnptr;
	void *entry = 0;

	/*
	 * Read remainder of header
	 */
	mustRead (fd, &header.cbuf[FORMAT_CHECK_SIZE],
				sizeof header.coff_filehdr - FORMAT_CHECK_SIZE);
	scnptr = sizeof header.coff_filehdr;

	/*
	 * Skip over optional header
	 */
	skipBytes (fd, coff2 (header.coff_filehdr.f_opthdr));
	scnptr += coff2 (header.coff_filehdr.f_opthdr);

	/*
	 * Read section headers and form list of sections to load
	 */
	nsections = coff2 (header.coff_filehdr.f_nscns);
	spbuf = NULL;
	spcount = 0;
	for (i = 0 ; i < nsections ; i++) {
		spbuf = mustRealloc (spbuf, (sizeof *sp) * (spcount + 1));
		sp = &spbuf[spcount];
		mustRead (fd, (char *)sp, sizeof *sp);
		scnptr += sizeof *sp;
		if (coff4 (sp->s_scnptr)) {
			if ((spcount == 0) || strcmp (sp->s_name, ".text") == 0)
				entry = (void *)coff4 (sp->s_paddr);
			spcount++;
		}
	}

	/*
	 * Read the sections
	 */
	for (i = 0 ; i < spcount ; i++) {
		unsigned long sectionBase;
		unsigned long sectionSize;

		/*
		 * Get section information
		 */
		sp = &spbuf[i];
		printf ("Load %lu bytes of section `%.*s' at 0x%lx.\n",
					coff4 (sp->s_size),
					(int)sizeof sp->s_name, sp->s_name,
					coff4 (sp->s_paddr));

		/*
		 * Skip to the beginning of the section
		 */
		sectionBase = coff4 (sp->s_scnptr);
		skipBytes (fd, sectionBase - scnptr);
		scnptr = sectionBase;

		/*
		 * Read the section
		 */
		sectionSize = coff4 (sp->s_size);
		mustRead (fd, (char *)coff4 (sp->s_paddr), sectionSize);
		scnptr += sectionSize;
	}
	close (fd);

	/*
	 * Start the program
	 */
	startProgram (entry);
}

/*
 **********************************************************************
 *                    LOAD AND RUN ELF EXECUTABLE                     *
 **********************************************************************
 */

static int elf_bigendian;

/*
 * Program header
 */
struct elf_pgmhdr {
	unsigned char	p_type[4];	/* Identifies program segment type */
	unsigned char	p_offset[4];	/* Segment file offset */
	unsigned char	p_vaddr[4];	/* Segment virtual address */
	unsigned char	p_paddr[4];	/* Segment physical address */
	unsigned char	p_filesz[4];	/* Segment size in file */
	unsigned char	p_memsz[4];	/* Segment size in memory */
	unsigned char	p_flags[4];	/* Segment flags */
	unsigned char	p_align[4];	/* Segment alignment, file & memory */
};

/*
 * Convert 2-byte ELF value
 */
static unsigned short
elf2 (const unsigned char *cp)
{
	if (elf_bigendian)
		return (cp[0] << 8) | cp[1];
	else
		return (cp[1] << 8) | cp[0];
}

/*
 * Convert 4-byte ELF value
 */
static unsigned long
elf4 (const unsigned char *cp)
{
	if (elf_bigendian)
		return (cp[0] << 24) | (cp[1] << 16) | (cp[2] << 8) | cp[3];
	else
		return (cp[3] << 24) | (cp[2] << 16) | (cp[1] << 8) | cp[0];
}

static void
loadelf (int fd)
{
	unsigned long phoffset;
	unsigned long phnum, headerbytes;
	unsigned long offset;
	struct elf_pgmhdr *phdr;
	int firstTime = 1;
	void *entry = 0;

	/*
	 * Read remainder of header
	 */
	mustRead (fd, &header.cbuf[FORMAT_CHECK_SIZE],
				sizeof header.elf_filehdr - FORMAT_CHECK_SIZE);
	offset = sizeof header.elf_filehdr;
	if (header.elf_filehdr.e_ident[4] != 1) {
		printf ("Not a 32-bit file.\n");
		return;
	}
	switch (header.elf_filehdr.e_ident[5]) {
	case 1: elf_bigendian = 0;	break;
	case 2: elf_bigendian = 1;	break;
	default:
		printf ("Not a 2's complement file.\n");
		return;
	}
	if (header.elf_filehdr.e_ident[6] != 1)
		printf ("Warning -- Not a version 1 file.\n");
#ifdef __mc68000__
	if (elf2 (header.elf_filehdr.e_machine) != 4) {
		printf ("Not an M68K file.\n");
		return;
	}
#endif
	if (elf2 (header.elf_filehdr.e_phentsize) != sizeof *phdr) {
		printf ("Bad program header size.\n");
		return;
	}
	phnum = elf2 (header.elf_filehdr.e_phnum);
	phoffset = elf4 (header.elf_filehdr.e_phoff);
	skipBytes (fd, phoffset - offset);
	offset = phoffset;

	/*
	 * Read program header(s)
	 */
	headerbytes = phnum * sizeof *phdr;
	phdr = mustRealloc (NULL, headerbytes);
	mustRead (fd, (char *)phdr, headerbytes);
	offset += headerbytes;

	/*
	 * Read program segments
	 */
	for ( ; phnum ; phnum--, phdr++) {
		unsigned long p_offset;
		unsigned long p_filesz;
		unsigned long p_vaddr;

		p_offset = elf4 (phdr->p_offset);
		p_filesz = elf4 (phdr->p_filesz);
		p_vaddr = elf4 (phdr->p_vaddr);

		/*
		 * Check that we're never moving backwards
		 */
		if (p_offset < offset) {
			printf ("Can't move backwards!\n");
			return;
		}
		skipBytes (fd, p_offset - offset);
		offset = p_offset;

		/*
		 * Is this segment to be loaded?
		 */
		if (elf4 (phdr->p_type) == 1) {
			printf ("Load %lu bytes at 0x%lx.\n", p_filesz, p_vaddr);
			mustRead (fd, (char *)p_vaddr, p_filesz);
			if (firstTime) {
				firstTime = 0;
				entry = (void *)p_vaddr;
			}
		}
		else {
			skipBytes (fd, p_filesz);
		}
		offset += p_filesz;
	}
	if (firstTime) {
		printf ("Nothing to load!\n ");
		return;
	}
	close (fd);

	/*
	 * Start the program
	 */
	startProgram (entry);
}

/*
 **********************************************************************
 *              OPEN FILE AND DETERMINE EXECUTABLE FORMAT             *
 **********************************************************************
 */
void
Download (const char *hostname, const char *name)
{
	char *fullname;
	int fd;

	/*
	 * Open the file
	 */
	if (name == NULL) {
		printf ("Nothing to read!\n");
		return;
	}
	printf ("Download `%s' from %s.\n", name, hostname);
	fullname = malloc (8 + strlen (hostname) + strlen (name));
	sprintf (fullname, "/TFTP/%s/%s", hostname, name);
	fd = open (fullname, O_RDONLY);
	free (fullname);
	if (fd < 0) {
		printf ("Open failed: %s\n", strerror (errno));
		return;
	}

	/*
	 * Read enough of the file to be able to determine the format
	 */
	mustRead (fd, header.cbuf, FORMAT_CHECK_SIZE);

	/*
	 * Load and start the program
	 */
	if ((header.lbuf[0] & 0xFFFF) == 0407)
		loadaout (fd);
	else if ((header.cbuf[0] == 0x1) && (header.cbuf[1] == 0x50))
		loadcoff (fd);
	else if ((header.cbuf[0] == 0x7F) && (header.cbuf[1] == 'E')
	      && (header.cbuf[2] == 'L') && (header.cbuf[3] == 'F'))
		loadelf (fd);
	else
		printf ("`%s' is not an executable file.\n", name);
}
