/*
 * RTEMS BOOTP/TFTP Bootstrap For Motorola MC68360
 *
 * This program may be distributed and used for any purpose.
 * I ask only that you:
 *      1. Leave this author information intact.
 *      2. Document any changes you make.
 *
 * W. Eric Norum
 * Saskatchewan Accelerator Laboratory
 * University of Saskatchewan
 * Saskatoon, Saskatchewan, CANADA
 * eric@skatter.usask.ca
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <bsp.h>
#include <m68360.h>
#include <rtems/error.h>
#include <rtems/rtems_bsdnet.h>
#include <rtems/tftp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "load.h"

#define CONFIGURE_TEST_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_TEST_NEEDS_CLOCK_DRIVER
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE

#define CONFIGURE_EXECUTIVE_RAM_SIZE    (128*1024)
#define CONFIGURE_MAXIMUM_SEMAPHORES    10

#define CONFIGURE_MICROSECONDS_PER_TICK 10486

#define CONFIGURE_INIT_TASK_STACK_SIZE (10*1024)
#define CONFIGURE_INIT_TASK_PRIORITY   30
#define NETWORK_TASK_PRIORITY          20

#define CONFIGURE_INIT
rtems_task Init (rtems_task_argument argument);

#include <confdefs.h>

/*
 * Configure a small memory requirement
 */
static struct rtems_bsdnet_ifconfig netdriver_config = {
  RTEMS_BSP_NETWORK_DRIVER_NAME,   /* Network driver name */
  RTEMS_BSP_NETWORK_DRIVER_ATTACH, /* Network driver attach function */
  NULL,                            /* Next driver */
  NULL,                            /* IP address */
  NULL,                            /* IP net mask */
  NULL,                            /* Hardware address */
  1,                               /* Ignore broadcast packets */
  1500,                            /* Standard Ethernet MTU */
  10,                              /* Number of receive descriptors */
  4,                               /* Number of transmit descriptors */
};
struct rtems_bsdnet_config rtems_bsdnet_config = {
  &netdriver_config,      /* Network interface */
  rtems_bsdnet_do_bootp,  /* Use BOOTP to get network configuration */
  NETWORK_TASK_PRIORITY,  /* Network task priority */
  32*1024,                /* MBUF space */
  64*1024,                /* MBUF cluster space */
};

static rtems_interval ticksPerSecond;

/*
 * RTEMS Startup Task
 */
rtems_task
Init (rtems_task_argument ignored)
{
	/*
	 * Show that we're alive
	 */
	printf ("RTEMS BOOTP/TFTP BOOTSTRAP -- %s\n", "$Revision: 1.10 $");

	/*
	 * Get some timing information
	 */
	rtems_clock_get (RTEMS_CLOCK_GET_TICKS_PER_SECOND, &ticksPerSecond);

	/*
	 * Start networking
	 */
	rtems_bsdnet_initialize_network ();
	rtems_bsdnet_initialize_tftp_filesystem ();

	/*
	 * Download the file
	 */
	Download (inet_ntoa (rtems_bsdnet_bootp_server_address),
					rtems_bsdnet_bootp_boot_file_name);

	/*
	 * Don't flood the network with requests
	 */
	rtems_task_wake_after (30 * ticksPerSecond);

	/*
	 * Try again
	 */
	exit (0);
}

/*
 * Pause for a while, then exit
 */
void
delayExit (void)
{
	rtems_task_wake_after (2* ticksPerSecond);
	rtems_panic ("LOAD FAILED!");
}

/*
 * Transfer control to the downloaded program
 */
void
startProgram (void *entry)
{
	int s;
	struct ifreq ifreq;
	unsigned long *exceptionVector;
	unsigned long sp, pc;

	/*
	 * Show interface statistics
	 */
	rtems_bsdnet_show_if_stats ();

	/*
	 * Shut down the ethernet driver
	 */
	s = socket (AF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		printf ("Can't create socket: error %s\n", strerror (errno));
	}
	else {
		strcpy (ifreq.ifr_name, RTEMS_BSP_NETWORK_DRIVER_NAME);
		if (ioctl (s, SIOCGIFFLAGS, &ifreq) < 0) {
			printf ("Can't get interface flags: error %s\n", strerror (errno));
		}
		else {
			ifreq.ifr_flags &= ~IFF_UP;
			if (ioctl (s, SIOCSIFFLAGS, &ifreq) < 0)
				printf ("Can't set interface flags: error %s\n", strerror (errno));
		}
		close (s);
	}

	/*
	 * Pick up initial SP and PC values
	 */
	exceptionVector =  (unsigned long *)entry;
	sp = exceptionVector[0];
	pc = exceptionVector[1];
	printf ("Start program.  PC:0x%lx  SP:0x%lx\n", pc, sp);

	/*
	 * Give printf some time to finish
	 */
	rtems_task_wake_after (ticksPerSecond);

	/*
	 * Reset the CPM
	 */
	M360ExecuteRISC (M360_CR_RST);

	/*
	 * Transfer control to downloaded program
	 */
	asm volatile (
		"movew	#0x2700,%%sr\n\t"	/* Disable interrupts, select Supervisor/Interrupt stack */
		"movec	%0,%%vbr\n\t"	/* Clear VBR */
		"movel	%1,%%sp\n\t"	/* Initialize stack pointer */
		"jmp	%2@\n"		/* Jump to program entry point */
		: : "d"(exceptionVector), "r"(sp), "a"(pc));
	rtems_panic ("Downloaded program didn't start!");
}
