/* Description of the internal Parameter Ram of the MPC823.
 *
 * Author: Josu Onandia <jonandia@fagorautomation.es>
 *
 * This file has been downloaded from the Motorola site, but at the end 
 * there are some useful constants and macros derived from:
 * 	c/src/lib/libcpu/powerpc/mpc821/include/mpc821.h
 * and
 * 	c/src/lib/libcpu/powerpc/mpc860/include/mpc860.h
 */ 
#ifndef _MPC823_H
#define _MPC823_H

#ifndef ASM

/*
 * Definitions of the parameter area RAM.
 * Note that different structures are overlaid
 * at the same offsets for the different modes
 * of operation.
 *
 * History:
 * 9/4/97       saw     Initial Version.  Based on mpc821 Rev. A header file.
 * 10/16/97     saw     Fixed Mapping Error in DMA section.
 * 10/16/97     saw     Fixed Mapping Error in Video Controller section.
 * 2/23/98      saw     Added Missing USB Definitions
 * 2/24/98      saw     Simplified PRAM Structure; Updated for CDR
 * 4/27/98      saw     Removed errant SCC2 protocol comment
 * 6/5/98       saw     Changed SIRAM unit width to 32-bits
 * 8/3/98       saw     Changed piscr from 32-bit to 16-bit register
 * 8/14/98      saw     modified dual port ram to agree with pg. 16-16 of UM
 * 8/24/98      saw     added guard against redundant inclusion
 * 9/14/98      saw     rewrote USB Endpoint Registers (USEPx) and
 *                      USB Endpoint Pointers (EPxPTR) Registers as arrays
 * 9/15/98      saw     rewrote PCMCIA window base and option regs as arrays
 *                      Deleted PCMCIA pgcra
 * 9/15/98      saw     rewrote MEMC base and option regs as arrays
 * 10/15/98     saw     added additional reserved word to USB Endpoint Params Block (USEP)
 *
 */

/*****************************************************************
	General Purpose SCC parameter RAM
*****************************************************************/

struct scc_pram {
	unsigned short	rbase;		/* RX BD base address */
	unsigned short	tbase;		/* TX BD base address */
	unsigned char	rfcr;		/* Rx function code */
	unsigned char	tfcr;		/* Tx function code */
	unsigned short	mrblr;		/* Rx buffer length */
	unsigned long	rstate;		/* Rx internal state */
	unsigned long	rptr;		/* Rx internal data pointer */
	unsigned short	rbptr;		/* rb BD Pointer */
	unsigned short	rcount;		/* Rx internal byte count */
	unsigned long	rtemp;		/* Rx temp */
	unsigned long	tstate;		/* Tx internal state */
	unsigned long	tptr;		/* Tx internal data pointer */
	unsigned short	tbptr;		/* Tx BD pointer */
	unsigned short	tcount;		/* Tx byte count */
	unsigned long	ttemp;		/* Tx temp */
	unsigned long	rcrc;		/* temp receive CRC */
	unsigned long	tcrc;		/* temp transmit CRC */
};


/*****************************************************************
	HDLC parameter RAM
*****************************************************************/

struct hdlc_pram {
	/*
	 * HDLC specific SCC parameter RAM
	 */
	unsigned char	RESERVED1[4];	/* Reserved area */
	unsigned long	c_mask;		/* CRC constant */
	unsigned long	c_pres;		/* CRC preset */
	unsigned short	disfc;		/* discarded frame counter */
	unsigned short	crcec;		/* CRC error counter */
	unsigned short	abtsc;		/* abort sequence counter */
	unsigned short	nmarc;		/* nonmatching address rx cnt */
	unsigned short	retrc;		/* frame retransmission cnt */
	unsigned short	mflr;		/* maximum frame length reg */
	unsigned short	max_cnt;	/* maximum length counter */
	unsigned short	rfthr;		/* received frames threshold */
	unsigned short	rfcnt;		/* received frames count */
	unsigned short	hmask;		/* user defined frm addr mask */
	unsigned short	haddr1;	/* user defined frm address 1 */
	unsigned short	haddr2;	/* user defined frm address 2 */
	unsigned short	haddr3;	/* user defined frm address 3 */
	unsigned short	haddr4;	/* user defined frm address 4 */
	unsigned short	tmp;	/* temp */
	unsigned short	tmp_mb;	/* temp */
};


/*****************************************************************
	ASYNC HDLC parameter RAM
*****************************************************************/

struct async_hdlc_pram {
	/*
	 * ASYNC HDLC specific SCC parameter RAM
	 */
	unsigned char	RESERVED1[4];	/* Reserved area */
	unsigned long	c_mask;		/* CRC constant */
	unsigned long	c_pres;		/* CRC preset */
	unsigned short	bof;		/* begining of flag character */
	unsigned short	eof;		/* end of flag character */
	unsigned short	esc;		/* control escape character */
	unsigned char	RESERVED2[4];	/* Reserved area */
	unsigned short	zero;		/* zero */
	unsigned char	RESERVED3[2];	/* Reserved area */
	unsigned short	rfthr;		/* received frames threshold */
	unsigned char	RESERVED4[4];	/* Reserved area */
	unsigned long	txctl_tbl;	/* Tx ctl char mapping table */
	unsigned long	rxctl_tbl;	/* Rx ctl char mapping table */
	unsigned short	nof;		/* Number of opening flags */
};


/*****************************************************************
	UART parameter RAM
*****************************************************************/

/*
 * bits in uart control characters table
 */
#define	CC_INVALID	0x8000		/* control character is valid */
#define	CC_REJ		0x4000		/* don't store char in buffer */
#define	CC_CHAR		0x00ff		/* control character */

struct scc_uart_pram {
	/*
	 * SCC UART specific SCC parameter RAM
	 */
	unsigned char	RESERVED1[8];	/* Reserved area */
	unsigned short	max_idl;	/* maximum idle characters */
	unsigned short	idlc;		/* rx idle counter (internal) */
	unsigned short	brkcr;		/* break count register */

	unsigned short	parec;		/* Rx parity error counter */
	unsigned short	frmec;		/* Rx framing error counter */
	unsigned short	nosec;		/* Rx noise counter */
	unsigned short	brkec;		/* Rx break character counter */
	unsigned short	brkln;		/* Reaceive break length */

	unsigned short	uaddr1;		/* address character 1 */
	unsigned short	uaddr2;		/* address character 2 */
	unsigned short	rtemp;		/* temp storage */
	unsigned short	toseq;		/* Tx out of sequence char */
	unsigned short	cc[8];		/* Rx control characters */
	unsigned short	rccm;		/* Rx control char mask */
	unsigned short	rccrp;		/* Rx control char register */
	unsigned short	rlbc;		/* Receive last break char */
};


/*****************************************************************
	SPI parameter RAM
*****************************************************************/

#define	SPI_R	0x8000		/* Ready bit in BD */

struct spi_pram {
	unsigned char   RESERVED[0x50]; /* RESERVED -- SCC2 Space */
	unsigned short	rbase;		/* Rx BD Base Address */
	unsigned short	tbase;		/* Tx BD Base Address */
	unsigned char	rfcr;		/* Rx function code */
	unsigned char	tfcr;		/* Tx function code */
	unsigned short	mrblr;		/* Rx buffer length */
	unsigned long	rstate;		/* Rx internal state */
	unsigned long	rptr;		/* Rx internal data pointer */
	unsigned short	rbptr;		/* rb BD Pointer */
	unsigned short	rcount;		/* Rx internal byte count */
	unsigned long	rtemp;		/* Rx temp */
	unsigned long	tstate;		/* Tx internal state */
	unsigned long	tptr;		/* Tx internal data pointer */
	unsigned short	tbptr;		/* Tx BD pointer */
	unsigned short	tcount;		/* Tx byte count */
	unsigned long	ttemp;		/* Tx temp */
        unsigned char   RESERVED1[8];
};

struct smc_pram {
        /*
         * General Purpose SMC PRAM
         */
	unsigned short	rbase;		/* Rx BD Base Address */
	unsigned short	tbase;		/* Tx BD Base Address */
	unsigned char	rfcr;		/* Rx function code */
	unsigned char	tfcr;		/* Tx function code */
	unsigned short	mrblr;		/* Rx buffer length */
	unsigned long	rstate;		/* Rx internal state */
	unsigned long	rptr;		/* Rx internal data pointer */
	unsigned short	rbptr;		/* rb BD Pointer */
	unsigned short	rcount;		/* Rx internal byte count */
	unsigned long	rtemp;		/* Rx temp */
	unsigned long	tstate;		/* Tx internal state */
	unsigned long	tptr;		/* Tx internal data pointer */
	unsigned short	tbptr;		/* Tx BD pointer */
	unsigned short	tcount;		/* Tx byte count */
	unsigned long	ttemp;		/* Tx temp */

        /*
         * UART-Specific SMC PRAM
         */
	unsigned short	max_idl;	/* Maximum IDLE Characters */
	unsigned short	idlc;		/* Temporary IDLE Counter */
	unsigned short	brkln;		/* Last Rx Break Length */
	unsigned short	brkec;		/* Rx Break Condition Counter */
	unsigned short	brkcr;		/* Break Count Register (Tx) */
	unsigned short	r_mask;		/* Temporary bit mask */
        unsigned char   RESERVED[0xc];  /* RESERVED */
};

/*****************************************************************
	DSP Parameter RAM
*****************************************************************/

struct dsp_pram {
	unsigned long	fdbase;		/* Function Descr Tbl Base Addr */
	unsigned long	fd_ptr;		/* (Internal) Function Descr Ptr */
	unsigned long	dstate;		/* (Internal) DSP State */
        unsigned char   RESERVED0[4];   /* RESERVED */
	unsigned short	dstatus;	/* (Internal) Current Function Descr Status */
	unsigned short	i;		/* (Internal) Current Function Descr Number of Iterations */
	unsigned short	tap;		/* (Internal) Current Function Descr Number of Taps */
	unsigned short	cbase;		/* (Internal) Current Function Descr Cbase Pointer */
	unsigned short	fdsbsm1;	/* (Internal) Current Function Descr Sample Buffer Size - 1 */
	unsigned short	xptr;		/* (Internal) Current Function Descr Pointer to Sample Pointer */
	unsigned short	fdobsm1;	/* (Internal) Current Function Descr Output Buffer Size - 1 */
	unsigned short	yptr;		/* (Internal) Current Function Descr Pointer to Output Buffer Pointer */
	unsigned short	m;		/* (Internal) Current Function Descr Sample Buffer Size - 1 */
	unsigned short	fdsbp;		/* (Internal) Current Function Descr Sample Buffer Pointer */
	unsigned short	n;		/* (Internal) Current Function Descr Output Buffer Size - 1 */
	unsigned short	fdobp;		/* (Internal) Current Function Descr Output Buffer Pointer */
	unsigned short	k;		/* (Internal) Current Function Descr Coefficient Buffer Size - 1 */
	unsigned short	fdcbp;		/* (Internal) Current Function Descr Coefficient Buffer Pointer */
        unsigned char RESERVED1[0x14];	/* RESERVED */
};



/*****************************************************************
	IDMA Parameter RAM (Autobuffer/Buffer-Chaining; Single Buffer)
*****************************************************************/

struct idma_ab_bc_pram {
	/*
	 * IDMA parameter RAM specifically for Autobuffer or 
         *     Buffer-Chaining Modes
	 */
	unsigned short	ibase;		/* BD Base Address */
	unsigned short	dcmr;		/* Channel Mode Register */
	unsigned long	sapr;		/* source internal address pointer */
	unsigned long	dapr;		/* destination internal address pointer */
	unsigned short	ibptr;		/* (Internal) Buffer Descriptor Pointer */
	unsigned short	write_sp;	/* (Internal) */
	unsigned long	s_byte_c;	/* Internal Source Byte Count */
	unsigned long	d_byte_c;	/* Internal Destination Byte Count */
	unsigned long	s_state;	/* Internal State */
	unsigned long	itemp[4];	/* (Internal) Temporary Data Storage */
	unsigned long	sr_mem;	        /* (Internal) Data Storage for Peripheral Write */
	unsigned short	read_sp;	/* (Internal) */
	unsigned short	dbsdr;		/* (Internal) Diff Between Source and Destination Residue */
	unsigned short	tsap;		/* (Internal) Temporary Storage Address Pointer */
	unsigned short	sbc;		/* (Internal) SR_MEM Byte Count */
	unsigned long	d_state;	/* Internal State */
        unsigned char   RESERVED[0x4];	/* Reserved */
};

struct idma_sb_pram {
	/*
	 * IDMA parameter RAM specifically for Single-Buffer Mode
         *     NOTE:  Definitions depend on the destination address generation
         *            mode -- progressive or interlaced
	 */
	unsigned long	bapr;
	unsigned long	bcr;
	unsigned long	dcmr;
	unsigned long	fbar;
	unsigned char	RESERVED0[2];
	unsigned short  nfld;
	unsigned short  lcr;
	unsigned short  l_cnt;
	unsigned short  bplr;
	unsigned short  rbr;
	unsigned char   RESERVED1[0x24];
};


struct ethernet_pram {
	/*
	 * ETHERNET specific parameter RAM
	 */
	unsigned long	c_pres;		/* preset CRC */
	unsigned long	c_mask;		/* constant mask for CRC */
	unsigned long	crcec;		/* CRC error counter */
	unsigned long	alec;		/* alighnment error counter */
	unsigned long	disfc;		/* discard frame counter */
	unsigned short	pads;		/* short frame PAD characters */
	unsigned short	ret_lim;	/* retry limit threshold */
	unsigned short	ret_cnt;	/* retry limit counter */
	unsigned short	mflr;		/* maximum frame length reg */
	unsigned short	minflr;		/* minimum frame length reg */
	unsigned short	maxd1;		/* maximum DMA1 length reg */
	unsigned short	maxd2;		/* maximum DMA2 length reg */
	unsigned short	maxd;		/* rx max DMA */
	unsigned short	dma_cnt;	/* rx dma counter */
	unsigned short	max_b;		/* max bd byte count */
	unsigned short	gaddr1;		/* group address filter 1 */
	unsigned short	gaddr2;		/* group address filter 2 */
	unsigned short	gaddr3;		/* group address filter 3 */
	unsigned short	gaddr4;		/* group address filter 4 */
	unsigned long	tbuf0_data0;	/* save area 0 - current frm */
	unsigned long	tbuf0_data1;	/* save area 1 - current frm */
	unsigned long	tbuf0_rba0;
	unsigned long	tbuf0_crc;
	unsigned short	tbuf0_bcnt;
	unsigned short	paddr_l;	/* physical address (LSB) */
	unsigned short	paddr_m;	/* physical address (Middle Byte) */
	unsigned short	paddr_h;	/* physical address (MSB) */
	unsigned short	p_per;		/* persistence */
	unsigned short	rfbd_ptr;	/* rx first bd pointer */
	unsigned short	tfbd_ptr;	/* tx first bd pointer */
	unsigned short	tlbd_ptr;	/* tx last bd pointer */
	unsigned long	tbuf1_data0;	/* save area 0 - next frame */
	unsigned long	tbuf1_data1;	/* save area 1 - next frame */
	unsigned long	tbuf1_rba0;
	unsigned long	tbuf1_crc;
	unsigned short	tbuf1_bcnt;
	unsigned short	tx_len;		/* tx frame length counter */
	unsigned short	iaddr1;		/* individual address filter 1*/
	unsigned short	iaddr2;		/* individual address filter 2*/
	unsigned short	iaddr3;		/* individual address filter 3*/
	unsigned short	iaddr4;		/* individual address filter 4*/
	unsigned short	boff_cnt;	/* back-off counter */
	unsigned short	taddr_l;	/* temp address (LSB) */
	unsigned short	taddr_m;	/* temp address (Middle Byte) */
	unsigned short	taddr_h;	/* temp address (MSB) */
};

struct transparent_pram {
	/*
	 * TRANSPARENT specific parameter RAM
	 */
	unsigned long	crc_p;		/* CRC Preset */
	unsigned long	crc_c;		/* CRC constant */
};

struct usb_pram {
	/*
         * USB specific parameter RAM
         */
	unsigned short  ep_ptr[4];	/* Endpoint 0..3 Pointers */
	unsigned long   rstate;		/* RX Internal State */
	unsigned long   rptr;		/* RX Internal Data Pointer */
	unsigned short  frame_n;	/* Frame Number */
	unsigned short  rbcnt;		/* RX Internal Byte Count */
	unsigned short  rtemp;		/* RX Temp */
      unsigned char RESERVED[0x68];
};

struct timer_pram {
	unsigned short	tm_base;	/* RISC timer table base adr */
	unsigned short	tm_ptr;		/* RISC timer table pointer */
	unsigned short	r_tmr;		/* RISC timer mode register */
	unsigned short	r_tmv;		/* RISC timer valid register */
	unsigned long	tm_cmd;		/* RISC timer cmd register */
	unsigned long	tm_cnt;		/* RISC timer internal cnt */
};

struct i2c_pram {
	unsigned short	rbase;		/* RX BD base address */
	unsigned short	tbase;		/* TX BD base address */
	unsigned char	rfcr;		/* Rx function code */
	unsigned char	tfcr;		/* Tx function code */
	unsigned short	mrblr;		/* Rx buffer length */
	unsigned long	rstate;		/* Rx internal state */
	unsigned long	rptr;		/* Rx internal data pointer */
	unsigned short	rbptr;		/* rb BD Pointer */
	unsigned short	rcount;		/* Rx internal byte count */
	unsigned long	rtemp;		/* Rx temp */
	unsigned long	tstate;		/* Tx internal state */
	unsigned long	tptr;		/* Tx internal data pointer */
	unsigned short	tbptr;		/* Tx BD pointer */
	unsigned short	tcount;		/* Tx byte count */
	unsigned long	ttemp;		/* Tx temp */
        unsigned char   RESERVED1[8];   /* RESERVED */
};

/*
 * definitions of EPPC memory structures
 */

typedef struct eppc {
/* BASE + 0x0000: INTERNAL REGISTERS */
/* SIU */
        volatile struct siu_regs {
		unsigned long	siumcr;	/* module configuration reg */
		unsigned long	sypcr;	/* System protection cnt */
		unsigned char   RESERVED0[0x6];
		unsigned short	swsr;	/* sw service */
		unsigned long	sipend;	/* Interrupt pend reg */
		unsigned long	simask;	/* Interrupt mask reg */
		unsigned long	siel;	/* Interrupt edge level mask reg */
		unsigned long	sivec;	/* Interrupt vector */
		unsigned long	tesr;	/* Transfer error status */
		unsigned char	RESERVED1[0xc];/* Reserved area */
		unsigned long	sdcr;	/* SDMA configuration reg */
		unsigned char   RESERVED2[0x4c];
        } siu;

/* PCMCIA */
        volatile struct pcmcia_regs {
            /* PCMCIA BASE and OPTION Registers for each of Windows 0:7 */
            struct pcmcia_window {
                unsigned long  pbr;    /* PCMCIA Base Reg */
                unsigned long  por;    /* PCMCIA Option Reg */
            } w[8];
		unsigned char	RESERVED0[0x24];/* Reserved area */
		unsigned long	pgcrb;      /* PCMCIA Slot B Control  Reg */
		unsigned long	pscr;       /* PCMCIA Status Reg */
		unsigned char	RESERVED1[4];/* Reserved area */
		unsigned long	pipr;       /* PCMCIA Pins Value Reg */
		unsigned char	RESERVED2[4];/* Reserved area */
		unsigned long	per;        /* PCMCIA Enable Reg */
		unsigned char	RESERVED3[4];/* Reserved area */
        } pcmcia;

/* MEMC */
        volatile struct memc_regs {
            /* Memory Controller BASE and OPTION Registers for each of Banks 0:7 */
            struct memory_controller_bank {
		    unsigned long	br;	/* base register */
		    unsigned long	or;	/* option register */
            } bank[8];
		unsigned char	RESERVED0[0x24];	/* Reserved area */
		unsigned long	mar;	/* Memory address */
		unsigned long	mcr;	/* Memory command */
		unsigned char	RESERVED1[0x4];	/* Reserved area */
		unsigned long	mamr;	/* Machine A mode */
		unsigned long	mbmr;	/* Machine B mode */
		unsigned short	mstat;	/* Memory status */
		unsigned short	mptpr;	/* Memory preidic timer prescalar */
		unsigned long	mdr;	/* Memory data */
		unsigned char	RESERVED2[0x80];/* Reserved area */
        } memc;

/* SYSTEM INTEGRATION TIMERS */
        volatile struct simt_regs {
		unsigned short	tbscr;	/* Time base stat&ctr */
        	unsigned char	RESERVED0[0x2];	/* Reserved area */
		unsigned long	tbreffu;	/* Time base reference upper */
		unsigned long	tbreffl;	/* Time base reference lower */
		unsigned char	RESERVED1[0x14];/* Reserved area */
		unsigned short	rtcsc;	/* Realtime clk stat&cntr 1 */
        	unsigned char	RESERVED2[0x2];	/* Reserved area */
		unsigned long	rtc;	/* Realtime clock */
		unsigned long	rtsec;	/* Realtime alarm seconds */
		unsigned long	rtcal;	/* Realtime alarm */
		unsigned char	RESERVED3[0x10];/* Reserved area */
                unsigned short  piscr;  /* PIT stat&ctrl */
                unsigned char   RESERVED3a[0x2]; /* Reserved area */
		unsigned long	pitc;	/* PIT counter */
		unsigned long	pitr;	/* PIT */
		unsigned char	RESERVED4[0x34];/* Reserved area */
        } simt;

/* CLOCKS, RESET */
        volatile struct clocks_and_reset_regs {
		unsigned long	sccr;	/* System clk cntrl */
		unsigned long	plprcr;	/* PLL reset&ctrl */
		unsigned long	rsr;	/* reset status */
		unsigned char	RESERVED[0x74];/* Reserved area */
        } clkr;

/* System Integration Timers Keys */
        volatile struct simtk_regs {
        	unsigned long  tbscrk;    /* Timebase Status&Ctrl Key */
        	unsigned long  tbreffuk;  /* Timebase Reference upper Key */
        	unsigned long  tbrefflk;  /* Timebase Reference lower Key */
        	unsigned long  tbk;       /* Timebase and Decrementer Key */
		unsigned char  RESERVED0[0x10];/* Reserved area */
        	unsigned long  rtcsck;    /* Real-Time Clock Status&Ctrl Key */
        	unsigned long  rtck;      /* Real-Time Clock Key */
        	unsigned long  rtseck;    /* Real-Time Alarm Seconds Key */
        	unsigned long  rtcalk;    /* Real-Time Alarm Key */
		unsigned char  RESERVED1[0x10];/* Reserved area */
        	unsigned long  piscrk;    /* Periodic Interrupt Status&Ctrl Key */
        	unsigned long  pitck;     /* Periodic Interrupt Count Key */
		unsigned char  RESERVED2[0x38];/* Reserved area */
        } simtk;


/* Clock and Reset Keys */
        volatile struct clkrk_regs {
        	unsigned long	sccrk;    /* System Clock Control Key */
        	unsigned long	plprcrk;  /* PLL, Low Power and Reset Control Key */
        	unsigned long	rsrk;     /* Reset Status Key */
		unsigned char	RESERVED[0x474];/* Reserved area */
	} clkrk;

/* Video Controller */
        volatile struct video_regs {
        	unsigned short vc_vccr;    /* VC Configuration Reg */
        	unsigned char RESERVED0[2];
        	unsigned char  vc_vsr;     /* VC Status Reg 	*/
        	unsigned char RESERVED1;
        	unsigned char  vc_vcmr;    /* VC Command Reg */
        	unsigned char RESERVED2;
        	unsigned long  vc_vbcb;    /* VC Background Color Buffer Reg */
        	unsigned char RESERVED3[4];
        	unsigned long  vc_vfcr0;   /* VC Frame Config Set 0 Reg */
        	unsigned long  vc_vfaa0;   /* VC Frame Buffer A Address Set 0 Reg
 */
        	unsigned long  vc_vfba0;   /* VC Frame Buffer B Address Set 0 Reg
 */
        	unsigned long  vc_vfcr1;   /* VC Frame Config Set 1 Reg */
        	unsigned long  vc_vfaa1;   /* VC Frame Buffer A Address Set 1 Reg
 */
        	unsigned long  vc_vfba1;   /* VC Frame Buffer B Address Set 1 Reg
 */
        	unsigned char RESERVED4[0x18];
        } video;


/* LCD Controller */
        volatile struct lcd_regs {
		unsigned long	lccr;	/* configuration Reg */
		unsigned long	lchcr;	/* Horizontal ctl Reg */
		unsigned long	lcvcr;	/* Vertical ctl Reg */
		unsigned char	RESERVED0[4];
		unsigned long	lcfaa;	/* Frame buffer A Address */
		unsigned long	lcfba;	/* Frame buffer B Address */
		unsigned char	lcsr;	/* Status Reg */
		unsigned char	RESERVED1[7];/* Reserved area */
        } lcd;

/* I2C */
        volatile struct i2c_regs {
		unsigned char	i2mod;	/* i2c mode */
		unsigned char	RESERVED59[3];
		unsigned char	i2add;	/* i2c address */
		unsigned char	RESERVED60[3];
		unsigned char	i2brg;	/* i2c brg */
		unsigned char	RESERVED61[3];
		unsigned char	i2com;	/* i2c command */
		unsigned char	RESERVED62[3];
		unsigned char	i2cer;	/* i2c event */
		unsigned char	RESERVED63[3];
		unsigned char	i2cmr;	/* i2c mask */
		unsigned char	RESERVED10[0x8b];/* Reserved area */
        } i2c_reg;

/* DMA */
        volatile struct dma_regs {
        	unsigned char  RESERVED0[0x4];/* Reserved area */
        	unsigned long  sdar;       /* SDMA address reg */
        	unsigned char  sdsr;       /* SDMA status reg */
        	unsigned char  RESERVED1[0x3];/* Reserved area */
        	unsigned char  sdmr;       /* SDMA mask reg */
        	unsigned char  RESERVED2[0x3];/* Reserved area */
        	unsigned char  idsr1;      /* IDMA1 status reg */
        	unsigned char  RESERVED3[0x3];/* Reserved area */
        	unsigned char  idmr1;      /* IDMA1 mask reg */
        	unsigned char  RESERVED4[0x3];/* Reserved area */
        	unsigned char  idsr2;      /* IDMA2 status reg */
        	unsigned char  RESERVED5[0x3];/* Reserved area */
        	unsigned char  idmr2;      /* IDMA2 mask reg */
        	unsigned char  RESERVED6[0x13];/* Reserved area */
        } dma;

/* CPM Interrupt Controller */
        volatile struct cpmic_regs {
		unsigned short	civr;	/* CP interrupt vector reg */
		unsigned char	RESERVED[0xe];/* Reserved area */
		unsigned long	cicr;	/* CP interrupt configuration reg */
		unsigned long	cipr;	/* CP interrupt pending reg */
		unsigned long	cimr;	/* CP interrupt mask reg */
		unsigned long	cisr;	/* CP interrupt in-service reg */
        } cpmic;

/* Parallel Ports A,C,D */
        volatile struct pio_regs {
/* Parallel Port A */
		unsigned short	padir;	/* port A data direction reg */
		unsigned short	papar;	/* port A pin assignment reg */
		unsigned short	paodr;	/* port A open drain reg */
		unsigned short	padat;	/* port A data register */
		unsigned char	RESERVED0[8];	/* Reserved area */
/* Parallel Port C */
		unsigned short	pcdir;	/* port C data direction reg */
		unsigned short	pcpar;	/* port C pin assignment reg */
		unsigned short	pcso;	/* port C special options */
		unsigned short	pcdat;	/* port C data register */
		unsigned short	pcint;	/* port C interrupt cntrl reg */
		unsigned char	RESERVED1[6];
/* Parallel Port D */
		unsigned short	pddir;	/* port D Data Direction reg */
		unsigned short	pdpar;	/* port D pin assignment reg */
		unsigned char	RESERVED2[2];
		unsigned short	pddat;	/* port D data reg */
		unsigned char	RESERVED3[8];	/* Reserved area */
        } pio;

/* CPM Timer */
        volatile struct cpmt_regs {
		unsigned short	tgcr;	/* timer global configuration  reg */
		unsigned char	RESERVED0[0xe];	/* Reserved area */

		unsigned short	tmr1;	/* timer 1 mode reg */
		unsigned short	tmr2;	/* timer 2 mode reg */
		unsigned short	trr1;	/* timer 1 reference reg */
		unsigned short	trr2;	/* timer 2 reference reg */
		unsigned short	tcr1;	/* timer 1 capture reg */
		unsigned short	tcr2;	/* timer 2 capture reg */
		unsigned short	tcn1;	/* timer 1 counter reg */
		unsigned short	tcn2;	/* timer 2 counter reg */

		unsigned short	tmr3;	/* timer 3 mode reg */
		unsigned short	tmr4;	/* timer 4 mode reg */
		unsigned short	trr3;	/* timer 3 reference reg */
		unsigned short	trr4;	/* timer 4 reference reg */
		unsigned short	tcr3;	/* timer 3 capture reg */
		unsigned short	tcr4;	/* timer 4 capture reg */
		unsigned short	tcn3;	/* timer 3 counter reg */
		unsigned short	tcn4;	/* timer 4 counter reg */

		unsigned short	ter1;	/* timer 1 event reg */
		unsigned short	ter2;	/* timer 2 event reg */
		unsigned short	ter3;	/* timer 3 event reg */
		unsigned short	ter4;	/* timer 4 event reg */
		unsigned char	RESERVED1[8];	/* Reserved area */
        } timer;

/* CP */
        volatile struct cp_regs {
		unsigned short	cpcr;		/* command register */
		unsigned char	RESERVED0[2];	/* Reserved area */
		unsigned short	rccr;	/* main configuration reg */
		unsigned char	RESERVED1;	/* Reserved */
		unsigned char	rmds;	/* RISC Microcode Development Support Status Register */
                unsigned char   RESERVED3[0xe];
		unsigned short	rter;	/* RISC timer event reg */
		unsigned char	RESERVED4[2];  /* Reserved */
		unsigned short	rtmr;	/* RISC timers mask reg */
		unsigned char	RESERVED5[0x14];	/* Reserved area */
        } cp;

/* BRG */
	volatile unsigned long	brgc1;		/* BRG1 configuration reg */
	volatile unsigned long	brgc2;		/* BRG2 configuration reg */
	volatile unsigned long	brgc3;		/* BRG3 configuration reg */
	volatile unsigned long	brgc4;		/* BRG4 configuration reg */

/* USB */
      struct usb_regs {
            unsigned char usmod;        /* USB Mode Reg */
            unsigned char usadr;        /* USB Address Reg */
            unsigned char uscom;        /* USB Command Reg */
            unsigned char    RESERVED0;
            unsigned short usep[4];     /* USB End Point 0..3 Registers */
            unsigned char    RESERVED1[4];
            unsigned short usber;       /* USB Event Reg */
            unsigned char    RESERVED2[2];
            unsigned short usbmr;       /* USB Mask Reg */
            unsigned char    RESERVED3;
            unsigned char usbs;         /* USB Status Reg */
            unsigned char    RESERVED4[8];
      } usb_regs;


/* SCC registers */
	volatile struct scc_regs {
		unsigned long	gsmr_l; /* SCC Gen mode (LOW) */
		unsigned long	gsmr_h; /* SCC Gen mode (HIGH) */
		unsigned short	psmr;	/* protocol specific mode register */
		unsigned char	RESERVED0[2]; /* Reserved area */
		unsigned short	todr; /* SCC transmit on demand */
		unsigned short	dsr;	/* SCC data sync reg */
		unsigned short	scce;	/* SCC event reg */
		unsigned char	RESERVED1[2];/* Reserved area */
		unsigned short	sccm;	/* SCC mask reg */
		unsigned char	RESERVED2;	/* Reserved area */
		unsigned char	sccs;	/* SCC status reg */
                /*
                 * InfraRed
                 */
                unsigned short	irmode;	/* SCC InfraRed Mode Register */
                unsigned short	irsip;	/* SCC InfraRed Serial Interaction Pulse Control Register */
        	unsigned char RESERVED8[0x44];
	} scc_reg;


/* SMC */
	volatile struct smc_regs {
		unsigned char	RESERVED34[0x2]; /* Reserved area */
		unsigned short	smcmr;	/* SMC mode reg */
		unsigned char	RESERVED35[0x2];/* Reserved area */
		unsigned char	smce;	/* SMC event reg */
		unsigned char	RESERVED36[0x3]; /* Reserved area */
		unsigned char	smcm;	/* SMC mask reg */
		unsigned char	RESERVED37[0x5]; /* Reserved area */
	} smc_reg[2];

/* SPI */
        volatile struct spi_regs {
		unsigned short	spmode;	/* SPI mode reg */
		unsigned char	RESERVED0[4];	/* Reserved area */
		unsigned char	spie;	/* SPI event reg */
		unsigned char	RESERVED1[3];	/* Reserved area */
		unsigned char	spim;	/* SPI mask reg */
		unsigned char	RESERVED2[2];	/* Reserved area */
		unsigned char	spcom;	/* SPI command reg */
		unsigned char	RESERVED3[4];	/* Reserved area */
        } spi_reg;

	unsigned char	RESERVED10[6];	/* Reserved area */

/* Port B */
        volatile struct port_b_regs {
		unsigned long	pbdir;	/* port b data direction reg */
		unsigned long	pbpar;	/* port b pin assignment reg */
		unsigned char	RESERVED0[2];	/* Reserved area */
		unsigned short	pbodr;	/* port b open drain reg */
		unsigned long	pbdat;	/* port b data reg */
		unsigned char	RESERVED1[0x18];	/* Reserved area */
        } pb;

/* Serial Interface */
        volatile struct serial_interface_regs {
		unsigned long	simode;	/* SI mode register */
		unsigned char	sigmr;	/* SI global mode register */
		unsigned char	RESERVED0; /* Reserved area */
		unsigned char	sistr;	/* SI status register */
		unsigned char	sicmr;	/* SI command register */
		unsigned char	RESERVED1[0x4]; /* Reserved area */
		unsigned long	sicr;	/* SI clock routing */
		unsigned long	sirp;	/* SI ram pointers */
        } si;

        volatile unsigned char RESERVED300[0xc];

        volatile unsigned char	vc_vcram[0x100]; /* Video Controller RAM */

        volatile unsigned long	si_siram[0x80]; /* SI routing RAM */

        volatile unsigned short lcd_lcolr[256]; /* LCD Color RAM */

	volatile unsigned char	RESERVED48[0x1000]; /* Reserved area */

/* BASE + 0x2000::0x3C00 (CDR and later) user data memory */
        volatile struct micrcode_user_data {
            unsigned char bd_ucode0[0x200]; /*user data bd's Ucode*/
            unsigned char bd_ucode1[0x200]; /*user data bd's Ucode*/
            unsigned char bd_ucode2[0x400]; /*user data bd's Ucode*/
            unsigned char bd0[0x600];       /*user data Ucode*/
            unsigned char bd_ucode3[0x200]; /*user data bd's Ucode*/
            unsigned char bd_ucode4[0x400]; /*user data bd's Ucode*/
            unsigned char bd_ucode5[0x400]; /*user data bd's Ucode*/
            unsigned char bd1[0x200];       /*user data Ucode*/
            unsigned char bd_ucode6[0x200]; /*user data bd's Ucode*/
        } udata;

/* BASE + 0x3c00: PARAMETER RAM */
        volatile struct parameter_ram {
            struct usb_pram usb;
	    struct i2c_pram i2c;
            unsigned char MISC[0x10];
	    union {
		struct idma_ab_bc_pram	ab_bc;
		struct idma_sb_pram	sb;
            } idma1;
            struct scc_pram scc2;
            union {
                /*
                 * SCC2 Protocols
                 */
	        struct async_hdlc_pram	a;
	        struct ethernet_pram	e;
	        struct hdlc_pram	h;
	        struct transparent_pram	t;
	        struct scc_uart_pram	u;
                /*
                 * SPI -- Note SPI and Ethernet PRAM Overlay
                 */
	        struct spi_pram	spi;
            } scc2_spi;
	    struct timer_pram timer;
            /*
             * IDMA2 Configuration Options:  Autobuffer, Buffer Chaining
             */
	    struct idma_ab_bc_pram idma2;
            unsigned char   RESERVED100[0x80];    /* RESERVED */
            struct smc_pram smc1;
	    struct dsp_pram dsp1;
            unsigned char RESERVED200[0x80];
            struct smc_pram smc2;
	    struct dsp_pram dsp2;
        } pram;
} EPPC;


typedef struct usb_endpoint_parameters_block {
	unsigned short	rbase;		/* RX BD Base Address */
	unsigned short	tbase;		/* TX BD Base Address */
	unsigned char	rfcr;		/* RX Function Code */
	unsigned char	tfcr;		/* TX Function Code */
	unsigned short	mrblr;		/* Maximum Receive Buffer Length */
	unsigned short	rbptr;		/* RX BD Pointer */
	unsigned short	tbptr;		/* TX BD Pointer */
	unsigned long	tstate;		/* TX Internal State */
	unsigned long	tptr;		/* TX Internal Data Pointer */
	unsigned short	tcrc;		/* TX Temp CRC */
	unsigned short	tbcnt;		/* TX Internal Byte Count */
	unsigned char	RESERVED[8];	/* Reserved */
} USEP;


#define MEMC_BR1_OFFSET 0x108

#define MDR_OFFSET 0x0000017cL
#define MCR_OFFSET 0x00000168L
#define MCR_BASIC_VALUE 0x00004100L

#define BR0_OFFSET 0x100L
#define OR0_OFFSET 0x104L
#define BR1_OFFSET 0x108L
#define OR1_OFFSET 0x10cL
#define BR2_OFFSET 0x110L
#define OR2_OFFSET 0x114L
#define BR3_OFFSET 0x118L
#define OR3_OFFSET 0x11cL
#define BR0_VALUE 0x02800001L
#define BR1_VALUE 0x02100001L
#define BR2_VALUE 0x00000081L
#define OR0_VALUE 0xffe00954L
#define OR1_VALUE 0xffff8110L
#define OR2_VALUE 0xffc00800L
#define BR3_VALUE_36200 0x00400081L
#define BR3_VALUE_36800 0x01000081L
#define OR3_VALUE_36200 0xffc00800L
#define OR3_VALUE_36800 0xff000800L
#define OR2_VALUE_36800 0xff000800L
#define OR2_VALUE_36400 OR2_VALUE_36800
#define MPTPR_OFFSET 0x0000017aL
#define MPTPR_VALUE 0x0800
#define MAMR_OFFSET 0x00000170L
#define MAMR_VALUE 0xc0a21114L
#define MAMR_VALUE_36200 0x60a21114L
#define MAMR_VALUE_36400 0xc0b21114L
#define MAMR_VALUE_36800 0x60b21114L

/* the default FLASH is 29020 - 2M */
#define OR0_VALUE_29040 0xffc00954L
#define OR0_VALUE_29060 0xff800954L

#define SIUMCR_OFFSET 0
#define SYPCR_OFFSET 4
#define TBSCR_OFFSET 0x200
#define RTCSC_OFFSET 0x220
#define PISCR_OFFSET 0x240
#define INTERNAL_MEM_ADR 0x2200000L

/* KEEP ALIVE POWER REGISTERS KEY */
#define KEEP_ALIVE_KEY 0x55ccaa33

/* END of the MOTOROLA file. The rest has been added by Josu Onandia,
 * derived from the descriptions of MPC821 and MPC860
 */
/*
*************************************************************************
*                 SDMA (SCC, SMC, SPI) Buffer Descriptors               *
*************************************************************************
*/
typedef struct m823BufferDescriptor_ {
  rtems_unsigned16        status;
  rtems_unsigned16        length;
  volatile void           *buffer;
} m823BufferDescriptor_t;

#define M823_BRG_RST            (1<<17)         /* Reset generator */
#define M823_BRG_EN             (1<<16)         /* Enable generator */
#define M823_BRG_EXTC_BRGCLK    (0<<14)         /* Source is BRGCLK */

#define M823_RFCR_MOT           (2<<3)
#define M823_TFCR_MOT           (2<<3)

/*
 * Bits in receive buffer descriptor status word
 */
#define M823_BD_EMPTY           (1<<15) /* Ethernet, SCC UART, SMC UART, SPI */
#define M823_BD_WRAP            (1<<13) /* Ethernet, SCC UART, SMC UART, SPI */
#define M823_BD_INTERRUPT       (1<<12) /* Ethernet, SCC UART, SMC UART, SPI */
#define M823_BD_LAST            (1<<11) /* Ethernet, SPI */
#define M823_BD_CONTROL_CHAR    (1<<11) /* SCC UART */
#define M823_BD_FIRST_IN_FRAME  (1<<10) /* Ethernet */
#define M823_BD_ADDRESS         (1<<10) /* SCC UART */
#define M823_BD_CONTINUOUS      (1<<9)  /* SCC UART, SMC UART, SPI */
#define M823_BD_MISS            (1<<8)  /* Ethernet */
#define M823_BD_IDLE            (1<<8)  /* SCC UART, SMC UART */
#define M823_BD_ADDRSS_MATCH    (1<<7)  /* SCC UART */
#define M823_BD_LONG            (1<<5)  /* Ethernet */
#define M823_BD_BREAK           (1<<5)  /* SCC UART, SMC UART */
#define M823_BD_NONALIGNED      (1<<4)  /* Ethernet */
#define M823_BD_FRAMING_ERROR   (1<<4)  /* SCC UART, SMC UART */
#define M823_BD_SHORT           (1<<3)  /* Ethernet */
#define M823_BD_PARITY_ERROR    (1<<3)  /* SCC UART, SMC UART */
#define M823_BD_CRC_ERROR       (1<<2)  /* Ethernet */
#define M823_BD_OVERRUN         (1<<1)  /* Ethernet, SCC UART, SMC UART, SPI */
#define M823_BD_COLLISION       (1<<0)  /* Ethernet */
#define M823_BD_CARRIER_LOST    (1<<0)  /* SCC UART, SMC UART */
#define M823_BD_MASTER_ERROR    (1<<0)  /* SPI */

/*
 * Bits in transmit buffer descriptor status word
 * Many bits have the same meaning as those in receiver buffer descriptors.
 */
#define M823_BD_READY           (1<<15) /* Ethernet, SCC UART, SMC UART, SPI */
#define M823_BD_PAD             (1<<14) /* Ethernet */
#define M823_BD_CTS_REPORT      (1<<11) /* SCC UART */
#define M823_BD_TX_CRC          (1<<10) /* Ethernet */
#define M823_BD_DEFER           (1<<9)  /* Ethernet */
#define M823_BD_HEARTBEAT       (1<<8)  /* Ethernet */
#define M823_BD_PREAMBLE        (1<<8)  /* SCC UART, SMC UART */
#define M823_BD_LATE_COLLISION  (1<<7)  /* Ethernet */
#define M823_BD_NO_STOP_BIT     (1<<7)  /* SCC UART */
#define M823_BD_RETRY_LIMIT     (1<<6)  /* Ethernet */
#define M823_BD_RETRY_COUNT(x)  (((x)&0x3C)>>2) /* Ethernet */
#define M823_BD_UNDERRUN        (1<<1)  /* Ethernet, SPI */
#define M823_BD_CARRIER_LOST    (1<<0)  /* Ethernet */
#define M823_BD_CTS_LOST        (1<<0)  /* SCC UART */

/*
*************************************************************************
*                     Serial Management Controllers                     *
*************************************************************************
*/
/*
 * Mode register
 */
#define M823_SMCMR_CLEN(x)              ((x)<<11)    /* Character length */
#define M823_SMCMR_2STOP                (1<<10)      /* 2 stop bits */
#define M823_SMCMR_PARITY               (1<<9)       /* Enable parity */
#define M823_SMCMR_EVEN                 (1<<8)       /* Even parity */
#define M823_SMCMR_SM_GCI               (0<<4)       /* GCI Mode */
#define M823_SMCMR_SM_UART              (2<<4)       /* UART Mode */
#define M823_SMCMR_SM_TRANSPARENT       (3<<4)       /* Transparent Mode */
#define M823_SMCMR_DM_LOOPBACK          (1<<2)       /* Local loopback mode */
#define M823_SMCMR_DM_ECHO              (2<<2)       /* Echo mode */
#define M823_SMCMR_TEN                  (1<<1)       /* Enable transmitter */
#define M823_SMCMR_REN                  (1<<0)       /* Enable receiver */


/*
*************************************************************************
*       RISC Communication Processor Module Command Register (CR)       *
*************************************************************************
*/
#define M823_CR_RST             (1<<15) /* Reset communication processor */
#define M823_CR_OP_INIT_RX_TX   (0<<8)  /* SCC2, USB, SMC UART, SMC GCI, SPI, I2C */
#define M823_CR_OP_INIT_RX      (1<<8)  /* SCC, USB, SMC UART, SPI, I2C */
#define M823_CR_OP_INIT_TX      (2<<8)  /* SCC, USB, SMC UART, SPI, I2C */
#define M823_CR_OP_INIT_HUNT    (3<<8)  /* SCC, USB, SMC UART */
#define M823_CR_OP_STOP_TX      (4<<8)  /* SCC, USB, SMC UART */
#define M823_CR_OP_GR_STOP_TX   (5<<8)  /* SCC, USB */
#define M823_CR_OP_INIT_IDMA    (5<<8)  /* IDMA */
#define M823_CR_OP_RESTART_TX   (6<<8)  /* SCC, USB, SMC UART */
#define M823_CR_OP_CLOSE_RX_BD  (7<<8)  /* SCC, USB, SMC UART, SPI, I2C */
#define M823_CR_OP_SET_GRP_ADDR (8<<8)  /* SCC */
#define M823_CR_OP_SET_TIMER    (8<<8)  /* Timer */
#define M823_CR_OP_GCI_TIMEOUT  (9<<8)  /* SMC GCI */
/* Josu: It's not in the MPC823 User's Guide #define M823_CR_OP_RESERT_BCS   (10<<8) * SCC */
#define M823_CR_OP_GCI_ABORT    (10<<8) /* SMC GCI */
#define M823_CR_OP_STOP_IDMA    (11<<8) /* IDMA */
#define M823_CR_OP_START_DSP    (12<<8) /* DSP */
#define M823_CR_OP_INIT_DSP     (13<<8) /* DSP */

/* Channel selection */
#define M823_CR_CHAN_I2C        (1<<4)
#define M823_CR_CHAN_IDMA1      (1<<4)
#define M823_CR_CHAN_SCC2       (4<<4)
#define M823_CR_CHAN_SPI        (5<<4)
#define M823_CR_CHAN_IDMA2      (5<<4)
#define M823_CR_CHAN_TIMER      (5<<4)
#define M823_CR_CHAN_SMC1       (9<<4)
#define M823_CR_CHAN_DSP1       (9<<4)
#define M823_CR_CHAN_SMC2       (13<<4)
#define M823_CR_CHAN_DSP2       (13<<4)
#define M823_CR_FLG             (1<<0)  /* Command flag */

/*
 * Receive and transmit function code register bits
 * These apply to the function code registers of all devices, not just SCC.
 */
#define M823_RFCR_BO(x)         ((x)<<3)
#define M823_RFCR_MOT           (2<<3)
#define M823_RFCR_DMA_SPACE(x)  (x)
#define M823_TFCR_BO(x)         ((x)<<3)
#define M823_TFCR_MOT           (2<<3)
#define M823_TFCR_DMA_SPACE(x)  (x)

#define M823_PISCR_PIRQ(x)      (1<<(15-x))  /* PIT interrupt level */
#define M823_PISCR_PS           (1<<7)    /* PIT Interrupt state */
#define M823_PISCR_PIE          (1<<2)    /* PIT interrupt enable */
#define M823_PISCR_PITF         (1<<1)    /* Stop timer when freeze asserted */
#define M823_PISCR_PTE          (1<<0)    /* PIT enable */

#define M823_TBSCR_TBIRQ(x)     (1<<(15-x))  /* TB interrupt level */
#define M823_TBSCR_REFA         (1<<7)    /* TB matches TBREFF0 */
#define M823_TBSCR_REFB         (1<<6)    /* TB matches TBREFF1 */
#define M823_TBSCR_REFAE        (1<<3)    /* Enable ints for REFA */
#define M823_TBSCR_REFBE        (1<<2)    /* Enable ints for REFB */
#define M823_TBSCR_TBF          (1<<1)    /* TB stops on FREEZE */
#define M823_TBSCR_TBE          (1<<0)    /* enable TB and decrementer */

#define M823_SIMASK_IRM0        (1<<31)
#define M823_SIMASK_LVM0        (1<<30)
#define M823_SIMASK_IRM1        (1<<29)
#define M823_SIMASK_LVM1        (1<<28)
#define M823_SIMASK_IRM2        (1<<27)
#define M823_SIMASK_LVM2        (1<<26)
#define M823_SIMASK_IRM3        (1<<25)
#define M823_SIMASK_LVM3        (1<<24)
#define M823_SIMASK_IRM4        (1<<23)
#define M823_SIMASK_LVM4        (1<<22)
#define M823_SIMASK_IRM5        (1<<21)
#define M823_SIMASK_LVM5        (1<<20)
#define M823_SIMASK_IRM6        (1<<19)
#define M823_SIMASK_LVM6        (1<<18)
#define M823_SIMASK_IRM7        (1<<17)
#define M823_SIMASK_LVM7        (1<<16)

extern volatile EPPC m823;

#endif /* ASM */

#endif /* _MPC823_H */
