/*  clock.c
 *
 *  This routine initializes the PIT on the MPC823.
 *  The tick frequency is specified by the bsp.
 *
 *  Author: Josu Onandia (jonandia@fagorautomation.es)
 *  Copyright (C) 2001 by Fagor Automation S. Coop.
 *  Derived from c/src/lib/libcpu/powerpc/mpc860/clock/clock.c:
 *
 *  Author: Jay Monkman (jmonkman@frasca.com)
 *  Copyright (C) 1998 by Frasca International, Inc.
 *
 *  Derived from c/src/lib/libcpu/ppc/ppc403/clock/clock.c:
 *
 *  Author: Andrew Bray <andy@i-cubed.co.uk>
 *
 *  COPYRIGHT (c) 1995 by i-cubed ltd.
 *
 *  To anyone who acknowledges that this file is provided "AS IS"
 *  without any express or implied warranty:
 *      permission to use, copy, modify, and distribute this file
 *      for any purpose is hereby granted without fee, provided that
 *      the above copyright notice and this notice appears in all
 *      copies, and that the name of i-cubed limited not be used in
 *      advertising or publicity pertaining to distribution of the
 *      software without specific, written prior permission.
 *      i-cubed limited makes no representations about the suitability
 *      of this software for any purpose.
 *
 *  Derived from c/src/lib/libcpu/hppa1_1/clock/clock.c:
 *
 *  COPYRIGHT (c) 1989-1999.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.OARcorp.com/rtems/license.html.
 *
 *  $Id: clock.c,v 1.6 2000/01/11 17:34:20 joel Exp $
 */

#include <rtems.h>
#include <clockdrv.h>
#include <rtems/libio.h>

#include <stdlib.h>                     /* for atexit() */
#include <mpc823.h>

volatile rtems_unsigned32 Clock_driver_ticks;

void Clock_exit( void );
 
/*
 * These are set by clock driver during its init
 */
 
rtems_device_major_number rtems_clock_major = ~0;
rtems_device_minor_number rtems_clock_minor;

/*
 *  ISR Handler
 */
rtems_isr Clock_isr(rtems_vector_number vector)
{
  m823.simt.piscr |= M823_PISCR_PS;
  Clock_driver_ticks++;
  rtems_clock_tick();
}

void Install_clock(rtems_isr_entry clock_isr)
{
  rtems_isr_entry previous_isr;
  rtems_unsigned32 pit_value;
  rtems_unsigned64 clk_freq;

  Clock_driver_ticks = 0;

/*
 * This algorithm is not accurate enough on FADS823 board,
 * so we use new one instead.
 */
#if 0
  pit_value = rtems_configuration_get_microseconds_per_tick() /
               rtems_cpu_configuration_get_clicks_per_usec();
#endif

/*
 * Note: FADS823 board has two different clock input sources,
 *       which is selected via DIP switch setting.
 */
#define BIT_RTSEL   (1 << 23)
#define BIT_RTDIV   (1 << 24)
#define OSCM_FREQ     32768 /* crystal oscillator frequency in Hz */
#define EXTCLK_FREQ 4000000 /* external clock frequency in Hz */
  /*
   * Check RTSEL bit of SCCR to determine input clock frequency.
   */
  if (m823.clkr.sccr & BIT_RTSEL) {
    clk_freq = EXTCLK_FREQ;
  } else {
    clk_freq = OSCM_FREQ;
  }

  /*
   * Clear RTDIV bit of SCCR to force divide ratio to 4.
   */
  m823.clkr.sccr &= ~BIT_RTDIV;

  /*
   * The strange equation is actually used to round the number using integer arithmetic,
   * which is equivalent to the floating point equation:
   *     pit_value = usec_per_tick / (usec_per_sec / (clk_freq / div_ratio)) + 0.5
   */
  pit_value = ((rtems_configuration_get_microseconds_per_tick() * clk_freq * 2) / (4 * RTEMS_MILLISECONDS_TO_MICROSECONDS(1000)) + 1) >> 1;
  if (pit_value == 0) {
    pit_value = 0xffff;
  } else {
    pit_value--;
  }

  /*
   * PIT counter is only 16 bits.
   */
  if (pit_value > 0xffff) {
    rtems_fatal_error_occurred(-1);
  }  

  /*
   * Install PIT interrupt handler.
   */
  rtems_interrupt_catch(clock_isr, PPC_IRQ_LVL3, &previous_isr);

  /*
   * Set PIT Counter register.
   */
  /* Josu: This register is ill-defined in mpc823.h. Although
   * is defined as 32 bits, only upper 16 bits count
   * m823.simt.pitc = pit_value;
   */
  m823.simt.pitc = pit_value << 16;

  /*
   * Set PIT irq level, enable PIT, PIT interrupts and clear interrupt status.
   */
  m823.simt.piscr = M823_PISCR_PIRQ(3) | M823_PISCR_PTE | M823_PISCR_PITF | M823_PISCR_PS | M823_PISCR_PIE;
  m823.siu.simask |= M823_SIMASK_LVM3;

  /*
   * Register Clock_exit() to atexit().
   */
  atexit(Clock_exit);
}

void
ReInstall_clock(rtems_isr_entry new_clock_isr)
{
  rtems_isr_entry previous_isr;
  rtems_unsigned32 isrlevel = 0;
  
  rtems_interrupt_disable(isrlevel);
  
  rtems_interrupt_catch(new_clock_isr, PPC_IRQ_LVL3, &previous_isr);
  
  rtems_interrupt_enable(isrlevel);
}


/*
 * Called via atexit()
 * Remove the clock interrupt handler by setting handler to NULL
 */
void
Clock_exit(void)
{
  /* disable PIT and PIT interrupts */
  m823.simt.piscr &= ~(M823_PISCR_PTE | M823_PISCR_PIE); 

  (void) set_vector(0, PPC_IRQ_LVL3, 1);
}

rtems_device_driver Clock_initialize(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void *pargp
)
{
  Install_clock( Clock_isr );
 
  /*
   * make major/minor avail to others such as shared memory driver
   */
 
  rtems_clock_major = major;
  rtems_clock_minor = minor;
 
  return RTEMS_SUCCESSFUL;
}
 
rtems_device_driver Clock_control(
  rtems_device_major_number major,
  rtems_device_minor_number minor,
  void *pargp
)
{
  rtems_libio_ioctl_args_t *args = pargp;
  
  if (args == 0)
    goto done;
  
  /*
   * This is hokey, but until we get a defined interface
   * to do this, it will just be this simple...
   */
  
  if (args->command == rtems_build_name('I', 'S', 'R', ' ')) {
    Clock_isr(PPC_IRQ_LVL3);
  }
  else if (args->command == rtems_build_name('N', 'E', 'W', ' ')) {
    ReInstall_clock(args->buffer);
  }
  
 done:
  return RTEMS_SUCCESSFUL;
}
