#define r3	3	/* General Purpose Register 3 */
#define r4	4	/* General Purpose Register 4 */
#define r5	5	/* General Purpose Register 5 */

#define LR	8	/* Link Register */
#define IC_CST	560	/* ICache Control Status Register */
#define DC_CST	568	/* DCache Control Status Register */
#define MI_CTR	784	/* MMU Instruction Control Register */
#define MD_CTR	792	/* MMU Data Control Register */
#define MD_AP	794	/* MMU Data Access Permission Register */
#define MD_EPN	795	/* MMU Data Effective Page Number Register */
#define MD_TWC	797	/* MMU Data Tablewalk Control Register */
#define MD_RPN	798	/* MMU Instruction Real Page Number Register */
#define MI_AP	786	/* MMU Instruction Access Permission Register */
#define MI_EPN	787	/* MMU Instruction Effective Page Number Reg */
#define MI_TWC	789	/* MMU Instruction Tablewalk Control Register */
#define MI_RPN	790	/* MMU Instruction Real Page Number Register */

#define CacheUnlockAllCmd	0x0A00	/* Cache Unlock_All command */
#define CacheDisableCmd		0x0400	/* Cache Disable command */
#define CacheInvAllCmd		0x0C00	/* Cache Invalidate_All command */
#define CacheEnableCmd		0x0200	/* DCache_Enable Command */

   .section .text,"ax",@progbits
   .globl cache_mmu_init
/*
#########################################################################
# Function: cache_mmu_init
#
# Description: Initializes the Data/Instruction Caches and MMU.
#########################################################################			
*/
cache_mmu_init:
  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Save the Link Register value to r5
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */
   mfspr    r5,LR

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Disable Instruction and Data Translation mode in MSR
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */
   mfmsr    r3
   addis    r4,0,0x0030
   andc     r3,r3,r4
   mtmsr    r3

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Load r4 with 0. Note it will always be 0 in the module.
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */
   addis    r4,0,0

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Instruction Cache Initialization. Note that Instruction Cache will be
   # enabled after the MMU is programmed.
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */

  /*
   #-----------------------------------------------------------------------
   # Program the Instruction Cache Control and Status Register
   #
   # - IEN (Instruction Cache Enable Status Bit) = 0 = Disabled
   #
   # - CMD (Data Cache Commands) = 0A00 = Unlock all cache lines and ways
   #
   # - All other bits are status related and are read only
   #-----------------------------------------------------------------------
   */
   addis    r3,0,CacheUnlockAllCmd
   mtspr    IC_CST,r3             
   isync

  /*
   #-----------------------------------------------------------------------
   # Program the Instruction Cache Control and Status Register
   #
   # - IEN (Instruction Cache Enable Status Bit) = 0 = Disabled
   #
   # - CMD (Data Cache Commands) = 0C00 = Invalidate all cache
   #
   # - All other bits are status related and are read only
   #-----------------------------------------------------------------------
   */
   addis    r3,0,CacheInvAllCmd
   mtspr    IC_CST,r3
   isync

  /*
   #-----------------------------------------------------------------------
   # Program the Instruction Cache Control and Status Register
   #
   # - IEN (Instruction Cache Enable Status Bit) = 0 = Disabled
   #
   # - CMD (Data Cache Commands) = 0400 = Disable the Instruction Cache
   #
   # - All other bits are status related and are read only
   #-----------------------------------------------------------------------
   */
   addis    r3,0,CacheDisableCmd  
   mtspr    IC_CST,r3
   isync

  /*
   #~~~~~~~~~~~~~~~~~~~~
   # Initialize the MMU
   #~~~~~~~~~~~~~~~~~~~~
   */
   bl       mmu_init

  /*
   #-----------------------------------------------------------------------
   # Program the Instruction Cache Control and Status Register
   #
   # - IEN (Instruction Cache Enable Status Bit) = will be enabled
   #
   # - CMD (Data Cache Commands) = 0200 = enable the Instruction cache
   #
   # - All other bits are status related and are read only
   #-----------------------------------------------------------------------
   */
   addis    r3,0,CacheEnableCmd
   mtspr    IC_CST,r3
   isync

  /*
   #----------------------------
   # return from cache_mmu_init
   #----------------------------
   */
   mtspr    LR,r5
   bclr     20,0

/*
###########################################################################
# Function: mmu_init
#
# Description: 
#
# This function initializes the Memory Management Unit. This function does
# not define the complete 4 GByte address range. It also does not account 
# for TLB misses and the exception to handle misses is not in this example.
# Only certain memory spaces have been defined and stored in the TLB table. 
# Obviously for a robust system, all address space would be accounted for
# and the user would take care in not accessing undefined space or memory
# locations not define in the TLB table, if the TLB table were the only
# MMU mechanism being used. This MMU example is only provided to give the 
# user an idea of how to initialize the MMU in designs with simple memory
# mapping. To get a sense of how we came up with these MMU definition, the
# user should consult the MPC860ADS hardware manual.
###########################################################################			
*/
mmu_init:
  /*
   #-------------------------------------------------------------------------
   # Invalidate all TLB entries: This needs to be done because Translation
   # Lookaside Buffers values are undefined. The MMU comes out of reset
   # disabled but if the MMU were enabled and TLB's were not invalidated 
   # first, erratic behavior would result.
   #-------------------------------------------------------------------------
   */
   tlbia
   isync

  /*
   #-------------------------------------------------------------------------
   # Program the MMU Instruction Control Register with the following values
   #
   # - GPM (Group Protection Mode) = 0 = PowerPC Mode
   #
   # - PPM (Page Protection Mode) = 0 = Page resolution protection
   #
   # - CIDEF (CI (Cache Inhibit) Default) = 0 = This is the instruction cache 
   #   attribute default value when the data MMU is disabled (MSR[dr]).
   #
   # - RSV2I (Reserve Two instruction TLB Entries) = 0 = ITLB_INDX 
   #   decremented modulo 8.
   #   
   # - PPCS (Privilege/Problem State Compare Mode) = 0 = Ignore problem/
   #   privilege state during address compare.
   #
   # - ITLB_INDX (Instruction TLB Index) = 31
   #-------------------------------------------------------------------------
   */
   addi     r3,0,0x1F00
   mtspr    MI_CTR,r3

  /*
   #---------------------------------------------------
   # Access Permission defined by page protection bits
   #---------------------------------------------------
   */
   oris     r3,r4,0x5555
   ori      r3,r3,0x5555
   mtspr    MI_AP,r3

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # DRAM first 8M: program memory
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */

  /*
   #-------------------------------------------------------------------------
   # Program the MI_EPN (MMU Instruction Effective Page Number Register) with
   # the following values:
   #
   # - EPN (Effective Page Number) = 0x00000 = establishing the 
   #   base address for the 1st 8 MByte DRAM bank.
   #
   # - EV (TLB Entry Valid Bit = 1 = The Instruction TLB is valid.
   #
   # - ASID (Address Space ID) = 0x0 = Address space ID of the data TLB Entry
   #   to be compared with the CASID field in the M_CASID register.
   #-------------------------------------------------------------------------
   */
   oris     r3,r4,0x0000
   addi     r3,r3,0x0200
   mtspr    MI_EPN,r3

  /*
   #-------------------------------------------------------------------------
   # Program the MI_TWC (MMU Instruction Tablewalk Control Register) with the
   # following values:
   #
   # - L2TB (Tablewalk Level 2 Base Value) = 0 = These are the most 
   #   significant bits of the level two pointer.
   #
   # - APG (Access Protection Group) = 0 = Kept at reset value.
   #
   # - G (Guarded) = 0 = Unguarded storage
   #
   # - PS (Page Size) = 11 = 8 Mbyte
   #
   # - V (Valid) = 1 = This entry is valid
   #-------------------------------------------------------------------------
   */
   addi     r3,0,0x000D
   mtspr    MI_TWC,r3

  /*
   #-------------------------------------------------------------------------
   # Program the MI_RPN (MMU Instruction Real Page Number Register) with the
   # following values:
   #
   # - RPN (Real Page Number) = 0x00000
   #
   # - PP (Protection Page) = 11 0 0 11 11 b = Because PPM = 0, PP takes on 
   #   following meaning:
   #
   #   --> Privileged     Problem  
   #       ----------     -------
   #       executable    executable
   #
   #   --> Contains PowerPC Encoding
   #
   #   --> all 4 subpages are valid
   #
   # - LPS (Large Page Size) = 1 = 16K or greater
   #
   # - SH (Shared Page) = 1 = ASID Comparison is Disabled for the entry.
   #
   # - CI (Cache Inhibit) = 0 = Data Cache is not inhibited
   #
   # - V (Entry Valid Indication) = 1 = valid
   #
   #-------------------------------------------------------------------------
   */
   oris     r3,r4,0x0000
   ori      r3,r3,0x0CFD
   mtspr    MI_RPN,r3
   isync

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Flash 8M, Read only Instructions
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */

  /*
   #-------------------------------------------------------------------------
   # Program the MI_EPN (MMU Instruction Effective Page Number Register) with
   # the following values:
   #
   # - EPN (Effective Page Number) = 0x02800 = establishing the 
   #   base address for the the complete flash range on this target.
   #
   # - EV (TLB Entry Valid Bit = 1 = The Instruction TLB is valid.
   #
   # - ASID (Address Space ID) = 0x0 = Address space ID of the data TLB Entry
   #   to be compared with the CASID field in the M_CASID register.
   #-------------------------------------------------------------------------
   */
   oris     r3,r4,0x0280
   ori      r3,r3,0x0200
   mtspr    MI_EPN,r3       

  /*
   #-------------------------------------------------------------------------
   # Program the MI_TWC (MMU Instruction Tablewalk Control Register) with the
   # following values:
   #
   # - L2TB (Tablewalk Level 2 Base Value) = 0 = These are the most 
   #   significant bits of the level two pointer.
   #
   # - APG (Access Protection Group) = 0 = Kept at reset value.
   #
   # - G (Guarded) = 0 = Unguarded storage
   #
   # - PS (Page Size) = 11 = 8 Mbyte
   #
   # - V (Valid) = 1 = This entry is valid
   #-------------------------------------------------------------------------
   */
   addi     r3,0,0x000D
   mtspr    MI_TWC,r3

  /*
   #-------------------------------------------------------------------------
   # Program the MI_RPN (MMU Instruction Real Page Number Register) with the
   # following values:
   #
   # - RPN (Real Page Number) = 0x02800
   #
   # - PP (Protection Page) = 11 0 0 11 11 b = Because PPM = 0, PP takes on 
   #   following meaning:
   #
   #   --> Privileged     Problem  
   #       ----------     -------
   #       executable    executable
   #
   #   --> Contains PowerPC Encoding
   #
   #   --> all 4 subpages are valid
   #
   # - LPS (Large Page Size) = 1 = 16K 
   #
   # - SH (Shared Page) = 1 = ASID Comparison is Disabled for the entry.
   #
   # - CI (Cache Inhibit) = 0 = Data Cache is not inhibited
   #
   # - V (Entry Valid Indication) = 1 = valid
   #
   #-------------------------------------------------------------------------
   */
   oris     r3,r4,0x0280
   ori      r3,r3,0x0CFD
   mtspr    MI_RPN,r3
   isync

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Enable Instruction Translation mode in MSR
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */
   mfmsr    r3
   ori      r3,r3,0x0020
   mtmsr    r3

  /*
   #----------------------
   # return from mmu_init
   #----------------------
   */
   bclr     20,0
